#!/bin/bash
# vim: dict+=/usr/share/beakerlib/dictionary.vim cpt=.,w,b,u,t,i,k
. /usr/share/beakerlib/beakerlib.sh || exit 1

rlJournalStart
    rlPhaseStartSetup
        export REPO="$(pwd)/../.."
        export REV="$(git rev-parse --short HEAD)"
        rlRun "set -o pipefail"
        if ! command -v pre-commit ; then
          if test -e /root/.local/bin/pre-commit; then
            rlRun "export PATH=/root/.local/bin:$PATH" 0 "Fix PATH to include /root/.local/bin with pre-commit"
          else
            rlFail "pre-commit is not on PATH nor in /root/.local/bin"
          fi
        fi
    rlPhaseEnd

for WHAT in "tests-" "plans-" "stories-" ''; do
    hook="tmt-${WHAT}lint"

    rlPhaseStartTest "Hook: $hook"
        rlRun "tmp=\$(mktemp -d)" 0 "Creating tmp directory"
        rlRun "pushd $tmp"

        rlRun "tmt init"
        rlRun "git init"
        rlRun "git config --local user.name LZachar"
        rlRun "git config --local user.email lzachar@redhat.com"
cat <<EOF > .pre-commit-config.yaml
repos:
  - repo: $REPO
    rev: "$REV"
    hooks:
    - id: "$hook"
EOF
        if [ -n "$WHAT" ]; then
            expected_command="tmt ${WHAT%-} lint"
        else
            expected_command="tmt lint"
        fi
        rlRun "cat .pre-commit-config.yaml"
        rlRun -s "pre-commit install"
        rlAssertGrep 'pre-commit installed' $rlRun_LOG
        rlRun -s "git add .pre-commit-config.yaml"
        rlRun -s "git commit -m nothing_to_check"
        # No *fmf file modified
        rlAssertGrep "$expected_command.*no files to check" $rlRun_LOG

        case $WHAT in
            tests-|"")
                rlRun "echo 'test: echo' > main.fmf"
                ;;
            stories-)
                rlRun "echo 'story: whatever' > main.fmf"
                ;;
            plans-)
                rlRun "echo -e 'execute:\n  how: tmt' > main.fmf"
                ;;
        esac

        rlRun "git add main.fmf"
        rlRun -s "git commit -m 'missing_fmf_root'" "1"
        # .fmf/version was not added
        rlAssertGrep "$expected_command.*Failed" $rlRun_LOG

        rlRun "git add .fmf/version"
        rlRun -s "git commit -m 'pass'"
        # All good
        rlAssertGrep "$expected_command.*Passed" $rlRun_LOG

        rlRun "echo foo: bar > wrong.fmf"
        rlRun "git add wrong.fmf"
        rlRun -s "git commit -m wrong" "1"
        # Test uses invalid attribute
        rlAssertGrep "$expected_command.*Failed" $rlRun_LOG
        rlAssertGrep 'fail .001 unknown key "foo" is used' $rlRun_LOG

        # Force broken test into repo
        rlRun -s "git commit --no-verify -m wrong" "0"

        # Add another good test, pre-commit should pass because /wrong
        # is not touched
        rlRun "echo summary: hello world > good.fmf"
        rlRun "git add good.fmf"
        rlRun -s "git commit -m 'add_good'"
        rlAssertGrep "$expected_command.*Passed" $rlRun_LOG

        # Modify main.fmf so /wrong is checked
        # /good is checked as well but since it passes it is not reported
        rlRun "echo summary: foo >> main.fmf"
        rlRun -s "git commit -a -m 'modify_main'" "1"
        rlAssertGrep "$expected_command.*Failed" $rlRun_LOG
        rlAssertNotGrep '/good' $rlRun_LOG
        rlAssertGrep '/wrong' $rlRun_LOG

        # Check it works in different fmf_root
        rlRun "mkdir tmt_root"
        rlRun -s "git mv .fmf/ tmt_root/.fmf/"
        # Should fail because fmf_root is in different location
        # (Also tests that pre-commit runs when .fmf/version is changed)
        rlRun -s "git commit -m 'wrong_fmf_root'" "1"
        rlAssertGrep "$expected_command.*Failed" $rlRun_LOG
        # Add the correct root
cat <<EOF > .pre-commit-config.yaml
repos:
  - repo: $REPO
    rev: "$REV"
    hooks:
    - id: "$hook"
      args: [ --root, tmt_root ]
EOF
        rlRun -s "git add .pre-commit-config.yaml"
        rlRun -s "git mv main.fmf tmt_root"
        rlRun -s "git commit -m 'pass different root'"
        rlAssertGrep "$expected_command.*Passed" $rlRun_LOG

        rlRun "popd"
        rlRun "rm -rf $tmp" 0 "Removing tmp directory"
    rlPhaseEnd
done
rlJournalEnd
