package v1alpha1

import (
	"context"
	"fmt"

	"go.starlark.net/starlark"
	"k8s.io/apimachinery/pkg/util/validation/field"

	"github.com/tilt-dev/tilt/internal/controllers/apicmp"
	"github.com/tilt-dev/tilt/internal/controllers/apiset"
	"github.com/tilt-dev/tilt/internal/tiltfile/starkit"
	"github.com/tilt-dev/tilt/pkg/apis/core/v1alpha1"
)

// Defines starlark functions that register API objects
// from the v1alpha1 API.
//
// Eventually, all the files in this module should be autogenerated from
// the data models.
type Plugin struct {
}

func NewPlugin() Plugin {
	return Plugin{}
}

func (e Plugin) NewState() interface{} {
	return apiset.ObjectSet{}
}

func (p Plugin) OnStart(env *starkit.Environment) error {
	return p.registerSymbols(env)
}

// Register an API object, so that it's reconciled against the API server when
// tiltfile execution completes.
func (p Plugin) register(t *starlark.Thread, obj apiset.Object) (starlark.Value, error) {
	objV, ok := obj.(validator)
	if ok {
		err := objV.Validate(context.TODO())
		if err != nil {
			return nil, err.ToAggregate()
		}
	}

	err := starkit.SetState(t, func(set apiset.ObjectSet) (apiset.ObjectSet, error) {
		typedSet := set.GetOrCreateTypedSet(obj)
		name := obj.GetName()
		existing, exists := typedSet[name]
		if exists {
			// tiltextension adds this for extensions, but new objects will not
			// have this annotation yet, so delete for comparison
			delete(existing.GetAnnotations(), v1alpha1.AnnotationManagedBy)
			if !apicmp.DeepEqual(obj, existing) {
				return set, fmt.Errorf("%s %q already registered", obj.GetGroupVersionResource().Resource, name)
			}
		}

		typedSet[name] = obj
		return set, nil
	})
	if err != nil {
		return nil, err
	}
	return starlark.None, nil
}

var _ starkit.StatefulPlugin = Plugin{}

func MustState(model starkit.Model) apiset.ObjectSet {
	state, err := GetState(model)
	if err != nil {
		panic(err)
	}
	return state
}

func GetState(m starkit.Model) (apiset.ObjectSet, error) {
	var state apiset.ObjectSet
	err := m.Load(&state)
	return state, err
}

type validator interface {
	Validate(ctx context.Context) field.ErrorList
}
