/*
    Copyright (C) 2022 Tenable, Inc.

	Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

		http://www.apache.org/licenses/LICENSE-2.0

	Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

package vulnerability

import (
	"reflect"
	"testing"
)

func TestGetImageDetails(t *testing.T) {
	type args struct {
		image        string
		imageDetails ImageDetails
	}
	tests := []struct {
		name string
		args args
		want ImageDetails
	}{
		{
			name: "gcr image without tag and digest",
			args: args{
				image:        "gcr.io/terrascan-test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "gcr.io",
				Repository: "terrascan-test",
			},
		},
		{
			name: "gcr image with tag and digest",
			args: args{
				image:        "gcr.io/terrascan-test:test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "gcr.io",
				Repository: "terrascan-test",
				Tag:        "test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "gcr image with tag",
			args: args{
				image:        "gcr.io/terrascan-test:test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "gcr.io",
				Repository: "terrascan-test",
				Tag:        "test",
			},
		},
		{
			name: "gcr image with Digest",
			args: args{
				image:        "gcr.io/terrascan-test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "gcr.io",
				Repository: "terrascan-test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "ecr image without tag and digest",
			args: args{
				image:        "245578968.dkr.ecr.us-east-2.amazonaws.com/terrascan-test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "245578968.dkr.ecr.us-east-2.amazonaws.com",
				Repository: "terrascan-test",
			},
		},
		{
			name: "ecr image with tag and digest",
			args: args{
				image:        "245578968.dkr.ecr.us-east-2.amazonaws.com/terrascan-test:test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "245578968.dkr.ecr.us-east-2.amazonaws.com",
				Repository: "terrascan-test",
				Tag:        "test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "ecr image with tag",
			args: args{
				image:        "245578968.dkr.ecr.us-east-2.amazonaws.com/terrascan-test:test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "245578968.dkr.ecr.us-east-2.amazonaws.com",
				Repository: "terrascan-test",
				Tag:        "test",
			},
		},
		{
			name: "ecr image with  digest",
			args: args{
				image:        "245578968.dkr.ecr.us-east-2.amazonaws.com/terrascan-test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "245578968.dkr.ecr.us-east-2.amazonaws.com",
				Repository: "terrascan-test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "acr image without tag and digest",
			args: args{
				image:        "test.azurecr.io/terrascan-test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "test.azurecr.io",
				Repository: "terrascan-test",
			},
		},
		{
			name: "acr image with tag and digest",
			args: args{
				image:        "test.azurecr.io/terrascan-test:test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "test.azurecr.io",
				Repository: "terrascan-test",
				Tag:        "test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "acr image with tag",
			args: args{
				image:        "test.azurecr.io/terrascan-test:test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "test.azurecr.io",
				Repository: "terrascan-test",
				Tag:        "test",
			},
		},
		{
			name: "acr image with Digest",
			args: args{
				image:        "test.azurecr.io/terrascan-test@sha256:3f13b4376446cf92b0cb9a5c4",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "test.azurecr.io",
				Repository: "terrascan-test",
				Digest:     "sha256:3f13b4376446cf92b0cb9a5c4",
			},
		},
		{
			name: "harbor image with tag",
			args: args{
				image:        "terrascan.com/test/terrascan:test",
				imageDetails: ImageDetails{},
			},
			want: ImageDetails{
				Registry:   "terrascan.com",
				Repository: "test/terrascan",
				Tag:        "test",
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := GetImageDetails(tt.args.image, tt.args.imageDetails); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("GetImageDetails() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGetDomain(t *testing.T) {
	type args struct {
		image string
	}
	tests := []struct {
		name string
		args args
		want string
	}{
		{name: "get gcr image domain",
			args: args{image: "gcr.io/terrascan-test"},
			want: "gcr.io",
		},
		{name: "get acr image domain",
			args: args{image: "test.azurecr.io/terrascan-test"},
			want: "test.azurecr.io",
		},
		{name: "get ecr image domain",
			args: args{image: "245578968.dkr.ecr.us-east-2.amazonaws.com/terrascan-test"},
			want: "245578968.dkr.ecr.us-east-2.amazonaws.com",
		},
		{name: "invalid image domain reference",
			args: args{image: ""},
			want: "",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := GetDomain(tt.args.image); got != tt.want {
				t.Errorf("GetDomain() = %v, want %v", got, tt.want)
			}
		})
	}
}
