// Copyright 2019 The Terraformer Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package okta

import (
	"github.com/GoogleCloudPlatform/terraformer/terraformutils"
	"github.com/okta/okta-sdk-golang/v2/okta"
)

type AuthorizationServerPolicyRuleGenerator struct {
	OktaService
}

func (g AuthorizationServerPolicyRuleGenerator) createResources(authorizationServerPolicyRuleList []*okta.AuthorizationServerPolicyRule, authorizationServerID string, authorizationServerName string, authorizationServerPolicyID string, authorizationServerPolicyName string) []terraformutils.Resource {
	var resources []terraformutils.Resource

	for _, authorizationServerPolicyRule := range authorizationServerPolicyRuleList {
		resources = append(resources, terraformutils.NewResource(
			authorizationServerPolicyRule.Id,
			normalizeResourceName("auth_server_"+authorizationServerName+"_policy_"+authorizationServerPolicyName+"_rule_"+authorizationServerPolicyRule.Name),
			"okta_auth_server_policy_rule",
			"okta",
			map[string]string{
				"auth_server_id": authorizationServerID,
				"policy_id":      authorizationServerPolicyID,
			},
			[]string{},
			map[string]interface{}{},
		))
	}
	return resources
}

func (g *AuthorizationServerPolicyRuleGenerator) InitResources() error {
	var resources []terraformutils.Resource
	ctx, client, e := g.Client()
	if e != nil {
		return e
	}

	authorizationServers, err := getAuthorizationServers(ctx, client)
	if err != nil {
		return err
	}

	for _, authorizationServer := range authorizationServers {

		authorizationServerPolicies, _, err := client.AuthorizationServer.ListAuthorizationServerPolicies(ctx, authorizationServer.Id)
		if err != nil {
			return err
		}

		for _, authorizationServerPolicy := range authorizationServerPolicies {
			output, _, err := client.AuthorizationServer.ListAuthorizationServerPolicyRules(ctx, authorizationServer.Id, authorizationServerPolicy.Id)
			if err != nil {
				return err
			}

			resources = append(resources, g.createResources(output, authorizationServer.Id, authorizationServer.Name, authorizationServerPolicy.Id, authorizationServerPolicy.Name)...)
		}
	}

	g.Resources = resources
	return nil
}
