package instance

import (
	"context"
	"sort"
	"strings"

	"github.com/fatih/color"
	"github.com/scaleway/scaleway-cli/internal/core"
	"github.com/scaleway/scaleway-cli/internal/human"
	"github.com/scaleway/scaleway-sdk-go/api/instance/v1"
	"github.com/scaleway/scaleway-sdk-go/scw"
)

//
// Marshalers
//

var (
	serverTypesAvailabilityMarshalSpecs = human.EnumMarshalSpecs{
		instance.ServerTypesAvailabilityAvailable: &human.EnumMarshalSpec{Attribute: color.FgGreen},
		instance.ServerTypesAvailabilityScarce:    &human.EnumMarshalSpec{Attribute: color.FgYellow, Value: "low stock"},
		instance.ServerTypesAvailabilityShortage:  &human.EnumMarshalSpec{Attribute: color.FgRed, Value: "out of stock"},
	}
)

//
// Builders
//

// serverTypeListBuilder transforms the server map into a list to display a
// table of server types instead of a flat key/value list.
// We need it for:
// - [APIGW-1932] hide deprecated instance for scw instance server-type list
func serverTypeListBuilder(c *core.Command) *core.Command {
	deprecatedNames := map[string]struct{}{
		"START1-L":    {},
		"START1-M":    {},
		"START1-S":    {},
		"START1-XS":   {},
		"VC1L":        {},
		"VC1M":        {},
		"VC1S":        {},
		"X64-120GB":   {},
		"X64-15GB":    {},
		"X64-30GB":    {},
		"X64-60GB":    {},
		"C1":          {},
		"C2M":         {},
		"C2L":         {},
		"C2S":         {},
		"ARM64-2GB":   {},
		"ARM64-4GB":   {},
		"ARM64-8GB":   {},
		"ARM64-16GB":  {},
		"ARM64-32GB":  {},
		"ARM64-64GB":  {},
		"ARM64-128GB": {},
	}

	c.Run = func(ctx context.Context, argsI interface{}) (interface{}, error) {
		type customServerType struct {
			Name            string                           `json:"name"`
			MonthlyPrice    *scw.Money                       `json:"monthly_price"`
			HourlyPrice     *scw.Money                       `json:"hourly_price"`
			LocalVolumeSize scw.Size                         `json:"local_volume_size"`
			CPU             uint32                           `json:"cpu"`
			GPU             *uint64                          `json:"gpu"`
			RAM             scw.Size                         `json:"ram"`
			Arch            instance.Arch                    `json:"arch"`
			Availability    instance.ServerTypesAvailability `json:"availability"`
		}

		api := instance.NewAPI(core.ExtractClient(ctx))

		// Get server types.
		request := argsI.(*instance.ListServersTypesRequest)
		listServersTypesResponse, err := api.ListServersTypes(request)
		if err != nil {
			return nil, err
		}
		serverTypes := []*customServerType(nil)

		// Get server availabilities.
		availabilitiesResponse, err := api.GetServerTypesAvailability(&instance.GetServerTypesAvailabilityRequest{})
		if err != nil {
			return nil, err
		}

		for name, serverType := range listServersTypesResponse.Servers {
			_, isDeprecated := deprecatedNames[name]
			if isDeprecated {
				continue
			}

			serverTypes = append(serverTypes, &customServerType{
				Name:            name,
				MonthlyPrice:    scw.NewMoneyFromFloat(float64(serverType.MonthlyPrice), "EUR", 2),
				HourlyPrice:     scw.NewMoneyFromFloat(float64(serverType.HourlyPrice), "EUR", 3),
				LocalVolumeSize: serverType.VolumesConstraint.MinSize,
				CPU:             serverType.Ncpus,
				GPU:             serverType.Gpu,
				RAM:             scw.Size(serverType.RAM),
				Arch:            serverType.Arch,
				Availability:    availabilitiesResponse.Servers[name].Availability,
			})
		}

		sort.Slice(serverTypes, func(i, j int) bool {
			categoryA := serverTypeCategory(serverTypes[i].Name)
			categoryB := serverTypeCategory(serverTypes[j].Name)
			if categoryA != categoryB {
				return categoryA < categoryB
			}
			return serverTypes[i].MonthlyPrice.ToFloat() < serverTypes[j].MonthlyPrice.ToFloat()
		})

		return serverTypes, nil
	}

	return c
}

func serverTypeCategory(serverTypeName string) (category string) {
	return strings.Split(serverTypeName, "-")[0]
}
