package goflags

import (
	"fmt"
	"strconv"
	"strings"

	"github.com/projectdiscovery/stringsutil"
)

// Port is a list of unique ports in a normalized format
type Port struct {
	kv map[int]struct{}
}

func (port Port) String() string {
	defaultBuilder := &strings.Builder{}
	defaultBuilder.WriteString("(")

	var items string
	for k := range port.kv {
		items += fmt.Sprintf("%d,", k)
	}
	defaultBuilder.WriteString(stringsutil.TrimSuffixAny(items, ",", "="))
	defaultBuilder.WriteString(")")
	return defaultBuilder.String()
}

// Set inserts a value to the port map. A number of formats are accepted.
func (port *Port) Set(value string) error {
	if port.kv == nil {
		port.kv = make(map[int]struct{})
	}
	port.normalizePortValue(value)
	return nil
}

// AsPorts returns the ports list after normalization
func (port *Port) AsPorts() []int {
	if port.kv == nil {
		return nil
	}
	ports := make([]int, 0, len(port.kv))
	for k := range port.kv {
		ports = append(ports, k)
	}
	return ports
}

// normalizePortValues normalizes and returns a list of ports for a value.
//
// Supported values -
//   1,2 => ports: 1, 2
//   1-10 => ports: 1 to 10
//   1- => ports: 1 to 65535
//   -/*/full => ports: 1 to 65535
//   topxxx => ports: top most xxx common ports
//   ftp,http => ports: 21, 80
//   ftp* => ports: 20, 21, 574, 989, 990, 8021
//   U:53,T:25 => ports: 53 udp, 25 tcp
func (port *Port) normalizePortValue(value string) {
	// Handle top-xxx/*/- cases
	switch value {
	case "full", "-", "*":
		value = portsFull
	case "top-100":
		value = portsNmapTop100
	case "top-1000":
		value = portsNmapTop1000
	}

	values := strings.Split(value, ",")
	for _, item := range values {
		if ports, ok := servicesMap[item]; ok {
			// Handle ftp,http,etc service names
			port.appendPortsToKV(ports)
		} else if strings.Contains(item, ":") {
			// Handle colon : based name like TCP:443
			port.parsePortColonSeparated(item)
		} else if strings.HasSuffix(item, "*") {
			// Handle wildcard service names
			port.parseWildcardService(item)
		} else if strings.Contains(item, "-") {
			// Handle dash based separated items
			port.parsePortDashSeparated(item)
		} else {
			// Handle normal ports
			port.parsePortNumberItem(item)
		}
	}
}

func (port *Port) appendPortsToKV(ports []int) {
	for _, p := range ports {
		port.kv[p] = struct{}{}
	}
}

// parseWildcardService parses wildcard based service names
func (port *Port) parseWildcardService(item string) {
	stripped := strings.TrimSuffix(item, "*")
	for service, ports := range servicesMap {
		if strings.HasPrefix(service, stripped) {
			port.appendPortsToKV(ports)
		}
	}
}

// parsePortDashSeparated parses dash separated ports
func (port *Port) parsePortDashSeparated(item string) {
	parts := strings.Split(item, "-")
	// Handle x- scenarios
	if len(parts) == 2 && parts[1] == "" {
		port.parsePortPairItems(parts[0], "65535")
	}
	// Handle x-x port pairs
	if len(parts) == 2 {
		port.parsePortPairItems(parts[0], parts[1])
	}
}

// parsePortColonSeparated parses colon separated ports
func (port *Port) parsePortColonSeparated(item string) {
	items := strings.Split(item, ":")
	if len(items) == 2 {
		parsed, err := strconv.Atoi(items[1])
		if err == nil && parsed > 0 {
			port.kv[parsed] = struct{}{}
		}
	}
}

// parsePortNumberItem parses a single port number
func (port *Port) parsePortNumberItem(item string) {
	parsed, err := strconv.Atoi(item)
	if err == nil && parsed > 0 {
		port.kv[parsed] = struct{}{}
	}
}

// parsePortPairItems parses port x-x pair items
func (port *Port) parsePortPairItems(first, second string) {
	firstParsed, err := strconv.Atoi(first)
	if err != nil {
		return
	}
	secondParsed, err := strconv.Atoi(second)
	if err != nil {
		return
	}
	for i := firstParsed; i <= secondParsed; i++ {
		port.kv[i] = struct{}{}
	}
}

const (
	portsFull        = "1-65535"
	portsNmapTop100  = "7,9,13,21-23,25-26,37,53,79-81,88,106,110-111,113,119,135,139,143-144,179,199,389,427,443-445,465,513-515,543-544,548,554,587,631,646,873,990,993,995,1025-1029,1110,1433,1720,1723,1755,1900,2000-2001,2049,2121,2717,3000,3128,3306,3389,3986,4899,5000,5009,5051,5060,5101,5190,5357,5432,5631,5666,5800,5900,6000-6001,6646,7070,8000,8008-8009,8080-8081,8443,8888,9100,9999-10000,32768,49152-49157"
	portsNmapTop1000 = "1,3-4,6-7,9,13,17,19-26,30,32-33,37,42-43,49,53,70,79-85,88-90,99-100,106,109-111,113,119,125,135,139,143-144,146,161,163,179,199,211-212,222,254-256,259,264,280,301,306,311,340,366,389,406-407,416-417,425,427,443-445,458,464-465,481,497,500,512-515,524,541,543-545,548,554-555,563,587,593,616-617,625,631,636,646,648,666-668,683,687,691,700,705,711,714,720,722,726,749,765,777,783,787,800-801,808,843,873,880,888,898,900-903,911-912,981,987,990,992-993,995,999-1002,1007,1009-1011,1021-1100,1102,1104-1108,1110-1114,1117,1119,1121-1124,1126,1130-1132,1137-1138,1141,1145,1147-1149,1151-1152,1154,1163-1166,1169,1174-1175,1183,1185-1187,1192,1198-1199,1201,1213,1216-1218,1233-1234,1236,1244,1247-1248,1259,1271-1272,1277,1287,1296,1300-1301,1309-1311,1322,1328,1334,1352,1417,1433-1434,1443,1455,1461,1494,1500-1501,1503,1521,1524,1533,1556,1580,1583,1594,1600,1641,1658,1666,1687-1688,1700,1717-1721,1723,1755,1761,1782-1783,1801,1805,1812,1839-1840,1862-1864,1875,1900,1914,1935,1947,1971-1972,1974,1984,1998-2010,2013,2020-2022,2030,2033-2035,2038,2040-2043,2045-2049,2065,2068,2099-2100,2103,2105-2107,2111,2119,2121,2126,2135,2144,2160-2161,2170,2179,2190-2191,2196,2200,2222,2251,2260,2288,2301,2323,2366,2381-2383,2393-2394,2399,2401,2492,2500,2522,2525,2557,2601-2602,2604-2605,2607-2608,2638,2701-2702,2710,2717-2718,2725,2800,2809,2811,2869,2875,2909-2910,2920,2967-2968,2998,3000-3001,3003,3005-3007,3011,3013,3017,3030-3031,3052,3071,3077,3128,3168,3211,3221,3260-3261,3268-3269,3283,3300-3301,3306,3322-3325,3333,3351,3367,3369-3372,3389-3390,3404,3476,3493,3517,3527,3546,3551,3580,3659,3689-3690,3703,3737,3766,3784,3800-3801,3809,3814,3826-3828,3851,3869,3871,3878,3880,3889,3905,3914,3918,3920,3945,3971,3986,3995,3998,4000-4006,4045,4111,4125-4126,4129,4224,4242,4279,4321,4343,4443-4446,4449,4550,4567,4662,4848,4899-4900,4998,5000-5004,5009,5030,5033,5050-5051,5054,5060-5061,5080,5087,5100-5102,5120,5190,5200,5214,5221-5222,5225-5226,5269,5280,5298,5357,5405,5414,5431-5432,5440,5500,5510,5544,5550,5555,5560,5566,5631,5633,5666,5678-5679,5718,5730,5800-5802,5810-5811,5815,5822,5825,5850,5859,5862,5877,5900-5904,5906-5907,5910-5911,5915,5922,5925,5950,5952,5959-5963,5987-5989,5998-6007,6009,6025,6059,6100-6101,6106,6112,6123,6129,6156,6346,6389,6502,6510,6543,6547,6565-6567,6580,6646,6666-6669,6689,6692,6699,6779,6788-6789,6792,6839,6881,6901,6969,7000-7002,7004,7007,7019,7025,7070,7100,7103,7106,7200-7201,7402,7435,7443,7496,7512,7625,7627,7676,7741,7777-7778,7800,7911,7920-7921,7937-7938,7999-8002,8007-8011,8021-8022,8031,8042,8045,8080-8090,8093,8099-8100,8180-8181,8192-8194,8200,8222,8254,8290-8292,8300,8333,8383,8400,8402,8443,8500,8600,8649,8651-8652,8654,8701,8800,8873,8888,8899,8994,9000-9003,9009-9011,9040,9050,9071,9080-9081,9090-9091,9099-9103,9110-9111,9200,9207,9220,9290,9415,9418,9485,9500,9502-9503,9535,9575,9593-9595,9618,9666,9876-9878,9898,9900,9917,9929,9943-9944,9968,9998-10004,10009-10010,10012,10024-10025,10082,10180,10215,10243,10566,10616-10617,10621,10626,10628-10629,10778,11110-11111,11967,12000,12174,12265,12345,13456,13722,13782-13783,14000,14238,14441-14442,15000,15002-15004,15660,15742,16000-16001,16012,16016,16018,16080,16113,16992-16993,17877,17988,18040,18101,18988,19101,19283,19315,19350,19780,19801,19842,20000,20005,20031,20221-20222,20828,21571,22939,23502,24444,24800,25734-25735,26214,27000,27352-27353,27355-27356,27715,28201,30000,30718,30951,31038,31337,32768-32785,33354,33899,34571-34573,35500,38292,40193,40911,41511,42510,44176,44442-44443,44501,45100,48080,49152-49161,49163,49165,49167,49175-49176,49400,49999-50003,50006,50300,50389,50500,50636,50800,51103,51493,52673,52822,52848,52869,54045,54328,55055-55056,55555,55600,56737-56738,57294,57797,58080,60020,60443,61532,61900,62078,63331,64623,64680,65000,65129,65389"
)

var servicesMap = map[string][]int{"bvcontrol": {1236}, "ivecon-port": {3258}, "int-rcv-cntrl": {3603}, "delos-dms": {3714}, "gw-log": {4844}, "vnc": {5900}, "raw-serial": {2167}, "p2pgroup": {3587}, "ibm-rsyscon": {9085}, "iqrm": {10117}, "wdbrpc": {17185}, "iso-ip": {147}, "esro-emsdp": {642}, "mainsoft-lm": {1593}, "primaserver": {6105}, "niprobe": {25903}, "sor-update": {3922}, "bitspeer": {2178}, "ovtopmd": {2532}, "ndl-tcp-ois-gw": {2738}, "tmesis-upshot": {2798}, "wg-netforce": {3359}, "jaugsremotec-2": {3473}, "n1-fwp": {4446}, "sun-as-jpda": {9191}, "mptn": {397}, "evm": {1139}, "ci3-software-2": {1302}, "vaultbase": {1771}, "atmtcp": {2812}, "ice-router": {4063}, "icl-twobase2": {25001}, "telnets": {992}, "capfast-lmd": {1756}, "lrp": {2090}, "thermo-calc": {6201}, "dsmipv6": {4191}, "senomix07": {8058}, "dna-cml": {436}, "sesi-lm": {1714}, "spice": {1923}, "mc-gt-srv": {2180}, "gds_db": {3050}, "itv-control": {3899}, "lpar2rrd": {8162}, "tcp-id-port": {1999}, "xnds": {2157}, "www-dev": {2784}, "incognitorv": {3139}, "silverpeakcomm": {4164}, "mni-prot-rout": {3764}, "scaleft": {4421}, "windows-icfw": {1002}, "isbconference1": {1244}, "t1distproc": {1274}, "psrserver": {2351}, "fastlynx": {2689}, "efcp": {3671}, "netxms-agent": {4700}, "afs3-prserver": {7002}, "i3-sessionmgr": {3952}, "bacnet": {47808}, "newlixengine": {2075}, "infocrypt": {2233}, "redstorm_info": {2348}, "netsteward": {2810}, "deskview": {3298}, "ipcs-command": {3743}, "xns-time": {52}, "upgrade": {2537}, "ccs-software": {2734}, "vseconnector": {2893}, "emwavemsg": {20480}, "prismiq-plugin": {3650}, "vatata": {4188}, "smsd": {596}, "afrog": {1042}, "eicon-slp": {1440}, "solera-epmap": {2132}, "gsiftp": {2811}, "tag-ups-1": {3573}, "drizzle": {4427}, "intu-ec-client": {8021}, "pentbox-sim": {6817}, "amandaidx": {10082}, "smip-agent": {164}, "hp-managed-node": {382, 782}, "utmpcd": {431}, "iasd": {432}, "acmaint_dbd": {774}, "telesis-licman": {1380}, "maitrd": {997}, "nfa": {1155}, "willy": {2518}, "nmsigport": {2817, 2839}, "dnp-sec": {19999}, "trinket-agent": {21212}, "vpad": {1516}, "gammafetchsvr": {1859}, "servicemeter": {2603}, "samd": {3275}, "solera-lpn": {4738}, "myq-termlink": {11108}, "hacl-cfg": {5302}, "fcp-srvr-inst2": {5503}, "npp": {92}, "qubes": {1341}, "imprs": {3164}, "aes-discovery": {3224}, "tns-adv": {3309}, "spocp": {4751}, "canit_store": {6568}, "mercury-disc": {9596}, "osm-oev": {9991}, "vat-control": {3457}, "metalbend": {7172}, "rxmon": {1311}, "kmscontrol": {1773}, "avocentkvm": {2068}, "scientia-ssdb": {2121}, "dyniplookup": {3295}, "monp": {3445}, "opsec-cvp": {18181}, "cis": {22305}, "CarbonCopy": {1680}, "ms-sna-base": {1478}, "repsvc": {6320}, "vantronix-mgmt": {8034}, "cumulus": {9287}, "tmophl7mts": {20046}, "liberty-lm": {1496}, "3ds-lm": {1538}, "jbroker": {2506}, "ans-console": {3440}, "sun-user-https": {7677}, "apani5": {9164}, "iniserve-port": {3560}, "acme": {9216}, "dsfgw": {438}, "dsf": {555}, "ideafarm-door": {902}, "ssserver": {1270}, "sixnetudr": {1658}, "policyserver": {3055}, "mctet-gateway": {3116}, "pscribe": {6163}, "clearcase": {371}, "dns2go": {1227}, "childkey-notif": {1891}, "indx-dds": {2454}, "madcap": {2535}, "lbc-measure": {2815}, "nq-ap": {36424}, "bnetfile": {1120}, "redstorm_join": {2346}, "bfd-echo": {3785}, "sdxauthd": {5540}, "pgbouncer": {6432}, "pmd": {7431}, "j-ac": {4107}, "hp-status": {5226}, "silverplatter": {416}, "ampr-inter": {1536}, "hb-engine": {1703}, "proxy-gateway": {2303}, "upsnotifyprot": {2648}, "can-ferret-ssl": {3661}, "heretic2": {28910}, "hermes": {1248}, "spectardata": {3834}, "aws-wsp": {4195}, "trim-ice": {4323}, "dsmcc-config": {13818}, "apc-9950": {9950}, "svs-omagent": {1625}, "lonworks": {2540}, "sah-lm": {3291}, "pxc-splr-ft": {4003}, "aol-3": {5193}, "ddi-tcp-3": {8890}, "heathview": {35000}, "cisco-tdp": {711}, "high-criteria": {2467}, "rscd": {5750}, "almobile-system": {9209}, "keysrvr": {19283}, "keyshadow": {19315}, "rtip": {771}, "galaxy-server": {3051}, "sapcomm": {3398}, "xmlblaster": {3412}, "newton-dock": {3679}, "mediat": {5157}, "genrad-mux": {176}, "omnivision": {1135}, "ebinsite": {2651}, "newgenpay": {3165}, "ilss": {4802}, "tve-announce": {2670}, "cleanerliverc": {3481}, "opsmessaging": {8090}, "tempest-port": {11600}, "apple-xsrvr-admin": {625}, "tsdos390": {1237}, "opennl": {1258}, "mysql-cm-agent": {1862}, "pwgippfax": {3951}, "aja-ntv4-disc": {4804}, "d-data-control": {4302}, "itactionserver1": {7280}, "dsp3270": {246}, "aeroflight-ret": {1219}, "netbill-prod": {1616}, "opencm": {3434}, "sixxsconfig": {3874}, "cppdp": {4051}, "semi-grpc": {8710}, "guibase": {9321}, "mshvlm": {6600}, "ortec-disc": {40853}, "topx": {2436}, "saprouter": {3299}, "edm-manager": {3460}, "wysdmc": {3916}, "irisa": {11000}, "optohost004": {22004, 22005}, "easyengine": {22222}, "flexlm8": {27008}, "novastorbakcup": {308}, "dpcp": {4099}, "tgcconnect": {4146}, "sfmsso": {5635}, "spg": {7016}, "font-service": {7100}, "vnsstr": {3321}, "dynamid": {9002}, "at-7": {207}, "polestar": {1060}, "mpshrsv": {1261}, "menandmice-dns": {1337}, "coord-svr": {2565}, "argis-te": {2581}, "ctiprogramload": {4452}, "ipfixs": {4740}, "weandsf": {48050}, "emcrmirccd": {10004}, "systat": {11}, "apc-2260": {2260}, "privilege": {2588}, "autocuesmi": {3103}, "sun-sr-jmx": {6488}, "t2-brm": {7933}, "apwi-imserver": {4391}, "emfis-cntl": {141}, "mc2studios": {1899}, "tr-rsrb-p3": {1989}, "fjippol-port1": {2750}, "silkp4": {2832}, "tl1-telnet": {3083}, "4-tieropmcli": {2934}, "prnrequest": {3910}, "bzflag": {5154}, "iatp-highpri": {6998}, "programmar": {15999}, "voxelstorm": {28200}, "aocp": {2712}, "xdtp": {3088}, "aipn-auth": {3833}, "beacon-port-2": {4426}, "prosharenotify": {5717}, "dmt": {7683}, "rds-ib": {18634}, "ss7ns": {477}, "ibm-mqseries2": {1881}, "univision": {2820}, "wsicopy": {3378}, "c-h-it-port": {3778}, "parsec-master": {6580}, "palace-5": {9996}, "MOS-lower": {10540}, "orasrv": {1525}, "xss-port": {3510}, "ehp-backup": {3638}, "wbem-http": {5988}, "fisa-svc": {7018}, "d-star": {9011}, "octopustentacle": {10933}, "sometimes-rpc20": {32778}, "icl-twobase5": {25004}, "csi-sgwp": {348}, "leecoposserver": {2212}, "gw": {3010}, "ipfix": {4739}, "quantastor": {8153}, "famdc": {10081}, "a1-bs": {5603}, "eor-game": {8149}, "roketz": {1730}, "intrastar": {1907}, "b2-license": {2204}, "mgcp-gateway": {2427}, "lsi-raid-mgmt": {2463}, "surfcontrolcpa": {3909}, "dtp-net": {8732}, "qb-db-server": {10160}, "tibsd": {11971}, "nbdb": {13785}, "ls3bcast": {3068}, "networklens": {3409}, "ncp": {524}, "icabrowser": {1604}, "shadowserver": {2027}, "appleugcontrol": {2336}, "wusage": {2396}, "simslink": {2676}, "ita-manager": {5052}, "targus-getdata": {5200}, "elektron-admin": {5398}, "efb-aci": {6159}, "pace-licensed": {31400}, "e1-interface": {38462}, "pssc": {645}, "nmap": {689}, "win-rpc": {1026}, "xtrms": {3424}, "vchat": {1168}, "nms_topo_serv": {1486}, "snap": {4752}, "coldfusion-auth": {44442, 44443}, "icad-el": {425}, "gss-http": {488}, "apparenet-tps": {3237}, "ordinox-dbase": {3355}, "em7-secom": {7700}, "ild": {24321}, "at-5": {205}, "winjaserver": {1290}, "ferrari-foam": {3216}, "irdmi2": {7999}, "apsolab-cols": {5471}, "dicom": {11112}, "appleqtcsrvr": {545}, "hcp-wismar": {686}, "spss-lm": {1759}, "qip-qdhcp": {2490}, "socorfs": {3379}, "iso-tp0s": {3782}, "tsaf": {12753}, "omad": {32768}, "webemshttp": {2851}, "nm-asses-admin": {3150}, "xcap-portal": {4888}, "trispen-sra": {9555}, "ttc-etap-ds": {2978}, "pxc-pin": {4005}, "samsung-unidex": {4010}, "piranha1": {4600}, "quailnet": {5464}, "tinyfw": {44334}, "audiojuggler": {3643}, "omviserver": {4428}, "parliant": {4681}, "varadero-0": {4837}, "rlm-admin": {5054}, "aequus-alt": {23457}, "dn6-smm-red": {196}, "opalis-robot": {314}, "scoremgr": {2034}, "fs-server": {8043}, "sec-t4net-clt": {9401}, "sns-dispatcher": {2657}, "strexec-s": {5027}, "ida-discover1": {5741}, "irdmi": {8000}, "canon-bjnp3": {8613}, "otp": {9390}, "dcutility": {1044}, "kermit": {1649}, "hao": {2245}, "sossecollector": {3166}, "qt-serveradmin": {1220}, "fj-hdnet": {1717}, "vcrp": {3073}, "trident-data": {7727}, "trdp-md": {17225}, "consul-insight": {5992}, "mzca-alert": {7282}, "mcidas": {112}, "tenebris_nts": {359}, "c3": {2472}, "mgxswitch": {3070}, "isoft-p2p": {3501}, "scte104": {5167}, "aesop": {8202}, "dl_agent": {3876}, "wfm": {4057}, "http": {80, 8008}, "matip-type-a": {350}, "l3-hbmon": {2370}, "tqdata": {2700}, "business": {3107}, "ttntspauto": {3474}, "cruise-update": {8380}, "svn": {3690}, "spectardb": {3835}, "mecomm": {668}, "eicon-server": {1438}, "pconnectmgr": {1562}, "etftp": {1818}, "qencp": {2120}, "dwmsgserver": {3228}, "scotty-ft": {14000}, "cloudcheck": {45514}, "etc-control": {6107}, "swtp-port1": {9281}, "apertus-ldp": {539}, "imap4-ssl": {585}, "watcom-sql": {1498}, "webaccess": {1739}, "ripd": {2602}, "geognosisman": {4325}, "mciautoreg": {1528}, "brcm-comm-port": {3188}, "igo-incognito": {4100}, "sco-peer-tta": {5427}, "ssh": {22}, "xlog": {482}, "islc": {1637}, "simplement-tie": {2756}, "vocaltec-gold": {6670}, "cisco-cloudsec": {8017}, "vop": {4433}, "rmiregistry": {1099}, "eklogin": {2105}, "elatelink": {2124}, "onesaf": {3244}, "ns-cfg-server": {3266}, "nuts_dem": {4132}, "cadkey-licman": {1399}, "msl_lmd": {1464}, "sybaseanywhere": {2638}, "sftu": {3326}, "lv-pici": {2145}, "gtrack-ne": {3592}, "multiplex": {171}, "cft-1": {1762}, "twrpc": {3479}, "lrs-paging": {3700}, "nim-wan": {6421}, "smtps": {465}, "hap": {661}, "telefinder": {1474}, "mobrien-chat": {2031}, "jaus": {3794}, "iphone-sync": {62078}, "split-ping": {6924}, "blp4": {8294}, "dsETOS": {378}, "kink": {910}, "novell-lu6.2": {1416}, "tivoli-npm": {1965}, "bbars": {3327}, "apple-vpns-rp": {4112}, "mapx": {36700}, "nimspooler": {48001}, "entrust-sps": {640}, "mtqp": {1038}, "winddlb": {1565}, "ping-pong": {3010}, "wmlserver": {4883}, "qft": {189}, "funk-license": {1787}, "csnotify": {2955}, "geniuslm": {3005}, "acp-discovery": {3822}, "metaedit-mu": {6360}, "osm-appsrvr": {9990}, "omasgport": {4090}, "cimple": {10125}, "pop3": {110}, "iad1": {1030}, "emwin": {2211}, "itm-lm": {2828}, "talarian-mcast1": {4015}, "houston": {4041}, "mongod": {27017, 27018, 27019, 28017}, "solid-e-engine": {1964}, "ici": {2200}, "caspssl": {1131}, "diagmond": {1508}, "tn-tl-r1": {1580}, "netbill-cred": {1614}, "isdc": {1636}, "altalink": {1845}, "atmp": {5150}, "mondex": {471}, "mylxamport": {2981}, "BackOrifice": {31337}, "lot105-ds-upd": {2053}, "conclave-cpp": {2491}, "cumulus-admin": {8954}, "nmc-disc": {10810}, "fw1-secureremote": {256}, "rrp": {648}, "ms-streaming": {1755}, "di-traceware": {3041}, "pcoip": {4172}, "odnsp": {9966}, "shivahose": {1549}, "worldfusion2": {2596}, "listmgr-port": {3767}, "juka": {48048}, "blp3": {8292}, "sgi-soap": {11110}, "914c-g": {211}, "msexch-routing": {691}, "pictrography": {1280}, "waste": {1337}, "wizard": {2001}, "netrix-sftm": {2328}, "perimlan": {4075}, "nacnl": {4361}, "stun-p1": {1990}, "mercantile": {3398}, "bcinameservice": {3415}, "bmc-jmx-port": {3604}, "ccp": {3947}, "sbi-agent": {3962}, "vcom-tunnel": {8001}, "netarx": {1040}, "vids-avtp": {1853}, "raid-sf": {2014}, "rdlap": {2321}, "cecsvc": {2571}, "epicon": {2912}, "telelpathstart": {5010}, "flexlm9": {27009}, "vdab": {1775}, "nbx-cc": {2093}, "ccowcmr": {2116}, "mcs-messaging": {3331}, "distccd": {3632}, "sixchat": {4605}, "sometimes-rpc8": {32772}, "novell-zen": {2544}, "minilock": {3798}, "admd": {5100}, "aed-512": {149}, "metricadbc": {2622}, "ifcp-port": {3420}, "apple-iphoto": {8770}, "hsrpv6": {2029}, "datusorb": {3282}, "indigo-vrmi": {8130}, "wv-csp-sms-cir": {3716}, "agentsease-db": {3997}, "upnp": {1900, 5000}, "vcmp": {2426}, "murx": {2743}, "cspmulti": {2807}, "officelink2000": {3320}, "cctv-port": {3559}, "g2tag": {4110}, "avt-profile-1": {5004}, "itinternet": {2691}, "qotd": {17}, "sqlnet": {66}, "netview-aix-2": {1662}, "cft-7": {1768}, "mailbox": {2004}, "xinuexpansion2": {2022}, "synapse-nhttp": {8280}, "hexen2": {26900}, "fatserv": {347}, "streetperfect": {1330}, "scrabble": {2026}, "mdap-port": {3235}, "distcc": {3632}, "aspentec-lm": {6142}, "apm-link": {32483}, "nmasoverip": {1242}, "mcntp": {5418}, "visweather": {9979}, "netapp-sync": {10006}, "decap": {403}, "h323hostcallsc": {1300}, "ssmd": {2187}, "ironmail": {3206}, "apwi-disc": {4394}, "paycash-wbp": {8129}, "novar-global": {23402}, "ftps-data": {989}, "sunclustergeo": {2084}, "ispmmgr": {3775}, "itwo-server": {4410}, "intel-rci-mp": {16991}, "commtact-http": {20002}, "pcs-sf-ui-man": {6655}, "rda": {630}, "fiveacross": {1193}, "ssslog-mgr": {1204}, "tht-treasure": {1832}, "service-ctrl": {2367}, "globalcatLDAP": {3268}, "ssslic-mgr": {1203}, "mpnjsc": {1952}, "lbm": {2465}, "solve": {2636}, "esip": {2950}, "wg-endpt-comms": {33000}, "onmux": {417}, "empire-empuma": {1691}, "mao": {2908}, "dmod-workspace": {3199}, "spdp": {5794}, "lanserver": {637}, "pptp": {1723}, "ppp": {3000}, "radius-dynauth": {3799}, "gkrellm": {19150}, "chargen": {19}, "accord-mgc": {1205}, "treehopper": {3959}, "fmpro-v6": {5013}, "ssp-client": {7801}, "axio-disc": {35100}, "simplifymedia": {8087}, "m3da": {44900}, "hiq": {1410}, "h323gatestat": {1719}, "gsigatekeeper": {2119}, "dtv-chan-req": {2253}, "sdbproxy": {3562}, "xecp-node": {3940}, "erunbook_agent": {9616}, "shrinkwrap": {358}, "ergolight": {2109}, "cart-o-rama": {3292}, "asap-sctp": {3863}, "limnerpressure": {8191}, "aruba-papi": {8211}, "sst": {266}, "mdbs_daemon": {800}, "omginitialrefs": {900}, "umsp": {2110}, "rmlnk": {2818}, "street-stream": {1736}, "tekpls": {1946}, "semaphore": {3255}, "netcelera": {3701}, "ftsync": {4086}, "pmdfmgt": {7633}, "vmpwscs": {214}, "dialogic-elmd": {1945}, "visinet-gui": {3601}, "movaz-ssc": {5252}, "icpps": {14143}, "olsv": {1160}, "qip-audup": {2765}, "apc-3506": {3506}, "jute": {5883}, "nbd": {10809}, "enpp": {2968}, "pnbscada": {3875}, "via-ftp": {63}, "vettcp": {78}, "sas-1": {1426}, "dxadmind": {1958}, "lv-ffx": {2144}, "metaconsole": {2850}, "tunstall-lwp": {5197}, "dgpf-exchg": {6785}, "sqlexec": {9088}, "febooti-aw": {36524}, "astergate-disc": {9106}, "flexlm2": {27002}, "nettgain-nms": {1879}, "brutus": {2003}, "newoak": {4001}, "pcoip-mgmt": {5172}, "downtools": {5245}, "devbasic": {5426}, "af": {1411}, "apollo-status": {2758}, "myblast": {3795}, "ewdgs": {4092}, "ssdtp": {6071}, "dgi-serv": {33333}, "pcmail-srv": {158}, "iee-qfx": {1284}, "tr-rsrb-port": {1996}, "anet-b": {3338}, "asnaacceler8db": {5042}, "cfw": {7563}, "neteh": {3828}, "mit-dov": {91}, "emc-gateway": {1273}, "goldleaf-licman": {1401}, "clvm-cfg": {1476}, "rprt": {3064}, "monetra": {8665}, "tpdu": {1430}, "pharmasoft": {1779}, "initlsmsad": {2793}, "connect-client": {3441}, "dayliteserver": {6113}, "sophos": {8192, 8193, 8194}, "iso-tsap": {102}, "compaqdiag": {2301, 49400}, "maincontrol": {2516}, "hfcs": {4900}, "metaedit-se": {6370}, "kerberos_master": {751}, "lecroy-vicp": {1861}, "ircu": {6665, 6666, 6667, 6668, 6669}, "pando-pub": {7680}, "quest-vista": {7980}, "bcslogc": {13216}, "sgcip": {16950}, "sfs-config": {452}, "vatp": {690}, "mbap-s": {802}, "lisp-control": {4342}, "mtportmon": {7421}, "sstp-2": {9801}, "rapido-ip": {2457}, "fcip-port": {3225}, "dvcprov-port": {3776}, "netops-broker": {5465}, "gnmi-gnoi": {9339}, "bmdss": {13823}, "rje": {5}, "tomato-springs": {3040}, "adobeserver-4": {3704}, "boe-eventsrv": {6402}, "sge_execd": {6445}, "mgcs-mfp-port": {6509}, "xns-auth": {56}, "univ-appserver": {1233}, "resource_mgr": {3019}, "dtserver-port": {4028}, "aurp": {387}, "atm-zip-office": {1520}, "newheights": {2114}, "rcip-itu": {2225}, "swx": {7300, 7301, 7302, 7303, 7304, 7305, 7306, 7307, 7308, 7309, 7310, 7311, 7312, 7313, 7314, 7315, 7316, 7317, 7318, 7319, 7320, 7321, 7322, 7323, 7324, 7325, 7326, 7327, 7328, 7329, 7330, 7331, 7332, 7333, 7334, 7335, 7336, 7337, 7338, 7339, 7340, 7341, 7342, 7343, 7344, 7345, 7346, 7347, 7348, 7349, 7350, 7351, 7352, 7353, 7354, 7355, 7356, 7357, 7358, 7359}, "printer_agent": {3396}, "exp1": {1021}, "mcs-calypsoicf": {3330}, "eq-office-4942": {4942}, "sgi-esphttp": {5554}, "ajp12": {8007}, "vrts-tdd": {14149}, "groove": {2492}, "perfd": {5227}, "nsiiops": {261}, "avian": {486}, "pehelp": {2307}, "qftest-lookup": {3543}, "t5-straton": {11173}, "sweetware-apps": {1221}, "ibm-db2-admin": {6789}, "isa100-gci": {17223}, "svrloc": {427}, "mpnjsomb": {2681}, "netassistant": {3283}, "omnivisionesx": {4395}, "aigairserver": {21221}, "citadel": {504}, "pacerforum": {1480}, "tvnetworkvideo": {3791}, "oidocsvc": {4142}, "optohost003": {22003}, "apani3": {9162}, "ms-rome": {569}, "snaresecure": {1684}, "cs-services": {3631}, "talon-webserver": {7015}, "radan-http": {8088}, "svcloud": {8404}, "base": {5429}, "cisco-redu": {5786}, "fujitsu-dev": {747}, "zephyr-srv": {2102}, "remotedeploy": {3789}, "nuauth": {4129}, "kdnet": {5364}, "netsupport2": {5421}, "bmc-perf-agent": {6767}, "talon-disc": {7011}, "hislip": {4880}, "pago-services1": {30001}, "macon": {456}, "csvr-proxy": {3190}, "embrace-dp-s": {3197}, "netwatcher-db": {3204}, "kingfisher": {4058}, "xandros-cms": {4389}, "whisker": {3233}, "charsetmgr": {3903}, "csnet-ns": {105}, "pip": {321, 1321}, "mobileip-agent": {434}, "sshell": {614}, "netview-aix-1": {1661}, "unifyadmin": {2696}, "rwp": {30100}, "qptlmd": {10055}, "cycleserv2": {772}, "ridgeway1": {2776}, "l3-hawk": {2842}, "pit-vpn": {2865}, "sos": {3838}, "herodotus-net": {3921}, "spramsd": {5770}, "fse": {7394}, "9pfs": {564}, "sa-msg-port": {1646}, "ms-rule-engine": {3132}, "itelserverport": {3719}, "starfish": {3981}, "pvxplusio": {4193}, "cuillamartin": {1356}, "darcorp-lm": {1679}, "ovsessionmgr": {2389}, "jdl-dbkitchen": {3086}, "netplay-port1": {3640}, "5afe-dir": {23294}, "con": {759}, "exp2": {1022}, "pptconference": {1711}, "kana": {2656}, "nasmanager": {1960}, "llm-pass": {2813}, "bmap": {3421}, "sip-tls": {5061}, "wnn6_Tw": {22321}, "crestron-cip": {41794}, "bex-webadmin": {6122}, "vp2p": {8473}, "cft-4": {1765}, "csoft1": {1837}, "3com-webview": {2339}, "vsamredirector": {2387}, "vipremoteagent": {3752}, "synapsis-edge": {5008}, "ub-dns-control": {8953}, "vnetd": {13724}, "ttc": {2483}, "interactionweb": {3508}, "utime": {519}, "kshell": {544}, "eudora-set": {592}, "wpgs": {780}, "jwalkserver": {1289}, "armadp": {1913}, "ramp": {7227}, "loadsrv": {480}, "submission": {587}, "jwserver": {1937}, "quaddb": {2497}, "sometimes-rpc3": {32770}, "cnrprotocol": {1096}, "alchemy": {3234}, "swdtp": {10104}, "tmosms0": {5580}, "teamviewer": {5938}, "msdp": {639}, "swldy-sias": {1250}, "listen": {2766}, "awg-proxy": {3277}, "contamac_icm": {4846}, "pcduo-old": {5400}, "rs-pias": {13217}, "miva-mqs": {1277}, "raid-cs": {2015}, "acp-port": {2071}, "eforward": {2181}, "linktest-s": {3747}, "syam-webserver": {3930}, "psl-management": {5507}, "X11:4": {6004}, "evb-elm": {1504}, "di-drm": {2226}, "aviva-sna": {2238}, "avsecuremgmt": {3211}, "parallel": {4989}, "magpie": {5092}, "affiliate": {6579}, "babel": {6696, 6697}, "lipsinc1": {1969}, "tdmoip": {2142}, "quasar-server": {3599}, "gvcp": {3956}, "spss": {5443}, "tl1-ssh": {6252}, "documentum_s": {10003}, "xrl": {1104}, "amx-webadmin": {2929}, "netwkpathengine": {3209}, "ctdb": {4379}, "opcua-tls": {4843}, "apsolab-tags": {5473}, "dccp-udp": {6511}, "mpp": {218}, "ddm-dfm": {447}, "entrust-ash": {710}, "peport": {1449}, "powerchute": {3052}, "rt-event": {3706}, "d2dconfig": {9387}, "xpra": {14500}, "suitcase": {2903}, "taep-as-svc": {5111}, "radec-corp": {5430}, "imqstomp": {7672}, "ddi-tcp-7": {8894}, "canditv": {24676}, "ncr_ccl": {2528}, "net-projection": {5363}, "sometimes-rpc10": {32773}, "lsnr": {1158}, "sdproxy": {1297}, "onehome-remote": {2198}, "dpserveadmin": {7021}, "paycash-online": {8128}, "netserialext2": {16361}, "mac-srvr-admin": {660}, "nlogin": {758}, "iec-104": {2404}, "sai_sentlm": {2640}, "palace-3": {9994}, "ncd-pref": {5997}, "opi-sock": {7429}, "iris-beep": {702}, "hypercube-lm": {1577}, "firefox": {1689}, "mapper-mapethd": {3985}, "mlchat-proxy": {4002}, "zenginkyo-1": {5020}, "itach": {8184}, "pegasus-ctl": {9279}, "tw-auth-key": {27999}, "sec-t4net-srv": {9400}, "sage-best-com2": {14034}, "fujitsu-mmpdc": {1657}, "optika-emedia": {1829}, "unite": {3217}, "apple-sasl": {3659}, "ncd-pref-tcp": {5977}, "imqtunnel": {7675}, "ss-idi-disc": {20012}, "cvmon": {1686}, "landesk-rc": {1761, 1762, 1763, 1764}, "dnet-keyproxy": {2064}, "neteh-ext": {3829}, "padl2sim": {5236}, "openwebnet": {20005}, "compaq-evm": {619}, "xribs": {2025}, "xpanel": {3737}, "gbmt-stars": {3912}, "mppolicy-v5": {5968}, "ctxlic": {7279}, "mfcobol": {86}, "rpi": {2214}, "X11": {6000}, "icl-twobase10": {25009}, "an-pcp": {3846}, "surfpass": {5030}, "sco-websrvrmgr": {620}, "notify": {773}, "mxomss": {1141}, "adapt-sna": {1365}, "gtp-control": {2123}, "cpudpencap": {2746}, "sac": {8097}, "oap-admin": {8567}, "secure-ts": {9318}, "suncacao-rmi": {11163}, "imap3": {220}, "secure-aux-bus": {664}, "quicktime": {1220}, "ip-blf": {2088}, "globalcatLDAPssl": {3269}, "l2c-disc": {4371}, "italk": {12345}, "sometimes-rpc9": {32773}, "plysrv-https": {6771}, "msss": {7742}, "commlinx-avl": {1190}, "de-spot": {2753}, "winpcs": {5166}, "esinstall": {5599}, "reversion": {5842}, "SunVTS-RMI": {6483}, "kerberos": {750}, "mgcp-callagent": {2727}, "mindarray-ca": {9445}, "xpilot": {15345}, "iclpv-nls": {1393}, "nucleus": {1463}, "uacs": {2768}, "smpp": {2775}, "galileo": {3519}, "syam-agent": {3894}, "iad2": {1031}, "nbt-pc": {5133}, "ieee11073-20701": {6464}, "acmsoda": {6969}, "sstp-1": {7743}, "pushns": {7997}, "ccproxy-http": {808}, "lofr-lm": {1752}, "raid-ac": {2012}, "LiebDevMgmt_A": {3029}, "oracleas-https": {7443}, "rmt": {411}, "scan-change": {2719}, "piranha2": {4601}, "dmp": {5031}, "iposplanet": {7031}, "X11:2": {6002}, "cucme-3": {7650}, "fln-spx": {221}, "sns-quote": {1967}, "brdptc": {2155}, "caps-lm": {3290}, "hp-clic": {3384}, "n1-rmgmt": {4447}, "dellwebadmin-2": {1279}, "bounzza": {2218}, "netiq-endpt": {10115}, "metagram": {99}, "cgn-config": {2183}, "wanscaler": {2312}, "sm-pas-3": {2940}, "choiceview-clt": {4316}, "sentlm-srv2srv": {5099}, "smbdirect": {5445}, "fac-restore": {5582}, "ftp": {21}, "donnyworld": {1821}, "bigbrother": {1984}, "gdp-port": {1997}, "gris": {2135}, "agps-port": {3425}, "h323callsigalt": {11720}, "s1-control": {36412}, "entp": {1865}, "atlinks": {4154}, "caaclang2": {5249}, "nucleus-sand": {1201}, "cadsi-lm": {1387}, "hbci": {3000}, "crip": {6253}, "adi-gxp-srvprt": {6769}, "gamesmith-port": {31765}, "dn6-nlm-aud": {195}, "amiganetfs": {2100}, "simple-push": {3687}, "fmtp": {8500}, "bctp-server": {10107}, "inovaport6": {23005}, "video-activmail": {1398}, "icg-iprelay": {2064}, "asap-sctp-tls": {3864}, "sphinxql": {9306}, "wfremotertm": {1046}, "navbuddy": {1288}, "kmip": {5696}, "quest-disc": {7040}, "janus-disc": {7181}, "bfd-lag": {6784}, "driveappserver": {1930}, "allstorcns": {2901}, "sentinel": {3588}, "openhpid": {4743}, "icmpd": {5813}, "xdsxdm": {6558}, "predatar-comms": {1753}, "orbix-cfg-ssl": {3078}, "rfe": {5002}, "ddi-tcp-6": {8893}, "med-ovw": {24004}, "davsrcs": {9802}, "pduncs": {16310}, "rpcbind": {111}, "lpcp": {1298}, "dls-monitor": {2048}, "qfp": {5083}, "vpa": {5164}, "amberon": {8301}, "net-device": {4350}, "xcap-control": {4889}, "hpvmmdata": {1126}, "aura": {2066}, "ads": {2550}, "quicksuite": {2900}, "wap-pushsecure": {2949}, "netclip": {2971}, "rothaga": {7117}, "noit-transport": {43191}, "locus-map": {125}, "livelan": {1555}, "rnrp": {2423}, "roce": {4791}, "Trinoo_Bcast": {27444}, "retp": {32811}, "composit-server": {2417}, "iwec": {4801}, "ocbinder": {183}, "mnotes": {603}, "elfiq-repl": {1148}, "pdap-np": {1526}, "picodbc": {1603}, "conf": {2008}, "mphlpdmc": {9344}, "sec-ntb-clnt": {32635}, "mc-client": {1180}, "mediaspace": {3594}, "oma-ulp": {7275}, "ms-sql-m": {1434}, "ibm_wrless_lan": {1461}, "radio-sm": {1596}, "theta-lm": {2296}, "dzoglserver": {3867}, "indy": {5963}, "keyserver": {584}, "tripwire": {1169}, "3com-net-mgmt": {2391}, "qnxnetman": {3385}, "icl-twobase8": {25007}, "relief": {1353}, "whosells": {2781}, "cl-db-remote": {4137}, "blackice-icecap": {8081}, "nping-echo": {9929}, "dhcpv6-server": {547}, "catchpole": {1185}, "sdt-lmd": {3319}, "pkix-cmc": {5318}, "ctf": {84}, "apple-licman": {1381}, "uma": {1797}, "securid": {5500}, "netspeak-acd": {21848}, "eq3-update": {43439}, "sgmp": {153}, "orbix-config": {3076}, "stat-cc": {4158}, "silverpeakpeer": {4163}, "pinghgl": {4306}, "boe-resssvr3": {6409}, "aminet": {2639}, "ncadg-ip-udp": {3063}, "strexec-d": {5026}, "net2display": {9086}, "insitu-conf": {1490}, "hlserver": {3047}, "vvr-control": {4145}, "pago-services2": {30002}, "sometimes-rpc16": {32776}, "jmact5": {2957}, "dnx": {3998}, "vision_elmd": {6673}, "ris": {180}, "itm-mcell-s": {828}, "excw": {1271}, "netiq-mc": {2735}, "esp-encap": {2797}, "imqstomps": {7673}, "cisco-snat": {15555}, "sur-meas": {243}, "ibm-res": {1405}, "vocaltec-admin": {1796}, "mpnjsocl": {2685}, "auriga-router": {5680}, "afs3-update": {7008}, "su-mit-tg": {89}, "phonebook": {767}, "bmc-messaging": {2059}, "citrix-rtmp": {2897}, "asctrl-agent": {5155}, "vrmg-ip": {24323}, "ms-olap1": {2393}, "mcs-fastmail": {3302}, "msp-os": {4686}, "hde-lcesrvr-1": {14936}, "embrace-dp-c": {3198}, "postgresql": {5432}, "dccm": {5679}, "openremote-ctrl": {8688}, "otlp": {6951}, "simco": {7626}, "accessnetwork": {699}, "imagequery": {2239}, "mynahautostart": {2388}, "tksocket": {2915}, "X11:8": {6008}, "spc": {6111}, "nav-data-cmd": {6317}, "nexentamv": {8457}, "fjappmgrbulk": {2510}, "qpasa-agent": {2612}, "lotusmtap": {3007}, "plato": {3285}, "sanavigator": {4033}, "hacl-test": {5305}, "brightcore": {5682}, "bmc-net-adm": {1769}, "direcpc-si": {2464}, "quartus-tcl": {2589}, "dicom-iscl": {2761}, "ispipes": {2853}, "couchdb": {5984}, "hacl-qs": {1238}, "egs": {1926}, "xaapi": {1934}, "ridgeway2": {2777}, "cisco-wafs": {4050}, "assuria-ins": {4704}, "wap-vcard": {9204}, "snmpdtls": {10161}, "cmmdriver": {1294}, "stun-p3": {1992}, "ott": {2428}, "netrek": {2592}, "cpq-tasksmart": {3201}, "mobileanalyzer": {7869}, "dnp": {20000}, "filesphere": {24242}, "permabit-cs": {5312}, "scohelp": {457}, "udp-sr-port": {1624}, "lnvmailmon": {2285}, "cdl-server": {3056}, "roverlog": {3677}, "admeng": {5102}, "ipfltbcst": {4068}, "MOS-soap": {10543}, "audit-transfer": {1146}, "acd-pm": {2259, 8793}, "tappi-boxnet": {2306}, "gprs-data": {3386}, "kpn-icw": {3699}, "nav-port": {3859}, "weave": {11095}, "a27-ran-ran": {28119}, "event-port": {2069}, "hicp": {3250}, "noteit": {4663}, "radpdf": {18104}, "sstats": {486}, "vieo-fe": {3245}, "va-pacbase": {3676}, "ss-idi": {20013}, "dbase": {217}, "sonuscallsig": {2569}, "snapd": {2599}, "data-insurance": {2764}, "sdreport": {5540}, "alljoyn-stm": {9955}, "cdbroker": {3376}, "grcp": {9123}, "qsnet-assist": {4356}, "ems": {4664}, "legent-1": {373}, "iss-realsecure": {902}, "seagulllms": {1291}, "prolink": {1678}, "dicom-tls": {2762}, "hacl-monitor": {3542}, "PS-Server": {7215}, "blp2": {8195}, "whosockami": {2009, 2019}, "directv-web": {3334}, "hpssmgmt": {4484}, "aol-2": {5192}, "cuelink-disc": {5271}, "fmsascon": {16001}, "xmquery": {2279}, "softaudit": {3419}, "netrjs-2": {72}, "blackjack": {1025}, "netopia-vo3": {1841}, "deslogin": {2005, 3005}, "ah-esp-encap": {2070}, "foliocorp": {2242}, "press": {3582}, "alaris-disc": {3613}, "smile": {3670}, "netserialext3": {16367}, "hip-nat-t": {10500}, "iris-lwz": {715}, "dlsrpn": {2065}, "santak-ups": {3038}, "cst-port": {3742}, "qvr": {5028}, "ampify": {8040}, "ptcnameservice": {597}, "jacobus-lm": {1578}, "netcabinet-com": {4409}, "ndl-ahp-svc": {6064}, "palcom-disc": {6657}, "agentx": {705}, "fcp-udp": {810}, "fronet": {4130}, "teamcoherence": {9222}, "jdp-disc": {7095}, "pmdmgr": {7426}, "dwr": {644}, "twamp-control": {862}, "socks": {1080}, "serverview-gf": {3171}, "kar2ouche": {4661}, "spdy": {6121}, "proremote": {8183}, "netgw": {741}, "ncpm-ft": {1744}, "kx": {2111}, "avocent-adsap": {3871}, "hes-clip": {8807}, "gsms": {16002}, "med-ci": {24005}, "icap": {1344}, "flukeserver": {2359}, "altav-tunnel": {3265}, "haipe-discover": {3623}, "choiceview-agt": {4314}, "cruise-swroute": {8377}, "proshareaudio": {5713}, "qbdb": {8019}, "tserver": {450}, "cft-6": {1767}, "ibridge-data": {2275}, "tl1-raw": {3082}, "ccmcomm": {3505}, "jprinter": {5309}, "sent-lm": {2316}, "sd-capacity": {2384}, "pxc-spvr-ft": {4002}, "enrp-sctp": {9901}, "dcsl-backup": {11202}, "iRAPP": {4073}, "xmms2": {9667}, "sdnskmp": {558}, "vpjp": {1345}, "hpppssvr": {2448}, "system-monitor": {2609}, "iwb-whiteboard": {2982}, "dell-rm-port": {3668}, "dsm-scm-target": {9987}, "rsip": {4555}, "ctxs-vpp": {4980}, "cmip-agent": {164}, "netbill-trans": {1612}, "sychrond": {3723}, "zmp": {3925}, "wello": {4177}, "ntske": {4460}, "securitychase": {5399}, "collaber": {7689}, "hagel-dump": {3036}, "anoto-rendezv": {3715}, "commerce": {542}, "eyelink": {589}, "wpages": {776}, "kvm-via-ip": {1132}, "kauth": {2120}, "miles-apart": {2621}, "domaintime": {9909}, "NFS-or-IIS": {1025}, "seagull-ais": {1208}, "netview-aix-3": {1663}, "h323q931": {1720}, "tl-ipcproxy": {4176}, "apsolab-rpc": {5474}, "telnetcpcd": {3696}, "conductor": {6970}, "urm": {606}, "vpps-via": {676}, "applix": {999}, "proshare-mc-1": {1673}, "slush": {1966}, "bb": {1984}, "quake2": {27910}, "videotex": {516}, "twcss": {3428}, "cyc": {3645}, "asc-slmd": {4448}, "pichat": {9009}, "entextlow": {12004}, "vscp": {9598}, "ptp": {15740}, "writesrv": {1334}, "efidiningport": {2553}, "netwatcher-mon": {3203}, "calltrax": {3675}, "rnm": {3844}, "zeus-admin": {9090}, "securenetpro-sensor": {975}, "rimf-ps": {2209}, "ospfd": {2604}, "intraintra": {3202}, "an-signaling": {4936}, "talon-engine": {7012}, "wmc-log-svc": {1338}, "h323hostcall": {1720}, "ea1": {1791}, "hp-webqosdb": {1877}, "sixid": {4606}, "core-of-source": {8767}, "aol-1": {5191}, "z-wave-s": {41230}, "netrjs-4": {74}, "cardax": {1072}, "dbcontrol-oms": {1158}, "re-conn-proto": {1306}, "syncserver": {2647}, "dsom-server": {3053}, "amt-blc-port": {2848}, "directvdata": {3287}, "oap": {3567}, "spw-dnspreload": {3849}, "awacs-ice": {4488}, "pcanywheredata": {5631}, "connect": {2137}, "svnetworks": {2973}, "directv-catlg": {3337}, "opswmanager": {3977}, "canon-bjnp4": {8614}, "raven-rdp": {3533}, "pxc-spvr": {4006}, "snmpssh-trap": {5162}, "sdadmind": {5550}, "galaxy4d": {8881}, "optocontrol": {22001}, "cpscomm": {5194}, "snmptls-trap": {10162}, "vsinet": {996}, "bytex": {1375}, "timbuktu-srv3": {1419}, "xss-srv-port": {3646}, "derby-repli": {4851}, "pvaccess": {5075}, "netview-aix-10": {1670}, "daishi": {2870}, "listcrt-port": {3913}, "powwow-client": {13223}, "sun-lm": {7588}, "ctechlicensing": {9346}, "bgs-nsi": {482}, "newlixconfig": {2076}, "nmmp": {3649}, "nnp": {3780}, "sasggprs": {3964}, "crusecontrol": {5231}, "monkeycom": {9898}, "sometimes-rpc12": {32774}, "acptsys": {2149}, "volley": {3625}, "vtsas": {5070}, "talarian-udp": {5101}, "swa-3": {9025}, "emc-vcas-tcp": {13218}, "prosharedata": {5715}, "dpm-acm": {6075}, "rsh-spx": {222}, "openport": {260}, "vid": {769}, "talnet": {1838}, "idac": {3881}, "quosa": {4841}, "jeol-nsddp-3": {6243}, "amt-soap-http": {16992}, "cplscrambler-lg": {1086}, "mtport-regist": {2791}, "adtech-test": {3357}, "sigma-port": {3614}, "pjlink": {4352}, "X11:7": {6007}, "pawserv": {345}, "eisport": {3525}, "openqueue": {8764}, "jetdirect": {9100, 9101, 9102, 9103, 9104, 9105, 9106, 9107}, "tcc-http": {24680}, "aspcoordination": {7235}, "exoline-udp": {26486}, "dfn": {1133}, "gwha": {1383}, "vistium-share": {1545}, "pay-per-view": {1564}, "versa-tek": {2610}, "dey-sapi": {4330}, "jboss-iiop": {3528}, "sscan": {3853}, "cacp": {190}, "corba-iiop": {683}, "mosaicsyssvc1": {1235}, "oi-2000": {2364}, "repliweb": {2837}, "btrieve": {3351}, "rome": {7663}, "gamegen1": {1738}, "jdmn-port": {4030}, "dpm-agent": {5719}, "armcenterhttps": {9295}, "csvr-sslproxy": {3191}, "prchat-user": {4455}, "netnews": {532}, "rmc": {657}, "imgames": {1077}, "aplx": {1134}, "noaaport": {2210}, "gadgetgate2way": {2678}, "munin": {4949}, "watchdoc-pod": {5743}, "hpstgmgr": {2600}, "esnm-zoning": {4023}, "ml-svnet": {4171}, "dtspcd": {6112}, "nfoldman": {7393}, "diagd": {31727}, "med-supp": {24003}, "dberegister": {1479}, "sm-pas-2": {2939}, "ii-admin": {3006}, "lanrevserver": {3971}, "oirtgsvc": {4141}, "sds": {5059}, "link": {245}, "rrirtr": {1693}, "max": {6074}, "apollo-admin": {8122}, "mcns-sec": {638}, "iris-xpc": {713}, "cisco-ipsla": {1167}, "tcim-control": {2729}, "ucentric-ds": {2879}, "uohost": {3314}, "tl1-raw-ssl": {6251}, "sossd-collect": {7981}, "lm-perfworks": {8204}, "printopia": {10631}, "EtherNetIP-2": {44818}, "isnetserv": {48128}, "locus-con": {127}, "shockwave2": {1257}, "event_listener": {3017}, "nomdb": {13786}, "ibm-mqseries": {1414}, "miteksys-lm": {1482}, "nimaux": {3902}, "rs-status": {45002}, "asipregistry": {687}, "lupa": {1212}, "netbill-keyrep": {1613}, "ambit-lm": {6831}, "ng-control": {38412}, "xns-mail": {58}, "krbupdate": {760}, "dmdocbroker": {1489}, "cgi-starapi": {3893}, "flexlm4": {27004}, "ndm-requester": {1363}, "auris": {2772}, "hpidsadmin": {2984}, "veritas-vis2": {2994}, "converge": {4774}, "ipdtp-port": {20202}, "crs": {507}, "exlm-agent": {3002}, "procos-lm": {3248}, "universe_suite": {4184}, "iconstructsrv": {6077}, "servicetags": {6481}, "net8-cman": {1830}, "rtcm-sc104": {2101}, "rsmtp": {2390}, "cautcpd": {3061}, "mctet-jserv": {3117}, "hfcs-manager": {4999}, "networklenss": {3410}, "dmaf-server": {3574}, "amp": {3811}, "lm-webwatcher": {8208}, "blocks": {10288}, "bf-game": {25954}, "bdp": {581}, "ms-sql-s": {1433}, "nkd": {1650}, "knetd": {2053}, "arepa-raft": {3025}, "iua": {9900}, "ovwdb": {2447}, "mpsysrmsvr": {3358}, "vpm-udp": {5046}, "abarsd": {8402}, "MOS-upper": {10541}, "nimbusdb": {48004}, "zixi-transport": {7088}, "glrpc": {9080}, "rsvd": {168}, "pcm": {1827}, "anet-h": {3341}, "e-builder": {4121}, "privatewire": {4449}, "iba-cfg": {7072}, "serverstart": {9213}, "serverview-icc": {3173}, "a21-an-1xbs": {4597}, "ads-s": {8016}, "monetra-admin": {8666}, "gue": {6080}, "agpolicy": {38203}, "fw1-mc-fwmodule": {257}, "mini-sql": {1114}, "d-cinema-rrp": {1173}, "dirgis": {2496}, "batman": {4305}, "jaxflow": {5229}, "netview-aix-4": {1664}, "lds-dump": {6544}, "crtech-nlm": {20810}, "terminaldb": {2008, 2018}, "elcn": {7101}, "smc-https": {6789}, "cab-protocol": {595}, "unet": {1189}, "rellpack": {2018}, "lpsrecommender": {2620}, "incp": {2932}, "bfd-control": {3784}, "aodv": {654}, "csi-lfap": {3145}, "nat-pmp-status": {5350}, "ct2nmcs": {7023}, "lds-distrib": {6543}, "aurora-balaena": {33123}, "openvms-sysipc": {557}, "qrh": {752}, "startron": {1057}, "anynetgateway": {1491}, "hppronetman": {3908}, "questdb2-lnchr": {5677}, "netrjs-1": {71}, "omserv": {764}, "hp-hcip-gwy": {1803}, "canon-mfnp": {8610}, "mxit": {9119}, "mvs-capacity": {10007}, "ftps": {990}, "nicelink": {1095}, "shiva_confsrvr": {1651}, "rdc-wh-eos": {3142}, "suncacao-jmxmp": {11162}, "filenet-rpc": {32769}, "sonar": {572}, "gcmonitor": {2660}, "jpegmpeg": {3155}, "Trinoo_Master": {27665}, "isoipsigport-1": {1106}, "starschool": {2270}, "smc-jmx": {6786}, "d-fence": {8555}, "philips-vc": {583}, "netconf-beep": {831}, "delta-mcp": {1324}, "visitview": {1631}, "bluectrlproxy": {2277}, "scx-proxy": {470}, "wkstn-mon": {2991}, "amt-cnf-prot": {3054}, "squid-htcp": {4827}, "patrol": {8160}, "wnn6-ds": {26208}, "icb": {7326}, "digital-vrc": {466}, "openvpn": {1194}, "telindus": {1728}, "opequus-server": {2400}, "eppc": {3031}, "quickbooksrds": {3790}, "grid-alt": {6269}, "nexgen": {6627}, "passwrd-policy": {1333}, "dec-mbadmin-h": {1656}, "windb": {2522}, "savant": {3391}, "xmp": {5270}, "ttl-publisher": {5462}, "laes-bf": {9536}, "sometimes-rpc15": {32776}, "udrive": {1867}, "pnaconsult-lm": {2937}, "talarian-mcast2": {4016}, "sco-aip": {5307}, "lpdg": {10805}, "cajo-discovery": {1198}, "fujitsu-dtc": {1513}, "ito-e-gui": {2531}, "backupedge": {3946}, "radmin": {4899}, "dbsyncarbiter": {4953}, "worldscores": {4545}, "a17-an-an": {4599}, "encrypted_admin": {1138}, "cft-5": {1766}, "masc": {2587}, "zieto-sock": {4072}, "cl-db-request": {4136}, "axysbridge": {4418}, "snmpssh": {5161}, "pnet-conn": {7797}, "tcpnethaspsrv": {475}, "ansoft-lm-1": {1083}, "ste-smsc": {1836}, "aicc-cmi": {3316}, "plcy-net-svcs": {4351}, "upnotifyp": {4445}, "mmcals": {2271}, "urbisnet": {2745}, "mira": {3454}, "tigv2": {4124}, "diagnose-proc": {6072}, "canto-roboflow": {8998}, "monitor": {561}, "nani": {2236}, "markem-dcp": {3836}, "fdt-rcatp": {4320}, "presence": {5298}, "password-chg": {586}, "wms": {1755}, "bintec-admin": {2107}, "ottp": {2951}, "dhe": {3252}, "amt-soap-https": {16993}, "hpvirtgrp": {5223}, "continuus": {5412}, "masqdialer": {224}, "remotefs": {556}, "norton-lambert": {2338}, "homesteadglory": {2597}, "dist-upgrade": {3624}, "synchromesh": {4548}, "senomix06": {8057}, "server-find": {8351}, "navegaweb-port": {3159}, "xrpc-registry": {3651}, "rpki-rtr-tls": {324}, "LSA-or-nterm": {1026}, "cplscrambler-in": {1087}, "ehtp": {1295}, "hybrid": {1424}, "informatik-lm": {1428}, "turbonote-1": {39681}, "asmp": {45000}, "aal-lm": {1469}, "orion-rmi-reg": {2413}, "goahead-fldup": {3057}, "safetynetp": {40000}, "cuseeme": {7648}, "zigbee-ips": {17756}, "netmagic": {1196}, "cns-srv-port": {2976}, "ups-engine": {3664}, "cardifftec-back": {4573}, "actifioudsagent": {5106}, "jeol-nsdtp-3": {6243}, "us-srv": {8083}, "find": {24922}, "ni-ftp": {47}, "ulp": {522}, "trp": {2156}, "cxtp": {5091}, "office-tools": {7789}, "rocrail": {8051}, "sps-tunnel": {2876}, "elvin_client": {2917}, "ice-location": {4061}, "mzca-action": {7282}, "jesmsjc": {27442}, "iracinghelper": {32034}, "login": {513}, "nas": {991}, "transact": {1869}, "signet-ctf": {2733}, "ms-licensing": {5720}, "agentview": {2331}, "config-port": {3577}, "landmarks": {3969}, "ddi-udp-3": {8890}, "swdtp-sv": {10009}, "lonworks2": {2541}, "nds_sso": {3024}, "grubd": {3136}, "sysrqd": {4094}, "edonkey": {4662, 4666}, "imqbrokerd": {7676}, "xcompute": {11235}, "gearman": {4730}, "citrixupp": {7228}, "ident": {113}, "ndsauth": {353}, "simp-all": {1959}, "vsiadmin": {2539}, "media-agent": {2789}, "sftdst-port": {3230}, "apogeex-port": {3184}, "cmip-man": {163}, "teedtap": {559}, "nerv": {1222}, "ifor-protocol": {1515}, "mentaclient": {2117}, "gotodevice": {2217}, "noteza": {5215}, "beorl": {5633}, "umeter": {571}, "lanyon-lantern": {1682}, "gtp-user": {2152}, "ldap-admin": {3407}, "snac": {3536}, "updog": {4414}, "nbt-wol": {6133}, "rexecj": {8230}, "erunbook_server": {9617}, "syserverremote": {6418}, "ftp-data": {20}, "ipx": {213}, "mrm": {679}, "de-noc": {1254}, "hpiod": {2208}, "iims": {4800}, "funk-dialout": {2909}, "directnet": {3447}, "npds-tracker": {3680}, "backup-express": {6123}, "jwpc-bin": {16021}, "kuang2": {17300}, "ptp-general": {320}, "cadkey-tablet": {1400}, "scientia-sdb": {1811}, "canocentral1": {1872}, "intelliadm-disc": {4746}, "xmpv7": {7430}, "iceedcp_rx": {31949}, "digital-notary": {1335}, "wins": {1512}, "xnmp": {1652}, "netobjects1": {2485}, "pcptcpservice": {4182}, "rrdp": {5313}, "grid": {6268}, "pmcs": {6355}, "daytime": {13}, "tftp": {69}, "ns": {760}, "rxe": {761}, "groove-dpp": {1211}, "shivadiscovery": {1502}, "senomix01": {8052}, "oracle-ms-ens": {8997}, "asmps": {45001}, "patrol-mq-gm": {2664}, "mpfwsas": {2952}, "mdtp": {3232}, "unihub-server": {2357}, "alta-smp": {3433}, "ohmtrigger": {4732}, "nsdeepfreezectl": {7724}, "snss": {11171}, "netprowler-manager": {61439}, "new-rwho": {550}, "fc-cli": {1371}, "unify-debug": {4867}, "ibar": {5784}, "acctopus-st": {6868}, "p2p-sip": {6084}, "scup-disc": {6315}, "adobeserver-1": {1102}, "gpfs": {1191}, "giop-ssl": {2482}, "dzdaemon": {3866}, "hillrserv": {4117}, "lisp-cons": {4342}, "sum": {6551}, "rdgs": {16385}, "tinc": {655}, "cimtrak": {3749}, "peocoll": {9631}, "abb-hw": {10020}, "connected": {16384}, "sharp-server": {3617}, "cpdlc": {5911}, "ardus-trns": {1115}, "unisql-java": {1979}, "ptk-alink": {3089}, "vsaiport": {3317}, "802-11-iapp": {3517}, "camac": {3545}, "tetrinet": {31457}, "arbortext-lm": {1557}, "interbase": {2041}, "hsl-storm": {2113}, "checkmk-agent": {6556}, "mpls-udp": {6635}, "gcm-app": {14145}, "bre": {4096}, "hid": {24322}, "cfdptkt": {120}, "hpvmmagent": {1125}, "signal": {2974}, "cognima": {3779}, "wifree": {11208}, "cadis-2": {1442}, "meetingmaker": {3292}, "talarian-mcast3": {4017}, "amqps": {5671}, "diameters": {5868}, "drm-production": {7171}, "bakbonenetvault": {20031}, "vrace": {9300}, "ivs-database": {38000}, "entomb": {775}, "deskshare": {1702}, "mmpft": {1815}, "blockade-bpsp": {2574}, "sj3": {3086}, "ewinstaller": {4091}, "frc-hp": {6704}, "isi-gl": {55}, "fjicl-tep-b": {1902}, "ivsd": {2241}, "extensisportfolio": {2903}, "edm-mgr-sync": {3464}, "ds-slp": {4406}, "mailq": {174}, "chshell": {562}, "drwcs": {2193}, "appman-server": {3312}, "issc": {9992}, "rtmp": {1935}, "avauthsrvprtcl": {2068}, "etcd-server": {2380}, "sdo-tls": {3896}, "libelle": {8282}, "cce4x": {12000}, "qsc": {787}, "equationbuilder": {1351}, "bears-01": {2852}, "rets": {6103}, "boks": {6500}, "PS-Capture-Pro": {7216}, "hydap": {15000}, "busboy": {998}, "indigo-server": {1176}, "global-dtserv": {1774}, "osdcp": {3432}, "napster": {6699}, "lazy-ptop": {7099}, "fxuptp": {19539}, "pcanywhere": {65301}, "iclcnet_svinfo": {887}, "pcc-mfp": {2256}, "isg-uda-server": {2551}, "m-wnn": {3732}, "bolt": {7687}, "sec-pc2fax-srv": {9402}, "amicon-fpsu-ra": {30003}, "work-sol": {400}, "neod2": {1048}, "hpocbus": {2206}, "docker": {2375, 2376}, "ms-s-sideshow": {5361}, "sitewatch-s": {3766}, "teradataordbms": {8002}, "nntp": {119}, "asip-webadmin": {311}, "meter": {570}, "kpop": {1109}, "pxc-sapxom": {2680}, "webdata": {3748}, "ssh-mgmt": {17235}, "ftnmtp": {8502}, "smpte": {420}, "microcom-sbp": {1680}, "cifs": {3020}, "ceph": {3300}, "finisar": {4682}, "inetfs": {7775}, "mpsserver": {6106}, "exoline-tcp": {26486}, "cableport-ax": {282}, "sas-3": {1501}, "natuslink": {2895}, "nm-assessor": {3151}, "smcluster": {4174}, "amahi-anywhere": {4563}, "saltd-licensing": {29000}, "netrockey6": {4425}, "tritium-can": {4876}, "sntp-heartbeat": {580}, "netinfo": {1033}, "EtherNetIP-1": {2222}, "serverview-rm": {3172}, "jmq-daemon-2": {3215}, "hp-dataprotect": {3612}, "mpls-udp-dtls": {6636}, "dixie": {96}, "rimsl": {2044}, "backroomnet": {3387}, "med-fsp-tx": {24002}, "tc_ads_discovery": {48899}, "attachmate-uts": {2304}, "squid-http": {3128}, "snagas": {108}, "pim-rp-disc": {496}, "mesavistaco": {1249}, "bcs-lmserver": {1951}, "ellpack": {2025}, "nbx-ser": {2095}, "afs3-callback": {7001}, "mc-appserver": {8763}, "biimenu": {18000}, "at-8": {208}, "psmond": {1788}, "o2server-port": {1894}, "statusd": {5414}, "softcm": {6110}, "ipluminary": {7420}, "syslog-conn": {601}, "spamtrap": {2568}, "sctp-tunneling": {9899}, "reftek": {2543}, "pro-ed": {8032}, "tor-control": {9051}, "winpoplanmess": {1152}, "iclpv-dm": {1389}, "cops-tls": {3183}, "ias-neighbor": {4596}, "appserv-http": {4848}, "bosswave": {28589}, "vipera": {12012}, "iclid": {18242}, "smtp": {25}, "dslremote-mgmt": {2420}, "honyaku": {2744}, "ccmail": {3264}, "dandv-tester": {3889}, "ida-agent": {5051}, "persona": {1916}, "p25cai": {6082}, "privoxy": {8118}, "pim-port": {8471}, "bveapi": {10880}, "tsrmagt": {2077}, "t1-e1-over-ip": {3175}, "targus-getdata2": {5202}, "frc-lp": {6706}, "entextxid": {12000}, "sift-uft": {608}, "boscap": {2990}, "ssql": {3352}, "rendezvous": {3689}, "odmr": {366}, "silkp1": {2829}, "3com-njack-2": {5265}, "tal-pod": {6149}, "kz-migr": {8102}, "warehouse-sss": {12321}, "netdb-export": {1329}, "cvmmon": {2300}, "nlg-data": {5299}, "neckar": {37475}, "fcmsys": {2344}, "a25-fap-fgw": {4502}, "alesquery": {5074}, "rbt-wanopt": {7810}, "opendeploy": {20014}, "sncp": {7560}, "canon-bjnp1": {8611}, "apri-lm": {1447}, "patrol-mq-nm": {2665}, "softrack-meter": {3884}, "topflow-ssl": {3885}, "vinainstall": {4344}, "ca-1": {5064}, "mnp-exchange": {2197}, "zfirm-shiprush3": {3841}, "pslicser": {4168}, "wiegand": {4390}, "na-localise": {5062}, "xadmin": {9105}, "xsmsvc": {6936}, "zsecure": {7173}, "registrar": {1712}, "pda-data": {3253}, "diameter": {3868}, "srdp": {3942}, "l3t-at-an": {4591}, "vnc-3": {5903}, "imagepump": {27345}, "sns-query": {2659}, "user-manager": {3272}, "edm-adm-notify": {3463}, "sagectlpanel": {3698}, "kpdp": {5253}, "mppolicy-mgr": {5969}, "cichlid": {1377}, "virtual-places": {1533}, "contclientms": {4665}, "man": {9535}, "cscp": {40841}, "pcihreq": {3085}, "vnc-1": {5901}, "csoauth": {7847}, "sep": {2089}, "sflow": {6343}, "otpatch": {2936}, "rbr-discovery": {3553}, "tam": {209}, "infoman": {1451}, "nfs": {2049}, "mpnjsomg": {2686}, "esps-portal": {2867}, "flashmsg": {2884}, "qsm-remote": {1166}, "ftsrv": {1359}, "iscape": {5047}, "ohsc": {18186}, "synoptics-trap": {412}, "netopia-vo5": {1843}, "brlp-1": {4102}, "sentinel-lm": {5093}, "netagent": {5771}, "ddi-tcp-4": {8891}, "nexgen-aux": {6629}, "cisco-avp": {8470}, "yak-chat": {258}, "hyper-g": {418}, "gilatskysurfer": {3013}, "dsc": {3390}, "rbsystem": {5693}, "boe-pagesvr": {6405}, "robix": {9599}, "pammrpc": {1633}, "bmc-grx": {6300}, "tidp": {7548}, "ciscocsdb": {43441}, "wmereceiving": {11997}, "pacmand": {1307}, "conferencetalk": {1713}, "zymed-zpp": {2133}, "l2c-control": {4371}, "capwap-control": {5246}, "tile-ml": {10261}, "fpo-fns": {1066}, "sitaraserver": {2629}, "cpqrpm-agent": {3256}, "scservp": {3637}, "edb-server2": {3711}, "minecraft": {25565}, "wafs": {4049}, "kdm": {1024, 2115}, "sftsrv": {1303}, "taurus-wh": {1610}, "slp-notify": {1847}, "livestats": {2795}, "mqe-agent": {3958}, "asap-tcp": {3863}, "lms": {4056}, "elan": {1378}, "sg-lm": {1659}, "blackboard": {2032}, "spock": {2507}, "vytalvaultbrtp": {2546}, "pcmk-remote": {3121}, "doglms": {6088}, "tor-trans": {9040}, "subseven": {16959, 27374}, "sometimes-rpc19": {32778}, "entrustmanager": {709}, "proshare2": {1460}, "answersoft-lm": {1781}, "rapidbase": {1953}, "gs-realtime": {30400}, "acap": {674}, "orbixd": {1570}, "netrisk": {1799}, "ssmc": {2187}, "ppsms": {3967}, "documentum": {10002}, "microsoft-ds": {445}, "hpvmmcontrol": {1124}, "cops": {3288}, "serverwsd2": {5362}, "vvr-data": {8199}, "dai-shell": {45824}, "citrixuppg": {7229}, "vitalanalysis": {2474}, "opentelemetry": {4317}, "smar-se-port2": {4988}, "x11": {6010, 6011, 6012, 6013, 6014, 6015, 6016, 6017, 6018, 6019, 6020, 6021, 6022, 6023, 6024, 6025, 6026, 6027, 6028, 6029, 6030, 6031, 6032, 6033, 6034, 6035, 6036, 6037, 6038, 6039, 6040, 6041, 6042, 6043, 6044, 6045, 6046, 6047, 6048, 6049, 6050, 6051, 6052, 6053, 6054, 6055, 6056, 6057, 6058, 6060, 6061, 6062, 6063}, "ibprotocol": {6714}, "iccrushmore": {6850}, "starbot": {2838}, "exapt-lmgr": {3759}, "soap-http": {7627}, "danf-ak2": {1041}, "spandataport": {3193}, "pdrncs": {3299}, "ccss-qsm": {4970}, "z39.50": {210}, "npmp-trap": {609}, "pipe_server": {2010}, "bluelance": {2877}, "mediavault-gui": {3673}, "partimage": {4025}, "mediacntrlnfsd": {2363}, "vmsvc": {7024}, "usicontentpush": {7998}, "gsidcap": {22128}, "DragonIDSConsole": {9111}, "sun-sea-port": {16161}, "sstsys-lm": {1692}, "dlswpn": {2067}, "btpp2audctr1": {2536}, "unimobilectrl": {2927}, "traversal": {4678}, "crystalreports": {6400}, "maybe-fw1": {265}, "device": {801}, "cas": {2418}, "pyrrho": {5433}, "blackice-alerts": {8082}, "dnox": {4022}, "swistrap": {6965}, "rtsp-alt": {8554}, "nta-ds": {7544}, "bmc-gms": {10129}, "cadis-1": {1441}, "bmc-net-svc": {1770}, "direcpc-dll": {1844}, "facsys-router": {2515}, "trap-daemon": {3600}, "wlanauth": {3810}, "ibm-app": {385}, "peerenabler": {3531}, "upstriggervsw": {3786}, "eis": {3982}, "cucme-4": {7651}, "emc-pp-mgmtsvc": {9083}, "hmip-routing": {43438}, "cisco-fna": {130}, "tnETOS": {377}, "blwnkl-port": {2625}, "vtr-emulator": {3122}, "ncu-2": {3196}, "pmip6-cntl": {5436}, "servstat": {633}, "sbl": {1039}, "audio-activmail": {1397}, "powergemplus": {2899}, "rlp": {39}, "smakynet": {122}, "iso-tp0": {146}, "vfo": {1056}, "iascontrol-oms": {1156}, "apollo-cc": {2754}, "geneve": {6081}, "bpjava-msvc": {13722}, "bmpp": {632}, "sdsc-lm": {1537}, "hecmtl-db": {1551}, "tripe": {4070}, "ctisystemmsg": {4451}, "efr": {5618}, "sometimes-rpc21": {32779}, "winrm": {47001}, "mbus": {47000}, "rap-listen": {1531}, "ncu-1": {3195}, "aipn-reg": {4113}, "xmcp": {4788}, "spt-automation": {5814}, "cwmp": {7547}, "apc-6548": {6548}, "intu-ec-svcdisc": {8020}, "auditd": {48}, "resacommunity": {1154}, "fjicl-tep-a": {1901}, "rmtserver": {2416}, "smntubootstrap": {2613}, "psi-ptt": {4374}, "unisys-eportal": {37654}, "domain-s": {853}, "cinegrfx-elmd": {2891}, "xic": {6115}, "martalk": {7073}, "sgi-lk": {11106}, "jdatastore": {2508}, "re-mail-ck": {50}, "husky": {1310}, "imtc-map": {2202}, "sun-sr-jms": {6484}, "MobilitySrv": {6997}, "aero": {8060}, "world-lm": {1462}, "mikey": {2269}, "cr-websystems": {2314}, "sso-service": {2710}, "powerchuteplus": {6547, 6548, 6549}, "trisoap": {10200}, "prosharevideo": {5714}, "sane-port": {6566}, "siam": {498}, "rootd": {1094}, "saphostctrl": {1128}, "nimrod-agent": {1617}, "elpro_tunnel": {4370}, "a16-an-an": {4598}, "pqsflows": {9640}, "serverviewdbms": {9212}, "esl-lm": {1455}, "xinuexpansion3": {2023}, "nicetec-mgmt": {2557}, "sms-xfer": {2702}, "fotogcad": {3878}, "nettest": {4138}, "is99c": {379}, "lsdp": {11430}, "p-net-local": {34378}, "wkars": {2720}, "splitlock": {3606}, "excerpts": {5401}, "ansyslmd": {1055}, "vpvd": {1518}, "dsatp": {2111}, "hivep": {12172}, "ewctsp": {6066}, "ptp-event": {319}, "ggf-ncp": {678}, "atc-appserver": {1171}, "nimsh": {3901}, "ldxp": {4042}, "salient-dtasrv": {5409}, "arcisdms": {262}, "health-polling": {1161}, "els": {1315}, "vfmobile": {5646}, "sometimes-rpc6": {32771}, "dbbrowse": {47557}, "vocaltec-phone": {22555}, "tell": {754}, "gdoi": {848}, "krb5gatekeeper": {1318}, "emprise-lls": {3585}, "trap-port": {3857}, "parsec-peer": {6581}, "cylink-c": {5420}, "cefd-vmp": {10023}, "auth": {113}, "sco-dtmgr": {617}, "apex-mesh": {912}, "ipcore": {2215}, "webyast": {4984}, "fmwp": {5015}, "flexlm7": {27007}, "bctp": {8999}, "CardWeb-IO": {9060}, "vmnet": {175}, "payrouter": {1246}, "stdptc": {2154}, "sabp-signal": {3452}, "gre-udp-dtls": {4755}, "rlm": {5053}, "fjinvmgr": {9396}, "sype-transport": {9911}, "arena-server": {11321}, "nimgtw": {48003}, "analogx": {6588}, "rise": {7473}, "src": {200}, "netlabs-lm": {1406}, "apx500api-1": {2264}, "ms-olap3": {2382}, "raventbs": {2713}, "lsp-ping": {3503}, "ddi-udp-5": {8892}, "iuhsctpassoc": {29169}, "surf": {1010}, "3l-l1": {1511}, "vat": {3456}, "rtraceroute": {3765}, "seosload": {8892}, "odbcpathway": {9628}, "decbsrv": {579}, "tftp-mcast": {1758}, "eisp": {3983}, "amt-redir-tls": {16995}, "ardus-mtrns": {1117}, "casp": {1130}, "iclpv-nlc": {1394}, "smart-lm": {1608}, "sso-control": {2711}, "boe-was": {6401}, "ezmeeting": {26261}, "Trinoo_Register": {31335}, "radware-rpm": {2188}, "ars-master": {3176}, "bones": {4914}, "scpi-raw": {5025}, "hpvroom": {5228}, "sun-answerbook": {8888}, "sasg": {3744}, "sops": {3944}, "controlit": {799}, "vdmplay": {1707}, "xiip": {1924}, "boldsoft-lm": {2961}, "dtp-dia": {3489}, "questnotify": {3554}, "logcabin": {5254}, "pmcdproxy": {44322}, "linuxconf": {98}, "fiorano-rtrsvc": {1855}, "tvbus": {2191}, "dfoxserver": {2960}, "capmux": {4728}, "irtrans": {21000}, "ha-cluster": {694}, "iden-ralp": {1725}, "gprs-sig": {3386}, "hacl-poll": {5315}, "unot": {5055}, "3gpp-w1ap": {37472}, "swift-rvf": {97}, "ibm-pps": {1376}, "rrimwm": {1694}, "supermon": {2709}, "qtp": {2935}, "drmsfsd": {4098}, "openmanage": {7273}, "beeyond": {2414}, "oap-s": {3568}, "eims-admin": {4199}, "kwtc": {4566}, "opcua-udp": {4840}, "sip": {5060}, "irc": {194, 529, 6665, 6666, 6667, 6668, 6669, 6670}, "dtn1": {2445}, "sm-pas-4": {2941}, "garmin-marine": {8322}, "avtp": {17220}, "netopia-vo4": {1842}, "cnrp": {2757}, "exce": {2769}, "ias-paging": {4595}, "xfer": {82}, "netboot-pxe": {3928}, "cncp": {4785}, "pk": {5272}, "datascaler-ctl": {6625}, "ibm-diradm": {3538}, "agcat": {3915}, "cadlock": {770, 1000}, "supfilesrv": {871}, "q55-pcc": {1253}, "hydra": {2374}, "autocuelog": {3104}, "tarantella": {3144}, "eenet": {5234}, "vmsvc-2": {7025}, "etlservicemgr": {9001}, "golem": {9005}, "IIS": {1027}, "p2pq": {1981}, "fjippol-swrly": {2747}, "stonefalls": {2986}, "edm-mgr-cntrl": {3465}, "espeech-rtp": {8417}, "custix": {528}, "LiebDevMgmt_C": {3027}, "edm-stager": {3461}, "soagateway": {5250}, "micromuse-ncps": {7979}, "csccfirewall": {40843}, "pserver": {3662}, "esp": {6567}, "ocs_amu": {429}, "jetform": {1706}, "caicci": {1721}, "konshus-lm": {2294}, "gwen-sonya": {2778}, "findviatv": {3350}, "cdc": {223}, "avenue": {2134}, "apani1": {9160}, "scscp": {26133}, "xtell": {4224}, "nitrogen": {7725}, "rtsps": {322}, "netview-aix-9": {1669}, "hl7": {2575}, "eristwoguns": {2650}, "rapidmq-reg": {3094}, "nokia-ann-ch1": {3405}, "wap-vcal": {9205}, "rushd": {696}, "aspeclmd": {1544}, "sms-remctrl": {2704}, "wnn6_Cn": {22289}, "imyx": {1143}, "vrts-registry": {2410}, "timelot": {3243}, "spugna": {3807}, "nfapi": {7701}, "route": {520}, "fhsp": {1807}, "qadmifoper": {2461}, "k3software-svr": {26262}, "ddi-udp-6": {8893}, "webphone": {21845}, "unicontrol": {2437, 2499}, "d2k-tapestry1": {3393}, "smartcard-port": {3516}, "faxstfx-port": {3684}, "ads-c": {5913}, "ircs-u": {6697}, "x2-control": {36422}, "neod1": {1047}, "broker_service": {3014}, "edix": {3123}, "apc-5455": {5455}, "radmind": {6222, 6662}, "emc-vcas-udp": {13218}, "boe-processsvr": {6406}, "saphostctrls": {1129}, "web2host": {1559}, "msiccp": {1731}, "mfserver": {2266}, "yo-main": {4040}, "mtcevrunqss": {4557}, "kryptolan": {398}, "symantec-sim": {3547}, "mythtv": {6543, 6544}, "mpidcmgr": {9343}, "priv-term": {57}, "kpasswd5": {464}, "ecolor-imager": {3263}, "v-one-spp": {3845}, "cfengine": {5308}, "xinupageserver": {2020}, "journee": {3042}, "interworld": {3548}, "mapper-nodemgr": {3984}, "a4-sdunode": {5605}, "tenfold": {658}, "exbit-escp": {1316}, "pxc-epmap": {2434}, "farenet": {5557}, "websnp": {8084}, "jeol-nsddp-2": {6242}, "ddi-udp-1": {8888}, "nip": {376}, "ldapssl": {636}, "sunclustermgr": {1097}, "argis-ds": {2582}, "bintec-capi": {2662}, "rcst": {3467}, "paragent": {9022}, "quake": {26000}, "menandmice-lpm": {1231}, "healthd": {1281}, "clearvisn": {2052}, "wms-messenger": {3219}, "ctp": {3772}, "nexus-portal": {4021}, "cert-initiator": {1639}, "netview-aix-8": {1668}, "fjitsuappmgr": {2425}, "thingkit": {4423}, "trdp-pd": {17224}, "entrust-aams": {681}, "stars": {4131}, "icl-twobase7": {25006}, "hp-collector": {381, 781}, "travsoft-ipx-t": {2644}, "e3consultants": {3157}, "cytel-lm": {3297}, "armcenterhttp": {9294}, "stel": {10005}, "pwdgen": {129}, "gxtelmd": {2356}, "signacert-agent": {5032}, "tarp": {6442}, "flexlm10": {27010}, "overnet": {16444}, "igrid": {19000}, "rightbrain": {1354}, "netop-rc": {1970, 6502}, "udrawgraph": {2542}, "acc-raid": {2800}, "casanswmgmt": {3669}, "oemcacao-jmxmp": {11172}, "mountd": {20048}, "vofr-gateway": {21590}, "cadview-3d": {649}, "kyoceranetdev": {1063}, "funk-logger": {1786}, "es-elmd": {1822}, "extensis": {2666}, "interserver": {3060}, "kiosk": {1061}, "rds2": {1541}, "mvel-lm": {1574}, "interintelli": {2633}, "asci-val": {1560}, "idp-infotrieve": {2966}, "hylafax": {4559}, "sun-sr-https": {6443}, "tolfab": {20167}, "osu-nms": {192}, "iascontrol": {1157}, "netview-aix-6": {1666}, "squid-snmp": {3401}, "flirtmitmir": {3840}, "mxodbc-connect": {6632}, "ms-theater": {2460}, "infiniswitchcl": {3602}, "notateit-disc": {4803}, "cps": {14250}, "msp": {18, 2438}, "finger": {79, 2003}, "nss-routing": {159}, "intv": {1585}, "tsilb": {2489}, "patrol-coll": {6162}, "serialgateway": {1243}, "mtn": {4691}, "avdecc": {17221}, "olsr": {698}, "availant-mgr": {1122}, "sunscalar-svc": {1860}, "mysql": {3306}, "openmailns": {5766}, "suncacao-csa": {11164}, "ddrepl": {4126}, "jaxer-manager": {4328}, "serialnumberd": {626}, "proshare-mc-2": {1674}, "3Com-nsd": {1742}, "vytalvaultpipe": {2548}, "pktcablemmcops": {3918}, "ravehd": {4037}, "getty-focus": {4332}, "decvms-sysmgt": {441}, "zented": {1229}, "discovery-port": {1925}, "vx-auth-port": {3207}, "digivote": {3223}, "oms-nonsecure": {5102}, "stuns": {5349}, "metaedit-ws": {6390}, "d2ddatatrans": {9388}, "silkmeter": {5461}, "senomix02": {8053}, "ftp-agent": {574}, "boinc": {1043, 31416}, "queueadm": {2230}, "lnvalarm": {2282}, "fjippol-polsvr": {2748}, "acms": {3980}, "bhmds": {310}, "sun-manageconsole": {898}, "openmail": {5729}, "nmea-onenet": {10111}, "gandalf-lm": {1421}, "unbind-cluster": {2138}, "hpstgmgr2": {2715}, "acp-conduit": {3823}, "iss-mgmt-ssl": {3995}, "playsta2-app": {4658}, "mipv6tls": {7872}, "pds": {9595}, "http-rpc-epmap": {593}, "alarm-clock-s": {2667}, "wcpp": {4185}, "camp": {4450}, "vnc-http-1": {5801}, "boe-cachesvr": {6403}, "idxp": {603}, "hp-webadmin": {1188}, "anet-l": {3339}, "vsnm-agent": {3375}, "mupdate": {3905}, "athand-mmp": {20999}, "prm-nm": {409}, "epncdp2": {3259}, "ironstorm": {3504}, "gog-multiplayer": {5687}, "afesc-mc": {6628}, "sftp": {115}, "leoip": {1886}, "vrtl-vmf-ds": {1956}, "roboeda": {2920}, "pafec-lm": {7511}, "gadugadu": {8074}, "papachi-p2p-srv": {8027}, "mles": {8077}, "ewall": {1328}, "down": {2022}, "mpfoncl": {2579}, "nvmsgd": {3519}, "phrelay": {4868}, "outlaws": {5310}, "dey-keyneg": {8750}, "cronus": {148}, "nsjtp-ctrl": {1687}, "idrs": {2995}, "sunlps-http": {3816}, "http-proxy": {8080}, "multicast-ping": {9903}, "fs-qos": {41111}, "crestron-cips": {41796}, "opc-job-start": {423}, "vrtl-vmf-sa": {2074}, "sccip-media": {3499}, "ieee-mih": {4551}, "sicct": {4742}, "rp-reputation": {6568}, "nssocketport": {3522}, "magaya-network": {3691}, "flexlm": {744}, "rdrmshc": {1075}, "ansoft-lm-2": {1084}, "altav-remmgt": {2456}, "nsc-ccs": {2604}, "csd-monitor": {3072}, "sgi-dmfmgr": {11109}, "crestron-ctps": {41797}, "realm-rusd": {688}, "bridgecontrol": {1073}, "loaprobe": {1634}, "radware-rpm-s": {2189}, "ice-srouter": {4064}, "pfcp": {8805}, "ms-sql2000": {9152}, "sphinxapi": {9312}, "objcall": {94}, "mortgageware": {367}, "netview-aix-7": {1667}, "fjippol-port2": {2751}, "cbos-ip-port": {3750}, "nssalertmgr": {4453}, "aequus": {23456}, "newacct": {100}, "ircs": {994}, "imtc-mcs": {1503}, "videte-cipc": {1927}, "isomair": {3589}, "sapms": {9310}, "qmvideo": {5689}, "ionixnetmon": {7410}, "go-login": {491}, "mimer": {1360}, "oracle": {1521, 1525, 2005}, "navisphere": {2162}, "icslap": {2869}, "gre-in-udp": {4754}, "wlbs": {2504}, "gte-samp": {2643}, "nati-dstp": {3015}, "enpc": {3289}, "ogs-server": {9008}, "i-zipqd": {13160}, "mpm-flags": {44}, "cinegrfx-lm": {1743}, "adaptecmgr": {2521}, "conspiracy": {4692}, "ppactivation": {5134}, "rmiaux": {10990}, "3par-rcopy": {5785}, "cucme-2": {7649}, "saris": {4442}, "boe-resssvr4": {6410}, "dsdn": {1292}, "oceansoft-lm": {1466}, "snifferserver": {2533}, "cluster-disc": {3374}, "six-degrees": {3611}, "l-acoustics": {4432}, "mrssrendezvous": {7392}, "rsf-1": {1195}, "dca": {1456}, "ingreslock": {1524}, "nsw-fe": {27}, "as-servermap": {449}, "itu-bicc-stc": {3097}, "topovista-data": {3906}, "aiagent": {7738}, "semantix": {361}, "sfs-smp-net": {451}, "kentrox-prot": {2502}, "remoteware-un": {2999}, "ltp": {4044}, "synotics-relay": {391}, "pop3s": {995}, "homeportal-web": {3941}, "cucme-1": {7648}, "amicon-fpsu-s": {30004}, "oracle-oms": {1159}, "tn-timing": {2739}, "gcsp": {3429}, "ndn": {6363}, "tgp": {1223}, "servexec": {2021}, "samsung-disc": {30832}, "nokia-ann-ch2": {3406}, "ssowatch": {3644}, "apple-imap-admin": {626}, "us-gv": {1370}, "npdbgmngr": {2293}, "toad": {2669}, "dialpad-voice2": {2861}, "csms2": {3400}, "llmnr": {5355}, "p4p-portal": {6671}, "op-probe": {7030}, "autodesk-lm": {1422}, "skip-mc-gikreq": {1660}, "active-net": {3322, 3323, 3324, 3325}, "ttat3lb": {3579}, "abcvoice-port": {3781}, "palace-6": {9997}, "controlone-con": {7551}, "hncp-dtls-port": {8232}, "hostname": {101}, "fxp": {286, 2849}, "spamassassin": {783}, "ndl-aps": {3096}, "dsx-agent": {3685}, "cedros_fds": {4140}, "DMExpress": {32636}, "ncd-diag-tcp": {5978}, "tipc": {6118}, "microtalon-com": {7014}, "pcsync-https": {8443}, "snmptls": {10161}, "tungsten-http": {9762}, "oemcacao-rmi": {11174}, "sdserv": {5530}, "personal-link": {281}, "mt-scaleserver": {2305}, "orbix-locator": {3075}, "hp-pxpib": {3101}, "gc-config": {3436}, "dtp": {3663}, "mck-ivpip": {2698}, "gsakmp": {3761}, "pxc-splr": {4007}, "icl-twobase1": {25000}, "kastenxpipe": {36865}, "td-replica": {268}, "h2gf-w-2m": {3179}, "wbem-https": {5989}, "lanner-lm": {4547}, "taskman-port": {2470}, "pangolin-laser": {3348}, "bmc-ea": {3683}, "timestenbroker": {3754}, "scp": {3820}, "cbp": {4419}, "pon-ictp": {7202}, "crestron-ctp": {41795}, "ginad": {634}, "dcs": {1367}, "uaiact": {1470}, "kali": {2213}, "olhost": {2661}, "mailprox": {3936}, "rplay": {5555}, "supportassist": {5700}, "cplscrambler-al": {1088}, "sacred": {1118}, "radio": {1595}, "ctcd": {1851}, "lhtp": {1983}, "secure-cfg-svr": {3978}, "opsec-uaa": {19191}, "faxcomservice": {6417}, "skip-cert-send": {6456}, "hems": {151}, "ddt": {1052}, "vrts-ipcserver": {1317}, "lbc-sync": {2779}, "cd3o-protocol": {3616}, "menandmice_noh": {4151}, "borland-dsj": {707}, "idmaps": {1884}, "lbc-control": {2780}, "sflm": {3162}, "jomamqmonitor": {4114}, "sysinfo-sp": {11967}, "tmosms1": {5581}, "mftp": {349, 5402}, "portgate-auth": {3710}, "brlp-3": {4104}, "swxadmin": {5043}, "oma-mlp-s": {9211}, "at-zis": {206}, "xmltec-xmlmail": {9091}, "memcache": {11211}, "ciscopop": {45000}, "picknfs": {1598}, "dbm": {2345}, "nms-dpnss": {2503}, "pcep": {4189}, "x2e-disc": {11877}, "nirp": {4043}, "dec-mbadmin": {1655}, "colubris": {3490}, "enguity-xccetp": {8041}, "boxp": {9380}, "ipcserver": {600}, "pkix-3-ca-ra": {829}, "wag-service": {2608}, "dyn-site": {3932}, "checkoutdb": {5505}, "at-nbp": {202}, "entrust-aaas": {680}, "ecnp": {2858}, "jeol-nsdtp-4": {6244}, "ethercat": {34980}, "ncpm-pm": {1591}, "boks_servc": {6501}, "ovladmgr": {7428}, "inovaport2": {23001}, "nilinkanalyst": {25902}, "sco-sysmgr": {616}, "precise-sft": {2315}, "apdap": {3948}, "syslog-tls": {6514}, "ajp13": {8009}, "robot-remote": {8270}, "bandwiz-system": {1929}, "estamp": {1982}, "websm": {9090}, "irdg-post": {2632}, "ecp": {3134}, "bmc-reporting": {4568}, "cis-secure": {22343}, "talikaserver": {22763}, "directplay": {2234}, "ansysli": {2325}, "messageasap": {6070}, "ddm-rdb": {446}, "gdomap": {538}, "msql": {1112, 4333}, "elad": {1893}, "remctl": {4373}, "sgi-eventmond": {5553}, "simbaexpress": {1583}, "midnight-tech": {3008}, "svnet": {3413}, "ampl-lic": {5195}, "pulsonixnls": {6140}, "lm-dta": {8206}, "isis-ambc": {1643}, "giop": {2481}, "ep-pcp": {3620}, "eserver-pap": {3666}, "playsta2-lob": {4659}, "celatalk": {3485}, "lansurveyor": {4347}, "prospero": {191}, "opc-job-track": {424}, "blueberry-lm": {1432}, "lnvstatus": {2283}, "ttc-etap": {2675}, "feitianrockey": {3152}, "dtn-bundle-udp": {4556}, "swismgr1": {6963}, "xmsg": {1716}, "globe": {2002}, "mmacomm": {4667}, "doceri-ctl": {7019}, "csccredir": {40842}, "time": {37}, "retrospect": {497}, "sometimes-rpc1": {1012}, "psprserver": {2354}, "avi-nms": {4413}, "dxspider": {8873}, "marathontp": {8384}, "essbase": {1423}, "invision": {1641}, "licensedaemon": {1986}, "taskmaster2000": {2402, 2403}, "plethora": {3480}, "mobile-p2p": {4688}, "dlep": {854}, "cymtec-port": {1898}, "sercomm-wlink": {2235}, "omscontact": {4161}, "iceedcp_tx": {31948}, "netprowler-manager2": {61440}, "attachmate-g32": {2317}, "orbiter": {2398}, "apparenet-ts": {3236}, "jeol-nsddp-1": {6241}, "ubroker": {7887}, "pdefmns": {16311}, "appleqtc": {458}, "naap": {1340}, "innosys": {1412}, "winddx": {1727}, "jerand-lm": {1810}, "xmpp-server": {5269}, "trc-netpoll": {2405}, "orion": {2407}, "gadgetgate1way": {2677}, "tmi": {8300}, "sgcp": {440}, "axon-lm": {1548}, "acis": {9953}, "autotrac-acp": {31020}, "tor-socks": {9050}, "fjdmimgr": {9374}, "xingmpeg": {1558}, "saiseh": {1644}, "cardbox-http": {3106}, "cl-db-attach": {4135}, "io-dist-group": {5728}, "grasp": {7017}, "wnn6_Kr": {22305}, "ontime": {1622}, "noadmin": {1921}, "ccu-comm-1": {4053}, "pulseaudio": {4713}, "varadero-1": {4838}, "rscs": {10201}, "fsportmap": {4349}, "rfb": {5900}, "nced": {404}, "pirp": {553}, "brcd": {1323}, "softdataphone": {1621}, "banyan-net": {2708}, "dbcontrol_agent": {3938}, "vmware-fdm": {8182}, "commplex-link": {5001}, "netsc-dev": {155}, "send": {169}, "aspen-services": {1749}, "wimaxasncp": {2231}, "mkm-discovery": {3837}, "talarian-mcast5": {4019}, "pscl-mgt": {4312}, "ife_icorp": {5165}, "x-bone-ctl": {265}, "ups": {401}, "ace-client": {2334}, "zebra": {2601}, "firstcall42": {2673}, "e-net": {3286}, "webpush": {1001}, "galileolog": {3520}, "connect-proxy": {5490}, "alljoyn": {9956}, "slmap": {36423}, "ibus": {8733}, "x510": {9877}, "datasurfsrvsec": {462}, "omfs": {723}, "airs": {1481}, "a14": {3597}, "vnc-http-2": {5802}, "sttunnel": {7471}, "pacom": {3435}, "ccm-port": {3575}, "rfx-lm": {1497}, "ncconfig": {1888}, "sugp": {1905}, "ecwcfg": {2263}, "ewnn": {2674}, "sdp-id-port": {3242}, "xprtld": {5634}, "osvr": {7728}, "ospf-lite": {8899}, "ndm-agent-port": {43189}, "armagetronad": {4534}, "adap": {6350}, "cmtp-mgt": {8501}, "swa-2": {9024}, "vsixml": {2996}, "ds-clnt": {4402}, "maybe-veritas": {4987, 4998}, "ovhpas": {7510}, "ew-mgmt": {43440}, "escp-ip": {621}, "nessus": {1241, 3001}, "jtag-server": {1309}, "cft-0": {1761}, "tvdumtray-port": {3492}, "netperf": {12865}, "netsaint": {1040}, "trim": {1137}, "namemunge": {3950}, "saratoga": {7542}, "vacdsm-sws": {670}, "ias-admind": {2141}, "tsb": {2741}, "ps-ams": {3658}, "sun-sr-iiops": {6486}, "item": {3848}, "mdnsresponder": {5354}, "submit": {773}, "nicetec-nmsvc": {2556}, "rblcheckd": {3768}, "mrip": {4986}, "mpidcagt": {9397}, "sometimes-rpc26": {32786}, "excerpt": {5400}, "ttlpriceproxy": {5463}, "ni-mail": {61}, "netconfsoaphttp": {832}, "femis": {1776}, "combox-web-acc": {2534}, "acl-manager": {4013}, "undo-lm": {5281}, "oemcacao-websvc": {11175}, "ezproxy": {26260}, "snifferdata": {2892}, "quest-data-hub": {3566}, "stx": {527}, "netviewdm1": {729}, "qsm-gui": {1165}, "eyetv": {2170}, "tivoconnect": {2190}, "ripngd": {2603}, "juxml-port": {3642}, "icms": {4486}, "aprigo-cs": {5086}, "icl-twobase9": {25008}, "abbaccuray": {1546}, "cypress-stat": {2017}, "priv-file": {59}, "print-srv": {170}, "iris-xpcs": {714}, "anthony-data": {1206}, "bbn-mmc": {1347}, "nms": {1429}, "appswitch-emp": {2616}, "evtp": {2834}, "cardbox": {3105}, "arcpd": {3513}, "iqobject": {48619}, "xns-courier": {165}, "smux": {199}, "rrisat": {1697}, "xtrm": {3423}, "nsp": {5012}, "sbackup": {5163}, "deos": {76}, "syscomlan": {1065}, "sophia-lm": {1408}, "ovalarmsrv-cmd": {2954}, "qsnet-workst": {4355}, "appserv-https": {4849}, "emp-server1": {6321}, "accu-lmgr": {7781}, "amidxtape": {10083}, "fsr": {7164}, "clusterxl": {18243}, "dhcp-failover2": {847}, "coauthor": {1529}, "pns": {2487}, "asam": {3451}, "azeti-bd": {4192}, "carracho": {6700, 6701}, "flexlm1": {27001}, "prm-sm": {408}, "hp-nnm-data": {2690}, "magbind": {3194}, "bfd-multi-ctl": {4784}, "netsupport": {5405}, "smc-http": {6788}, "hnm": {6791}, "rds-ip": {18635}, "ardt": {1826}, "pn-requester": {2717}, "nesh-broker": {3507}, "avanti_cdp": {4065}, "npsp": {4088}, "frcs": {4915}, "qdb2service": {45825}, "pcsync-http": {8444}, "callwaveiam": {9283}, "intersan": {1331}, "vpac": {1517}, "mcagent": {1820}, "dialog-port": {2098}, "msfw-replica": {2173}, "emp-server2": {6322}, "nfsrdma": {20049}, "siemensgsm": {28240}, "tacacs": {49}, "ipp": {631}, "noagent": {1917}, "jetcmeserver": {1936}, "trinity-dist": {4711}, "uniport": {9629}, "apsolab-col": {5470}, "apc-9951": {9951}, "kerberos-sec": {88}, "webmachine": {1963}, "njenet-ssl": {2252}, "thrtx": {4139}, "tram": {4567}, "dhct-alerts": {4676}, "murray": {1123}, "jt400-ssl": {3471}, "msr-plugin-port": {3931}, "about": {2019}, "citrixadmin": {2513}, "vrml-multi-use": {4200, 4201, 4202, 4203, 4204, 4205, 4206, 4207, 4208, 4209, 4210, 4211, 4212, 4213, 4214, 4215, 4216, 4217, 4218, 4219, 4220, 4221, 4222, 4223, 4224, 4225, 4226, 4227, 4228, 4229, 4230, 4231, 4232, 4233, 4234, 4235, 4236, 4237, 4238, 4239, 4240, 4241, 4242, 4243, 4244, 4245, 4246, 4247, 4248, 4249, 4250, 4251, 4252, 4253, 4254, 4255, 4256, 4257, 4258, 4259, 4260, 4261, 4262, 4263, 4264, 4265, 4266, 4267, 4268, 4269, 4270, 4271, 4272, 4273, 4274, 4275, 4276, 4277, 4278, 4279, 4280, 4281, 4282, 4283, 4284, 4285, 4286, 4287, 4288, 4289, 4290, 4291, 4292, 4293, 4294, 4295, 4296, 4297, 4298, 4299}, "vstat": {7779}, "ftp-proxy": {8021}, "puparp": {998}, "td-postman": {1049}, "iph-policy-adm": {2963}, "anet-m": {3340}, "apc-7846": {7846}, "netspeak-cps": {21849}, "rap": {38, 256}, "powerburst": {485}, "gemini-lm": {1590}, "opcon-xps": {3100}, "dwnmshttp": {3227}, "gw-call-port": {3745}, "neon24x7": {3213}, "suucp": {4031}, "nat-t-ike": {4500}, "pcanywherestat": {5632}, "faximum": {7437}, "mbap": {502}, "sysopt": {3281}, "lavenir-lm": {3373}, "smar-se-port1": {4987}, "apc-6549": {6549}, "parsec-game": {6582}, "systemics-sox": {5406}, "apani4": {9163}, "warehouse": {12322}, "powwow-server": {13224}, "oc-lm": {1448}, "netx-agent": {2586}, "syam-smc": {3895}, "patrolview": {4097}, "swgps": {15126}, "isdninfo": {6105, 6106}, "netsc-prod": {154}, "qbikgdp": {368}, "amt": {2268}, "sysorb": {3241}, "directv-tick": {3336}, "eportcomm": {4666}, "opsec-ela": {18187}, "instl_bootc": {1068}, "qip-login": {2366}, "netobjects2": {2486}, "apparenet-as": {3238}, "wbem-exp-https": {5990}, "arcserve": {6050}, "progistics": {3973}, "minger": {4069}, "hmmp-op": {613}, "scol": {1200}, "nvd": {2184, 2329}, "dna": {2287}, "corbaloc": {2809}, "sm-pas-5": {2942}, "lorica-in-sec": {4081}, "deploymentmap": {4570}, "shiprush-d-ch": {5841}, "backupexec": {6101}, "opsec-ufp": {18182}, "galaxy7-data": {38201}, "hp-sca": {19411}, "acr-nema": {104}, "ets": {1569}, "rhp-iibp": {1912}, "apx500api-2": {2265}, "wsm-server-ssl": {5007}, "osaut": {6679}, "netconf-tls": {6513}, "mtl8000-matrix": {8115}, "ipcd": {576}, "timbuktu-srv1": {1417}, "vergencecm": {2771}, "adrep": {3954}, "lorica-in": {4080}, "mnet-discovery": {5237}, "oob-ws-http": {623}, "bess": {3960}, "m-oap": {5567}, "udt_os": {1382, 3900}, "must-backplane": {3515}, "mqe-broker": {3957}, "lorica-out-sec": {4083}, "at-rtmp": {201}, "zincite-a": {1034}, "edtools": {1142}, "seclayer-tls": {3496}, "mtsserver": {4602}, "screencast": {1368}, "fryeserv": {2788}, "hp-san-mgmt": {3037}, "apc-3052": {3052}, "serverview-as": {3169}, "maxim-asics": {3276}, "opsession-prxy": {3307}, "fatpipe": {3353}, "abatemgr": {3655}, "intelsync": {3692}, "plbserve-port": {3933}, "ivs-insertion": {38001}, "g5m": {2732}, "ifsf-hb-port": {3486}, "ninaf": {5627}, "mipi-debug": {7606}, "abyss": {9999}, "spike": {4683}, "esri_sde": {5151}, "java-or-OTGfileshare": {1050}, "smart-diagnose": {2721}, "cdid": {3315}, "sdo": {3635}, "spw-dialer": {3796}, "netconf-ch-ssh": {4334}, "tnp-discover": {8320}, "scintilla": {19007}, "axis-wimp-port": {10260}, "tacacs-ds": {65}, "mount": {635}, "caiccipc": {1202}, "wap-push-http": {4035}, "cisco-vpath-tun": {6633}, "oma-dcdocbs": {7278}, "trendchip-dcp": {3608}, "apollo-relay": {10252}, "netwall": {533}, "apex-edge": {913}, "propel-msgsys": {1268}, "netcomm2": {1676}, "linkname": {1903}, "iqserver": {2527}, "mc-comm": {9632}, "ddns-v3": {2164}, "silkp3": {2831}, "aap": {2878}, "slinterbase": {3065}, "capwap-data": {5247}, "synapse-nhttps": {8243}, "lmp": {701}, "biap-mp": {1962}, "vytalvaultvsmp": {2547}, "aamp": {3939}, "sieve": {4190}, "rtsp": {554}, "jstel": {1064}, "dlsrap": {1973}, "and-lm": {2646}, "biolink-auth": {3411}, "pluribus": {3469}, "msg-auth": {31}, "xyplex-mux": {173}, "ntalk": {518}, "ufsd": {1008}, "aibkup": {4071}, "ew-disc-cmd": {43440}, "oracle-em2": {1754}, "idcp": {2326}, "pmcp": {3821}, "recvr-rc": {43000}, "imaps": {993}, "nowcontact": {3167}, "sdp-portmapper": {3935}, "iax": {4569}, "sge_qmaster": {6444}, "ezmeeting-2": {10101}, "xtreelic": {996}, "redstorm_find": {2347}, "venus-se": {2431}, "suitjd": {3354}, "dcap": {22125}, "icl-twobase6": {25005}, "trivnet1": {8200}, "aic-oncrpc": {2786}, "mc-brk-srv": {3180}, "synel-data": {3734}, "ctdhercules": {3773}, "sgi-arrayd": {5434}, "toad-bi-appsrvr": {8066}, "acter": {4671}, "westec-connect": {5566}, "aeroflight-ads": {1218}, "unisys-lm": {1823}, "caupc-remote": {2122}, "cadencecontrol": {2318}, "tcpdataserver": {3805}, "spectraport": {3851}, "a3-sdunode": {5604}, "hnmp": {6790}, "versatalk": {3738}, "jaxer-web": {4327}, "remote-kis": {185}, "ibm-db2": {523, 50000}, "device2": {2030}, "LiebDevMgmt_DM": {3028}, "fjhpjp": {3067}, "ssmpp": {3550}, "phrelaydbg": {4869}, "scotty-disc": {14002}, "gruber-cashreg": {28010}, "rcts": {2258}, "cdn": {2412}, "att-mt-sms": {5586}, "dt-mgmtsvc": {6325}, "netbackup": {13701, 13702, 13705, 13706, 13708, 13709, 13710, 13711, 13712, 13713, 13714, 13715, 13716, 13717, 13718, 13720, 13721, 13722, 13782, 13783}, "cora": {19220}, "bvtsonar": {1149}, "dellpwrappks": {1266}, "isis-am": {1642}, "cpq-wbem": {2301}, "smwan": {3979}, "rwhois": {4321}, "talk": {517}, "pana": {716}, "nec-raidplus": {2730}, "blizwow": {3724}, "sunwebadmin": {8800}, "osxwebadmin": {16080}, "jtnetd-server": {5033}, "ldoms-migr": {8101}, "webster": {765, 2627}, "circle-x": {2931}, "wsdl-event": {4879}, "pgps": {9280}, "peoctlr": {9630}, "filenet-tms": {32768}, "td-service": {267}, "shell": {514}, "gmrupdateserv": {1070}, "flr_agent": {4901}, "powerschool": {5071}, "3com-tsmux": {106}, "gist": {270}, "radio-bc": {1596}, "spcsdlobby": {2888}, "watchme-7272": {7272}, "gopher": {70}, "tcoaddressbook": {1977}, "nas-metering": {2286}, "corel_vncadmin": {2654}, "sia-ctrl-plane": {4787}, "veritas_pbx": {1556}, "hello": {1789}, "opsession-clnt": {3303}, "tnos-dps": {7903}, "showcockpit-net": {22333}, "cert-responder": {1640}, "netop-school": {1971}, "snapp": {2333}, "stvp": {3158}, "fcopy-server": {5745}, "statsci1-lm": {6144}, "nfsd-status": {1110}, "rugameonline": {5156}, "br-channel": {5425}, "wnn6_DS": {26208}, "snmpdtls-trap": {10162}, "sometimes-rpc18": {32777}, "hdl-srv": {2641}, "dyna-lm": {3395}, "csms": {3399}, "wysdma": {3741}, "rsqlserver": {4430}, "fcp-addr-srvr2": {5501}, "statsci2-lm": {6145}, "svbackup": {8405}, "serverview-asn": {3170}, "cpqrpm-server": {3257}, "mc3ss": {3521}, "raven-rmp": {3532}, "ndsconnect": {3890}, "cc-tracking": {4870}, "flexlm0": {27000}, "sometimes-rpc5": {32771}, "nimhub": {48002}, "privateark": {1858}, "powerclientcsf": {2443}, "res-sap": {3163}, "lumimgrd": {4741}, "socp-c": {4882}, "sync-em7": {7707}, "netcp": {395, 740}, "precise-vip": {2924}, "smip": {7734}, "news": {144, 2009}, "srcp": {4303}, "infobright": {5029}, "rads": {12302}, "soap-beep": {605}, "mdc-portmapper": {685}, "raqmon-pdu": {7744}, "mxxrlogin": {1035}, "rbakcup2": {2774}, "ariliamulti": {3140}, "mindprint": {8033}, "direcpc-video": {1825}, "sunfm-port": {3934}, "ias-session": {4594}, "zigbee-ip": {17755}, "sometimes-rpc17": {32777}, "bmc-ar": {2494}, "ms-v-worlds": {2525}, "arepa-cas": {3030}, "symantec-sfdb": {5629}, "winqedit": {7395}, "scpi-telnet": {5024}, "rms-agent": {5994}, "integra-sme": {484}, "sco-websrvrmg3": {598}, "b2n": {1179}, "iwserver": {2166}, "chmd": {3099}, "vxlan": {4789}, "miami-bcast": {6083}, "dsx_monitor": {31685}, "http-alt": {591, 8000, 8008, 8080}, "downtools-disc": {5245}, "fcp-cics-gw1": {5504}, "ldoms-mgmt": {6482}, "timbuktu": {407}, "pn-requester2": {2718}, "sky-transport": {3556}, "nessus-xmlrpc": {8834}, "sgmp-traps": {160}, "dbreporter": {1379}, "mvx-lm": {1510}, "webmethods-b2b": {2907}, "roboer": {2919}, "sae-urn": {4500}, "b-novative-ls": {1896}, "lstp": {2559}, "pcc-image-port": {3892}, "intecom-ps1": {5056}, "board-roar": {9700}, "etebac5": {1216}, "servserv": {2011}, "wwiotalk": {5413}, "nls-tl": {7549}, "aws-brf": {22800}, "rpki-rtr": {323}, "moldflow-lm": {1576}, "powerguardian": {1777}, "ms-olap2": {2394}, "appss-lm": {3879}, "elanlm": {4346}, "alphatech-lm": {1653}, "ev-services": {5114}, "itactionserver2": {7281}, "jamlink": {8091}, "client-wakeup": {9694}, "softpc": {215}, "prism-deploy": {3133}, "qaz": {7597}, "edbsrvr": {12010}, "la-maint": {51}, "global-cd-port": {3229}, "spramsca": {5769}, "gnutella2": {6347}, "scriptview": {7741}, "p-net-remote": {34379}, "direct": {242}, "iss-console-mgr": {903}, "csdmbase": {1467, 1471}, "ciphire-serv": {3888}, "proaxess": {3961}, "prchat-server": {4456}, "qke-llc-v3": {2523}, "ttg-protocol": {2862}, "med-net-svc": {24006}, "apw-registry": {3758}, "perrla": {4313}, "dhcps": {67}, "nim": {1058}, "utcd": {1506}, "cosmocall": {2324}, "bgpd": {2605}, "aironetddp": {2887}, "elipse-rec": {6515}, "popup-reminders": {7787}, "entexthigh": {12002}, "medevolve": {13930}, "sometimes-rpc27": {32787}, "cma": {1050}, "cequint-cityid": {4074}, "jms": {5673}, "vns-tp": {7802}, "qsnet-trans": {4354}, "ita-agent": {5051}, "krb_prop": {754}, "dc": {2001}, "weblogin": {2054}, "onbase-dds": {2185}, "4-tieropmgw": {2933}, "tig": {3943}, "zen-pawn": {7628}, "iadt-tls": {9614}, "senomix05": {8056}, "aurora": {9084}, "icg-bridge": {2063}, "citrixima": {2512}, "netaspi": {2902}, "ndl-als": {3431}, "isrp-port": {3788}, "boe-resssvr2": {6408}, "k-patentssensor": {40023}, "boomerang": {1304}, "sns-admin": {2658}, "truckstar": {4725}, "coherence": {7574}, "klio": {7697}, "nod-provider": {8980}, "photuris": {468}, "lanmessenger": {2372}, "utsftp": {2529}, "sonardata": {2863}, "pmas": {4066}, "ifsp": {4744}, "hkp": {11371}, "stm_pproc": {3080}, "brlp-2": {4103}, "idfp": {549}, "hmmp-ind": {612}, "sixtrak": {1594}, "encore": {1740}, "oracle-em1": {1748}, "shaperai": {43210}, "display": {7236}, "git": {9418}, "pt2-discover": {1101}, "isis": {2042}, "isis-bcast": {2043}, "ndtp": {2882}, "rib-slm": {3296}, "dsmeter_iatc": {4060}, "digilent-adept": {33890}, "dproxy": {1296}, "ideesrv": {2337}, "apollo-gms": {2759}, "client-ctrl": {3730}, "joost": {4166}, "cssc": {5637}, "fly": {4396}, "mpls-pm": {6634}, "sometimes-rpc2": {737}, "edb-server1": {1635}, "slingshot": {1705}, "radius-acct": {1813}, "dvt-system": {3246}, "lorica-out": {4082}, "afs3-rmtsys": {7009}, "empowerid": {7080}, "vtp": {16666}, "set": {257}, "icg-swp": {2062}, "rsms": {10201}, "wmedistribution": {11998}, "inovaport1": {23000}, "nufw": {4128}, "airport-admin": {5009}, "dsp": {33}, "nest-protocol": {489}, "tunnel": {604}, "gtaua": {2186}, "pvsw-inet": {2441}, "taserver": {3552}, "brcd-vr-req": {4415}, "mgemanagement": {4680}, "dis": {393}, "afp": {548}, "hp-sci": {1299}, "ratl": {2449}, "vaprtm": {3654}, "fintrx": {3787}, "actifio-c2c": {5103}, "opalis-rbt-ipc": {5314}, "efs": {520}, "autodesk-nlm": {2080}, "webtie": {3342}, "filecast": {3401}, "aegate": {4549}, "seclayer-tcp": {3495}, "adobeserver-5": {3705}, "rmpp": {1121}, "gat-lmd": {1708}, "griffin": {2458}, "csoftragent": {3004}, "efi-lm": {3392}, "cloanto-lm": {3397}, "tapeware": {3817}, "gsmp": {6068}, "atex_elmd": {1385}, "zarkov": {2989}, "chevinservices": {3349}, "caacws": {5248}, "kftp": {6621}, "stt": {1607}, "fagordnc": {3873}, "icl-twobase3": {25002}, "iapp": {2313}, "xmlink-connect": {3953}, "ipdr-sp": {4737}, "meta-corp": {6141}, "hexarc": {7397}, "airsync": {2175}, "newlixreg": {2671}, "servistaitsm": {3636}, "hart-ip": {5094}, "swispol": {6966}, "razor": {3555}, "fcis-disc": {4727}, "dcp": {93}, "fnet-remote-ui": {1174}, "vsat-control": {1880}, "h2250-annex-g": {2099}, "net-steward": {2128}, "denali-server": {3444}, "storageos": {5705}, "xtreamx": {5793}, "cisco-aqos": {9081}, "ora-lm": {1446}, "lv-not": {2146}, "storman": {4178}, "cvd": {8400}, "mosmig": {4660}, "oms": {4662}, "msg-icp": {29}, "ardusmul": {1835}, "tcoregagent": {1976}, "advant-lm": {2295}, "discp-client": {2601}, "beacon-port": {3124}, "jetstream": {6901}, "sauterdongle": {25576}, "applus": {2037}, "msmq-mgmt": {2107}, "ats": {2201}, "mcns-tel-ret": {3311}, "emb-proj-cmd": {5116}, "netapp-icmgmt": {11104}, "banyan-rpc": {567}, "call-logging": {2552}, "cxws": {4673}, "xkotodrcp": {5344}, "xdas": {7629}, "ofsd": {2322}, "tinymessage": {5104}, "ncd-conf-tcp": {5979}, "boe-cms": {6400}, "rtimeviewer": {6900}, "isode-dua": {17007}, "sco-inetmgr": {615}, "cruise-config": {8378}, "wap-wsp-wtp-s": {9203}, "wbem-rmi": {5987}, "http-wmap": {8990}, "iscsi": {860, 3260}, "gridgen-elmd": {1542}, "isysg-lm": {1609}, "essp": {2969}, "gbs-stp": {3484}, "izm": {4109}, "valisys-lm": {1457}, "westell-stats": {1875}, "apt-cacher": {3142}, "mpl-gprs-port": {3924}, "dserver": {4309}, "storview": {9293}, "vacdsm-app": {671}, "call-sig-trans": {2517}, "onscreen": {5080}, "sunwebadmins": {8989}, "astrolink": {27876}, "observium-agent": {36602}, "mtrace": {33435}, "snmp": {161}, "groupwise": {1677}, "ecsqdmn": {1882}, "tl1": {2361}, "glishd": {2833}, "jeol-nsddp-4": {6244}, "appiq-mgmt": {4674}, "pixelpusher": {5078}, "ticf-2": {493}, "ff-fms": {1090}, "facilityview": {1561}, "connection": {2607}, "cogitate": {3039}, "dj-ilm": {3362}, "openit": {7478}, "msft-gc-ssl": {3269}, "comcam-io": {3605}, "lispworks-orb": {3672}, "avt-profile-2": {5005}, "ncd-conf": {5999}, "adws": {9389}, "directplaysrvr": {47624}, "bnet": {415}, "micromuse-lm": {1534}, "netopia-vo1": {1839}, "submitserver": {2028}, "hp-device-disc": {3329}, "f5-iquery": {4353}, "pspserver": {2353}, "vad": {14154}, "bacula-fd": {9102}, "cosir": {10321}, "doom": {666}, "plato-lm": {1819}, "datacaptor": {1857}, "ecomm": {3477}, "bis-sync": {5585}, "puppet": {8140}, "corerjd": {284}, "slp": {1605}, "n2nremote": {1685}, "azeti": {4192}, "barracuda-bbs": {5120}, "apc-5454": {5454}, "hpdevms": {5316, 5317}, "fsc-port": {9217}, "visionpyramid": {1247}, "os-licman": {1384}, "can-nds": {1918}, "rockwell-csp2": {2223}, "firepower": {2615}, "rmonitor_secure": {5145}, "saposs": {3397}, "teredo": {3544}, "covia": {64}, "zserv": {346}, "pdp": {1675}, "netcomm1": {1676}, "cgn-stat": {2182}, "necp": {3262}, "zicom": {3774}, "esmmanager": {5600}, "dpserve": {7020}, "vslmp": {312}, "chip-lm": {1572}, "smallchat": {4412}, "pkix-timestamp": {318}, "iclpv-pm": {1392}, "fax": {4557}, "ppsuitemsg": {5863}, "cloudsignaling": {7550}, "wcbackup": {8912}, "fcopys-server": {5746}, "fibotrader-com": {6715}, "echo": {7}, "ioc-sea-lm": {1579}, "dic-aida": {1941}, "evtp-data": {2835}, "odi-port": {3187}, "xnm-clear-text": {3221}, "eba": {45678}, "prizma": {2039}, "ssp": {3249}, "rtmp-port": {3500}, "rtps-dd-mt": {7402}, "sometimes-rpc28": {32787}, "unicall": {4343}, "prelude": {4690}, "dhcpc": {68}, "x25-svc-port": {1998}, "community": {2459}, "fjsvmpor": {2946}, "soniqsync": {3803}, "nvcnet": {3999}, "ssad": {4750}, "ms-alerter": {5359}, "sua": {14001}, "odeumservlink": {3523}, "bv-ds": {3992}, "gacp": {190}, "netware-ip": {396}, "search-agent": {1234}, "sightline": {1645}, "navisphere-sec": {2163}, "nm-game-admin": {3148}, "sns-agent": {5417}, "afs3-fileserver": {7000}, "bpdbm": {13721}, "tonidods": {24465}, "grcmp": {9122}, "3link": {15363}, "matip-type-b": {351}, "interhdl_elmd": {1454}, "ndsp": {2881}, "infoexch": {3667}, "abcsoftware": {3996}, "nsrexecd": {7937}, "garcon": {999}, "cmadmin": {2617}, "m2ua": {2904}, "fxaengine-net": {3402}, "obex": {650}, "biotic": {5087}, "ininmessaging": {5597}, "t1distproc60": {32249}, "pksd": {11371}, "carrius-rshell": {1197}, "poweronnud": {3168}, "flamenco-proxy": {3210}, "ibm-diradm-ssl": {3539}, "konspire2b": {6085}, "ethoscan": {6935}, "solid-mux": {1029}, "thrp": {3963}, "kademlia": {3246}, "icq": {4000}, "lyskom": {4894}, "car": {5090}, "ricardo-lm": {6148}, "sun-sr-http": {6480}, "nati-svrloc": {3580}, "matrix_vnet": {4360}, "vpp": {677}, "filex-lport": {1887}, "imsldoc": {2035}, "codima-rtp": {2415}, "opsession-srvr": {3304}, "slim-devices": {3483}, "prex-tcp": {4487}, "ics": {5639}, "daqstream": {7411}, "mbg-ctrl": {3569}, "sandpolis": {8768}, "xbox": {3074}, "wacp": {3633}, "domain": {53}, "xaudio": {1103}, "fhc": {1499}, "rap-service": {1530}, "easy-soft-mux": {2168}, "watchdog-nt": {2723}, "amx-rms": {3839}, "netxms-sync": {4702}, "dj-ice": {5419}, "cfg-cloud": {8015}, "sd": {9876}, "dsmcc-session": {13819}, "emperion": {1282}, "ms-sna-server": {1477}, "qotps": {2724}, "wfc": {4847}, "ea": {17729}, "eftp": {37601}, "ariel2": {421}, "prat": {1264}, "igi-lm": {1404}, "must-p2p": {3514}, "gcp-rphy": {8190}, "pando-sec": {8276}, "sql-net": {150}, "hpss-ndapi": {1217}, "daap": {3689}, "caldsoft-backup": {22537}, "asterix": {8600}, "ripng": {521}, "netattachsdmp": {3066}, "fg-fps": {3293}, "vxlan-gpe": {4790}, "cp-spxrpts": {5079}, "racf": {18136}, "pxc-roid": {4004}, "coap": {5683}, "dmidi": {1199}, "intellistor-lm": {1539}, "perf-port": {1995}, "msfw-storage": {2171}, "rcc-host": {2332}, "tip2": {3372}, "afs3-kaserver": {7004}, "bitcoin": {8333, 18333}, "n2receive": {9286}, "ailith": {17555}, "checkpoint-rtm": {18241}, "avinstalldisc": {3502}, "caevms": {5251}, "3com-amp3": {629}, "pammratc": {1632}, "idware-router": {2079}, "pclemultimedia": {2558}, "tns-server": {3308}, "upnotifyps": {3356}, "optohost002": {22002}, "hncp-udp-port": {8231}, "dsmcc-download": {13821}, "simple-tx-rx": {2257}, "ttc-ssl": {2484}, "pnrp-port": {3540}, "bpmd": {3593}, "cas-mapi": {3682}, "nim-vdrshell": {6420}, "hosts2-ns": {81}, "deviceshare": {552}, "aria": {2624}, "domiq": {44544}, "pop3pw": {106}, "micom-pfs": {490}, "multiling-http": {777}, "xap-ha": {3639}, "pqsp": {28001}, "xo-wave": {3763}, "xoms": {16619}, "sanity": {643}, "emsd-port": {1928}, "madge-ltd": {2453}, "kastenchasepad": {2918}, "csd-mgmt-port": {3071}, "escvpnet": {3629}, "sometimes-rpc14": {32775}, "sometimes-rpc22": {32779}, "ndm-server": {1364}, "hpssd": {2207}, "neto-dcs": {3814}, "simon": {4753}, "afs3-vlserver": {7003}, "ndmp": {10000}, "nod-client": {8981}, "tor-orport": {9001}, "cft-2": {1763}, "nati-logos": {2343}, "citriximaclient": {2598}, "ncdmirroring": {2706}, "gsmtap": {4729}, "nuxsl": {5991}, "insis": {9215}, "ftrapid-2": {1747}, "a26-fap-fgw": {4726}, "notateit": {4803}, "casrmagent": {7167}, "xybrid-cloud": {9925}, "ip-provision": {43190}, "netrjs-3": {73}, "ratio-adp": {1108}, "pegasus": {9278}, "ldgateway": {9592}, "board-voip": {9750}, "vestasdlp": {17184}, "dhanalakshmi": {34567}, "netbios-ns": {137}, "tams": {2726}, "wsdapi": {5357}, "s-openmail": {5767}, "dell-eql-asm": {7569}, "indigo-vbcp": {8131}, "pvuniwien": {1081}, "agri-gateway": {3026}, "a1-msc": {5602}, "winpharaoh": {6065}, "sruth": {38800}, "imsp": {406}, "whoami": {565}, "timeflies": {1362}, "scipticslsrvr": {2577}, "xmpcr-interface": {3877}, "cnap": {7262}, "cbserver": {3388}, "3comfaxrpc": {3446}, "magenta-logic": {313}, "kerberos-adm": {749}, "ultrex": {1327}, "cera-bcm": {1794}, "fc-faultnotify": {2819}, "redwood-chat": {3032}, "prsvp": {3455}, "can-nds-ssl": {3660}, "bis-web": {5584}, "clever-tcpip": {6688}, "iwg1": {7071}, "stun-port": {1994}, "tscchat": {2330}, "fjmpss": {2509}, "a13-an": {3125}, "seaview": {3143}, "erp-scale": {5135}, "faxportwinport": {1620}, "dif-port": {2251}, "blockade": {2911}, "tnp": {8321}, "inovaport4": {23003}, "cslg": {24754}, "repcmd": {641}, "bootserver": {2016}, "ovalarmsrv": {2953}, "irp": {4604}, "ccss-qmm": {4969}, "xmail-ctrl": {6017}, "dnsix": {90}, "scinet": {7708}, "gt-proxy": {9889}, "bmc_ctd_ldap": {6301}, "scp-config": {10001}, "graphics": {41}, "tacnews": {98}, "qwave": {2177}, "d-cinema-csp": {4170}, "d-data": {4301}, "dpm": {5718}, "alfin": {16003}, "yawn": {31029}, "rsvp-encap-2": {1699}, "bts-appserver": {1961}, "slc-ctrlrloops": {2827}, "pfu-prcallback": {3208}, "adcp": {7508}, "canon-bjnp2": {8612}, "pt-tls": {271}, "rlzdbase": {635}, "vcs-app": {14141}, "areaguard-neo": {23546}, "pkt-krb-ipsec": {1293}, "proxima-lm": {1445}, "netiq": {2220}, "dnc-port": {3448}, "ctp-state": {4047}, "bioserver": {6946}, "instl_boots": {1067}, "lnvconsole": {2281}, "tsp": {3653}, "igrs": {3880}, "hddtemp": {7634}, "capioverlan": {1147}, "avantageb2b": {2131}, "videobeans": {3058}, "wsmlb": {3806}, "tnp1-port": {4024}, "frc-mp": {6705}, "bears-02": {3146}, "ds-srvr": {4401}, "i-net-2000-npr": {5069}, "flexlm6": {27006}, "caci-lm": {1554}, "inova-ip-disco": {2716}, "dvapps": {3831}, "jini-discovery": {4160}, "altovacentral": {4689}, "netprowler-sensor": {61441}, "gss-xlicen": {128}, "cal": {588}, "llsurfup-https": {1184}, "buschtrommel": {4747}, "salient-mux": {5422}, "netspeak-cs": {21847}, "kitim": {35354}, "skkserv": {1178}, "rtnt-1": {3137}, "tiepie": {5450}, "oma-ilp-s": {7277}, "generalsync": {7962}, "xybrid-rt": {9978}, "tn-tl-w2": {474}, "udpradio": {1833}, "applusservice": {4087}, "geognosis": {4326}, "flexlm5": {27005}, "tdaccess": {2910}, "netplan": {2983}, "sas-remote-hlp": {3755}, "quest-agent": {3843}, "flashfiler": {24677}, "ingres-net": {134}, "ivmanager": {1276}, "tunstall-pnc": {1846}, "idotdist": {2590}, "q3ade": {7794}, "infi-async": {8067}, "santools": {4079}, "hpvirtctrl": {5224}, "ekshell": {545, 2106}, "pe-mike": {1305}, "rgtp": {1431}, "gtegsc-lm": {1452}, "sunscalar-dns": {1870}, "touchnetplus": {2158}, "supercell": {7967}, "connendp": {693}, "dpi-proxy": {1795}, "msantipiracy": {2222}, "btpp2sectrans": {2444}, "bngsync": {10439}, "instantia": {1240}, "beserver-msg-q": {3527}, "shareapp": {3595}, "iw-mmogame": {3596}, "lansurveyorxml": {3815}, "jmq-daemon-1": {3214}, "sicct-sdp": {4742}, "swa-4": {9026}, "xqosd": {31416}, "netbios-ssn": {139}, "ff-annunc": {1089}, "di-ase": {3046}, "admind": {3279, 8403}, "bcs": {4677}, "hughes-ap": {5105}, "prosharerequest": {5716}, "ms-shuttle": {568}, "ace-proxy": {2335}, "odn-castraq": {2498}, "topflow": {2885}, "aairnet-2": {3619}, "jibe-eb": {3777}, "tunalyzer": {5748}, "silhouette": {7500}, "pmcd": {44321}, "virtual-time": {1852}, "mqtt": {1883}, "infolibria": {2319}, "futrix": {2358}, "winshadow": {3261}, "metatude-mds": {6382}, "emc-xsw-dconfig": {11623}, "metastorm": {2511}, "pwrsevent": {2694}, "visicron-vs": {4307}, "enfs": {5233}, "3gpp-monp": {8809}, "sms-chat": {2703}, "mctet-master": {3115}, "peerbook-port": {3135}, "idps": {3797}, "senomix08": {8059}, "manyone-xml": {8911}, "bacula-dir": {9101}, "av-emb-config": {2050}, "redstorm_diag": {2349}, "vs-server": {3280}, "vt-ssl": {3509}, "ventoso": {4621}, "3par-evts": {5781}, "reachout": {43188}, "rsc-robot": {1793}, "raid-cc": {2006, 2011}, "clp": {2567}, "mns-mail": {2593}, "dts": {2594}, "senomix04": {8055}, "ulpnet": {483}, "remoteanything": {3996, 3997, 3998, 3999, 4000}, "mysqlx": {33060}, "giga-pocket": {3862}, "dcs-config": {3988}, "ansatrader": {124}, "ddm-ssl": {448}, "skronk": {460}, "jlicelmd": {1567}, "radacct": {1646, 1813, 1813}, "rsap": {1647}, "ezmessagesrv": {4085}, "omstopology": {4162}, "condor": {9618}, "sdtvwcam": {24666}, "murmur": {64738}, "decauth": {316}, "dnap": {1172}, "btprjctrl": {2803}, "wap-push": {2948}, "bullant-srap": {2964}, "tvpm": {21800}, "autobuild": {5115}, "vpnz": {1224}, "essweb-gw": {1772}, "ibp": {2572}, "aic-np": {2785}, "commandport": {3416}, "nuts_bootp": {4133}, "servergraph": {1251}, "s8-client-port": {3153}, "lm-instmgr": {8205}, "uec": {8778}, "cyborg-systems": {9888}, "radwiz-nms-srv": {2736}, "gpsd": {2947}, "asa": {386}, "uucp": {540}, "qsm-proxy": {1164}, "editbench": {1350}, "concurrent-lm": {1648}, "pc-mta-addrmap": {2246}, "wip-port": {3414}, "streettalk": {566}, "e-woa": {3728}, "acp-policy": {3824}, "pjj-player": {4416}, "tesla-sys-msg": {7631}, "spremotetablet": {46998}, "atc-lm": {1170}, "cspmlockmgr": {1272}, "objectmanager": {2038}, "msi-selectplay": {2871}, "http-mgmt": {280}, "houdini-lm": {1715}, "bacula-sd": {9103}, "eapol-relay": {10253}, "emc-xsw-dcache": {11723}, "amt-redir-tcp": {16994}, "traceroute": {33434}, "ams": {1037}, "mps-raft": {1700}, "cti-redwood": {2563}, "npep-messaging": {2868}, "gamelobby": {2914}, "ent-engine": {3665}, "acp": {599}, "hri-port": {3439}, "fazzt-admin": {4039}, "alta-ana-lm": {1346}, "gv-us": {1369}, "nexstorindltd": {2360}, "sun-sr-admin": {6489}, "mevent": {7900}, "bullant-rap": {2965}, "edm-std-notify": {3462}, "nut": {3493}, "ampl-tableproxy": {5196}, "hacl-gs": {5301}, "mctfeed": {5598}, "doip-data": {13400}, "alpes": {463}, "ljk-login": {472}, "iclpv-wsm": {1395}, "lipsinc": {1968}, "heartbeat": {3740}, "cadsisvr": {16789}, "datasurfsrv": {461}, "shockwave": {1626}, "epnsdp": {2051}, "altserviceboot": {4011}, "lutap": {4912}, "openmailg": {5755}, "mosaixcc": {2561}, "wsman": {5985}, "daylitetouch": {6117}, "ca-audit-da": {8025}, "xfr": {682}, "jps": {2205}, "di-msg": {2227}, "sabams": {2760}, "codemeter-cmwan": {22351}, "sptx": {40404}, "wago-io-system": {2455}, "fjmpcm": {2975}, "precise-i3": {3607}, "dddp": {9131}, "quake3": {27960}, "alias": {1187}, "ci3-software-1": {1301}, "apc-2161": {2161}, "raxa-mgmt": {6099}, "skip-cert-recv": {6455}, "f1-control": {38472}, "priv-print": {35}, "localinfosrvr": {1487}, "netbill-auth": {1615}, "fjippol-cnsl": {2749}, "hacl-local": {5304}, "io-dist-data": {5728}, "ibm-ssd": {1260}, "rich-cp": {2057}, "altcp": {4165}, "ufmp": {6306}, "tick-port": {3200}, "directv-soft": {3335}, "rndc": {953}, "cichild-lm": {1523}, "oracle-vp2": {1808}, "pvsw": {2520}, "doc1lm": {3161}, "firemonrcc": {3192}, "multip-msg": {3733}, "brlp-0": {4101}, "plysrv-http": {6770}, "synotics-broker": {392}, "ehome-ms": {2228}, "fast-rem-serv": {2495}, "db-lsp-disc": {17500}, "fmpro-fdal": {2399}, "alarm": {2740}, "tsb2": {2742}, "expresspay": {2755}, "cqg-netlan": {2823}, "mindfilesys": {7391}, "msnp": {1863}, "sentinel-ent": {3712}, "lanschool": {11796}, "cce3x": {1139}, "crmsbits": {2422}, "ltctcp": {3487}, "actnet": {5411}, "RETS-or-BackupExec": {6103}, "muse": {6888}, "mpnjsosv": {2684}, "msolap-ptp2": {2725}, "tnos-sp": {7901}, "ntz-p2p-storage": {6778}, "mle": {19788}, "itm-mcell-u": {1828}, "psbserver": {2350}, "netchat": {2451}, "hp-3000-telnet": {2564}, "jt400": {3470}, "msdts1": {3882}, "sunproxyadmin": {8081}, "profile": {136}, "alarm-clock-c": {2668}, "1ci-smcs": {3091}, "ms-la": {3535}, "data-port": {3578}, "smartcard-tls": {4116}, "unify": {181}, "de-cache-query": {1255}, "sonus": {2653}, "rusb-sys-port": {3422}, "senip": {3898}, "mcp": {4458}, "apc-6547": {6547}, "sns_credit": {1076}, "docstor": {1488}, "metrics-pas": {1824}, "ka0wuc": {2822}, "synchronite": {4106}, "synchronet-db": {6100}, "vrtp": {2255}, "md-cg-http": {2688}, "z-wave": {4123}, "interwise": {7778}, "stat-scanner": {4157}, "badm_priv": {6505}, "nextstep": {178}, "messageservice": {2311}, "venus": {2430}, "slc-systemlog": {2826}, "pkagent": {3118}, "isns": {3205}, "ezrelay": {10103}, "integrius-stp": {17234}, "m3ap": {36444}, "s-net": {166}, "gsi": {1850}, "agriserver": {3021}, "ovsam-d-agent": {3870}, "locus-disc": {5058}, "sage-best-com1": {14033}, "miroconnect": {1532}, "finle-lm": {1784}, "responsenet": {3045}, "drip": {3949}, "m2mservices": {8383}, "discard": {9}, "opalis-rdv": {536}, "epc": {1267}, "csdm": {1468, 1472}, "beeyond-media": {1943}, "astromed-main": {2864}, "asprovatalk": {1079}, "proshare1": {1459}, "bv-is": {3990}, "bv-smcsrv": {3991}, "tapestry": {1922}, "lbf": {2466}, "jboss-iiop-ssl": {3529}, "tftps": {3713}, "cindycollab": {3770}, "vhd": {3802}, "sybasesrvmon": {4950}, "xpl": {3865}, "nifty-hmi": {4134}, "dhcpv6-client": {546}, "netvenuechat": {1023}, "stargatealerts": {1654}, "tdp-suite": {1814}, "ismaeasdaqtest": {1950}, "nw-license": {3697}, "ghvpn": {12009}, "prm-sm-np": {1402}, "aas": {1601}, "facsys-ntp": {2514}, "aritts": {8423}, "asap-tcp-tls": {3864}, "hpe-dp-bura": {5565}, "embl-ndt": {394}, "hks-lm": {1722}, "res": {1942}, "sybase": {2638}, "surveyinst": {3212}, "trap-port-mom": {3858}, "sun-sr-iiop-aut": {6487}, "erpc": {121}, "ctt-broker": {1932}, "iph-policy-cli": {2962}, "netiq-qcheck": {10114}, "altova-lm": {35355}, "abatjss": {3656}, "npqes-test": {4703}, "mpm-snd": {46}, "purenoise": {663}, "ultimad": {1737}, "etcd-client": {2379}, "cesdcdman": {2921}, "bmcpatrolagent": {3181}, "aruba-server": {7166}, "imqtunnels": {7674}, "vopied": {13783}, "intersys-cache": {1972}, "tacticalauth": {2392}, "icpv2": {3130}, "verismart": {3270}, "ordinox-server": {3274}, "abb-escp": {6316}, "ctlptc": {2153}, "gtrack-server": {3591}, "rcp": {469}, "uucp-rlogin": {541}, "neoiface": {1285}, "mil-2045-47001": {1581}, "nsjtp-data": {1688}, "amdsched": {1931}, "cresco-control": {38002}, "stryker-com": {3854}, "talarian-mcast4": {4018}, "oidsr": {4143}, "gaia": {4340}, "unieng": {5730}, "visd": {9284}, "firewall1-rdp": {259}, "utmpsd": {430}, "t128-gateway": {1627}, "fprams": {4122}, "openmath": {1473}, "drmsmc": {1878}, "ema-sent-lm": {2526}, "vidigo": {3231}, "asr": {7800}, "fmsas": {16000}, "hp-pdl-datastr": {9100}, "k3software-cli": {26263}, "sun-dr": {665}, "vtu-comms": {2216}, "ibridge-mgmt": {2276}, "targus-getdata3": {5203}, "esmagent": {5601}, "classic": {9087}, "ndmps": {30000}, "presonus-ucnet": {47809}, "p2pevolvenet": {8004}, "vocaltec-hos": {25793}, "synapse": {2880}, "cspclmulti": {2890}, "qsoft": {3059}, "watcomdebug": {3563}, "vrts-auth-port": {4032}, "aes-x170": {7107}, "marcam-lm": {1444}, "trusted-web": {3011}, "medimageportal": {7720}, "ars-vista": {27782}, "asihpi": {44600}, "kis": {186}, "dab-sti-c": {1076}, "dbref": {2365}, "sse-app-config": {3852}, "jmevt2": {6962}, "dta-systems": {13929}, "bf-master": {25955}, "fyre-messanger": {2731}, "rfio": {3147}, "shadowstream": {4366}, "sgi-dgl": {5232}, "sossd-disc": {7982}, "ogs-client": {9007}, "cybro-a-bus": {8442}, "MOS-aux": {10542}, "lontalk-norm": {1628}, "bcs-broker": {1704}, "swr-port": {3491}, "sitewatch": {3792}, "damewaremobgtwy": {6130}, "mcftp": {6622}, "entrusttime": {309}, "accessbuilder": {888}, "buddy-draw": {1854}, "ias-reg": {2140}, "winfs": {5009}, "ddi-tcp-2": {8889}, "canocentral0": {1871}, "bim-pem": {3783}, "symb-sb-port": {3923}, "ctsd": {5137}, "XmlIpcRegSvc": {9092}, "supdup": {95}, "nsstp": {1036}, "dx-instrument": {1325}, "cyaserv": {2584}, "mlsn": {32801}, "icpp": {14142}, "btx": {20005}, "jetformpreview": {2097}, "workflowdir": {4417}, "wired": {4871}, "oma-rlp": {7273}, "hinp": {9954}, "dsmcc-passthru": {13820}, "onehome-help": {2199}, "rsisysaccess": {2752}, "arkivio": {3426}, "amc": {5506}, "nusrp": {49001}, "zion-lm": {1425}, "cssp": {4078}, "ncd-diag": {5998}, "bmc-perf-mgrd": {6768}, "octopus": {10008}, "nimcontroller": {48000}, "us-cli": {8082}, "otv": {8472}, "webobjects": {1085}, "proxim": {1732}, "mtp": {1911}, "wimd": {2980}, "netscript": {4118}, "omabcastltkm": {4359}, "rjcdb-vcards": {9208}, "binkp": {24554}, "wlcp": {36411}, "bhevent": {357}, "vnwk-prapi": {2538}, "integral": {3459}, "pharos": {4443}, "3par-mgmt-ssl": {5783}, "sometimes-rpc24": {32780}, "ivcollector": {1275}, "bccp": {4175}, "issa": {9991}, "asa-gateways": {7234}, "apc-7845": {7845}, "bgmp": {264}, "rmiactivation": {1098}, "sti-envision": {1312}, "l2c-data": {4372}, "slscc": {4408}, "vfbp": {6678}, "enrp": {9901}, "m2ap": {36443}, "distinct32": {9998}, "tn-tl-fd1": {476}, "support": {1529}, "simba-cs": {1543}, "amx-weblinx": {2930}, "ca-idms": {3709}, "sdo-ssh": {3897}, "netserialext4": {16368}, "fcp": {510}, "eapsp": {2291}, "pslserver": {2352}, "redstone-cpss": {2928}, "sdl-ets": {5081}, "altbsdp": {7799}, "lnvmaps": {2284}, "ncl": {2397}, "influence": {3345}, "ibm3494": {3494}, "meregister": {669}, "ftrapid-1": {1746}, "emcsymapiport": {2707}, "mice": {5022}, "cm": {5910}, "park-agent": {5431}, "opsec-sam": {18183}, "who": {513}, "optiwave-lm": {2524}, "fazzt-ptp": {4038}, "compx-lockview": {4308}, "apwi-rxserver": {4392}, "hacl-probe": {5303}, "ansanotify": {116}, "pvxpluscs": {4093}, "eq-office-4940": {4940}, "uarps": {219}, "jmact6": {2958}, "spiral-admin": {3438}, "opus-services": {3718}, "fw1-mc-gui": {258}, "corba-iiop-ssl": {684}, "netinfo-local": {1033}, "ncdloadbalance": {2683}, "ezproxy-2": {10102}, "manet": {269}, "zephyr-hm": {2104}, "lutcp": {4913}, "rets-ssl": {12109}, "robotraconteur": {48653}, "activememory": {2859}, "netiq-endpoint": {10113}, "stmf": {501}, "accelenet": {1182}, "tn-tl-fd2": {1584}, "ssm-cssps": {2478}, "tclprodebugger": {2576}, "oce-snmp-trap": {2697}, "helix": {10860}, "sometimes-rpc4": {32770}, "gv-pf": {18262}, "inovaport3": {23002}, "x-bone-api": {2165}, "gprs-cube": {3751}, "centerline": {3987}, "sagxtsds": {4952}, "cleverdetect": {6690}, "xmpp": {8010}, "idtp": {25604}, "slinkysearch": {1225}, "as-debug": {4026}, "ehs-ssl": {4536}, "pwgwims": {4951}, "caerpc": {42510}, "IISrpc-or-vat": {3456}, "listcrt-port-2": {3914}, "osp": {5045}, "ddi-udp-2": {8889}, "hpnpd": {22370}, "davsrc": {9800}, "sometimes-rpc11": {32774}, "knet-cmp": {157}, "netviewdm3": {731}, "ascomalarm": {4077}, "opencore": {4089}, "ccag-pib": {7169}, "sdr": {9010}, "pnbs": {6124}, "smc-admin": {6787}, "unitary": {126}, "remote-winsock": {1745}, "netopia-vo2": {1840}, "ums": {2248}, "sitaramgmt": {2630}, "dialpad-voice1": {2860}, "traingpsdata": {9277}, "rtps-dd-ut": {7401}, "sossd-agent": {7982}, "ideafarm-panic": {903}, "sas-2": {1436}, "sd-elmd": {1681}, "sms-rcinfo": {2701}, "dec-notes": {3333}, "hrpd-ith-at-an": {4592}, "npmp": {8450}, "apc-necmp": {18888}, "rt-viewer": {35001}, "e-mdu": {3727}, "m4-network-as": {4345}, "brvread": {1054}, "taligent-lm": {1475}, "ontobroker": {2267}, "sd-data": {2385}, "spearway": {2440}, "nm-game-server": {3149}, "telops-lmd": {7491}, "found": {4411}, "netmo-http": {6842}, "nmsd": {1239}, "moy-corp": {2488}, "lionhead": {2611}, "dvt-data": {3247}, "wap-push-https": {4036}, "PowerAlert-nsa": {4150}, "aries-kfinder": {7570}, "ucs-isc": {8070}, "hotline": {1234, 5500}, "hotu-chat": {3449}, "exasoftport1": {3920}, "lockd": {4045}, "ssr-servermgr": {45966}, "copycat": {9093}, "cirrossp": {10443}, "lontalk-urgnt": {1629}, "sdserver": {2309}, "ms-olap4": {2383}, "hhb-handheld": {4148}, "ms-smlbiz": {5356}, "swismgr2": {6964}, "netapp-icdata": {11105}, "plex": {32400}, "intrinsa": {503}, "sna-cs": {1553}, "viziblebrowser": {1868}, "pmip6-data": {5437}, "X11:3": {6003}, "cruise-diags": {8379}, "asf-rmcp": {623}, "supfiledbg": {1127}, "privatechat": {1735}, "silkp2": {2830}, "identify": {2987}, "eq-office-4941": {4941}, "genie": {402}, "rt-helper": {35006}, "bzr": {4155}, "waascluster": {5787}, "ris-cm": {748}, "netware-csp": {1366}, "opentable": {2368}, "hs-port": {2570}, "sim-control": {3110}, "cernsysmgmtagt": {3830}, "mcer-port": {6510}, "acctopus-cc": {6868}, "heythings": {18516}, "ka-sddp": {31016}, "nst": {4687}, "senomix03": {8054}, "ldaps": {636}, "mctp": {1100}, "glbp": {3222}, "dashpas-port": {3498}, "ccu-comm-2": {4054}, "trim-event": {4322}, "amt-esd-prot": {1082}, "geneous": {3381}, "microtalon-dis": {7013}, "optima-vnet": {1051}, "invokator": {2006}, "cs-live": {2129}, "soundsvirtual": {17185}, "netconf-ssh": {830}, "vmrdp": {2179}, "tec5-sdctp": {9668}, "encrypted-llrp": {5085}, "intermapper": {8181}, "dayna": {244}, "mobilip-mn": {435}, "cycleserv": {763}, "lnvpoller": {2280}, "rs-rmi": {3736}, "swx-gate": {4538}, "ultraseek-http": {8765}, "sometimes-rpc23": {32780}, "ac-cluster": {18463}, "telnet": {23}, "skytelnet": {1618}, "bflckmgr": {3966}, "vts-rpc": {5780}, "truecm": {8804}, "chipper": {17219}, "icp": {1112}, "chromagrafx": {1373}, "ewcappsrv": {1876}, "admdog": {5101}, "globecast-id": {6109}, "dpsi": {315}, "drp": {1974}, "troff": {2014}, "netport-id": {3129}, "oveadmgr": {7427}, "xw-control": {36462}, "xvttp": {508}, "timbuktu-srv4": {1420}, "cisco-net-mgmt": {1741}, "can-ferret": {1920}, "ovsdb": {6640}, "atul": {7543}, "unidata-ldm": {388}, "descent3": {2092}, "u-dbap": {3584}, "netiq-voipa": {10116}, "X11:9": {6009}, "ovobs": {30999}, "rpasswd": {774}, "pciarray": {1552}, "sds-admin": {2705}, "cspuni": {2806}, "netxms-mgmt": {4701}, "atsc-mh-ssc": {4937}, "jvl-mactalk": {47100}, "rnmap": {3418}, "cnckadserver": {7168}, "xns-ch": {54}, "inspect": {1602}, "close-combat": {1944}, "eye2eye": {1948}, "nmsserver": {2244}, "tl1-lv": {3081}, "cpsp": {17222}, "ep-nsp": {3621}, "bv-agent": {3993}, "qsnet-cond": {4357}, "ayiya": {5072}, "minotaur-sa": {5136}, "ncpm-hip": {1683}, "icon-discover": {2799}, "wsynch": {3111}, "adobeserver-3": {3703}, "winshadow-hd": {3861}, "csrpc": {5063}, "memcachedb": {21201}, "jmb-cds1": {8900}, "nupaper-ss": {12121}, "stgxfws": {1226}, "www-ldap-gw": {1760}, "lv-auth": {2147}, "xsync": {3721}, "canon-capt": {3756}, "x500ms": {5757}, "gw-asv": {4842}, "lmcs": {4877}, "omnisky": {2056}, "csoft-plusclnt": {2699}, "aimpp-port-req": {2847}, "ibm-dial-out": {3267}, "apwi-rxspooler": {4393}, "icshostsvc": {4553}, "htuilsrv": {5023}, "bo2k": {14141, 15151, 54320, 54321}, "l2f": {1701}, "sentinelsrm": {1947}, "airshot": {3975}, "scup": {6315}, "bittorrent-tracker": {6881}, "afs3-volser": {7005}, "odette-ftp": {3305}, "palace-2": {9993}, "imip-channels": {11320}, "mysql-proxy": {6446}, "openflow": {6653}, "veracity": {1062}, "avocent-proxy": {1078}, "swiftnet": {1751}, "comotionmaster": {2261}, "igcp": {2801}, "4talk": {3284}, "mcreport": {8003}, "microsan": {20001}, "bintec-tapi": {2663}, "iconp": {3972}, "transmit-port": {5282}, "wpl-analytics": {8006}, "candrp": {42509}, "tbrpf": {712}, "hivestor": {4884}, "imip": {11319}, "notezilla-lan": {21010}, "sometimes-rpc25": {32786}, "bwnfs": {650}, "prm-nm-np": {1403}, "stone-design-1": {1492}, "powerexchange": {2480}, "rbt-smc": {7870}, "rxapi": {10010}, "iiimsf": {50002}, "ups-onlinet": {7010}, "bhfhs": {248}, "bnetgame": {1119}, "dvl-activemail": {1396}, "connect-server": {3442}, "dlms-cosem": {4059}, "iadt": {4169}, "emcads": {3945}, "zabbix-trapper": {10051}, "ua-secureagent": {19194}, "netplay-port2": {3641}, "timed": {525}, "adobeserver-2": {1103}, "ies-lm": {1443}, "myrtle": {1831}, "msfw-s-storage": {2172}, "ncacn-ip-tcp": {3062}, "can-dch": {1919}, "ssm-els": {2479}, "nsc-posa": {2605}, "deslogind": {3006}, "bsfsvr-zn-ssl": {5321}, "qmtps": {6209}, "wrspice": {6114}, "legent-2": {374}, "cisco-sccp": {2000}, "maytagshuffle": {2591}, "itm-mccs": {3084}, "trnsprntproxy": {3346}, "apcupsd": {3551}, "ftranhc": {1105}, "cdfunc": {2045}, "sdmmp": {5573}, "com-bardac-dw": {48556}, "courier": {530}, "glogger": {2033}, "magicnotes": {3023}, "cs-auth-svr": {3113}, "boe-resssvr1": {6407}, "palace-1": {9992}, "owamp-control": {861}, "stun": {3478}, "odette-ftps": {6619}, "zabbix-agent": {10050}, "blp5": {48129}, "lmsocialserver": {1111}, "aker-cdp": {2473}, "d2000kernel": {3119}, "wssauthsvc": {4537}, "hpoms-ci-lstn": {5403}, "snet-sensor-mgmt": {10000}, "dragonfly": {8913}, "apani2": {9161}, "jvclient": {1940}, "dict": {2628}, "elvin_server": {2916}, "personnel": {3109}, "canex-watch": {3583}, "cvsup": {5999}, "ccmrmi": {3154}, "castorproxy": {3450}, "sqlexec-ssl": {9089}, "avesterra": {20057}, "ctx-bridge": {3127}, "publiqare-sync": {4329}, "netmo-default": {6841}, "mxi": {8005}, "nfsd-keepalive": {1110}, "kjtsiteserver": {1339}, "cfs": {3049, 7546}, "bmcpatrolrnvu": {3182}, "urld-port": {3534}, "bmc-data-coll": {3695}, "scoi2odialog": {360}, "objective-dbc": {1388}, "track": {3462, 20670}, "acp-proto": {4046}, "dtspc": {6112}, "ischat": {1336}, "rtsclient": {2501}, "nhserver": {2672}, "rrac": {5678}, "med-ltp": {24000}, "cockroach": {26257}, "codaauth2": {370}, "siipat": {1733}, "search": {2010}, "asoki-sma": {3087}, "ssrip": {3318}, "wsdapi-s": {5358}, "tragic": {2642}, "rt-event-s": {3707}, "pythonds": {7464}, "dsmcc-ccp": {13822}, "guttersnex": {35356}, "arca-api": {8023}, "dasp": {439}, "kpasswd": {761}, "dynamic3d": {2150}, "sis-emt": {2545}, "csregagent": {3022}, "bmc-onekey": {3561}, "socalia": {5100}, "heliosd": {6440}, "datascaler-db": {6624}, "axon-tunnel": {16665}, "mediabox": {46999}, "k-block": {287}, "sonus-logging": {2290}, "linktest": {3746}, "qtms-bootstrap": {3850}, "omid": {5428}, "sqlserv": {118}, "decladebug": {410}, "asdis": {2192}, "ggz": {5688}, "cohesity-agent": {21213}, "MaxumSP": {4179}, "htcp": {4827}, "passgo": {511}, "netviewdm2": {730}, "florence": {1228}, "tr-rsrb-p2": {1988}, "veritas-ucl": {2148}, "shofarplayer": {4105}, "vista-4gl": {24249}, "caistoragemgr": {7162}, "wap-wsp": {9200}, "c1222-acse": {1153}, "swarm": {2377}, "p2pcommunity": {3955}, "ubxd": {4034}, "ivocalize": {5049}, "lm-x": {6200}, "oma-mlp": {9210}, "linogridengine": {12300}, "minivend": {7786}, "sw-orion": {17777}, "statsrv": {133}, "bgp": {179}, "caller9": {2906}, "ds-admin": {4404}, "fcp-srvr-inst1": {5502}, "watchdoc": {5744}, "sdclient": {2310}, "pq-lic-mgmt": {2687}, "ufastro-instr": {3720}, "iadt-disc": {4169}, "rrifmm": {1696}, "netml": {2288}, "aairnet-1": {3618}, "ffserver": {3825}, "lanrevagent": {3970}, "wmereporting": {11999}, "m2pa": {3565}, "rt-sound": {35002}, "isakmp": {500}, "ldp": {646}, "tabula": {1437}, "l3-exprt": {2840}, "isi-irp": {3226}, "sxmp": {3273}, "sbcap": {29168}, "hp-alarm-mgr": {383}, "infoseek": {414}, "oraclenames": {1575}, "simbaservices": {1599}, "sitaradir": {2631}, "s102": {23272}, "netrcs": {742}, "vrxpservman": {4147}, "wxbrief": {4368}, "resorcs": {4733}, "neo4j": {7474}, "xiostatus": {2341}, "cesdinv": {2856}, "slnp": {3025, 3045, 8076}, "hlibmgr": {3634}, "neto-wol-server": {3812}, "ctdp": {7022}, "ff-sm": {1091}, "etp": {1798}, "bv-queryengine": {3989}, "rfa": {4672}, "db-lsp": {17500}, "at-3": {203}, "ppcontrol": {2505}, "ksysguard": {3112}, "X11:59": {6059}, "itose": {4348}, "gds-adppiw-db": {4550}, "dellwebadmin-1": {1278}, "msync": {2072}, "msfw-array": {2174}, "backburner": {2635}, "vpsipport": {2649}, "rapidmq-center": {3093}, "light": {4670}, "iso-tsap-c2": {399}, "dvs": {481}, "launchbird-lm": {3739}, "ds-user": {4403}, "rsync": {873}, "opsmgr": {1270}, "syncserverssl": {2679}, "cp-spxdpy": {4378}, "tag-pm": {5073}, "dbisamserver2": {12006}, "ariel3": {422}, "oracle-vp1": {1809}, "ls3": {3069}, "matahari": {49000}, "canon-cpp-disc": {8609}, "tempo": {526}, "cambertx-lm": {1734}, "netiq-ncap": {2219}, "plgproxy": {2790}, "mcs-mailsvr": {3332}, "lxi-evntsvc": {5044}, "genisar-port": {3475}, "ms-ilm": {5725}, "kingdomsonline": {30260}, "xmpp-client": {5222}, "https": {443}, "fjicl-tep-c": {1904}, "abr-secure": {1955}, "manage-exec": {2342}, "ttcmremotectrl": {3468}, "inst-discovery": {4878}, "ida-discover2": {5742}, "rasadv": {9753}, "blaze": {1150}, "bbn-mmx": {1348}, "ibm-abtact": {1586}, "magicom": {2243}, "stat-results": {4156}, "bip": {4376}, "fc-ser": {1372}, "raid-cd": {2013}, "netadmin": {2450}, "msgsrvr": {8787}, "dfserver": {21554}, "profinet-rt": {34962}, "dtpt": {5721}, "virprot-lm": {7121}, "bh611": {354}, "arns": {384}, "iiop": {535}, "tns-cml": {590}, "virtualtape": {2386}, "iiw-port": {3186}, "rrh": {753}, "trustestablish": {2573}, "ttnrepository": {2943}, "msft-dpm-cert": {6076}, "lm-sserver": {8207}, "profinet-rtm": {34963}, "childkey-ctrl": {1892}, "index-pc-wb": {2127}, "kofax-svr": {2424}, "mon": {2583, 9255}, "hctl": {4197}, "ora-oap": {5575}, "CodeMeter": {22350}, "cisco-sys": {132}, "cloanto-net-1": {356}, "apc-agent": {2161}, "lan900_remote": {2395}, "fortisphere-vm": {4084}, "netconf-ch-tls": {4335}, "bingbang": {29999}, "realsecure": {2998}, "nattyserver": {3753}, "mirrtex": {4310}, "vnc-http-3": {5803}, "afs3-bos": {7007}, "zep": {17754}, "rmonitor": {560}, "secureidprop": {5510}, "X11:6": {6006}, "zenginkyo-2": {5021}, "wago-service": {6626}, "resvc": {691}, "dwf": {1450}, "nbx-au": {2094}, "smartpackets": {3218}, "smauth-port": {3929}, "burp": {4971}, "hiperscan-id": {8293}, "ddi-udp-4": {8891}, "apc-9952": {9952}, "sqlsrv": {156}, "opentrac": {3855}, "omhs": {5723}, "flex-lm": {27000, 27001, 27002, 27003, 27004, 27005, 27006, 27007, 27008, 27009}, "hde-lcesrvr-2": {14937}, "opsec-lea": {18184}, "pov-ray": {494}, "netconfsoapbeep": {833}, "cypress": {2015}, "bpcp-trap": {2845}, "hyperip": {3919}, "scte30": {5168}, "gntp": {23053}, "paging-port": {3771}, "v5ua": {5675}, "netview-aix-5": {1665}, "xinuexpansion4": {2024}, "eli": {2087}, "mmcal": {2272}, "sm-pas-1": {2938}, "BESApi": {3408}, "sercomm-scadmin": {6108}, "hub-open-net": {8313}, "scc-security": {582}, "ipass": {2549}, "appliance-cfg": {2898}, "cgms": {3003}, "iqnet-port": {3804}, "ns-server": {5415}, "hdap": {263}, "dtk": {365}, "geolocate": {3108}, "d3winosfi": {3458}, "vrt": {4991}, "rsvp_tunnel": {363}, "silc": {706}, "pctrader": {3048}, "apparenet-ui": {3239}, "nacagent": {4407}, "oktaauthenticat": {8769}, "autocuetime": {3104}, "vpa-disc": {5164}, "proofd": {1093}, "confluent": {1484}, "cadabra-lm": {1563}, "s3db": {2278}, "attachmate-s2s": {2419}, "seaodbc": {2471}, "ddi-udp-7": {8894}, "wso2esb-console": {9444}, "spmp": {656}, "cognex-insight": {1069}, "comotionback": {2262}, "cesdcdtrn": {2922}, "kfxaclicensing": {3581}, "secrmmsafecopya": {38865}, "npmp-local": {610}, "nssagentmgr": {4454}, "fs-mgmt": {8044}, "proactivate": {24678}, "exonet": {26489}, "polipo": {8123}, "samba-swat": {901}, "dkmessenger": {1177}, "eoss": {1210}, "lv-jc": {2143}, "unglue": {2655}, "qcp": {5082}, "dtn-bundle-tcp": {4556}, "virtualuser": {5423}, "ieee-mms": {651}, "lotusnotes": {1352}, "nrcabq-lm": {1458}, "zebrasrv": {2600}, "vrts-at-port": {2821}, "triomotion": {3240}, "intrepid-ssl": {11751}, "rdb-dbs-disp": {1571}, "worldfusion1": {2595}, "aztec": {3512}, "adtempusclient": {3760}, "nat-pmp": {5351}, "msgclnt": {8786}, "zannet": {317}, "tnmpv2": {3686}, "scenccs": {7129}, "micromuse-ncpw": {9600}, "iec-104-sec": {19998}, "armtechdaemon": {9292}, "sma-spw": {9522}, "triquest-lm": {1588}, "vrtstrapserver": {1885}, "dectalk": {2007}, "iwlistener": {2866}, "fg-gip": {3294}, "pnet-enc": {7798}, "snews": {563}, "acmaint_transd": {775}, "wilkenlistener": {1890}, "jeol-nsdtp-1": {6241}, "ovbus": {7501}, "smsqp": {11201}, "creativeserver": {453, 3364}, "priv-mail": {24}, "orbplus-iiop": {1597}, "epl-slp": {3819}, "ohimsrv": {506}, "docent": {2151}, "pc-telecommute": {2299}, "ovsam-mgmt": {3869}, "lsi-bobcat": {5574}, "sasp": {3860}, "idp": {4067}, "bdir_priv": {6507}, "cp-cluster": {8116}, "ipdcesgbs": {9214}, "cl-1": {172}, "wta-wsp-wtp-s": {2923}, "fujitsu-neat": {3382}, "sun-as-iiops": {3708}, "tsepisp": {4422}, "rtps-discovery": {7400}, "ntp": {123}, "tr-rsrb-p1": {1987}, "kme-trap-port": {2081}, "opcua-tcp": {4840}, "lcm-server": {7365}, "speedtrace": {33334}, "codasrv-se": {2433}, "veritas-udp1": {2802}, "dyna-access": {3310}, "gerhcs": {4985}, "ca-2": {5065}, "synchronet-rtc": {6101}, "eicon-x25": {1439}, "global-wlink": {1909}, "j-lan-p": {2808}, "tolteces": {4375}, "ca-audit-ds": {8026}, "bspne-pcc": {1252}, "siebel-ns": {2320}, "netcheque": {4008}, "https-alt": {8443}, "banyan-vip": {573}, "vemmi": {575}, "vqp": {1589}, "osmosis-aeea": {3034}, "telaconsole": {5428}, "sun-sr-iiop": {6485}, "afore-vdp-disc": {4362}, "cbt": {7777}, "tn-tl-w1": {474}, "L2TP": {1701}, "tpmd": {1906}, "sqdr": {2728}, "ecovisiong6-1": {2896}, "sapeps": {3399}, "ac-tech": {2796}, "serverdocs": {10548}, "netmount": {2061}, "pktcable-cops": {2126}, "wap-vcal-s": {9207}, "llrp": {5084}, "wsmans": {5986}, "iso-ill": {499}, "syslog": {514}, "symplex": {1507}, "impera": {1710}, "ismaeasdaqlive": {1949}, "ddgn": {4167}, "iatp-normalpri": {6999}, "ocserver": {184}, "vpps-qua": {672}, "afs": {1483}, "pdnet": {2843}, "armi-server": {3174}, "pads": {7237}, "bpcd": {13782}, "wimsic": {1326}, "lam": {2040}, "jediserver": {2406}, "talarian-mqs": {2493}, "lbc-watchdog": {2816}, "joaJewelSuite": {6583}, "skynetflow": {8111}, "ismserver": {9500}, "gppitnp": {103}, "nimreg": {1059}, "datalens": {2229}, "dcsoftware": {3793}, "vrpn": {3883}, "smaclmgr": {4660}, "profinet-cm": {34964}, "amqp": {5672}, "mumps": {188}, "cryptoadmin": {624}, "h225gatedisc": {1718}, "RDCenter": {4756}, "varadero-2": {4839}, "gradecam": {5117}, "snmp-tcp-port": {1993}, "ttc-etap-ns": {2977}, "wv-csp-udp-cir": {3717}, "dbisamserver1": {12005}, "spsc": {478}, "pdps": {1314}, "lmdp": {2623}, "mccwebsvr-port": {3570}, "proxy-plus": {4480}, "dhcp-failover": {647}, "elcsd": {704}, "streamcomm-ds": {9612}, "3gpp-cbsp": {48049}, "wrs_registry": {2340}, "msft-gc": {3268}, "hyperscsi-port": {5674}, "wnn6": {22273}, "vipera-ssl": {12013}, "solaris-audit": {16162}, "pftp": {662}, "multidropper": {1035}, "re101": {1343}, "xinuexpansion1": {2021}, "netbookmark": {3131}, "gld": {6267}, "inedo": {46336}, "srvc_registry": {3018}, "X11:1": {6001}, "sabarsd": {8401}, "bprd": {13720}, "brf-gw": {22951}, "texar": {333}, "ulistserv": {372}, "dka": {1263}, "hsrp": {1985}, "ccproxy-ftp": {2121}, "chimera-hwm": {4009}, "infowave": {2082}, "badm_pub": {6506}, "cabsm-comm": {7161}, "mailbox-lm": {505}, "ecmport": {3524}, "X11:5": {6005}, "boe-filesvr": {6404}, "kopek-httphead": {27504}, "dbstar": {1415}, "squid-ipc": {3130}, "simple-push-s": {3688}, "vsi-omega": {7566}, "qadmifevent": {2462}, "kv-server": {3360}, "l5nas-parchan": {9747}, "qip-msgd": {2468}, "hippad": {2988}, "nburn_id": {20034}, "ncube-lm": {1521}, "responselogic": {2886}, "kv-agent": {3361}, "ff-lr-port": {3622}, "busycal": {4990}, "wsm-server": {5006}, "pcduo": {5405}, "sdt": {5568}, "clutild": {7174}, "diamondport": {33331}, "quotad": {762}, "xtlserv": {6116}, "tsa": {6689}, "realserver": {7070}, "tnos-dp": {7902}, "wap-wsp-s": {9202}, "netblox": {4441}, "hpoms-dps-lstn": {5404}, "atls": {216}, "innosys-acl": {1413}, "pearldoc-xact": {1980}, "netmon": {2606}, "radix": {2872}, "csoft-prev": {3271}, "rdmnet-ctrl": {5569}, "xs-openstorage": {1619}, "pcttunnell": {2274}, "smart-install": {4786}, "cacsambroker": {7163}, "fodms": {7200}, "intel_rci": {24386}, "swrmi": {1866}, "rmopagt": {2959}, "cogsys-lm": {3377}, "pscupd": {3453}, "personalos-001": {3557}, "informer": {3856}, "esro-gen": {259}, "wininstall-ipc": {3674}, "nvm-express": {4420}, "jmb-cds2": {8901}, "loreji-panel": {7026}, "fs-agent": {8042}, "xact-backup": {911}, "asi": {1827}, "pdb": {3033}, "bxp": {4027}, "idig_mux": {4152}, "apsolab-tag": {5472}, "lm-mon": {31620}, "rt-classmanager": {35004}, "sybasedbsynch": {2439}, "snifferclient": {2452}, "desktop-dna": {2763}, "wv-csp-sms": {3590}, "p6ssmc": {4311}, "eportcommdata": {4669}, "galaxy-network": {5235}, "repscmd": {653}, "ft-role": {2429}, "ssm-cvs": {2477}, "wta-wsp-s": {2805}, "ov-nnm-websrv": {3443}, "nss": {4159}, "omsdk": {5724}, "patrol-ism": {6161}, "sceanics": {5435}, "wherehoo": {5859}, "ocs_cmu": {428}, "metaagent": {1897}, "abr-api": {1954}, "unisql": {1978}, "vspread": {2695}, "boosterware": {2913}, "mdm": {7871}, "opsec-omi": {18185}, "rtelnet": {107}, "surebox": {5453}, "pdtp": {6086}, "pd-admin": {9597}, "mtrgtrans": {19398}, "med-fsp-rx": {24001}, "xn-control": {38422}, "aurora-cmgr": {364}, "kwdb-commn": {1127}, "xserveraid": {3722}, "pwgpsi": {3800}, "jtnetd-status": {5034}, "sapv1": {9875}, "icl-twobase4": {25003}, "collaborator": {622}, "cvc": {1495}, "worldwire": {2371}, "fdtracks": {5579}, "ms-ilm-sts": {5726}, "jeol-nsdtp-2": {6242}, "ismc": {1638}, "csbphonemaster": {1724}, "wormux": {3826}, "texai": {5048}, "oma-ilp": {7276}, "windlm": {1785}, "haipe-otnk": {3769}, "svdrp": {6419}, "d2k-tapestry2": {3394}, "vnas": {577}, "ardus-cntl": {1116}, "citrix-ica": {1494}, "jwclient": {1938}, "rfmp": {2249}, "codasrv": {2432}, "novation": {1322}, "centra": {1709}, "rtsserv": {2500}, "toruxserver": {5153}, "sometimes-rpc13": {32775}, "commonspace": {1592}, "sdfunc": {2046}, "f5-globalsite": {2792}, "bpcp-poll": {2844}, "esp-lm": {3383}, "citysearch": {3974}, "klogin": {543}, "ardusuni": {1834}, "d2000webserver": {3120}, "nati-vi-server": {3363}, "arcp": {7070}, "pwdis": {3735}, "owserver": {4304}, "smartsdp": {426}, "opennl-voice": {1259}, "netview-aix-11": {1671}, "elm-momentum": {1914}, "ltcudp": {3487}, "artifact-msg": {3518}, "dns-llq": {5352}, "tungsten-https": {9443}, "elxmgmt": {23333}, "llm-csv": {2814}, "index-net": {2970}, "rtc-pm-port": {3891}, "precise-comm": {5630}, "cim-rs": {5993}, "twds": {8937}, "fuscript": {1144}, "bmc_patroldb": {1313}, "msims": {1582}, "directplay8": {6073}, "exoconfig": {26487}, "etb4j": {16309}, "whois": {43}, "disclose": {667}, "netview-aix-12": {1672}, "srp-feedback": {2737}, "targus-getdata1": {5201}, "entextmed": {12003}, "natdataservice": {3927}, "aol": {5190}, "apsolab-data": {5475}, "sygatefw": {39213}, "ap": {47806}, "x9-icue": {1145}, "fjdocdist": {1848}, "optech-port1-lm": {2237}, "netwave-ap-mgmt": {2411}, "bsfserver-zn": {5320}, "ldss": {6087}, "ssports-bcast": {8808}, "swtp-port2": {9282}, "creativepartnr": {455, 3366}, "ttyinfo": {2012}, "3d-nfsd": {2323}, "veritas-tcp1": {2802}, "sysscanner": {3251}, "fmp": {4745}, "newbay-snc-mc": {16900}, "satvid-datalnk": {3367, 3368, 3369, 3370, 3371}, "imap": {143}, "pdap": {344}, "dei-icda": {618}, "iclcnet-locate": {886}, "netmap_lm": {1493}, "vcnet-link-v10": {2554}, "snare": {509}, "dali": {2378}, "trap": {4020}, "kca-service": {9878}, "z-wave-tunnel": {44123}, "bmc-perf-sd": {10128}, "paradym-31": {1864}, "mobile-file-dl": {2926}, "start-network": {3615}, "seraph": {4076}, "coaps": {5684}, "kftp-data": {6620}, "uucp-path": {117}, "passgo-tivoli": {627}, "tributary": {2580}, "lkcmserver": {3278}, "enrp-sctp-tls": {9902}, "inovaport5": {23004}, "binderysupport": {2302}, "gf": {3530}, "taiclock": {4014}, "actifioreplic": {5107}, "3exmp": {5221}, "hello-port": {652}, "mysql-cluster": {1186}, "amx-axbnet": {1320}, "rkb-oscs": {1817}, "remote-collab": {2250}, "imoguia-port": {3907}, "vpvc": {1519}, "ivs-video": {2232}, "msrp": {2855}, "ipether232port": {3497}, "qsnet-nucl": {4358}, "emfis-data": {140}, "qnts-orb": {1262}, "bootclient": {2017}, "voispeed-port": {3541}, "mentaserver": {2118}, "compaq-https": {2381}, "tmo-icon-sync": {5583}, "biff": {512}, "musiconline": {1806}, "ipr-dglt": {3678}, "filemaker": {5003}, "onep-tls": {15002}, "onpsocket": {5014}, "mdns": {5353}, "dls": {197, 2047}, "mib-streaming": {2292}, "veronica": {2770}, "pcle-infex": {3189}, "cmc-port": {3576}, "autopac": {4685}, "montage-lm": {6147}, "pss": {7880}, "vnyx": {8699}, "quakeworld": {27500}, "anet": {212}, "iad3": {1032}, "mpc-lifenet": {1213}, "ssdispatch": {3430}, "ms-sideshow": {5360}, "trip": {6069}, "fpitp": {1045}, "netmpi": {3827}, "https-wmap": {8991}, "infotos": {18881}, "ecmp": {6160}, "rna": {25471}, "mit-ml-dev": {83, 85}, "metasage": {1207}, "ibm-cics": {1435}, "d2k-datamover2": {2298}, "admins-lms": {2692}, "bitforestsrv": {5068}, "watershed-lm": {6143}, "trivnet2": {8201}, "netuitive": {1286}, "sd-request": {2384}, "priority-e-com": {2618}, "indura": {3156}, "easl": {3693}, "ktickets-rest": {4331}, "zoomcp": {9666}, "filenet-powsrm": {32767}, "wcr-remlib": {4845}, "dbabble": {8132}, "subntbcst_tftp": {247}, "uis": {390}, "pcnfs": {640}, "tlisrv": {1527}, "ms-cluster-net": {3343}, "ati-ip-to-ncpe": {3965}, "cpdi-pidas-cm": {3609}, "cs-remote-db": {3630}, "snmptrap": {162}, "fw1-or-bgmp": {264}, "uuidgen": {697}, "apc-2160": {2160}, "h248-binary": {2945}, "slslavemon": {3102}, "sun-as-iiops-ca": {3808}, "wspipe": {4431}, "krb524": {4444}, "fg-sysupdate": {6550}, "a15": {3598}, "fksp-audit": {3729}, "rpc2portmap": {369}, "cvc_hostd": {442}, "activesync-notify": {1034}, "d2k-datamover1": {2297}, "compaq-scp": {2766}, "msdtc": {3372}, "accel": {4108}, "xgrid": {4111}, "menandmice-upg": {4603}, "foresyte-clear": {5407}, "canna": {5680}, "panagolin-ident": {9021}, "acronis-backup": {44445}, "tsspmap": {1568}, "radius": {1645, 1812}, "ciphire-data": {3887}, "openmailpxy": {5768}, "lvision-lm": {6471}, "sxuptp": {19540}, "netbios-dgm": {138}, "tpip": {594}, "jaleosnd": {1623}, "dhct-status": {4675}, "fairview": {38202}, "rsftp": {26}, "raid-am": {2007, 2013}, "ace-server": {2475}, "hp-server": {5225}, "sns-gateway": {5416}, "concomp1": {1802}, "bues_service": {2446}, "vxcrnbuport": {3652}, "corelccam": {4300}, "atm-uhas": {11367}, "vdmmesh": {18668}, "j-link": {19020}, "pcia-rxp-b": {1332}, "sslp": {1750}, "cnhrp": {1757}, "ndnp": {2883}, "fcis": {4727}, "dlip": {7201}, "xmapi": {1933}, "teleniumdaemon": {2060}, "3com-njack-1": {5264}, "princity-agent": {6716}, "netbus": {12345, 12346}, "cba8": {9593}, "iafdbase": {480}, "stun-p2": {1991}, "echonet": {3610}, "na-er-tip": {3725}, "isqlplus": {5560}, "redis": {6379}, "infomover": {2854}, "pm-cmdsvr": {5112}, "npmp-gui": {611}, "ibm-dt-2": {1792}, "recipe": {2240}, "antidotemgrsvr": {2247}, "seispoc": {2254}, "vrcommerce": {2530}, "imink": {8615}, "proactivesrvr": {2722}, "nsesrvr": {9988}, "imdocsvc": {2637}, "csc_proxy": {4187}, "hybrid-pop": {473}, "sddp": {1163}, "mloadd": {1427}, "pra_elmd": {1587}, "fjmpjps": {1873}, "g-talk": {2421}, "sun-as-nodeagt": {4850}, "asgenf": {5727}, "2ping": {15998}, "halflife": {27015}, "vocaltec-wconf": {22555}, "oraclenet8cman": {1630}, "balour": {4324}, "freeciv": {5555, 5556}, "freezexservice": {7726}, "lgtomapper": {7938}, "jwpc": {16020}, "funkproxy": {1505}, "pda-gate": {4012}, "httpx": {4180}, "mmaeds": {4668}, "noteshare": {8474}, "shrewd-control": {22335}, "socp-t": {4881}, "entextnetwk": {12001}, "sgi-storman": {1178}, "connlcli": {1358}, "ill": {1611}, "msmq": {1801}, "gxs-data-port": {2073}, "ept-machine": {3628}, "accuracer": {12007}, "smsp": {413}, "enl": {1804}, "uorb": {3313}, "sns-channels": {3380}, "minirem": {4120}, "sometimes-rpc7": {32772}, "tip-app-server": {3160}, "rt-devicemapper": {35003}, "mm-admin": {534}, "ltp-deepspace": {1113}, "llsurfup-http": {1183}, "xds": {2130}, "b2-runtime": {2203}, "dxmessagebase1": {2874}, "oem-agent": {3872}, "ipt-anri-anri": {4593}, "tcpmux": {1}, "ipdd": {578}, "dossier": {1175}, "iberiagames": {1726}, "jvserver": {1939}, "ibm-mgr": {3801}, "enl-name": {1805}, "nbx-dir": {2096}, "netx-server": {2585}, "websphere-snmp": {3427}, "activesync": {5679}, "lsp-self-ping": {8503}, "checksum": {1386}, "macromedia-fcs": {1935}, "array-manager": {3726}, "rap-ip": {3813}, "nta-us": {7545}, "landesk-cba": {38037, 38292, 38293}, "ipcd3": {1209}, "dawn": {1908}, "netangel": {2442}, "ws-discovery": {3702}, "opswagent": {3976}, "clever-ctrace": {6687}, "candp": {42508}, "piccolo": {2787}, "workflow": {3466}, "assuria-slm": {4119}, "opsview-envoy": {4125}, "wap-vcard-s": {9206}, "bl-idm": {142}, "is99s": {380}, "watilapp": {1269}, "odsi": {1308}, "l3-ranger": {2841}, "umm-port": {3098}, "nxlmd": {28000}, "genie-lm": {1453}, "xxnetserver": {3832}, "MOS-soap-opt": {10544}, "kfserver": {5343}, "zeroconf": {5353}, "periscope": {1230}, "linx": {1361}, "zephyr-clt": {2103}, "pcs-pcw": {2566}, "xnm-ssl": {3220}, "profilemac": {4749}, "boks_clntd": {6503}, "mstmg-sstp": {6601}, "ntz-tracker": {6777}, "dpp": {8908}, "first-defense": {1232}, "psdbserver": {2355}, "avenyo": {2992}, "nrpe": {5666}, "isdd": {8148}, "abbs": {4885}, "lonewolf-lm": {6146}, "mylex-mapd": {467}, "oftep-rpc": {950}, "iop": {2055}, "rkinit": {2108}, "grf-port": {3757}, "dvbservdsc": {3937}, "mgesupervision": {4679}, "nsrp": {7170}, "ock": {1000}, "scanstat-1": {1215}, "globmsgsvc": {2519}, "bnt-manager": {3344}, "gbs-smp": {3762}, "omnilink-port": {3904}, "nmea-0183": {10110}, "clariion-evr01": {6389}, "t2-drm": {7932}, "nameserver": {42}, "caids-sensor": {1192}, "copy": {8445}, "msi-cps-rm": {8675}, "snpp": {444}, "mapper-ws_ethd": {3986}, "boxbackupstore": {4186}, "netserialext1": {16360}, "priv-term-l": {87}, "novell-ipx-cmd": {2645}, "hrd-ncs": {6324}, "msgsys": {9594}, "optilogic": {2435}, "egptlm": {3328}, "menandmice-mon": {4552}, "mtcevrunqman": {4558}, "origo-sync": {11103}, "smpnameres": {901}, "abacus-remote": {2894}, "esimport": {3564}, "turbonote-2": {34249}, "acplt": {7509}, "openstack-id": {35357}, "msrpc": {135}, "datex-asn": {355}, "comscm": {437}, "twsdss": {3012}, "lv-frontpanel": {3079}, "beyond-remote": {5424}, "tentacle": {41121}, "fasttrack": {1214}, "lansource": {1485}, "rrilwm": {1695}, "labrat": {2560}, "cqg-netlan-1": {2824}, "sun-as-jmxrmi": {8686}, "acas": {62}, "rescap": {283}, "pegboard": {1357}, "remographlm": {2373}, "jaugsremotec-1": {3472}, "rbakcup1": {2773}, "symantec-av": {2967}, "dls-mon": {198}, "at-echo": {204}, "ph": {481}, "nqs": {607}, "concert": {786}, "ultrabac": {1910}, "msfw-control": {3847}, "de-server": {1256}, "streletz": {6344}, "crystalenterprise": {6401}, "swa-1": {9023}, "rancher": {9345}, "ique": {18769}, "bex-xr": {15660}, "megaco-h248": {2944}, "dbdb": {6104}, "xoraya": {11876}, "Elite": {31337}, "cognex-dataman": {44444}, "novar-dbase": {23400}, "snip-slave": {33656}, "conference": {531}, "alpha-sms": {1849}, "brp": {3043}, "patrol-snmp": {8161}, "litecoin": {9333, 19333}, "asgcypresstcps": {11489}, "dots-signal": {4646}, "bdir_pub": {6508}, "netstat": {15}, "gdbremote": {2159}, "rsom": {2889}, "veritas-vis1": {2993}, "nhci": {3842}, "mbl-battd": {4153}, "peerwire": {9104}, "bilobit": {24577}, "vlsi-lm": {1500}, "fiorano-msgsvc": {1856}, "phonex-port": {3177}, "jamserverport": {3627}, "msdfsr": {5722}, "amcs": {8766}, "ncld": {405}, "rockwell-csp1": {2221}, "pk-electronics": {2634}, "dxmessagebase2": {2875}, "fjcp": {3648}, "cuelink": {5271}, "assoc-disc": {24850}, "nnsp": {433}, "routematch": {1287}, "rsvp-encap-1": {1698}, "catalyst": {2836}, "nei-management": {3886}, "sun-mc-grp": {5306}, "commtact-https": {20003}, "digiman": {2362}, "ovrimosdbman": {2956}, "mcp-port": {3558}, "ip-qsig": {4029}, "cddbp-alt": {8880}, "n2h2server": {9285}, "allpeers": {36001}, "rvs-isdn-dcp": {2578}, "ice-slocation": {4062}, "mmcc": {5050}, "flcrs": {5638}, "hp-sco": {19410}, "hp-sessmon": {19412}, "nmsp": {537, 1790}, "ms-lsa": {1028, 1029}, "remcap": {4731}, "hawk": {7630}, "manyone-http": {8910}, "jcp": {19541}, "contentserver": {454, 3365}, "dctp": {675}, "pipes": {1465}, "smpppd": {3185}, "ni-visa-remote": {3537}, "authentx": {5067}, "ampr-info": {1535}, "dvr-esm": {2804}, "radmin-port": {4899}, "hacl-hb": {5300}, "dbsa-lm": {1407}, "here-lm": {1409}, "bvcdaemon-port": {3626}, "immedianet-bcn": {3657}, "sdlog": {5520}, "xmpp-bosh": {5280}, "vnc-2": {5902}, "aci": {187}, "hyperwave-isp": {692}, "amx-icsp": {1319}, "corelvideo": {1566}, "ianywhere-dbns": {3968}, "prRegister": {4457}, "owms": {7878}, "netspeak-is": {21846}, "otmp": {29167}, "issd": {1600}, "hp-hcip": {1782}, "hpidsagent": {2985}, "pda-sys": {3254}, "aftmux": {3917}, "flexlm3": {27003}, "isbconference2": {1245}, "rapi": {2176}, "xprint-server": {8100}, "fs-rh-srv": {3488}, "ds-mail": {4405}, "audionews": {114}, "cft-3": {1764}, "harp": {1816}, "tpcsrvr": {2078}, "kip": {2112}, "mysql-im": {2273}, "priv-rje": {77}, "3comnetman": {1181}, "fujitsu-dtcns": {1514}, "vision_server": {6672}, "sgsap": {29118}, "timbuktu-srv2": {1418}, "agslb": {4149}, "wap-wsp-wtp": {9201}, "ticf-1": {492}, "health-trap": {1162}, "ace-svr-prop": {2476}, "aises": {2783}, "oa-system": {8022}, "d-s-n": {8086}, "discp-server": {2602}, "pmsm-webrctl": {2972}, "unikeypro": {4127}, "rfid-rp1": {4684}, "foresyte-sec": {5408}, "doc-server": {7165}, "v2g-secc": {15118}, "mpm": {45}, "priv-dial": {75}, "cybercash": {551}, "tcoflashagent": {1975}, "uadtc": {2767}, "starquiz-port": {3526}, "dnet-tstproxy": {3064}, "vulture": {3482}, "srssend": {362}, "ampr-rcmd": {459}, "cimplex": {673}, "ieee-mms-ssl": {695}, "fjswapsnp": {1874}, "h263-video": {2979}, "megaregsvrport": {3572}, "splitlock-gw": {3647}, "interact": {4052}, "xtgui": {4095}, "cawas": {12168}, "assyst-dr": {4485}, "stanag-5066": {5066}, "srmp": {193}, "iafserver": {479}, "remote-as": {1053}, "laplink": {1547}, "unix-status": {1957}, "nppmp": {3476}, "sntlkeyssrvr": {9450}, "rbr-debug": {44553}, "hassle": {375}, "saft": {487}, "3m-image-lm": {1550}, "panasas": {3095}, "nomad": {5209}, "s-bfd": {7784}, "megardsvr-port": {3571}, "e-design-web": {6703}, "itap-ddtp": {10100}, "WibuKey": {22347}, "magiccontrol": {4902}, "tunatic": {5747}, "pop2": {109}, "bftp": {152}, "qmqp": {628}, "cvspserver": {2401}, "iss-realsec": {2998}, "ds-srv": {4400}, "kti-icad-srvr": {6701}, "palace-4": {9995}, "niobserver": {25901}, "pmwebapi": {44323}, "synctest": {45045}, "lcs-ap": {9082}, "bsquare-voip": {1071}, "ibm-mqisdp": {1883}, "bruce": {2619}, "mma-discovery": {4173}, "telelpathattack": {5011}, "gnutella": {6346}, "shivasound": {1549}, "sns-protocol": {2409}, "fjsv-gssagt": {3035}, "vce": {11111}, "xdmcp": {177}, "ldap": {389}, "scenidm": {7128}, "tasp-net": {25900}, "ariel1": {419}, "intuitive-edge": {1355}, "e-dpnet": {2036}, "brain": {2169}, "gmmp": {4183}, "nvc": {8711}, "unify-adapter": {1889}, "compaq-wcp": {2555}, "smap": {3731}, "distinct": {9999}, "printer": {515}, "appworxsrv": {2136}, "ias-auth": {2139}, "delibo": {2562}, "cp-spxsvr": {4377}, "cslistener": {9000}, "gdrive-sync": {37483}, "rds": {1540}, "efi-mg": {2224}, "simctlp": {2857}, "notify_srvr": {3016}, "stss": {3090}, "msfrs": {4554}, "exec": {512}, "unizensus": {1151}, "autocueds": {3437}, "sfm-db-server": {5636}, "emgmsg": {6656}, "espeasy-p2p": {8266}, "ipulse-ics": {20222}, "audit": {182}, "gbjd816": {2626}, "social-alarm": {5146}, "stresstester": {5397}, "frc-hicp": {7244}, "ucontrol": {13894}, "e-design-net": {6702}, "metasys": {11001}, "xmlrpc-beep": {602}, "obrpd": {1092}, "dpkeyserv": {1780}, "facelink": {1915}, "rtnt-2": {3138}, "lisp-data": {4341}, "vcscmd": {14150}, "rdm-tfs": {21553}, "omviagent": {4429}, "jaxflow-data": {5230}, "3par-mgmt": {5782}, "afs3-errors": {7006}, "prgp": {7747}, "cruise-enum": {8376}, "dlpx-sp": {8415}, "warmspotMgmt": {1074}, "rna-lm": {1522}, "itscomm-ns": {1573}, "citynl": {1729}, "pxc-ntfy": {3009}, "genstat": {7283}, "nimbusdbctrl": {48005}, "novar-alarm": {23401}, "compressnet": {2, 3}, "cisco-tna": {131}, "hhb-gateway": {1136}, "prodigy-intrnet": {1778}, "emprise-lsc": {3586}, "cds": {4115}, "ansys-lm": {1800}, "ehs": {4535}, "vnc-http": {5800}, "esbroker": {1342}, "sbook": {1349}, "tn-tl-r2": {1580}, "tellumat-nms": {3549}, "wincim": {4144}, "epmd": {4369}, "purityrpc": {8117}, "apollo-data": {8121}, "isoipsigport-2": {1107}, "orbix-loc-ssl": {3077}, "espeech": {8416}, "p4runtime": {9559}, "apocd": {3809}, "ccu-comm-3": {4055}, "autonoc": {1140}, "ng-umds": {1690}, "mti-tcs-comm": {2469}, "m3ua": {2905}, "ccmad": {3114}, "csvr": {3417}, "oma-rlp-s": {7274}, "accuracer-dbms": {12008}, "robcad-lm": {1509}, "prp": {2091}, "dict-lookup": {2289}, "aimpp-hello": {2846}, "faircom-db": {19790}, "invision-ag": {45054}, "gnunet": {2086}, "fis": {5912}, "amanda": {10080}, "gap": {10800}, "h323gatedisc": {1718}, "idonix-metanet": {2112}, "interpathpanel": {2652}, "rebol": {2997}, "restconf-ch-tls": {4336}, "salient-usrmgr": {5410}, "xingcsm": {2327}, "raventdm": {2714}, "ndl-aas": {3128}, "webmail-2": {3511}, "ncxcp": {5681}, "secure-mqtt": {8883}, "htrust": {5628}, "synchronet-upd": {6102}, "uaac": {145}, "emce": {2004}, "vmodem": {3141}, "winport": {3926}, "sf-lm": {4546}, "intecom-ps2": {5057}, "snapenetio": {22000}, "idmgratm": {32896}, "dtag-ste-sb": {352}, "iclpv-sc": {1390}, "bts-x73": {3681}, "raadmin": {5676}, "tambora": {9020}, "inspider": {49150}, "jmact3": {6961}, "crinis-hb": {3818}, "rww": {4125}, "rid": {4590}, "filemq": {5670}, "dali-port": {5777}, "acnet": {6801}, "macbak": {4181}, "dt-vra": {6326}, "namp": {167}, "slm-api": {1606}, "radsec": {2083}, "sdhelp": {2308}, "ms-wbt-server": {3389}, "prnstatus": {3911}, "intecourier": {495}, "optimanet": {2408}, "sde-discovery": {5152}, "indi": {7624}, "qo-secure": {7913}, "pumpkindb": {9981}, "ca-web-update": {14414}, "epp": {700, 3044}, "dssiapi": {1265}, "phoenix-rpc": {3347}, "apc-5456": {5456}, "suncacao-snmp": {11161}, "suncacao-websvc": {11165}, "rt-labtracker": {35005}, "newwavesearch": {2058}, "neveroffline": {2614}, "radclientport": {3178}, "dmaf-caster": {3574}, "ktelnet": {6623}, "dpap": {8770}, "dec_dlm": {625}, "accelenet-data": {1182}, "iclpv-sas": {1391}, "ada-cip": {2085}, "lockstep": {2125}, "ccnx": {9695}, "productinfo": {1283}, "molly": {1374}, "everydayrc": {2782}, "wsscomfrmwk": {6602}}
