package git_commands

import (
	"testing"

	"github.com/jesseduffield/lazygit/pkg/commands/models"
	"github.com/jesseduffield/lazygit/pkg/commands/oscommands"
	"github.com/jesseduffield/lazygit/pkg/utils"
	"github.com/stretchr/testify/assert"
)

const tagsOutput = `v0.34
v0.33
v0.32.2
v0.32.1
v0.32
testtag
`

func TestGetTags(t *testing.T) {
	type scenario struct {
		testName      string
		gitVersion    *GitVersion
		runner        *oscommands.FakeCmdObjRunner
		expectedTags  []*models.Tag
		expectedError error
	}

	scenarios := []scenario{
		{
			testName:   "should return no tags if there are none",
			gitVersion: &GitVersion{2, 7, 0, ""},
			runner: oscommands.NewFakeRunner(t).
				Expect(`git tag --list --sort=-creatordate`, "", nil),
			expectedTags:  []*models.Tag{},
			expectedError: nil,
		},
		{
			testName:   "should return no tags if there are none (< 2.7.0)",
			gitVersion: &GitVersion{2, 6, 7, ""},
			runner: oscommands.NewFakeRunner(t).
				Expect(`git tag --list --sort=-v:refname`, "", nil),
			expectedTags:  []*models.Tag{},
			expectedError: nil,
		},
		{
			testName:   "should return tags if present",
			gitVersion: &GitVersion{2, 7, 0, ""},
			runner: oscommands.NewFakeRunner(t).
				Expect(`git tag --list --sort=-creatordate`, tagsOutput, nil),
			expectedTags: []*models.Tag{
				{Name: "v0.34"},
				{Name: "v0.33"},
				{Name: "v0.32.2"},
				{Name: "v0.32.1"},
				{Name: "v0.32"},
				{Name: "testtag"},
			},
			expectedError: nil,
		},
		{
			testName:   "should return tags if present (< 2.7.0)",
			gitVersion: &GitVersion{2, 6, 7, ""},
			runner: oscommands.NewFakeRunner(t).
				Expect(`git tag --list --sort=-v:refname`, tagsOutput, nil),
			expectedTags: []*models.Tag{
				{Name: "v0.34"},
				{Name: "v0.33"},
				{Name: "v0.32.2"},
				{Name: "v0.32.1"},
				{Name: "v0.32"},
				{Name: "testtag"},
			},
			expectedError: nil,
		},
	}

	for _, scenario := range scenarios {
		scenario := scenario
		t.Run(scenario.testName, func(t *testing.T) {
			loader := NewTagLoader(
				utils.NewDummyCommon(),
				scenario.gitVersion,
				oscommands.NewDummyCmdObjBuilder(scenario.runner),
			)

			tags, err := loader.GetTags()

			assert.Equal(t, scenario.expectedTags, tags)
			assert.Equal(t, scenario.expectedError, err)

			scenario.runner.CheckForMissingCalls()
		})
	}
}
