/*
Copyright 2020 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package azure

import (
	"context"
	"fmt"

	"github.com/Azure/azure-sdk-for-go/services/compute/mgmt/2020-06-01/compute"
	"github.com/Azure/azure-sdk-for-go/services/network/mgmt/2020-06-01/network"
	"k8s.io/klog/v2"
	"k8s.io/kops/protokube/pkg/gossip"
)

type client interface {
	ListVMScaleSets(ctx context.Context) ([]compute.VirtualMachineScaleSet, error)
	ListVMSSNetworkInterfaces(ctx context.Context, vmScaleSetName string) ([]network.Interface, error)
}

var _ client = &Client{}

// SeedProvider is an Azure implementation of gossip.SeedProvider.
type SeedProvider struct {
	client client
	tags   map[string]string
}

var _ gossip.SeedProvider = &SeedProvider{}

// NewSeedProvider returns a new SeedProvider.
func NewSeedProvider(client client, tags map[string]string) (*SeedProvider, error) {
	return &SeedProvider{
		client: client,
		tags:   tags,
	}, nil
}

// GetSeeds returns a slice of strings used as seeds of Gossip.
// This follows the implementation of AWS and creates seeds from
// private IPs of VMs in the cluster.
func (p *SeedProvider) GetSeeds() ([]string, error) {
	ctx := context.TODO()
	vmsses, err := p.client.ListVMScaleSets(ctx)
	if err != nil {
		return nil, fmt.Errorf("error listing VM Scale Sets: %s", err)
	}

	var vmssNames []string
	for _, vmss := range vmsses {
		if p.isVMSSForCluster(&vmss) {
			vmssNames = append(vmssNames, *vmss.Name)
		}
	}
	klog.V(2).Infof("Found %d VM Scale Sets for the cluster (out of %d)", len(vmssNames), len(vmsses))

	var seeds []string
	for _, vmssName := range vmssNames {
		ifaces, err := p.client.ListVMSSNetworkInterfaces(ctx, vmssName)
		if err != nil {
			return nil, fmt.Errorf("error listing VMSS network interfaces: %s", err)
		}
		for _, iface := range ifaces {
			for _, i := range *iface.IPConfigurations {
				seeds = append(seeds, *i.PrivateIPAddress)
			}
		}
	}
	return seeds, nil
}

func (p *SeedProvider) isVMSSForCluster(vmss *compute.VirtualMachineScaleSet) bool {
	found := 0
	for k, v := range vmss.Tags {
		if p.tags[k] == *v {
			found++
		}
	}
	// TODO(kenji): Filter by ProvisioningState if necessary.
	return found == len(p.tags)
}
