package azure

import (
	"strings"

	"github.com/rs/zerolog/log"

	"github.com/infracost/infracost/internal/resources/azure"
	"github.com/infracost/infracost/internal/schema"
)

func getStorageQueueRegistryItem() *schema.RegistryItem {
	return &schema.RegistryItem{
		Name:      "azurerm_storage_queue",
		CoreRFunc: newStorageQueue,
		ReferenceAttributes: []string{
			"storage_account_name",
		},
	}
}

func newStorageQueue(d *schema.ResourceData) schema.CoreResource {
	region := lookupRegion(d, []string{"storage_account_name"})

	accountReplicationType := "LRS"
	accountKind := "StorageV2"

	if len(d.References("storage_account_name")) > 0 {
		storageAccount := d.References("storage_account_name")[0]

		accountTier := storageAccount.Get("account_tier").String()
		if strings.EqualFold(accountTier, "premium") {
			log.Warn().Msgf("Skipping resource %s. Storage Queues don't support %s tier", d.Address, accountTier)
			return nil
		}

		accountReplicationType = storageAccount.Get("account_replication_type").String()
		accountKind = storageAccount.Get("account_kind").String()
	}

	switch strings.ToLower(accountReplicationType) {
	case "ragrs":
		accountReplicationType = "RA-GRS"
	case "ragzrs":
		accountReplicationType = "RA-GZRS"
	}

	return &azure.StorageQueue{
		Address:                d.Address,
		Region:                 region,
		AccountKind:            accountKind,
		AccountReplicationType: accountReplicationType,
	}
}
