//go:build ignore
// +build ignore

// This program generates pwrules_gen.go. It can be invoked by running
// go generate.
package main

import (
	"encoding/json"
	"net/http"
	"os"
	"text/template"
	"time"
)

const (
	aliasURL  = "https://raw.githubusercontent.com/apple/password-manager-resources/main/quirks/websites-with-shared-credential-backends.json"
	changeURL = "https://raw.githubusercontent.com/apple/password-manager-resources/main/quirks/change-password-URLs.json"
	rulesURL  = "https://raw.githubusercontent.com/apple/password-manager-resources/main/quirks/password-rules.json"
)

func main() {
	aliases, err := fetchAliases()
	if err != nil {
		panic(err)
	}
	changes, err := fetchChangeURLs()
	if err != nil {
		panic(err)
	}
	rules, err := fetchRules()
	if err != nil {
		panic(err)
	}

	f, err := os.Create("pwrules_gen.go")
	if err != nil {
		panic(err)
	}
	defer f.Close()

	pkgTpl.Execute(f, struct {
		Timestamp time.Time
		URLs      []string
		Aliases   map[string][]string
		Changes   map[string]string
		Rules     map[string]jsonRule
	}{
		Timestamp: time.Now().UTC(),
		URLs: []string{
			aliasURL,
			changeURL,
			rulesURL,
		},
		Aliases: aliases,
		Changes: changes,
		Rules:   rules,
	})
}

func fetchAliases() (map[string][]string, error) {
	resp, err := http.Get(aliasURL)
	if err != nil {
		return nil, err
	}
	var ja [][]string
	if err := json.NewDecoder(resp.Body).Decode(&ja); err != nil {
		return nil, err
	}
	aliases := make(map[string][]string, len(ja))
	for _, as := range ja {
		for _, a := range as {
			aliases[a] = as
		}
	}
	return aliases, nil
}

func fetchChangeURLs() (map[string]string, error) {
	resp, err := http.Get(changeURL)
	if err != nil {
		return nil, err
	}
	var change map[string]string
	if err := json.NewDecoder(resp.Body).Decode(&change); err != nil {
		return nil, err
	}
	return change, nil
}

type jsonRule struct {
	Exact bool   `json:"exact-domain-match-only"`
	Rules string `json:"password-rules"`
}

func fetchRules() (map[string]jsonRule, error) {
	resp, err := http.Get(rulesURL)
	if err != nil {
		return nil, err
	}
	var jr map[string]jsonRule
	if err := json.NewDecoder(resp.Body).Decode(&jr); err != nil {
		return nil, err
	}
	return jr, nil
}

// cf. https://blog.carlmjohnson.net/post/2016-11-27-how-to-use-go-generate/
var pkgTpl = template.Must(template.New("").Parse(`// Code generated by go generate gen.go. DO NOT EDIT.
// This package was generated by go generate gen.go at
// {{ .Timestamp }}
// using data from
// {{- range .URLs }}
// {{ . }}
// {{- end }}
package pwrules

var genAliases = map[string][]string{
{{- range $key, $value := .Aliases }}
  "{{ $key }}": []string{
  {{- range $value }}
    "{{ . }}",
  {{- end }}
  },
{{- end }}
}

var genChange = map[string]string{
{{- range $key, $value := .Changes }}
	"{{ $key }}": "{{ $value }}",
{{- end }}
}

var genRules = map[string]string{
{{- range $key, $value := .Rules }}
	"{{ $key }}": {{ printf "%q" $value.Rules }},
{{- end }}
}

var genRulesExact = map[string]bool{
{{- range $key, $value := .Rules }}
	"{{ $key }}": {{ printf "%t" $value.Exact }},
{{- end }}
}
`))
