// Copyright 2019 The gVisor Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package linux provides syscall tables for amd64 Linux.
package linux

import (
	"gvisor.dev/gvisor/pkg/abi"
	"gvisor.dev/gvisor/pkg/abi/linux"
	"gvisor.dev/gvisor/pkg/errors/linuxerr"
	"gvisor.dev/gvisor/pkg/hostarch"
	"gvisor.dev/gvisor/pkg/sentry/arch"
	"gvisor.dev/gvisor/pkg/sentry/kernel"
	"gvisor.dev/gvisor/pkg/sentry/syscalls"
)

const (
	// LinuxSysname is the OS name advertised by gVisor.
	LinuxSysname = "Linux"

	// LinuxRelease is the Linux release version number advertised by gVisor.
	LinuxRelease = "4.4.0"

	// LinuxVersion is the version info advertised by gVisor.
	LinuxVersion = "#1 SMP Sun Jan 10 15:06:54 PST 2016"
)

// AMD64 is a table of Linux amd64 syscall API with the corresponding syscall
// numbers from Linux 4.4.
var AMD64 = &kernel.SyscallTable{
	OS:   abi.Linux,
	Arch: arch.AMD64,
	Version: kernel.Version{
		// Version 4.4 is chosen as a stable, longterm version of Linux, which
		// guides the interface provided by this syscall table. The build
		// version is that for a clean build with default kernel config, at 5
		// minutes after v4.4 was tagged.
		Sysname: LinuxSysname,
		Release: LinuxRelease,
		Version: LinuxVersion,
	},
	AuditNumber: linux.AUDIT_ARCH_X86_64,
	Table: map[uintptr]kernel.Syscall{
		0:   syscalls.SupportedPoint("read", Read, PointRead),
		1:   syscalls.Supported("write", Write),
		2:   syscalls.PartiallySupportedPoint("open", Open, PointOpen, "Options O_DIRECT, O_NOATIME, O_PATH, O_TMPFILE, O_SYNC are not supported.", nil),
		3:   syscalls.Supported("close", Close),
		4:   syscalls.Supported("stat", Stat),
		5:   syscalls.Supported("fstat", Fstat),
		6:   syscalls.Supported("lstat", Lstat),
		7:   syscalls.Supported("poll", Poll),
		8:   syscalls.Supported("lseek", Lseek),
		9:   syscalls.PartiallySupported("mmap", Mmap, "Generally supported with exceptions. Options MAP_FIXED_NOREPLACE, MAP_SHARED_VALIDATE, MAP_SYNC MAP_GROWSDOWN, MAP_HUGETLB are not supported.", nil),
		10:  syscalls.Supported("mprotect", Mprotect),
		11:  syscalls.Supported("munmap", Munmap),
		12:  syscalls.Supported("brk", Brk),
		13:  syscalls.Supported("rt_sigaction", RtSigaction),
		14:  syscalls.Supported("rt_sigprocmask", RtSigprocmask),
		15:  syscalls.Supported("rt_sigreturn", RtSigreturn),
		16:  syscalls.PartiallySupported("ioctl", Ioctl, "Only a few ioctls are implemented for backing devices and file systems.", nil),
		17:  syscalls.Supported("pread64", Pread64),
		18:  syscalls.Supported("pwrite64", Pwrite64),
		19:  syscalls.Supported("readv", Readv),
		20:  syscalls.Supported("writev", Writev),
		21:  syscalls.Supported("access", Access),
		22:  syscalls.SupportedPoint("pipe", Pipe, PointPipe),
		23:  syscalls.Supported("select", Select),
		24:  syscalls.Supported("sched_yield", SchedYield),
		25:  syscalls.Supported("mremap", Mremap),
		26:  syscalls.PartiallySupported("msync", Msync, "Full data flush is not guaranteed at this time.", nil),
		27:  syscalls.PartiallySupported("mincore", Mincore, "Stub implementation. The sandbox does not have access to this information. Reports all mapped pages are resident.", nil),
		28:  syscalls.PartiallySupported("madvise", Madvise, "Options MADV_DONTNEED, MADV_DONTFORK are supported. Other advice is ignored.", nil),
		29:  syscalls.PartiallySupported("shmget", Shmget, "Option SHM_HUGETLB is not supported.", nil),
		30:  syscalls.PartiallySupported("shmat", Shmat, "Option SHM_RND is not supported.", nil),
		31:  syscalls.PartiallySupported("shmctl", Shmctl, "Options SHM_LOCK, SHM_UNLOCK are not supported.", nil),
		32:  syscalls.SupportedPoint("dup", Dup, PointDup),
		33:  syscalls.SupportedPoint("dup2", Dup2, PointDup2),
		34:  syscalls.Supported("pause", Pause),
		35:  syscalls.Supported("nanosleep", Nanosleep),
		36:  syscalls.Supported("getitimer", Getitimer),
		37:  syscalls.Supported("alarm", Alarm),
		38:  syscalls.Supported("setitimer", Setitimer),
		39:  syscalls.Supported("getpid", Getpid),
		40:  syscalls.Supported("sendfile", Sendfile),
		41:  syscalls.PartiallySupported("socket", Socket, "Limited support for AF_NETLINK, NETLINK_ROUTE sockets. Limited support for SOCK_RAW.", nil),
		42:  syscalls.SupportedPoint("connect", Connect, PointConnect),
		43:  syscalls.SupportedPoint("accept", Accept, PointAccept),
		44:  syscalls.Supported("sendto", SendTo),
		45:  syscalls.Supported("recvfrom", RecvFrom),
		46:  syscalls.Supported("sendmsg", SendMsg),
		47:  syscalls.PartiallySupported("recvmsg", RecvMsg, "Not all flags and control messages are supported.", nil),
		48:  syscalls.PartiallySupported("shutdown", Shutdown, "Not all flags and control messages are supported.", nil),
		49:  syscalls.PartiallySupportedPoint("bind", Bind, PointBind, "Autobind for abstract Unix sockets is not supported.", nil),
		50:  syscalls.Supported("listen", Listen),
		51:  syscalls.Supported("getsockname", GetSockName),
		52:  syscalls.Supported("getpeername", GetPeerName),
		53:  syscalls.SupportedPoint("socketpair", SocketPair, PointSocketpair),
		54:  syscalls.PartiallySupported("setsockopt", SetSockOpt, "Not all socket options are supported.", nil),
		55:  syscalls.PartiallySupported("getsockopt", GetSockOpt, "Not all socket options are supported.", nil),
		56:  syscalls.PartiallySupportedPoint("clone", Clone, PointClone, "Mount namespace (CLONE_NEWNS) not supported. Options CLONE_PARENT, CLONE_SYSVSEM not supported.", nil),
		57:  syscalls.SupportedPoint("fork", Fork, PointFork),
		58:  syscalls.SupportedPoint("vfork", Vfork, PointVfork),
		59:  syscalls.SupportedPoint("execve", Execve, PointExecve),
		60:  syscalls.Supported("exit", Exit),
		61:  syscalls.Supported("wait4", Wait4),
		62:  syscalls.Supported("kill", Kill),
		63:  syscalls.Supported("uname", Uname),
		64:  syscalls.Supported("semget", Semget),
		65:  syscalls.PartiallySupported("semop", Semop, "Option SEM_UNDO not supported.", nil),
		66:  syscalls.Supported("semctl", Semctl),
		67:  syscalls.Supported("shmdt", Shmdt),
		68:  syscalls.Supported("msgget", Msgget),
		69:  syscalls.Supported("msgsnd", Msgsnd),
		70:  syscalls.Supported("msgrcv", Msgrcv),
		71:  syscalls.Supported("msgctl", Msgctl),
		72:  syscalls.PartiallySupportedPoint("fcntl", Fcntl, PointFcntl, "Not all options are supported.", nil),
		73:  syscalls.PartiallySupported("flock", Flock, "Locks are held within the sandbox only.", nil),
		74:  syscalls.PartiallySupported("fsync", Fsync, "Full data flush is not guaranteed at this time.", nil),
		75:  syscalls.PartiallySupported("fdatasync", Fdatasync, "Full data flush is not guaranteed at this time.", nil),
		76:  syscalls.Supported("truncate", Truncate),
		77:  syscalls.Supported("ftruncate", Ftruncate),
		78:  syscalls.Supported("getdents", Getdents),
		79:  syscalls.Supported("getcwd", Getcwd),
		80:  syscalls.SupportedPoint("chdir", Chdir, PointChdir),
		81:  syscalls.SupportedPoint("fchdir", Fchdir, PointFchdir),
		82:  syscalls.Supported("rename", Rename),
		83:  syscalls.Supported("mkdir", Mkdir),
		84:  syscalls.Supported("rmdir", Rmdir),
		85:  syscalls.Supported("creat", Creat),
		86:  syscalls.PartiallySupported("link", Link, "Limited support with Gofer. Link count and linked files may get out of sync because gVisor is not aware of external hardlinks.", nil),
		87:  syscalls.Supported("unlink", Unlink),
		88:  syscalls.Supported("symlink", Symlink),
		89:  syscalls.Supported("readlink", Readlink),
		90:  syscalls.Supported("chmod", Chmod),
		91:  syscalls.PartiallySupported("fchmod", Fchmod, "Options S_ISUID and S_ISGID not supported.", nil),
		92:  syscalls.Supported("chown", Chown),
		93:  syscalls.Supported("fchown", Fchown),
		94:  syscalls.Supported("lchown", Lchown),
		95:  syscalls.Supported("umask", Umask),
		96:  syscalls.Supported("gettimeofday", Gettimeofday),
		97:  syscalls.Supported("getrlimit", Getrlimit),
		98:  syscalls.PartiallySupported("getrusage", Getrusage, "Fields ru_maxrss, ru_minflt, ru_majflt, ru_inblock, ru_oublock are not supported. Fields ru_utime and ru_stime have low precision.", nil),
		99:  syscalls.PartiallySupported("sysinfo", Sysinfo, "Fields loads, sharedram, bufferram, totalswap, freeswap, totalhigh, freehigh not supported.", nil),
		100: syscalls.Supported("times", Times),
		101: syscalls.PartiallySupported("ptrace", Ptrace, "Options PTRACE_PEEKSIGINFO, PTRACE_SECCOMP_GET_FILTER not supported.", nil),
		102: syscalls.Supported("getuid", Getuid),
		103: syscalls.PartiallySupported("syslog", Syslog, "Outputs a dummy message for security reasons.", nil),
		104: syscalls.Supported("getgid", Getgid),
		105: syscalls.SupportedPoint("setuid", Setuid, PointSetuid),
		106: syscalls.SupportedPoint("setgid", Setgid, PointSetgid),
		107: syscalls.Supported("geteuid", Geteuid),
		108: syscalls.Supported("getegid", Getegid),
		109: syscalls.Supported("setpgid", Setpgid),
		110: syscalls.Supported("getppid", Getppid),
		111: syscalls.Supported("getpgrp", Getpgrp),
		112: syscalls.SupportedPoint("setsid", Setsid, PointSetsid),
		113: syscalls.Supported("setreuid", Setreuid),
		114: syscalls.Supported("setregid", Setregid),
		115: syscalls.Supported("getgroups", Getgroups),
		116: syscalls.Supported("setgroups", Setgroups),
		117: syscalls.SupportedPoint("setresuid", Setresuid, PointSetresuid),
		118: syscalls.Supported("getresuid", Getresuid),
		119: syscalls.SupportedPoint("setresgid", Setresgid, PointSetresgid),
		120: syscalls.Supported("getresgid", Getresgid),
		121: syscalls.Supported("getpgid", Getpgid),
		122: syscalls.ErrorWithEvent("setfsuid", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/260"}), // TODO(b/112851702)
		123: syscalls.ErrorWithEvent("setfsgid", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/260"}), // TODO(b/112851702)
		124: syscalls.Supported("getsid", Getsid),
		125: syscalls.Supported("capget", Capget),
		126: syscalls.Supported("capset", Capset),
		127: syscalls.Supported("rt_sigpending", RtSigpending),
		128: syscalls.Supported("rt_sigtimedwait", RtSigtimedwait),
		129: syscalls.Supported("rt_sigqueueinfo", RtSigqueueinfo),
		130: syscalls.Supported("rt_sigsuspend", RtSigsuspend),
		131: syscalls.Supported("sigaltstack", Sigaltstack),
		132: syscalls.Supported("utime", Utime),
		133: syscalls.PartiallySupported("mknod", Mknod, "Device creation is not generally supported. Only regular file and FIFO creation are supported.", nil),
		134: syscalls.Error("uselib", linuxerr.ENOSYS, "Obsolete", nil),
		135: syscalls.ErrorWithEvent("personality", linuxerr.EINVAL, "Unable to change personality.", nil),
		136: syscalls.ErrorWithEvent("ustat", linuxerr.ENOSYS, "Needs filesystem support.", nil),
		137: syscalls.PartiallySupported("statfs", Statfs, "Depends on the backing file system implementation.", nil),
		138: syscalls.PartiallySupported("fstatfs", Fstatfs, "Depends on the backing file system implementation.", nil),
		139: syscalls.ErrorWithEvent("sysfs", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/165"}),
		140: syscalls.PartiallySupported("getpriority", Getpriority, "Stub implementation.", nil),
		141: syscalls.PartiallySupported("setpriority", Setpriority, "Stub implementation.", nil),
		142: syscalls.CapError("sched_setparam", linux.CAP_SYS_NICE, "", nil),
		143: syscalls.PartiallySupported("sched_getparam", SchedGetparam, "Stub implementation.", nil),
		144: syscalls.PartiallySupported("sched_setscheduler", SchedSetscheduler, "Stub implementation.", nil),
		145: syscalls.PartiallySupported("sched_getscheduler", SchedGetscheduler, "Stub implementation.", nil),
		146: syscalls.PartiallySupported("sched_get_priority_max", SchedGetPriorityMax, "Stub implementation.", nil),
		147: syscalls.PartiallySupported("sched_get_priority_min", SchedGetPriorityMin, "Stub implementation.", nil),
		148: syscalls.ErrorWithEvent("sched_rr_get_interval", linuxerr.EPERM, "", nil),
		149: syscalls.PartiallySupported("mlock", Mlock, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		150: syscalls.PartiallySupported("munlock", Munlock, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		151: syscalls.PartiallySupported("mlockall", Mlockall, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		152: syscalls.PartiallySupported("munlockall", Munlockall, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		153: syscalls.CapError("vhangup", linux.CAP_SYS_TTY_CONFIG, "", nil),
		154: syscalls.Error("modify_ldt", linuxerr.EPERM, "", nil),
		155: syscalls.Error("pivot_root", linuxerr.EPERM, "", nil),
		156: syscalls.Error("sysctl", linuxerr.EPERM, "Deprecated. Use /proc/sys instead.", nil),
		157: syscalls.PartiallySupported("prctl", Prctl, "Not all options are supported.", nil),
		158: syscalls.PartiallySupported("arch_prctl", ArchPrctl, "Options ARCH_GET_GS, ARCH_SET_GS not supported.", nil),
		159: syscalls.CapError("adjtimex", linux.CAP_SYS_TIME, "", nil),
		160: syscalls.PartiallySupported("setrlimit", Setrlimit, "Not all rlimits are enforced.", nil),
		161: syscalls.SupportedPoint("chroot", Chroot, PointChroot),
		162: syscalls.PartiallySupported("sync", Sync, "Full data flush is not guaranteed at this time.", nil),
		163: syscalls.CapError("acct", linux.CAP_SYS_PACCT, "", nil),
		164: syscalls.CapError("settimeofday", linux.CAP_SYS_TIME, "", nil),
		165: syscalls.PartiallySupported("mount", Mount, "Not all options or file systems are supported.", nil),
		166: syscalls.PartiallySupported("umount2", Umount2, "Not all options or file systems are supported.", nil),
		167: syscalls.CapError("swapon", linux.CAP_SYS_ADMIN, "", nil),
		168: syscalls.CapError("swapoff", linux.CAP_SYS_ADMIN, "", nil),
		169: syscalls.CapError("reboot", linux.CAP_SYS_BOOT, "", nil),
		170: syscalls.Supported("sethostname", Sethostname),
		171: syscalls.Supported("setdomainname", Setdomainname),
		172: syscalls.CapError("iopl", linux.CAP_SYS_RAWIO, "", nil),
		173: syscalls.CapError("ioperm", linux.CAP_SYS_RAWIO, "", nil),
		174: syscalls.CapError("create_module", linux.CAP_SYS_MODULE, "", nil),
		175: syscalls.CapError("init_module", linux.CAP_SYS_MODULE, "", nil),
		176: syscalls.CapError("delete_module", linux.CAP_SYS_MODULE, "", nil),
		177: syscalls.Error("get_kernel_syms", linuxerr.ENOSYS, "Not supported in Linux > 2.6.", nil),
		178: syscalls.Error("query_module", linuxerr.ENOSYS, "Not supported in Linux > 2.6.", nil),
		179: syscalls.CapError("quotactl", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_admin for most operations
		180: syscalls.Error("nfsservctl", linuxerr.ENOSYS, "Removed after Linux 3.1.", nil),
		181: syscalls.Error("getpmsg", linuxerr.ENOSYS, "Not implemented in Linux.", nil),
		182: syscalls.Error("putpmsg", linuxerr.ENOSYS, "Not implemented in Linux.", nil),
		183: syscalls.Error("afs_syscall", linuxerr.ENOSYS, "Not implemented in Linux.", nil),
		184: syscalls.Error("tuxcall", linuxerr.ENOSYS, "Not implemented in Linux.", nil),
		185: syscalls.Error("security", linuxerr.ENOSYS, "Not implemented in Linux.", nil),
		186: syscalls.Supported("gettid", Gettid),
		187: syscalls.Supported("readahead", Readahead),
		188: syscalls.PartiallySupported("setxattr", SetXattr, "Only supported for tmpfs.", nil),
		189: syscalls.PartiallySupported("lsetxattr", LSetXattr, "Only supported for tmpfs.", nil),
		190: syscalls.PartiallySupported("fsetxattr", FSetXattr, "Only supported for tmpfs.", nil),
		191: syscalls.PartiallySupported("getxattr", GetXattr, "Only supported for tmpfs.", nil),
		192: syscalls.PartiallySupported("lgetxattr", LGetXattr, "Only supported for tmpfs.", nil),
		193: syscalls.PartiallySupported("fgetxattr", FGetXattr, "Only supported for tmpfs.", nil),
		194: syscalls.PartiallySupported("listxattr", ListXattr, "Only supported for tmpfs", nil),
		195: syscalls.PartiallySupported("llistxattr", LListXattr, "Only supported for tmpfs", nil),
		196: syscalls.PartiallySupported("flistxattr", FListXattr, "Only supported for tmpfs", nil),
		197: syscalls.PartiallySupported("removexattr", RemoveXattr, "Only supported for tmpfs", nil),
		198: syscalls.PartiallySupported("lremovexattr", LRemoveXattr, "Only supported for tmpfs", nil),
		199: syscalls.PartiallySupported("fremovexattr", FRemoveXattr, "Only supported for tmpfs", nil),
		200: syscalls.Supported("tkill", Tkill),
		201: syscalls.Supported("time", Time),
		202: syscalls.PartiallySupported("futex", Futex, "Robust futexes not supported.", nil),
		203: syscalls.PartiallySupported("sched_setaffinity", SchedSetaffinity, "Stub implementation.", nil),
		204: syscalls.PartiallySupported("sched_getaffinity", SchedGetaffinity, "Stub implementation.", nil),
		205: syscalls.Error("set_thread_area", linuxerr.ENOSYS, "Expected to return ENOSYS on 64-bit", nil),
		206: syscalls.PartiallySupported("io_setup", IoSetup, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		207: syscalls.PartiallySupported("io_destroy", IoDestroy, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		208: syscalls.PartiallySupported("io_getevents", IoGetevents, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		209: syscalls.PartiallySupported("io_submit", IoSubmit, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		210: syscalls.PartiallySupported("io_cancel", IoCancel, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		211: syscalls.Error("get_thread_area", linuxerr.ENOSYS, "Expected to return ENOSYS on 64-bit", nil),
		212: syscalls.CapError("lookup_dcookie", linux.CAP_SYS_ADMIN, "", nil),
		213: syscalls.Supported("epoll_create", EpollCreate),
		214: syscalls.ErrorWithEvent("epoll_ctl_old", linuxerr.ENOSYS, "Deprecated.", nil),
		215: syscalls.ErrorWithEvent("epoll_wait_old", linuxerr.ENOSYS, "Deprecated.", nil),
		216: syscalls.ErrorWithEvent("remap_file_pages", linuxerr.ENOSYS, "Deprecated since Linux 3.16.", nil),
		217: syscalls.Supported("getdents64", Getdents64),
		218: syscalls.Supported("set_tid_address", SetTidAddress),
		219: syscalls.Supported("restart_syscall", RestartSyscall),
		220: syscalls.Supported("semtimedop", Semtimedop),
		221: syscalls.PartiallySupported("fadvise64", Fadvise64, "Not all options are supported.", nil),
		222: syscalls.Supported("timer_create", TimerCreate),
		223: syscalls.Supported("timer_settime", TimerSettime),
		224: syscalls.Supported("timer_gettime", TimerGettime),
		225: syscalls.Supported("timer_getoverrun", TimerGetoverrun),
		226: syscalls.Supported("timer_delete", TimerDelete),
		227: syscalls.Supported("clock_settime", ClockSettime),
		228: syscalls.Supported("clock_gettime", ClockGettime),
		229: syscalls.Supported("clock_getres", ClockGetres),
		230: syscalls.Supported("clock_nanosleep", ClockNanosleep),
		231: syscalls.Supported("exit_group", ExitGroup),
		232: syscalls.Supported("epoll_wait", EpollWait),
		233: syscalls.Supported("epoll_ctl", EpollCtl),
		234: syscalls.Supported("tgkill", Tgkill),
		235: syscalls.Supported("utimes", Utimes),
		236: syscalls.Error("vserver", linuxerr.ENOSYS, "Not implemented by Linux", nil),
		237: syscalls.PartiallySupported("mbind", Mbind, "Stub implementation. Only a single NUMA node is advertised, and mempolicy is ignored accordingly, but mbind() will succeed and has effects reflected by get_mempolicy.", []string{"gvisor.dev/issue/262"}),
		238: syscalls.PartiallySupported("set_mempolicy", SetMempolicy, "Stub implementation.", nil),
		239: syscalls.PartiallySupported("get_mempolicy", GetMempolicy, "Stub implementation.", nil),
		240: syscalls.ErrorWithEvent("mq_open", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),         // TODO(b/29354921)
		241: syscalls.ErrorWithEvent("mq_unlink", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),       // TODO(b/29354921)
		242: syscalls.ErrorWithEvent("mq_timedsend", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),    // TODO(b/29354921)
		243: syscalls.ErrorWithEvent("mq_timedreceive", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}), // TODO(b/29354921)
		244: syscalls.ErrorWithEvent("mq_notify", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),       // TODO(b/29354921)
		245: syscalls.ErrorWithEvent("mq_getsetattr", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),   // TODO(b/29354921)
		246: syscalls.CapError("kexec_load", linux.CAP_SYS_BOOT, "", nil),
		247: syscalls.Supported("waitid", Waitid),
		248: syscalls.Error("add_key", linuxerr.EACCES, "Not available to user.", nil),
		249: syscalls.Error("request_key", linuxerr.EACCES, "Not available to user.", nil),
		250: syscalls.Error("keyctl", linuxerr.EACCES, "Not available to user.", nil),
		251: syscalls.CapError("ioprio_set", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_nice or cap_sys_admin (depending)
		252: syscalls.CapError("ioprio_get", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_nice or cap_sys_admin (depending)
		253: syscalls.PartiallySupportedPoint("inotify_init", InotifyInit, PointInotifyInit, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		254: syscalls.PartiallySupportedPoint("inotify_add_watch", InotifyAddWatch, PointInotifyAddWatch, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		255: syscalls.PartiallySupportedPoint("inotify_rm_watch", InotifyRmWatch, PointInotifyRmWatch, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		256: syscalls.CapError("migrate_pages", linux.CAP_SYS_NICE, "", nil),
		257: syscalls.SupportedPoint("openat", Openat, PointOpenat),
		258: syscalls.Supported("mkdirat", Mkdirat),
		259: syscalls.Supported("mknodat", Mknodat),
		260: syscalls.Supported("fchownat", Fchownat),
		261: syscalls.Supported("futimesat", Futimesat),
		262: syscalls.Supported("fstatat", Fstatat),
		263: syscalls.Supported("unlinkat", Unlinkat),
		264: syscalls.Supported("renameat", Renameat),
		265: syscalls.PartiallySupported("linkat", Linkat, "See link(2).", nil),
		266: syscalls.Supported("symlinkat", Symlinkat),
		267: syscalls.Supported("readlinkat", Readlinkat),
		268: syscalls.Supported("fchmodat", Fchmodat),
		269: syscalls.Supported("faccessat", Faccessat),
		270: syscalls.Supported("pselect", Pselect),
		271: syscalls.Supported("ppoll", Ppoll),
		272: syscalls.PartiallySupported("unshare", Unshare, "Mount, cgroup namespaces not supported. Network namespaces supported but must be empty.", nil),
		273: syscalls.Supported("set_robust_list", SetRobustList),
		274: syscalls.Supported("get_robust_list", GetRobustList),
		275: syscalls.Supported("splice", Splice),
		276: syscalls.Supported("tee", Tee),
		277: syscalls.PartiallySupported("sync_file_range", SyncFileRange, "Full data flush is not guaranteed at this time.", nil),
		278: syscalls.ErrorWithEvent("vmsplice", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/138"}), // TODO(b/29354098)
		279: syscalls.CapError("move_pages", linux.CAP_SYS_NICE, "", nil),                               // requires cap_sys_nice (mostly)
		280: syscalls.Supported("utimensat", Utimensat),
		281: syscalls.Supported("epoll_pwait", EpollPwait),
		282: syscalls.PartiallySupportedPoint("signalfd", Signalfd, PointSignalfd, "Semantics are slightly different.", []string{"gvisor.dev/issue/139"}),
		283: syscalls.SupportedPoint("timerfd_create", TimerfdCreate, PointTimerfdCreate),
		284: syscalls.SupportedPoint("eventfd", Eventfd, PointEventfd),
		285: syscalls.PartiallySupported("fallocate", Fallocate, "Not all options are supported.", nil),
		286: syscalls.SupportedPoint("timerfd_settime", TimerfdSettime, PointTimerfdSettime),
		287: syscalls.SupportedPoint("timerfd_gettime", TimerfdGettime, PointTimerfdGettime),
		288: syscalls.SupportedPoint("accept4", Accept4, PointAccept4),
		289: syscalls.PartiallySupportedPoint("signalfd4", Signalfd4, PointSignalfd4, "Semantics are slightly different.", []string{"gvisor.dev/issue/139"}),
		290: syscalls.SupportedPoint("eventfd2", Eventfd2, PointEventfd2),
		291: syscalls.Supported("epoll_create1", EpollCreate1),
		292: syscalls.SupportedPoint("dup3", Dup3, PointDup3),
		293: syscalls.SupportedPoint("pipe2", Pipe2, PointPipe2),
		294: syscalls.PartiallySupportedPoint("inotify_init1", InotifyInit1, PointInotifyInit1, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		295: syscalls.Supported("preadv", Preadv),
		296: syscalls.Supported("pwritev", Pwritev),
		297: syscalls.Supported("rt_tgsigqueueinfo", RtTgsigqueueinfo),
		298: syscalls.ErrorWithEvent("perf_event_open", linuxerr.ENODEV, "No support for perf counters", nil),
		299: syscalls.PartiallySupported("recvmmsg", RecvMMsg, "Not all flags and control messages are supported.", nil),
		300: syscalls.ErrorWithEvent("fanotify_init", linuxerr.ENOSYS, "Needs CONFIG_FANOTIFY", nil),
		301: syscalls.ErrorWithEvent("fanotify_mark", linuxerr.ENOSYS, "Needs CONFIG_FANOTIFY", nil),
		302: syscalls.SupportedPoint("prlimit64", Prlimit64, PointPrlimit64),
		303: syscalls.Error("name_to_handle_at", linuxerr.EOPNOTSUPP, "Not supported by gVisor filesystems", nil),
		304: syscalls.Error("open_by_handle_at", linuxerr.EOPNOTSUPP, "Not supported by gVisor filesystems", nil),
		305: syscalls.CapError("clock_adjtime", linux.CAP_SYS_TIME, "", nil),
		306: syscalls.PartiallySupported("syncfs", Syncfs, "Depends on backing file system.", nil),
		307: syscalls.PartiallySupported("sendmmsg", SendMMsg, "Not all flags and control messages are supported.", nil),
		308: syscalls.ErrorWithEvent("setns", linuxerr.EOPNOTSUPP, "Needs filesystem support", []string{"gvisor.dev/issue/140"}), // TODO(b/29354995)
		309: syscalls.Supported("getcpu", Getcpu),
		310: syscalls.ErrorWithEvent("process_vm_readv", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/158"}),
		311: syscalls.ErrorWithEvent("process_vm_writev", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/158"}),
		312: syscalls.CapError("kcmp", linux.CAP_SYS_PTRACE, "", nil),
		313: syscalls.CapError("finit_module", linux.CAP_SYS_MODULE, "", nil),
		314: syscalls.ErrorWithEvent("sched_setattr", linuxerr.ENOSYS, "gVisor does not implement a scheduler.", []string{"gvisor.dev/issue/264"}), // TODO(b/118902272)
		315: syscalls.ErrorWithEvent("sched_getattr", linuxerr.ENOSYS, "gVisor does not implement a scheduler.", []string{"gvisor.dev/issue/264"}), // TODO(b/118902272)
		316: syscalls.ErrorWithEvent("renameat2", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/263"}),                                           // TODO(b/118902772)
		317: syscalls.Supported("seccomp", Seccomp),
		318: syscalls.Supported("getrandom", GetRandom),
		319: syscalls.Supported("memfd_create", MemfdCreate),
		320: syscalls.CapError("kexec_file_load", linux.CAP_SYS_BOOT, "", nil),
		321: syscalls.CapError("bpf", linux.CAP_SYS_ADMIN, "", nil),
		322: syscalls.SupportedPoint("execveat", Execveat, PointExecveat),
		323: syscalls.ErrorWithEvent("userfaultfd", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/266"}), // TODO(b/118906345)
		324: syscalls.PartiallySupported("membarrier", Membarrier, "Not supported on all platforms.", nil),
		325: syscalls.PartiallySupported("mlock2", Mlock2, "Stub implementation. The sandbox lacks appropriate permissions.", nil),

		// Syscalls implemented after 325 are "backports" from versions
		// of Linux after 4.4.
		326: syscalls.ErrorWithEvent("copy_file_range", linuxerr.ENOSYS, "", nil),
		327: syscalls.Supported("preadv2", Preadv2),
		328: syscalls.PartiallySupported("pwritev2", Pwritev2, "Flag RWF_HIPRI is not supported.", nil),
		329: syscalls.ErrorWithEvent("pkey_mprotect", linuxerr.ENOSYS, "", nil),
		330: syscalls.ErrorWithEvent("pkey_alloc", linuxerr.ENOSYS, "", nil),
		331: syscalls.ErrorWithEvent("pkey_free", linuxerr.ENOSYS, "", nil),
		332: syscalls.Supported("statx", Statx),
		333: syscalls.ErrorWithEvent("io_pgetevents", linuxerr.ENOSYS, "", nil),
		334: syscalls.PartiallySupported("rseq", RSeq, "Not supported on all platforms.", nil),

		// Linux skips ahead to syscall 424 to sync numbers between arches.
		424: syscalls.ErrorWithEvent("pidfd_send_signal", linuxerr.ENOSYS, "", nil),
		425: syscalls.ErrorWithEvent("io_uring_setup", linuxerr.ENOSYS, "", nil),
		426: syscalls.ErrorWithEvent("io_uring_enter", linuxerr.ENOSYS, "", nil),
		427: syscalls.ErrorWithEvent("io_uring_register", linuxerr.ENOSYS, "", nil),
		428: syscalls.ErrorWithEvent("open_tree", linuxerr.ENOSYS, "", nil),
		429: syscalls.ErrorWithEvent("move_mount", linuxerr.ENOSYS, "", nil),
		430: syscalls.ErrorWithEvent("fsopen", linuxerr.ENOSYS, "", nil),
		431: syscalls.ErrorWithEvent("fsconfig", linuxerr.ENOSYS, "", nil),
		432: syscalls.ErrorWithEvent("fsmount", linuxerr.ENOSYS, "", nil),
		433: syscalls.ErrorWithEvent("fspick", linuxerr.ENOSYS, "", nil),
		434: syscalls.ErrorWithEvent("pidfd_open", linuxerr.ENOSYS, "", nil),
		435: syscalls.ErrorWithEvent("clone3", linuxerr.ENOSYS, "", nil),
		436: syscalls.Supported("close_range", CloseRange),
		441: syscalls.Supported("epoll_pwait2", EpollPwait2),
	},
	Emulate: map[hostarch.Addr]uintptr{
		0xffffffffff600000: 96,  // vsyscall gettimeofday(2)
		0xffffffffff600400: 201, // vsyscall time(2)
		0xffffffffff600800: 309, // vsyscall getcpu(2)
	},
	Missing: func(t *kernel.Task, sysno uintptr, args arch.SyscallArguments) (uintptr, error) {
		t.Kernel().EmitUnimplementedEvent(t)
		return 0, linuxerr.ENOSYS
	},
}

// ARM64 is a table of Linux arm64 syscall API with the corresponding syscall
// numbers from Linux 4.4.
var ARM64 = &kernel.SyscallTable{
	OS:   abi.Linux,
	Arch: arch.ARM64,
	Version: kernel.Version{
		Sysname: LinuxSysname,
		Release: LinuxRelease,
		Version: LinuxVersion,
	},
	AuditNumber: linux.AUDIT_ARCH_AARCH64,
	Table: map[uintptr]kernel.Syscall{
		0:   syscalls.PartiallySupported("io_setup", IoSetup, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		1:   syscalls.PartiallySupported("io_destroy", IoDestroy, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		2:   syscalls.PartiallySupported("io_submit", IoSubmit, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		3:   syscalls.PartiallySupported("io_cancel", IoCancel, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		4:   syscalls.PartiallySupported("io_getevents", IoGetevents, "Generally supported with exceptions. User ring optimizations are not implemented.", []string{"gvisor.dev/issue/204"}),
		5:   syscalls.PartiallySupported("setxattr", SetXattr, "Only supported for tmpfs.", nil),
		6:   syscalls.PartiallySupported("lsetxattr", LSetXattr, "Only supported for tmpfs.", nil),
		7:   syscalls.PartiallySupported("fsetxattr", FSetXattr, "Only supported for tmpfs.", nil),
		8:   syscalls.PartiallySupported("getxattr", GetXattr, "Only supported for tmpfs.", nil),
		9:   syscalls.PartiallySupported("lgetxattr", LGetXattr, "Only supported for tmpfs.", nil),
		10:  syscalls.PartiallySupported("fgetxattr", FGetXattr, "Only supported for tmpfs.", nil),
		11:  syscalls.PartiallySupported("listxattr", ListXattr, "Only supported for tmpfs", nil),
		12:  syscalls.PartiallySupported("llistxattr", LListXattr, "Only supported for tmpfs", nil),
		13:  syscalls.PartiallySupported("flistxattr", FListXattr, "Only supported for tmpfs", nil),
		14:  syscalls.PartiallySupported("removexattr", RemoveXattr, "Only supported for tmpfs", nil),
		15:  syscalls.PartiallySupported("lremovexattr", LRemoveXattr, "Only supported for tmpfs", nil),
		16:  syscalls.PartiallySupported("fremovexattr", FRemoveXattr, "Only supported for tmpfs", nil),
		17:  syscalls.Supported("getcwd", Getcwd),
		18:  syscalls.CapError("lookup_dcookie", linux.CAP_SYS_ADMIN, "", nil),
		19:  syscalls.SupportedPoint("eventfd2", Eventfd2, PointEventfd2),
		20:  syscalls.Supported("epoll_create1", EpollCreate1),
		21:  syscalls.Supported("epoll_ctl", EpollCtl),
		22:  syscalls.Supported("epoll_pwait", EpollPwait),
		23:  syscalls.SupportedPoint("dup", Dup, PointDup),
		24:  syscalls.SupportedPoint("dup3", Dup3, PointDup3),
		25:  syscalls.PartiallySupportedPoint("fcntl", Fcntl, PointFcntl, "Not all options are supported.", nil),
		26:  syscalls.PartiallySupportedPoint("inotify_init1", InotifyInit1, PointInotifyInit1, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		27:  syscalls.PartiallySupportedPoint("inotify_add_watch", InotifyAddWatch, PointInotifyAddWatch, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		28:  syscalls.PartiallySupportedPoint("inotify_rm_watch", InotifyRmWatch, PointInotifyRmWatch, "Inotify events are only available inside the sandbox. Hard links are treated as different watch targets in gofer fs.", nil),
		29:  syscalls.PartiallySupported("ioctl", Ioctl, "Only a few ioctls are implemented for backing devices and file systems.", nil),
		30:  syscalls.CapError("ioprio_set", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_nice or cap_sys_admin (depending)
		31:  syscalls.CapError("ioprio_get", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_nice or cap_sys_admin (depending)
		32:  syscalls.PartiallySupported("flock", Flock, "Locks are held within the sandbox only.", nil),
		33:  syscalls.Supported("mknodat", Mknodat),
		34:  syscalls.Supported("mkdirat", Mkdirat),
		35:  syscalls.Supported("unlinkat", Unlinkat),
		36:  syscalls.Supported("symlinkat", Symlinkat),
		37:  syscalls.Supported("linkat", Linkat),
		38:  syscalls.Supported("renameat", Renameat),
		39:  syscalls.PartiallySupported("umount2", Umount2, "Not all options or file systems are supported.", nil),
		40:  syscalls.PartiallySupported("mount", Mount, "Not all options or file systems are supported.", nil),
		41:  syscalls.Error("pivot_root", linuxerr.EPERM, "", nil),
		42:  syscalls.Error("nfsservctl", linuxerr.ENOSYS, "Removed after Linux 3.1.", nil),
		43:  syscalls.PartiallySupported("statfs", Statfs, "Depends on the backing file system implementation.", nil),
		44:  syscalls.PartiallySupported("fstatfs", Fstatfs, "Depends on the backing file system implementation.", nil),
		45:  syscalls.Supported("truncate", Truncate),
		46:  syscalls.Supported("ftruncate", Ftruncate),
		47:  syscalls.PartiallySupported("fallocate", Fallocate, "Not all options are supported.", nil),
		48:  syscalls.Supported("faccessat", Faccessat),
		49:  syscalls.SupportedPoint("chdir", Chdir, PointChdir),
		50:  syscalls.SupportedPoint("fchdir", Fchdir, PointFchdir),
		51:  syscalls.SupportedPoint("chroot", Chroot, PointChroot),
		52:  syscalls.PartiallySupported("fchmod", Fchmod, "Options S_ISUID and S_ISGID not supported.", nil),
		53:  syscalls.Supported("fchmodat", Fchmodat),
		54:  syscalls.Supported("fchownat", Fchownat),
		55:  syscalls.Supported("fchown", Fchown),
		56:  syscalls.SupportedPoint("openat", Openat, PointOpenat),
		57:  syscalls.Supported("close", Close),
		58:  syscalls.CapError("vhangup", linux.CAP_SYS_TTY_CONFIG, "", nil),
		59:  syscalls.SupportedPoint("pipe2", Pipe2, PointPipe2),
		60:  syscalls.CapError("quotactl", linux.CAP_SYS_ADMIN, "", nil), // requires cap_sys_admin for most operations
		61:  syscalls.Supported("getdents64", Getdents64),
		62:  syscalls.Supported("lseek", Lseek),
		63:  syscalls.SupportedPoint("read", Read, PointRead),
		64:  syscalls.Supported("write", Write),
		65:  syscalls.Supported("readv", Readv),
		66:  syscalls.Supported("writev", Writev),
		67:  syscalls.Supported("pread64", Pread64),
		68:  syscalls.Supported("pwrite64", Pwrite64),
		69:  syscalls.Supported("preadv", Preadv),
		70:  syscalls.Supported("pwritev", Pwritev),
		71:  syscalls.Supported("sendfile", Sendfile),
		72:  syscalls.Supported("pselect", Pselect),
		73:  syscalls.Supported("ppoll", Ppoll),
		74:  syscalls.PartiallySupportedPoint("signalfd4", Signalfd4, PointSignalfd4, "Semantics are slightly different.", []string{"gvisor.dev/issue/139"}),
		75:  syscalls.ErrorWithEvent("vmsplice", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/138"}), // TODO(b/29354098)
		76:  syscalls.Supported("splice", Splice),
		77:  syscalls.Supported("tee", Tee),
		78:  syscalls.Supported("readlinkat", Readlinkat),
		79:  syscalls.Supported("fstatat", Fstatat),
		80:  syscalls.Supported("fstat", Fstat),
		81:  syscalls.PartiallySupported("sync", Sync, "Full data flush is not guaranteed at this time.", nil),
		82:  syscalls.PartiallySupported("fsync", Fsync, "Full data flush is not guaranteed at this time.", nil),
		83:  syscalls.PartiallySupported("fdatasync", Fdatasync, "Full data flush is not guaranteed at this time.", nil),
		84:  syscalls.PartiallySupported("sync_file_range", SyncFileRange, "Full data flush is not guaranteed at this time.", nil),
		85:  syscalls.SupportedPoint("timerfd_create", TimerfdCreate, PointTimerfdCreate),
		86:  syscalls.SupportedPoint("timerfd_settime", TimerfdSettime, PointTimerfdSettime),
		87:  syscalls.SupportedPoint("timerfd_gettime", TimerfdGettime, PointTimerfdGettime),
		88:  syscalls.Supported("utimensat", Utimensat),
		89:  syscalls.CapError("acct", linux.CAP_SYS_PACCT, "", nil),
		90:  syscalls.Supported("capget", Capget),
		91:  syscalls.Supported("capset", Capset),
		92:  syscalls.ErrorWithEvent("personality", linuxerr.EINVAL, "Unable to change personality.", nil),
		93:  syscalls.Supported("exit", Exit),
		94:  syscalls.Supported("exit_group", ExitGroup),
		95:  syscalls.Supported("waitid", Waitid),
		96:  syscalls.Supported("set_tid_address", SetTidAddress),
		97:  syscalls.PartiallySupported("unshare", Unshare, "Mount, cgroup namespaces not supported. Network namespaces supported but must be empty.", nil),
		98:  syscalls.PartiallySupported("futex", Futex, "Robust futexes not supported.", nil),
		99:  syscalls.Supported("set_robust_list", SetRobustList),
		100: syscalls.Supported("get_robust_list", GetRobustList),
		101: syscalls.Supported("nanosleep", Nanosleep),
		102: syscalls.Supported("getitimer", Getitimer),
		103: syscalls.Supported("setitimer", Setitimer),
		104: syscalls.CapError("kexec_load", linux.CAP_SYS_BOOT, "", nil),
		105: syscalls.CapError("init_module", linux.CAP_SYS_MODULE, "", nil),
		106: syscalls.CapError("delete_module", linux.CAP_SYS_MODULE, "", nil),
		107: syscalls.Supported("timer_create", TimerCreate),
		108: syscalls.Supported("timer_gettime", TimerGettime),
		109: syscalls.Supported("timer_getoverrun", TimerGetoverrun),
		110: syscalls.Supported("timer_settime", TimerSettime),
		111: syscalls.Supported("timer_delete", TimerDelete),
		112: syscalls.Supported("clock_settime", ClockSettime),
		113: syscalls.Supported("clock_gettime", ClockGettime),
		114: syscalls.Supported("clock_getres", ClockGetres),
		115: syscalls.Supported("clock_nanosleep", ClockNanosleep),
		116: syscalls.PartiallySupported("syslog", Syslog, "Outputs a dummy message for security reasons.", nil),
		117: syscalls.PartiallySupported("ptrace", Ptrace, "Options PTRACE_PEEKSIGINFO, PTRACE_SECCOMP_GET_FILTER not supported.", nil),
		118: syscalls.CapError("sched_setparam", linux.CAP_SYS_NICE, "", nil),
		119: syscalls.PartiallySupported("sched_setscheduler", SchedSetscheduler, "Stub implementation.", nil),
		120: syscalls.PartiallySupported("sched_getscheduler", SchedGetscheduler, "Stub implementation.", nil),
		121: syscalls.PartiallySupported("sched_getparam", SchedGetparam, "Stub implementation.", nil),
		122: syscalls.PartiallySupported("sched_setaffinity", SchedSetaffinity, "Stub implementation.", nil),
		123: syscalls.PartiallySupported("sched_getaffinity", SchedGetaffinity, "Stub implementation.", nil),
		124: syscalls.Supported("sched_yield", SchedYield),
		125: syscalls.PartiallySupported("sched_get_priority_max", SchedGetPriorityMax, "Stub implementation.", nil),
		126: syscalls.PartiallySupported("sched_get_priority_min", SchedGetPriorityMin, "Stub implementation.", nil),
		127: syscalls.ErrorWithEvent("sched_rr_get_interval", linuxerr.EPERM, "", nil),
		128: syscalls.Supported("restart_syscall", RestartSyscall),
		129: syscalls.Supported("kill", Kill),
		130: syscalls.Supported("tkill", Tkill),
		131: syscalls.Supported("tgkill", Tgkill),
		132: syscalls.Supported("sigaltstack", Sigaltstack),
		133: syscalls.Supported("rt_sigsuspend", RtSigsuspend),
		134: syscalls.Supported("rt_sigaction", RtSigaction),
		135: syscalls.Supported("rt_sigprocmask", RtSigprocmask),
		136: syscalls.Supported("rt_sigpending", RtSigpending),
		137: syscalls.Supported("rt_sigtimedwait", RtSigtimedwait),
		138: syscalls.Supported("rt_sigqueueinfo", RtSigqueueinfo),
		139: syscalls.Supported("rt_sigreturn", RtSigreturn),
		140: syscalls.PartiallySupported("setpriority", Setpriority, "Stub implementation.", nil),
		141: syscalls.PartiallySupported("getpriority", Getpriority, "Stub implementation.", nil),
		142: syscalls.CapError("reboot", linux.CAP_SYS_BOOT, "", nil),
		143: syscalls.Supported("setregid", Setregid),
		144: syscalls.SupportedPoint("setgid", Setgid, PointSetgid),
		145: syscalls.Supported("setreuid", Setreuid),
		146: syscalls.SupportedPoint("setuid", Setuid, PointSetuid),
		147: syscalls.SupportedPoint("setresuid", Setresuid, PointSetresuid),
		148: syscalls.Supported("getresuid", Getresuid),
		149: syscalls.SupportedPoint("setresgid", Setresgid, PointSetresgid),
		150: syscalls.Supported("getresgid", Getresgid),
		151: syscalls.ErrorWithEvent("setfsuid", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/260"}), // TODO(b/112851702)
		152: syscalls.ErrorWithEvent("setfsgid", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/260"}), // TODO(b/112851702)
		153: syscalls.Supported("times", Times),
		154: syscalls.Supported("setpgid", Setpgid),
		155: syscalls.Supported("getpgid", Getpgid),
		156: syscalls.Supported("getsid", Getsid),
		157: syscalls.SupportedPoint("setsid", Setsid, PointSetsid),
		158: syscalls.Supported("getgroups", Getgroups),
		159: syscalls.Supported("setgroups", Setgroups),
		160: syscalls.Supported("uname", Uname),
		161: syscalls.Supported("sethostname", Sethostname),
		162: syscalls.Supported("setdomainname", Setdomainname),
		163: syscalls.Supported("getrlimit", Getrlimit),
		164: syscalls.PartiallySupported("setrlimit", Setrlimit, "Not all rlimits are enforced.", nil),
		165: syscalls.PartiallySupported("getrusage", Getrusage, "Fields ru_maxrss, ru_minflt, ru_majflt, ru_inblock, ru_oublock are not supported. Fields ru_utime and ru_stime have low precision.", nil),
		166: syscalls.Supported("umask", Umask),
		167: syscalls.PartiallySupported("prctl", Prctl, "Not all options are supported.", nil),
		168: syscalls.Supported("getcpu", Getcpu),
		169: syscalls.Supported("gettimeofday", Gettimeofday),
		170: syscalls.CapError("settimeofday", linux.CAP_SYS_TIME, "", nil),
		171: syscalls.CapError("adjtimex", linux.CAP_SYS_TIME, "", nil),
		172: syscalls.Supported("getpid", Getpid),
		173: syscalls.Supported("getppid", Getppid),
		174: syscalls.Supported("getuid", Getuid),
		175: syscalls.Supported("geteuid", Geteuid),
		176: syscalls.Supported("getgid", Getgid),
		177: syscalls.Supported("getegid", Getegid),
		178: syscalls.Supported("gettid", Gettid),
		179: syscalls.PartiallySupported("sysinfo", Sysinfo, "Fields loads, sharedram, bufferram, totalswap, freeswap, totalhigh, freehigh not supported.", nil),
		180: syscalls.ErrorWithEvent("mq_open", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),         // TODO(b/29354921)
		181: syscalls.ErrorWithEvent("mq_unlink", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),       // TODO(b/29354921)
		182: syscalls.ErrorWithEvent("mq_timedsend", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),    // TODO(b/29354921)
		183: syscalls.ErrorWithEvent("mq_timedreceive", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}), // TODO(b/29354921)
		184: syscalls.ErrorWithEvent("mq_notify", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),       // TODO(b/29354921)
		185: syscalls.ErrorWithEvent("mq_getsetattr", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/136"}),   // TODO(b/29354921)
		186: syscalls.Supported("msgget", Msgget),
		187: syscalls.Supported("msgctl", Msgctl),
		188: syscalls.Supported("msgrcv", Msgrcv),
		189: syscalls.Supported("msgsnd", Msgsnd),
		190: syscalls.Supported("semget", Semget),
		191: syscalls.Supported("semctl", Semctl),
		192: syscalls.Supported("semtimedop", Semtimedop),
		193: syscalls.PartiallySupported("semop", Semop, "Option SEM_UNDO not supported.", nil),
		194: syscalls.PartiallySupported("shmget", Shmget, "Option SHM_HUGETLB is not supported.", nil),
		195: syscalls.PartiallySupported("shmctl", Shmctl, "Options SHM_LOCK, SHM_UNLOCK are not supported.", nil),
		196: syscalls.PartiallySupported("shmat", Shmat, "Option SHM_RND is not supported.", nil),
		197: syscalls.Supported("shmdt", Shmdt),
		198: syscalls.PartiallySupported("socket", Socket, "Limited support for AF_NETLINK, NETLINK_ROUTE sockets. Limited support for SOCK_RAW.", nil),
		199: syscalls.SupportedPoint("socketpair", SocketPair, PointSocketpair),
		200: syscalls.PartiallySupportedPoint("bind", Bind, PointBind, "Autobind for abstract Unix sockets is not supported.", nil),
		201: syscalls.Supported("listen", Listen),
		202: syscalls.SupportedPoint("accept", Accept, PointAccept),
		203: syscalls.SupportedPoint("connect", Connect, PointConnect),
		204: syscalls.Supported("getsockname", GetSockName),
		205: syscalls.Supported("getpeername", GetPeerName),
		206: syscalls.Supported("sendto", SendTo),
		207: syscalls.Supported("recvfrom", RecvFrom),
		208: syscalls.PartiallySupported("setsockopt", SetSockOpt, "Not all socket options are supported.", nil),
		209: syscalls.PartiallySupported("getsockopt", GetSockOpt, "Not all socket options are supported.", nil),
		210: syscalls.PartiallySupported("shutdown", Shutdown, "Not all flags and control messages are supported.", nil),
		211: syscalls.Supported("sendmsg", SendMsg),
		212: syscalls.PartiallySupported("recvmsg", RecvMsg, "Not all flags and control messages are supported.", nil),
		213: syscalls.Supported("readahead", Readahead),
		214: syscalls.Supported("brk", Brk),
		215: syscalls.Supported("munmap", Munmap),
		216: syscalls.Supported("mremap", Mremap),
		217: syscalls.Error("add_key", linuxerr.EACCES, "Not available to user.", nil),
		218: syscalls.Error("request_key", linuxerr.EACCES, "Not available to user.", nil),
		219: syscalls.Error("keyctl", linuxerr.EACCES, "Not available to user.", nil),
		220: syscalls.PartiallySupportedPoint("clone", Clone, PointClone, "Mount namespace (CLONE_NEWNS) not supported. Options CLONE_PARENT, CLONE_SYSVSEM not supported.", nil),
		221: syscalls.SupportedPoint("execve", Execve, PointExecve),
		222: syscalls.PartiallySupported("mmap", Mmap, "Generally supported with exceptions. Options MAP_FIXED_NOREPLACE, MAP_SHARED_VALIDATE, MAP_SYNC MAP_GROWSDOWN, MAP_HUGETLB are not supported.", nil),
		223: syscalls.PartiallySupported("fadvise64", Fadvise64, "Not all options are supported.", nil),
		224: syscalls.CapError("swapon", linux.CAP_SYS_ADMIN, "", nil),
		225: syscalls.CapError("swapoff", linux.CAP_SYS_ADMIN, "", nil),
		226: syscalls.Supported("mprotect", Mprotect),
		227: syscalls.PartiallySupported("msync", Msync, "Full data flush is not guaranteed at this time.", nil),
		228: syscalls.PartiallySupported("mlock", Mlock, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		229: syscalls.PartiallySupported("munlock", Munlock, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		230: syscalls.PartiallySupported("mlockall", Mlockall, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		231: syscalls.PartiallySupported("munlockall", Munlockall, "Stub implementation. The sandbox lacks appropriate permissions.", nil),
		232: syscalls.PartiallySupported("mincore", Mincore, "Stub implementation. The sandbox does not have access to this information. Reports all mapped pages are resident.", nil),
		233: syscalls.PartiallySupported("madvise", Madvise, "Options MADV_DONTNEED, MADV_DONTFORK are supported. Other advice is ignored.", nil),
		234: syscalls.ErrorWithEvent("remap_file_pages", linuxerr.ENOSYS, "Deprecated since Linux 3.16.", nil),
		235: syscalls.PartiallySupported("mbind", Mbind, "Stub implementation. Only a single NUMA node is advertised, and mempolicy is ignored accordingly, but mbind() will succeed and has effects reflected by get_mempolicy.", []string{"gvisor.dev/issue/262"}),
		236: syscalls.PartiallySupported("get_mempolicy", GetMempolicy, "Stub implementation.", nil),
		237: syscalls.PartiallySupported("set_mempolicy", SetMempolicy, "Stub implementation.", nil),
		238: syscalls.CapError("migrate_pages", linux.CAP_SYS_NICE, "", nil),
		239: syscalls.CapError("move_pages", linux.CAP_SYS_NICE, "", nil), // requires cap_sys_nice (mostly)
		240: syscalls.Supported("rt_tgsigqueueinfo", RtTgsigqueueinfo),
		241: syscalls.ErrorWithEvent("perf_event_open", linuxerr.ENODEV, "No support for perf counters", nil),
		242: syscalls.SupportedPoint("accept4", Accept4, PointAccept4),
		243: syscalls.PartiallySupported("recvmmsg", RecvMMsg, "Not all flags and control messages are supported.", nil),
		260: syscalls.Supported("wait4", Wait4),
		261: syscalls.SupportedPoint("prlimit64", Prlimit64, PointPrlimit64),
		262: syscalls.ErrorWithEvent("fanotify_init", linuxerr.ENOSYS, "Needs CONFIG_FANOTIFY", nil),
		263: syscalls.ErrorWithEvent("fanotify_mark", linuxerr.ENOSYS, "Needs CONFIG_FANOTIFY", nil),
		264: syscalls.Error("name_to_handle_at", linuxerr.EOPNOTSUPP, "Not supported by gVisor filesystems", nil),
		265: syscalls.Error("open_by_handle_at", linuxerr.EOPNOTSUPP, "Not supported by gVisor filesystems", nil),
		266: syscalls.CapError("clock_adjtime", linux.CAP_SYS_TIME, "", nil),
		267: syscalls.PartiallySupported("syncfs", Syncfs, "Depends on backing file system.", nil),
		268: syscalls.ErrorWithEvent("setns", linuxerr.EOPNOTSUPP, "Needs filesystem support", []string{"gvisor.dev/issue/140"}), // TODO(b/29354995)
		269: syscalls.PartiallySupported("sendmmsg", SendMMsg, "Not all flags and control messages are supported.", nil),
		270: syscalls.ErrorWithEvent("process_vm_readv", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/158"}),
		271: syscalls.ErrorWithEvent("process_vm_writev", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/158"}),
		272: syscalls.CapError("kcmp", linux.CAP_SYS_PTRACE, "", nil),
		273: syscalls.CapError("finit_module", linux.CAP_SYS_MODULE, "", nil),
		274: syscalls.ErrorWithEvent("sched_setattr", linuxerr.ENOSYS, "gVisor does not implement a scheduler.", []string{"gvisor.dev/issue/264"}), // TODO(b/118902272)
		275: syscalls.ErrorWithEvent("sched_getattr", linuxerr.ENOSYS, "gVisor does not implement a scheduler.", []string{"gvisor.dev/issue/264"}), // TODO(b/118902272)
		276: syscalls.ErrorWithEvent("renameat2", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/263"}),                                           // TODO(b/118902772)
		277: syscalls.Supported("seccomp", Seccomp),
		278: syscalls.Supported("getrandom", GetRandom),
		279: syscalls.Supported("memfd_create", MemfdCreate),
		280: syscalls.CapError("bpf", linux.CAP_SYS_ADMIN, "", nil),
		281: syscalls.SupportedPoint("execveat", Execveat, PointExecveat),
		282: syscalls.ErrorWithEvent("userfaultfd", linuxerr.ENOSYS, "", []string{"gvisor.dev/issue/266"}), // TODO(b/118906345)
		283: syscalls.PartiallySupported("membarrier", Membarrier, "Not supported on all platforms.", nil),
		284: syscalls.PartiallySupported("mlock2", Mlock2, "Stub implementation. The sandbox lacks appropriate permissions.", nil),

		// Syscalls after 284 are "backports" from versions of Linux after 4.4.
		285: syscalls.ErrorWithEvent("copy_file_range", linuxerr.ENOSYS, "", nil),
		286: syscalls.Supported("preadv2", Preadv2),
		287: syscalls.PartiallySupported("pwritev2", Pwritev2, "Flag RWF_HIPRI is not supported.", nil),
		288: syscalls.ErrorWithEvent("pkey_mprotect", linuxerr.ENOSYS, "", nil),
		289: syscalls.ErrorWithEvent("pkey_alloc", linuxerr.ENOSYS, "", nil),
		290: syscalls.ErrorWithEvent("pkey_free", linuxerr.ENOSYS, "", nil),
		291: syscalls.Supported("statx", Statx),
		292: syscalls.ErrorWithEvent("io_pgetevents", linuxerr.ENOSYS, "", nil),
		293: syscalls.PartiallySupported("rseq", RSeq, "Not supported on all platforms.", nil),

		// Linux skips ahead to syscall 424 to sync numbers between arches.
		424: syscalls.ErrorWithEvent("pidfd_send_signal", linuxerr.ENOSYS, "", nil),
		425: syscalls.ErrorWithEvent("io_uring_setup", linuxerr.ENOSYS, "", nil),
		426: syscalls.ErrorWithEvent("io_uring_enter", linuxerr.ENOSYS, "", nil),
		427: syscalls.ErrorWithEvent("io_uring_register", linuxerr.ENOSYS, "", nil),
		428: syscalls.ErrorWithEvent("open_tree", linuxerr.ENOSYS, "", nil),
		429: syscalls.ErrorWithEvent("move_mount", linuxerr.ENOSYS, "", nil),
		430: syscalls.ErrorWithEvent("fsopen", linuxerr.ENOSYS, "", nil),
		431: syscalls.ErrorWithEvent("fsconfig", linuxerr.ENOSYS, "", nil),
		432: syscalls.ErrorWithEvent("fsmount", linuxerr.ENOSYS, "", nil),
		433: syscalls.ErrorWithEvent("fspick", linuxerr.ENOSYS, "", nil),
		434: syscalls.ErrorWithEvent("pidfd_open", linuxerr.ENOSYS, "", nil),
		435: syscalls.ErrorWithEvent("clone3", linuxerr.ENOSYS, "", nil),
		436: syscalls.Supported("close_range", CloseRange),
		441: syscalls.Supported("epoll_pwait2", EpollPwait2),
	},
	Emulate: map[hostarch.Addr]uintptr{},
	Missing: func(t *kernel.Task, sysno uintptr, args arch.SyscallArguments) (uintptr, error) {
		t.Kernel().EmitUnimplementedEvent(t)
		return 0, linuxerr.ENOSYS
	},
}

func init() {
	kernel.RegisterSyscallTable(AMD64)
	kernel.RegisterSyscallTable(ARM64)
}
