// Copyright 2024 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package misc

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"golang.org/x/tools/gopls/internal/protocol"
	. "golang.org/x/tools/gopls/internal/test/integration"
)

// This test exercises the filtering of code actions in generated files.
// Most code actions, being potential edits, are discarded, but
// some (GoTest, GoDoc) are pure queries, and so are allowed.
func TestCodeActionsInGeneratedFiles(t *testing.T) {
	const src = `
-- go.mod --
module example.com
go 1.19

-- src.go --
package a

func f() { g() }
func g() {}
-- gen.go --
// Code generated by hand; DO NOT EDIT.
package a

func f() { g() }
func g() {}
`

	Run(t, src, func(t *testing.T, env *Env) {
		check := func(filename string, wantKind ...protocol.CodeActionKind) {
			env.OpenFile(filename)
			loc := env.RegexpSearch(filename, `g\(\)`)
			actions, err := env.Editor.CodeAction(env.Ctx, loc, nil)
			if err != nil {
				t.Fatal(err)
			}

			type kinds = map[protocol.CodeActionKind]bool
			got := make(kinds)
			for _, act := range actions {
				got[act.Kind] = true
			}
			want := make(kinds)
			for _, kind := range wantKind {
				want[kind] = true
			}

			if diff := cmp.Diff(want, got); diff != "" {
				t.Errorf("%s: unexpected CodeActionKinds: (-want +got):\n%s",
					filename, diff)
				t.Log(actions)
			}
		}
		check("src.go", protocol.GoDoc, protocol.RefactorExtract, protocol.RefactorInline)
		check("gen.go", protocol.GoDoc) // just "View package documentation"
	})
}
