package main

import (
	"bytes"
	"fmt"
	"go/format"
	"os"
)

func main() {
	var err error
	switch name := os.Args[1]; name {
	case "ctmul":
		err = ctmul()
	default:
		panic("unknown template: " + name)
	}
	if err != nil {
		panic(err)
	}
}

func ctmul() error {
	type u128 struct {
		hi, lo uint64
	}
	m := []u128{
		{0x2108421084210842, 0x1084210842108421},
		{0x4210842108421084, 0x2108421084210842},
		{0x8421084210842108, 0x4210842108421084},
		{0x0842108421084210, 0x8421084210842108},
		{0x1084210842108421, 0x0842108421084210},
	}

	var b bytes.Buffer
	b.WriteString("// Code generated by gen.go. DO NOT EDIT.\n\n")
	b.WriteString("package polyval\n")
	b.WriteString("import \"math/bits\"\n")
	b.WriteString("// ctmulGeneric returns the constant time 128-bit product of \n")
	b.WriteString("// x and y in GF(2^128).\n")
	b.WriteString("//\n")
	b.WriteString("// The idea comes from Thomas Pornin's constant-time blog post\n")
	b.WriteString("// with 64-bit fixes from Tim Taubert's blog post on formally\n")
	b.WriteString("// verified GHASH.\n")
	b.WriteString("//\n")
	b.WriteString("// See https://www.bearssl.org/constanttime.html\n")
	b.WriteString("// See https://timtaubert.de/blog/2017/06/verified-binary-multiplication-for-ghash/\n")
	b.WriteString("func ctmulGeneric(x, y uint64) (z1, z0 uint64) {\n")

	b.WriteString("// Split both x and y into 5 words with four-bit holes.\n")
	for i := 0; i < 5; i++ {
		fmt.Fprintf(&b, "x%d := x & %#0.16x\n", i, m[i].lo)
		fmt.Fprintf(&b, "y%d := y & %#0.16x\n", i, m[i].lo)
	}
	b.WriteByte('\n')

	// Generate the 25 128-bit multiplications:
	//
	// t0 := x0*y0 ^ x1*y1 ... x4*y4
	// t1 := x0*y4 ^ x1*y0 ... x4*y3
	//                     ...
	// t4 := x0*y1 ^ x1*y2 ... x4*y0
	//
	y := 0
	for i := 0; i < 5; i++ {
		// Print out a comment describing this block.
		fmt.Fprintf(&b, "// t%d := ", i)
		for x := 0; x < 5; x++ {
			if x > 0 {
				b.WriteString(" ^ ")
			}
			fmt.Fprintf(&b, "(x%d*y%d)", x, y)
		}
		fmt.Fprintf(&b, "\n// z |= t%d & 0x%x%x\n", i, m[i].hi, m[i].lo)

		// The 128-bit multiplications.
		var thi, tlo []string
		for x := 0; x < 5; x++ {
			t := fmt.Sprintf("x%dy%d", x, y)
			hi := t + "hi"
			lo := t + "lo"
			fmt.Fprintf(&b, "%s, %s := bits.Mul64(x%d, y%d)\n",
				hi, lo, x, y)
			thi = append(thi, hi)
			tlo = append(tlo, lo)
			if y == 0 {
				y = 4
			} else {
				y--
			}
		}
		y++

		// XOR the high halves.
		fmt.Fprintf(&b, "z1 |= (%s", thi[0])
		for _, t := range thi[1:] {
			fmt.Fprintf(&b, " ^ %s", t)
		}
		fmt.Fprintf(&b, ") & %#0.16x\n", m[i].hi)

		// XOR the low halves.
		fmt.Fprintf(&b, "z0 |= (%s", tlo[0])
		for _, t := range tlo[1:] {
			fmt.Fprintf(&b, " ^ %s", t)
		}
		fmt.Fprintf(&b, ") & %#0.16x\n", m[i].lo)

		b.WriteByte('\n')
	}
	b.WriteString("return\n}\n")

	buf, err := format.Source(b.Bytes())
	if err != nil {
		return err
	}
	return os.WriteFile("zctmul.go", buf, 0644)
}
