---
id: schema-def
title: Introduction
---

## Quick Summary

Schema describes the definition of one entity type in the graph, like `User` or `Group`,
and can contain the following configurations:
- Entity fields (or properties), like: name or age of a `User`.
- Entity edges (or relations), like: `User`'s groups, or `User`'s friends.
- Database specific options, like: indexes or unique indexes.

Here's an example of a schema:

```go
package schema

import (
	"entgo.io/ent"
	"entgo.io/ent/schema/field"
	"entgo.io/ent/schema/edge"
	"entgo.io/ent/schema/index"
)

type User struct {
	ent.Schema
}

func (User) Fields() []ent.Field {
	return []ent.Field{
		field.Int("age"),
		field.String("name"),
		field.String("nickname").
			Unique(),
	}
}

func (User) Edges() []ent.Edge {
	return []ent.Edge{
		edge.To("groups", Group.Type),
		edge.To("friends", User.Type),
	}
}

func (User) Indexes() []ent.Index {
	return []ent.Index{
		index.Fields("age", "name").
			Unique(),
	}
}
```

Entity schemas are usually stored inside `ent/schema` directory under
the root directory of your project, and can be generated by `entc` as follows:

```console
go run entgo.io/ent/cmd/ent init User Group
```

:::note 
Please note, that some schema names (like `Client`) are not available due to 
[internal use](https://pkg.go.dev/entgo.io/ent/entc/gen#ValidSchemaName). You can circumvent reserved names by using an 
annotation as mentioned [here](schema-annotations.md#custom-table-name).
:::

## It's Just Another ORM

If you are used to the definition of relations over edges, that's fine.
The modeling is the same. You can model with `ent` whatever you can model
with other traditional ORMs.
There are many examples in this website that can help you get started
in the [Edges](schema-edges.md) section.
