// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

// Code generated by mk_audit_msg_types - DO NOT EDIT.

package auparse

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
)

var errInvalidAuditMessageTypName = errors.New("invalid message type")

// AuditMessageType represents an audit message type used by the kernel.
type AuditMessageType uint16

// List of AuditMessageTypes.
const (
	AUDIT_GET                      AuditMessageType = 1000
	AUDIT_SET                      AuditMessageType = 1001
	AUDIT_LIST                     AuditMessageType = 1002
	AUDIT_ADD                      AuditMessageType = 1003
	AUDIT_DEL                      AuditMessageType = 1004
	AUDIT_USER                     AuditMessageType = 1005
	AUDIT_LOGIN                    AuditMessageType = 1006
	AUDIT_WATCH_INS                AuditMessageType = 1007
	AUDIT_WATCH_REM                AuditMessageType = 1008
	AUDIT_WATCH_LIST               AuditMessageType = 1009
	AUDIT_SIGNAL_INFO              AuditMessageType = 1010
	AUDIT_ADD_RULE                 AuditMessageType = 1011
	AUDIT_DEL_RULE                 AuditMessageType = 1012
	AUDIT_LIST_RULES               AuditMessageType = 1013
	AUDIT_TRIM                     AuditMessageType = 1014
	AUDIT_MAKE_EQUIV               AuditMessageType = 1015
	AUDIT_TTY_GET                  AuditMessageType = 1016
	AUDIT_TTY_SET                  AuditMessageType = 1017
	AUDIT_SET_FEATURE              AuditMessageType = 1018
	AUDIT_GET_FEATURE              AuditMessageType = 1019
	AUDIT_USER_AUTH                AuditMessageType = 1100
	AUDIT_USER_ACCT                AuditMessageType = 1101
	AUDIT_USER_MGMT                AuditMessageType = 1102
	AUDIT_CRED_ACQ                 AuditMessageType = 1103
	AUDIT_CRED_DISP                AuditMessageType = 1104
	AUDIT_USER_START               AuditMessageType = 1105
	AUDIT_USER_END                 AuditMessageType = 1106
	AUDIT_USER_AVC                 AuditMessageType = 1107
	AUDIT_USER_CHAUTHTOK           AuditMessageType = 1108
	AUDIT_USER_ERR                 AuditMessageType = 1109
	AUDIT_CRED_REFR                AuditMessageType = 1110
	AUDIT_USYS_CONFIG              AuditMessageType = 1111
	AUDIT_USER_LOGIN               AuditMessageType = 1112
	AUDIT_USER_LOGOUT              AuditMessageType = 1113
	AUDIT_ADD_USER                 AuditMessageType = 1114
	AUDIT_DEL_USER                 AuditMessageType = 1115
	AUDIT_ADD_GROUP                AuditMessageType = 1116
	AUDIT_DEL_GROUP                AuditMessageType = 1117
	AUDIT_DAC_CHECK                AuditMessageType = 1118
	AUDIT_CHGRP_ID                 AuditMessageType = 1119
	AUDIT_TEST                     AuditMessageType = 1120
	AUDIT_TRUSTED_APP              AuditMessageType = 1121
	AUDIT_USER_SELINUX_ERR         AuditMessageType = 1122
	AUDIT_USER_CMD                 AuditMessageType = 1123
	AUDIT_USER_TTY                 AuditMessageType = 1124
	AUDIT_CHUSER_ID                AuditMessageType = 1125
	AUDIT_GRP_AUTH                 AuditMessageType = 1126
	AUDIT_SYSTEM_BOOT              AuditMessageType = 1127
	AUDIT_SYSTEM_SHUTDOWN          AuditMessageType = 1128
	AUDIT_SYSTEM_RUNLEVEL          AuditMessageType = 1129
	AUDIT_SERVICE_START            AuditMessageType = 1130
	AUDIT_SERVICE_STOP             AuditMessageType = 1131
	AUDIT_GRP_MGMT                 AuditMessageType = 1132
	AUDIT_GRP_CHAUTHTOK            AuditMessageType = 1133
	AUDIT_MAC_CHECK                AuditMessageType = 1134
	AUDIT_ACCT_LOCK                AuditMessageType = 1135
	AUDIT_ACCT_UNLOCK              AuditMessageType = 1136
	AUDIT_USER_DEVICE              AuditMessageType = 1137
	AUDIT_SOFTWARE_UPDATE          AuditMessageType = 1138
	AUDIT_LAST_USER_MSG            AuditMessageType = 1199
	AUDIT_DAEMON_START             AuditMessageType = 1200
	AUDIT_DAEMON_END               AuditMessageType = 1201
	AUDIT_DAEMON_ABORT             AuditMessageType = 1202
	AUDIT_DAEMON_CONFIG            AuditMessageType = 1203
	AUDIT_DAEMON_RECONFIG          AuditMessageType = 1204
	AUDIT_DAEMON_ROTATE            AuditMessageType = 1205
	AUDIT_DAEMON_RESUME            AuditMessageType = 1206
	AUDIT_DAEMON_ACCEPT            AuditMessageType = 1207
	AUDIT_DAEMON_CLOSE             AuditMessageType = 1208
	AUDIT_DAEMON_ERR               AuditMessageType = 1209
	AUDIT_LAST_DAEMON              AuditMessageType = 1299
	AUDIT_SYSCALL                  AuditMessageType = 1300
	AUDIT_PATH                     AuditMessageType = 1302
	AUDIT_IPC                      AuditMessageType = 1303
	AUDIT_SOCKETCALL               AuditMessageType = 1304
	AUDIT_CONFIG_CHANGE            AuditMessageType = 1305
	AUDIT_SOCKADDR                 AuditMessageType = 1306
	AUDIT_CWD                      AuditMessageType = 1307
	AUDIT_EXECVE                   AuditMessageType = 1309
	AUDIT_IPC_SET_PERM             AuditMessageType = 1311
	AUDIT_MQ_OPEN                  AuditMessageType = 1312
	AUDIT_MQ_SENDRECV              AuditMessageType = 1313
	AUDIT_MQ_NOTIFY                AuditMessageType = 1314
	AUDIT_MQ_GETSETATTR            AuditMessageType = 1315
	AUDIT_KERNEL_OTHER             AuditMessageType = 1316
	AUDIT_FD_PAIR                  AuditMessageType = 1317
	AUDIT_OBJ_PID                  AuditMessageType = 1318
	AUDIT_TTY                      AuditMessageType = 1319
	AUDIT_EOE                      AuditMessageType = 1320
	AUDIT_BPRM_FCAPS               AuditMessageType = 1321
	AUDIT_CAPSET                   AuditMessageType = 1322
	AUDIT_MMAP                     AuditMessageType = 1323
	AUDIT_NETFILTER_PKT            AuditMessageType = 1324
	AUDIT_NETFILTER_CFG            AuditMessageType = 1325
	AUDIT_SECCOMP                  AuditMessageType = 1326
	AUDIT_PROCTITLE                AuditMessageType = 1327
	AUDIT_FEATURE_CHANGE           AuditMessageType = 1328
	AUDIT_REPLACE                  AuditMessageType = 1329
	AUDIT_KERN_MODULE              AuditMessageType = 1330
	AUDIT_FANOTIFY                 AuditMessageType = 1331
	AUDIT_LAST_EVENT               AuditMessageType = 1399
	AUDIT_AVC                      AuditMessageType = 1400
	AUDIT_SELINUX_ERR              AuditMessageType = 1401
	AUDIT_AVC_PATH                 AuditMessageType = 1402
	AUDIT_MAC_POLICY_LOAD          AuditMessageType = 1403
	AUDIT_MAC_STATUS               AuditMessageType = 1404
	AUDIT_MAC_CONFIG_CHANGE        AuditMessageType = 1405
	AUDIT_MAC_UNLBL_ALLOW          AuditMessageType = 1406
	AUDIT_MAC_CIPSOV4_ADD          AuditMessageType = 1407
	AUDIT_MAC_CIPSOV4_DEL          AuditMessageType = 1408
	AUDIT_MAC_MAP_ADD              AuditMessageType = 1409
	AUDIT_MAC_MAP_DEL              AuditMessageType = 1410
	AUDIT_MAC_IPSEC_ADDSA          AuditMessageType = 1411
	AUDIT_MAC_IPSEC_DELSA          AuditMessageType = 1412
	AUDIT_MAC_IPSEC_ADDSPD         AuditMessageType = 1413
	AUDIT_MAC_IPSEC_DELSPD         AuditMessageType = 1414
	AUDIT_MAC_IPSEC_EVENT          AuditMessageType = 1415
	AUDIT_MAC_UNLBL_STCADD         AuditMessageType = 1416
	AUDIT_MAC_UNLBL_STCDEL         AuditMessageType = 1417
	AUDIT_MAC_CALIPSO_ADD          AuditMessageType = 1418
	AUDIT_MAC_CALIPSO_DEL          AuditMessageType = 1419
	AUDIT_LAST_SELINUX             AuditMessageType = 1499
	AUDIT_AA                       AuditMessageType = 1500
	AUDIT_APPARMOR_AUDIT           AuditMessageType = 1501
	AUDIT_APPARMOR_ALLOWED         AuditMessageType = 1502
	AUDIT_APPARMOR_DENIED          AuditMessageType = 1503
	AUDIT_APPARMOR_HINT            AuditMessageType = 1504
	AUDIT_APPARMOR_STATUS          AuditMessageType = 1505
	AUDIT_APPARMOR_ERROR           AuditMessageType = 1506
	AUDIT_LAST_APPARMOR            AuditMessageType = 1599
	AUDIT_FIRST_KERN_CRYPTO_MSG    AuditMessageType = 1600
	AUDIT_LAST_KERN_CRYPTO_MSG     AuditMessageType = 1699
	AUDIT_ANOM_PROMISCUOUS         AuditMessageType = 1700
	AUDIT_ANOM_ABEND               AuditMessageType = 1701
	AUDIT_ANOM_LINK                AuditMessageType = 1702
	AUDIT_LAST_KERN_ANOM_MSG       AuditMessageType = 1799
	AUDIT_INTEGRITY_DATA           AuditMessageType = 1800
	AUDIT_INTEGRITY_METADATA       AuditMessageType = 1801
	AUDIT_INTEGRITY_STATUS         AuditMessageType = 1802
	AUDIT_INTEGRITY_HASH           AuditMessageType = 1803
	AUDIT_INTEGRITY_PCR            AuditMessageType = 1804
	AUDIT_INTEGRITY_RULE           AuditMessageType = 1805
	AUDIT_INTEGRITY_LAST_MSG       AuditMessageType = 1899
	AUDIT_KERNEL                   AuditMessageType = 2000
	AUDIT_ANOM_LOGIN_FAILURES      AuditMessageType = 2100
	AUDIT_ANOM_LOGIN_TIME          AuditMessageType = 2101
	AUDIT_ANOM_LOGIN_SESSIONS      AuditMessageType = 2102
	AUDIT_ANOM_LOGIN_ACCT          AuditMessageType = 2103
	AUDIT_ANOM_LOGIN_LOCATION      AuditMessageType = 2104
	AUDIT_ANOM_MAX_DAC             AuditMessageType = 2105
	AUDIT_ANOM_MAX_MAC             AuditMessageType = 2106
	AUDIT_ANOM_AMTU_FAIL           AuditMessageType = 2107
	AUDIT_ANOM_RBAC_FAIL           AuditMessageType = 2108
	AUDIT_ANOM_RBAC_INTEGRITY_FAIL AuditMessageType = 2109
	AUDIT_ANOM_CRYPTO_FAIL         AuditMessageType = 2110
	AUDIT_ANOM_ACCESS_FS           AuditMessageType = 2111
	AUDIT_ANOM_EXEC                AuditMessageType = 2112
	AUDIT_ANOM_MK_EXEC             AuditMessageType = 2113
	AUDIT_ANOM_ADD_ACCT            AuditMessageType = 2114
	AUDIT_ANOM_DEL_ACCT            AuditMessageType = 2115
	AUDIT_ANOM_MOD_ACCT            AuditMessageType = 2116
	AUDIT_ANOM_ROOT_TRANS          AuditMessageType = 2117
	AUDIT_ANOM_LOGIN_SERVICE       AuditMessageType = 2118
	AUDIT_LAST_ANOM_MSG            AuditMessageType = 2199
	AUDIT_RESP_ANOMALY             AuditMessageType = 2200
	AUDIT_RESP_ALERT               AuditMessageType = 2201
	AUDIT_RESP_KILL_PROC           AuditMessageType = 2202
	AUDIT_RESP_TERM_ACCESS         AuditMessageType = 2203
	AUDIT_RESP_ACCT_REMOTE         AuditMessageType = 2204
	AUDIT_RESP_ACCT_LOCK_TIMED     AuditMessageType = 2205
	AUDIT_RESP_ACCT_UNLOCK_TIMED   AuditMessageType = 2206
	AUDIT_RESP_ACCT_LOCK           AuditMessageType = 2207
	AUDIT_RESP_TERM_LOCK           AuditMessageType = 2208
	AUDIT_RESP_SEBOOL              AuditMessageType = 2209
	AUDIT_RESP_EXEC                AuditMessageType = 2210
	AUDIT_RESP_SINGLE              AuditMessageType = 2211
	AUDIT_RESP_HALT                AuditMessageType = 2212
	AUDIT_RESP_ORIGIN_BLOCK        AuditMessageType = 2213
	AUDIT_RESP_ORIGIN_BLOCK_TIMED  AuditMessageType = 2214
	AUDIT_LAST_ANOM_RESP           AuditMessageType = 2299
	AUDIT_USER_ROLE_CHANGE         AuditMessageType = 2300
	AUDIT_ROLE_ASSIGN              AuditMessageType = 2301
	AUDIT_ROLE_REMOVE              AuditMessageType = 2302
	AUDIT_LABEL_OVERRIDE           AuditMessageType = 2303
	AUDIT_LABEL_LEVEL_CHANGE       AuditMessageType = 2304
	AUDIT_USER_LABELED_EXPORT      AuditMessageType = 2305
	AUDIT_USER_UNLABELED_EXPORT    AuditMessageType = 2306
	AUDIT_DEV_ALLOC                AuditMessageType = 2307
	AUDIT_DEV_DEALLOC              AuditMessageType = 2308
	AUDIT_FS_RELABEL               AuditMessageType = 2309
	AUDIT_USER_MAC_POLICY_LOAD     AuditMessageType = 2310
	AUDIT_ROLE_MODIFY              AuditMessageType = 2311
	AUDIT_USER_MAC_CONFIG_CHANGE   AuditMessageType = 2312
	AUDIT_LAST_USER_LSPP_MSG       AuditMessageType = 2399
	AUDIT_CRYPTO_TEST_USER         AuditMessageType = 2400
	AUDIT_CRYPTO_PARAM_CHANGE_USER AuditMessageType = 2401
	AUDIT_CRYPTO_LOGIN             AuditMessageType = 2402
	AUDIT_CRYPTO_LOGOUT            AuditMessageType = 2403
	AUDIT_CRYPTO_KEY_USER          AuditMessageType = 2404
	AUDIT_CRYPTO_FAILURE_USER      AuditMessageType = 2405
	AUDIT_CRYPTO_REPLAY_USER       AuditMessageType = 2406
	AUDIT_CRYPTO_SESSION           AuditMessageType = 2407
	AUDIT_CRYPTO_IKE_SA            AuditMessageType = 2408
	AUDIT_CRYPTO_IPSEC_SA          AuditMessageType = 2409
	AUDIT_LAST_CRYPTO_MSG          AuditMessageType = 2499
	AUDIT_VIRT_CONTROL             AuditMessageType = 2500
	AUDIT_VIRT_RESOURCE            AuditMessageType = 2501
	AUDIT_VIRT_MACHINE_ID          AuditMessageType = 2502
	AUDIT_VIRT_INTEGRITY_CHECK     AuditMessageType = 2503
	AUDIT_VIRT_CREATE              AuditMessageType = 2504
	AUDIT_VIRT_DESTROY             AuditMessageType = 2505
	AUDIT_VIRT_MIGRATE_IN          AuditMessageType = 2506
	AUDIT_VIRT_MIGRATE_OUT         AuditMessageType = 2507
	AUDIT_LAST_VIRT_MSG            AuditMessageType = 2599
	AUDIT_LAST_USER_MSG2           AuditMessageType = 2999
)

var auditMessageTypeToName = map[AuditMessageType]string{
	AUDIT_AA:                       "APPARMOR",
	AUDIT_ACCT_LOCK:                "ACCT_LOCK",
	AUDIT_ACCT_UNLOCK:              "ACCT_UNLOCK",
	AUDIT_ADD:                      "ADD",
	AUDIT_ADD_GROUP:                "ADD_GROUP",
	AUDIT_ADD_RULE:                 "ADD_RULE",
	AUDIT_ADD_USER:                 "ADD_USER",
	AUDIT_ANOM_ABEND:               "ANOM_ABEND",
	AUDIT_ANOM_ACCESS_FS:           "ANOM_ACCESS_FS",
	AUDIT_ANOM_ADD_ACCT:            "ANOM_ADD_ACCT",
	AUDIT_ANOM_AMTU_FAIL:           "ANOM_AMTU_FAIL",
	AUDIT_ANOM_CRYPTO_FAIL:         "ANOM_CRYPTO_FAIL",
	AUDIT_ANOM_DEL_ACCT:            "ANOM_DEL_ACCT",
	AUDIT_ANOM_EXEC:                "ANOM_EXEC",
	AUDIT_ANOM_LINK:                "ANOM_LINK",
	AUDIT_ANOM_LOGIN_ACCT:          "ANOM_LOGIN_ACCT",
	AUDIT_ANOM_LOGIN_FAILURES:      "ANOM_LOGIN_FAILURES",
	AUDIT_ANOM_LOGIN_LOCATION:      "ANOM_LOGIN_LOCATION",
	AUDIT_ANOM_LOGIN_SERVICE:       "ANOM_LOGIN_SERVICE",
	AUDIT_ANOM_LOGIN_SESSIONS:      "ANOM_LOGIN_SESSIONS",
	AUDIT_ANOM_LOGIN_TIME:          "ANOM_LOGIN_TIME",
	AUDIT_ANOM_MAX_DAC:             "ANOM_MAX_DAC",
	AUDIT_ANOM_MAX_MAC:             "ANOM_MAX_MAC",
	AUDIT_ANOM_MK_EXEC:             "ANOM_MK_EXEC",
	AUDIT_ANOM_MOD_ACCT:            "ANOM_MOD_ACCT",
	AUDIT_ANOM_PROMISCUOUS:         "ANOM_PROMISCUOUS",
	AUDIT_ANOM_RBAC_FAIL:           "ANOM_RBAC_FAIL",
	AUDIT_ANOM_RBAC_INTEGRITY_FAIL: "ANOM_RBAC_INTEGRITY_FAIL",
	AUDIT_ANOM_ROOT_TRANS:          "ANOM_ROOT_TRANS",
	AUDIT_APPARMOR_ALLOWED:         "APPARMOR_ALLOWED",
	AUDIT_APPARMOR_AUDIT:           "APPARMOR_AUDIT",
	AUDIT_APPARMOR_DENIED:          "APPARMOR_DENIED",
	AUDIT_APPARMOR_ERROR:           "APPARMOR_ERROR",
	AUDIT_APPARMOR_HINT:            "APPARMOR_HINT",
	AUDIT_APPARMOR_STATUS:          "APPARMOR_STATUS",
	AUDIT_AVC:                      "AVC",
	AUDIT_AVC_PATH:                 "AVC_PATH",
	AUDIT_BPRM_FCAPS:               "BPRM_FCAPS",
	AUDIT_CAPSET:                   "CAPSET",
	AUDIT_CHGRP_ID:                 "CHGRP_ID",
	AUDIT_CHUSER_ID:                "CHUSER_ID",
	AUDIT_CONFIG_CHANGE:            "CONFIG_CHANGE",
	AUDIT_CRED_ACQ:                 "CRED_ACQ",
	AUDIT_CRED_DISP:                "CRED_DISP",
	AUDIT_CRED_REFR:                "CRED_REFR",
	AUDIT_CRYPTO_FAILURE_USER:      "CRYPTO_FAILURE_USER",
	AUDIT_CRYPTO_IKE_SA:            "CRYPTO_IKE_SA",
	AUDIT_CRYPTO_IPSEC_SA:          "CRYPTO_IPSEC_SA",
	AUDIT_CRYPTO_KEY_USER:          "CRYPTO_KEY_USER",
	AUDIT_CRYPTO_LOGIN:             "CRYPTO_LOGIN",
	AUDIT_CRYPTO_LOGOUT:            "CRYPTO_LOGOUT",
	AUDIT_CRYPTO_PARAM_CHANGE_USER: "CRYPTO_PARAM_CHANGE_USER",
	AUDIT_CRYPTO_REPLAY_USER:       "CRYPTO_REPLAY_USER",
	AUDIT_CRYPTO_SESSION:           "CRYPTO_SESSION",
	AUDIT_CRYPTO_TEST_USER:         "CRYPTO_TEST_USER",
	AUDIT_CWD:                      "CWD",
	AUDIT_DAC_CHECK:                "DAC_CHECK",
	AUDIT_DAEMON_ABORT:             "DAEMON_ABORT",
	AUDIT_DAEMON_ACCEPT:            "DAEMON_ACCEPT",
	AUDIT_DAEMON_CLOSE:             "DAEMON_CLOSE",
	AUDIT_DAEMON_CONFIG:            "DAEMON_CONFIG",
	AUDIT_DAEMON_END:               "DAEMON_END",
	AUDIT_DAEMON_ERR:               "DAEMON_ERR",
	AUDIT_DAEMON_RECONFIG:          "DAEMON_RECONFIG",
	AUDIT_DAEMON_RESUME:            "DAEMON_RESUME",
	AUDIT_DAEMON_ROTATE:            "DAEMON_ROTATE",
	AUDIT_DAEMON_START:             "DAEMON_START",
	AUDIT_DEL:                      "DEL",
	AUDIT_DEL_GROUP:                "DEL_GROUP",
	AUDIT_DEL_RULE:                 "DEL_RULE",
	AUDIT_DEL_USER:                 "DEL_USER",
	AUDIT_DEV_ALLOC:                "DEV_ALLOC",
	AUDIT_DEV_DEALLOC:              "DEV_DEALLOC",
	AUDIT_EOE:                      "EOE",
	AUDIT_EXECVE:                   "EXECVE",
	AUDIT_FANOTIFY:                 "FANOTIFY",
	AUDIT_FD_PAIR:                  "FD_PAIR",
	AUDIT_FEATURE_CHANGE:           "FEATURE_CHANGE",
	AUDIT_FIRST_KERN_CRYPTO_MSG:    "FIRST_KERN_CRYPTO_MSG",
	AUDIT_FS_RELABEL:               "FS_RELABEL",
	AUDIT_GET:                      "GET",
	AUDIT_GET_FEATURE:              "GET_FEATURE",
	AUDIT_GRP_AUTH:                 "GRP_AUTH",
	AUDIT_GRP_CHAUTHTOK:            "GRP_CHAUTHTOK",
	AUDIT_GRP_MGMT:                 "GRP_MGMT",
	AUDIT_INTEGRITY_DATA:           "INTEGRITY_DATA",
	AUDIT_INTEGRITY_HASH:           "INTEGRITY_HASH",
	AUDIT_INTEGRITY_LAST_MSG:       "INTEGRITY_LAST_MSG",
	AUDIT_INTEGRITY_METADATA:       "INTEGRITY_METADATA",
	AUDIT_INTEGRITY_PCR:            "INTEGRITY_PCR",
	AUDIT_INTEGRITY_RULE:           "INTEGRITY_RULE",
	AUDIT_INTEGRITY_STATUS:         "INTEGRITY_STATUS",
	AUDIT_IPC:                      "IPC",
	AUDIT_IPC_SET_PERM:             "IPC_SET_PERM",
	AUDIT_KERNEL:                   "KERNEL",
	AUDIT_KERNEL_OTHER:             "KERNEL_OTHER",
	AUDIT_KERN_MODULE:              "KERN_MODULE",
	AUDIT_LABEL_LEVEL_CHANGE:       "LABEL_LEVEL_CHANGE",
	AUDIT_LABEL_OVERRIDE:           "LABEL_OVERRIDE",
	AUDIT_LAST_ANOM_MSG:            "LAST_ANOM_MSG",
	AUDIT_LAST_ANOM_RESP:           "LAST_ANOM_RESP",
	AUDIT_LAST_APPARMOR:            "LAST_APPARMOR",
	AUDIT_LAST_CRYPTO_MSG:          "LAST_CRYPTO_MSG",
	AUDIT_LAST_DAEMON:              "LAST_DAEMON",
	AUDIT_LAST_EVENT:               "LAST_EVENT",
	AUDIT_LAST_KERN_ANOM_MSG:       "LAST_KERN_ANOM_MSG",
	AUDIT_LAST_KERN_CRYPTO_MSG:     "LAST_KERN_CRYPTO_MSG",
	AUDIT_LAST_SELINUX:             "LAST_SELINUX",
	AUDIT_LAST_USER_LSPP_MSG:       "LAST_USER_LSPP_MSG",
	AUDIT_LAST_USER_MSG:            "LAST_USER_MSG",
	AUDIT_LAST_USER_MSG2:           "LAST_USER_MSG2",
	AUDIT_LAST_VIRT_MSG:            "LAST_VIRT_MSG",
	AUDIT_LIST:                     "LIST",
	AUDIT_LIST_RULES:               "LIST_RULES",
	AUDIT_LOGIN:                    "LOGIN",
	AUDIT_MAC_CALIPSO_ADD:          "MAC_CALIPSO_ADD",
	AUDIT_MAC_CALIPSO_DEL:          "MAC_CALIPSO_DEL",
	AUDIT_MAC_CHECK:                "MAC_CHECK",
	AUDIT_MAC_CIPSOV4_ADD:          "MAC_CIPSOV4_ADD",
	AUDIT_MAC_CIPSOV4_DEL:          "MAC_CIPSOV4_DEL",
	AUDIT_MAC_CONFIG_CHANGE:        "MAC_CONFIG_CHANGE",
	AUDIT_MAC_IPSEC_ADDSA:          "MAC_IPSEC_ADDSA",
	AUDIT_MAC_IPSEC_ADDSPD:         "MAC_IPSEC_ADDSPD",
	AUDIT_MAC_IPSEC_DELSA:          "MAC_IPSEC_DELSA",
	AUDIT_MAC_IPSEC_DELSPD:         "MAC_IPSEC_DELSPD",
	AUDIT_MAC_IPSEC_EVENT:          "MAC_IPSEC_EVENT",
	AUDIT_MAC_MAP_ADD:              "MAC_MAP_ADD",
	AUDIT_MAC_MAP_DEL:              "MAC_MAP_DEL",
	AUDIT_MAC_POLICY_LOAD:          "MAC_POLICY_LOAD",
	AUDIT_MAC_STATUS:               "MAC_STATUS",
	AUDIT_MAC_UNLBL_ALLOW:          "MAC_UNLBL_ALLOW",
	AUDIT_MAC_UNLBL_STCADD:         "MAC_UNLBL_STCADD",
	AUDIT_MAC_UNLBL_STCDEL:         "MAC_UNLBL_STCDEL",
	AUDIT_MAKE_EQUIV:               "MAKE_EQUIV",
	AUDIT_MMAP:                     "MMAP",
	AUDIT_MQ_GETSETATTR:            "MQ_GETSETATTR",
	AUDIT_MQ_NOTIFY:                "MQ_NOTIFY",
	AUDIT_MQ_OPEN:                  "MQ_OPEN",
	AUDIT_MQ_SENDRECV:              "MQ_SENDRECV",
	AUDIT_NETFILTER_CFG:            "NETFILTER_CFG",
	AUDIT_NETFILTER_PKT:            "NETFILTER_PKT",
	AUDIT_OBJ_PID:                  "OBJ_PID",
	AUDIT_PATH:                     "PATH",
	AUDIT_PROCTITLE:                "PROCTITLE",
	AUDIT_REPLACE:                  "REPLACE",
	AUDIT_RESP_ACCT_LOCK:           "RESP_ACCT_LOCK",
	AUDIT_RESP_ACCT_LOCK_TIMED:     "RESP_ACCT_LOCK_TIMED",
	AUDIT_RESP_ACCT_REMOTE:         "RESP_ACCT_REMOTE",
	AUDIT_RESP_ACCT_UNLOCK_TIMED:   "RESP_ACCT_UNLOCK_TIMED",
	AUDIT_RESP_ALERT:               "RESP_ALERT",
	AUDIT_RESP_ANOMALY:             "RESP_ANOMALY",
	AUDIT_RESP_EXEC:                "RESP_EXEC",
	AUDIT_RESP_HALT:                "RESP_HALT",
	AUDIT_RESP_KILL_PROC:           "RESP_KILL_PROC",
	AUDIT_RESP_ORIGIN_BLOCK:        "RESP_ORIGIN_BLOCK",
	AUDIT_RESP_ORIGIN_BLOCK_TIMED:  "RESP_ORIGIN_BLOCK_TIMED",
	AUDIT_RESP_SEBOOL:              "RESP_SEBOOL",
	AUDIT_RESP_SINGLE:              "RESP_SINGLE",
	AUDIT_RESP_TERM_ACCESS:         "RESP_TERM_ACCESS",
	AUDIT_RESP_TERM_LOCK:           "RESP_TERM_LOCK",
	AUDIT_ROLE_ASSIGN:              "ROLE_ASSIGN",
	AUDIT_ROLE_MODIFY:              "ROLE_MODIFY",
	AUDIT_ROLE_REMOVE:              "ROLE_REMOVE",
	AUDIT_SECCOMP:                  "SECCOMP",
	AUDIT_SELINUX_ERR:              "SELINUX_ERR",
	AUDIT_SERVICE_START:            "SERVICE_START",
	AUDIT_SERVICE_STOP:             "SERVICE_STOP",
	AUDIT_SET:                      "SET",
	AUDIT_SET_FEATURE:              "SET_FEATURE",
	AUDIT_SIGNAL_INFO:              "SIGNAL_INFO",
	AUDIT_SOCKADDR:                 "SOCKADDR",
	AUDIT_SOCKETCALL:               "SOCKETCALL",
	AUDIT_SOFTWARE_UPDATE:          "SOFTWARE_UPDATE",
	AUDIT_SYSCALL:                  "SYSCALL",
	AUDIT_SYSTEM_BOOT:              "SYSTEM_BOOT",
	AUDIT_SYSTEM_RUNLEVEL:          "SYSTEM_RUNLEVEL",
	AUDIT_SYSTEM_SHUTDOWN:          "SYSTEM_SHUTDOWN",
	AUDIT_TEST:                     "TEST",
	AUDIT_TRIM:                     "TRIM",
	AUDIT_TRUSTED_APP:              "TRUSTED_APP",
	AUDIT_TTY:                      "TTY",
	AUDIT_TTY_GET:                  "TTY_GET",
	AUDIT_TTY_SET:                  "TTY_SET",
	AUDIT_USER:                     "USER",
	AUDIT_USER_ACCT:                "USER_ACCT",
	AUDIT_USER_AUTH:                "USER_AUTH",
	AUDIT_USER_AVC:                 "USER_AVC",
	AUDIT_USER_CHAUTHTOK:           "USER_CHAUTHTOK",
	AUDIT_USER_CMD:                 "USER_CMD",
	AUDIT_USER_DEVICE:              "USER_DEVICE",
	AUDIT_USER_END:                 "USER_END",
	AUDIT_USER_ERR:                 "USER_ERR",
	AUDIT_USER_LABELED_EXPORT:      "USER_LABELED_EXPORT",
	AUDIT_USER_LOGIN:               "USER_LOGIN",
	AUDIT_USER_LOGOUT:              "USER_LOGOUT",
	AUDIT_USER_MAC_CONFIG_CHANGE:   "USER_MAC_CONFIG_CHANGE",
	AUDIT_USER_MAC_POLICY_LOAD:     "USER_MAC_POLICY_LOAD",
	AUDIT_USER_MGMT:                "USER_MGMT",
	AUDIT_USER_ROLE_CHANGE:         "USER_ROLE_CHANGE",
	AUDIT_USER_SELINUX_ERR:         "USER_SELINUX_ERR",
	AUDIT_USER_START:               "USER_START",
	AUDIT_USER_TTY:                 "USER_TTY",
	AUDIT_USER_UNLABELED_EXPORT:    "USER_UNLABELED_EXPORT",
	AUDIT_USYS_CONFIG:              "USYS_CONFIG",
	AUDIT_VIRT_CONTROL:             "VIRT_CONTROL",
	AUDIT_VIRT_CREATE:              "VIRT_CREATE",
	AUDIT_VIRT_DESTROY:             "VIRT_DESTROY",
	AUDIT_VIRT_INTEGRITY_CHECK:     "VIRT_INTEGRITY_CHECK",
	AUDIT_VIRT_MACHINE_ID:          "VIRT_MACHINE_ID",
	AUDIT_VIRT_MIGRATE_IN:          "VIRT_MIGRATE_IN",
	AUDIT_VIRT_MIGRATE_OUT:         "VIRT_MIGRATE_OUT",
	AUDIT_VIRT_RESOURCE:            "VIRT_RESOURCE",
	AUDIT_WATCH_INS:                "WATCH_INS",
	AUDIT_WATCH_LIST:               "WATCH_LIST",
	AUDIT_WATCH_REM:                "WATCH_REM",
}

func (t AuditMessageType) String() string {
	name, found := auditMessageTypeToName[t]
	if found {
		return name
	}

	return fmt.Sprintf("UNKNOWN[%d]", uint16(t))
}

func (t AuditMessageType) MarshalText() (text []byte, err error) {
	return []byte(strings.ToLower(t.String())), nil
}

func (t *AuditMessageType) UnmarshalText(text []byte) error {
	messageType, err := GetAuditMessageType(string(text))
	if err != nil {
		return err
	}
	*t = messageType
	return nil
}

var auditMessageNameToType = map[string]AuditMessageType{
	"APPARMOR":                 AUDIT_AA,
	"ACCT_LOCK":                AUDIT_ACCT_LOCK,
	"ACCT_UNLOCK":              AUDIT_ACCT_UNLOCK,
	"ADD":                      AUDIT_ADD,
	"ADD_GROUP":                AUDIT_ADD_GROUP,
	"ADD_RULE":                 AUDIT_ADD_RULE,
	"ADD_USER":                 AUDIT_ADD_USER,
	"ANOM_ABEND":               AUDIT_ANOM_ABEND,
	"ANOM_ACCESS_FS":           AUDIT_ANOM_ACCESS_FS,
	"ANOM_ADD_ACCT":            AUDIT_ANOM_ADD_ACCT,
	"ANOM_AMTU_FAIL":           AUDIT_ANOM_AMTU_FAIL,
	"ANOM_CRYPTO_FAIL":         AUDIT_ANOM_CRYPTO_FAIL,
	"ANOM_DEL_ACCT":            AUDIT_ANOM_DEL_ACCT,
	"ANOM_EXEC":                AUDIT_ANOM_EXEC,
	"ANOM_LINK":                AUDIT_ANOM_LINK,
	"ANOM_LOGIN_ACCT":          AUDIT_ANOM_LOGIN_ACCT,
	"ANOM_LOGIN_FAILURES":      AUDIT_ANOM_LOGIN_FAILURES,
	"ANOM_LOGIN_LOCATION":      AUDIT_ANOM_LOGIN_LOCATION,
	"ANOM_LOGIN_SERVICE":       AUDIT_ANOM_LOGIN_SERVICE,
	"ANOM_LOGIN_SESSIONS":      AUDIT_ANOM_LOGIN_SESSIONS,
	"ANOM_LOGIN_TIME":          AUDIT_ANOM_LOGIN_TIME,
	"ANOM_MAX_DAC":             AUDIT_ANOM_MAX_DAC,
	"ANOM_MAX_MAC":             AUDIT_ANOM_MAX_MAC,
	"ANOM_MK_EXEC":             AUDIT_ANOM_MK_EXEC,
	"ANOM_MOD_ACCT":            AUDIT_ANOM_MOD_ACCT,
	"ANOM_PROMISCUOUS":         AUDIT_ANOM_PROMISCUOUS,
	"ANOM_RBAC_FAIL":           AUDIT_ANOM_RBAC_FAIL,
	"ANOM_RBAC_INTEGRITY_FAIL": AUDIT_ANOM_RBAC_INTEGRITY_FAIL,
	"ANOM_ROOT_TRANS":          AUDIT_ANOM_ROOT_TRANS,
	"APPARMOR_ALLOWED":         AUDIT_APPARMOR_ALLOWED,
	"APPARMOR_AUDIT":           AUDIT_APPARMOR_AUDIT,
	"APPARMOR_DENIED":          AUDIT_APPARMOR_DENIED,
	"APPARMOR_ERROR":           AUDIT_APPARMOR_ERROR,
	"APPARMOR_HINT":            AUDIT_APPARMOR_HINT,
	"APPARMOR_STATUS":          AUDIT_APPARMOR_STATUS,
	"AVC":                      AUDIT_AVC,
	"AVC_PATH":                 AUDIT_AVC_PATH,
	"BPRM_FCAPS":               AUDIT_BPRM_FCAPS,
	"CAPSET":                   AUDIT_CAPSET,
	"CHGRP_ID":                 AUDIT_CHGRP_ID,
	"CHUSER_ID":                AUDIT_CHUSER_ID,
	"CONFIG_CHANGE":            AUDIT_CONFIG_CHANGE,
	"CRED_ACQ":                 AUDIT_CRED_ACQ,
	"CRED_DISP":                AUDIT_CRED_DISP,
	"CRED_REFR":                AUDIT_CRED_REFR,
	"CRYPTO_FAILURE_USER":      AUDIT_CRYPTO_FAILURE_USER,
	"CRYPTO_IKE_SA":            AUDIT_CRYPTO_IKE_SA,
	"CRYPTO_IPSEC_SA":          AUDIT_CRYPTO_IPSEC_SA,
	"CRYPTO_KEY_USER":          AUDIT_CRYPTO_KEY_USER,
	"CRYPTO_LOGIN":             AUDIT_CRYPTO_LOGIN,
	"CRYPTO_LOGOUT":            AUDIT_CRYPTO_LOGOUT,
	"CRYPTO_PARAM_CHANGE_USER": AUDIT_CRYPTO_PARAM_CHANGE_USER,
	"CRYPTO_REPLAY_USER":       AUDIT_CRYPTO_REPLAY_USER,
	"CRYPTO_SESSION":           AUDIT_CRYPTO_SESSION,
	"CRYPTO_TEST_USER":         AUDIT_CRYPTO_TEST_USER,
	"CWD":                      AUDIT_CWD,
	"DAC_CHECK":                AUDIT_DAC_CHECK,
	"DAEMON_ABORT":             AUDIT_DAEMON_ABORT,
	"DAEMON_ACCEPT":            AUDIT_DAEMON_ACCEPT,
	"DAEMON_CLOSE":             AUDIT_DAEMON_CLOSE,
	"DAEMON_CONFIG":            AUDIT_DAEMON_CONFIG,
	"DAEMON_END":               AUDIT_DAEMON_END,
	"DAEMON_ERR":               AUDIT_DAEMON_ERR,
	"DAEMON_RECONFIG":          AUDIT_DAEMON_RECONFIG,
	"DAEMON_RESUME":            AUDIT_DAEMON_RESUME,
	"DAEMON_ROTATE":            AUDIT_DAEMON_ROTATE,
	"DAEMON_START":             AUDIT_DAEMON_START,
	"DEL":                      AUDIT_DEL,
	"DEL_GROUP":                AUDIT_DEL_GROUP,
	"DEL_RULE":                 AUDIT_DEL_RULE,
	"DEL_USER":                 AUDIT_DEL_USER,
	"DEV_ALLOC":                AUDIT_DEV_ALLOC,
	"DEV_DEALLOC":              AUDIT_DEV_DEALLOC,
	"EOE":                      AUDIT_EOE,
	"EXECVE":                   AUDIT_EXECVE,
	"FANOTIFY":                 AUDIT_FANOTIFY,
	"FD_PAIR":                  AUDIT_FD_PAIR,
	"FEATURE_CHANGE":           AUDIT_FEATURE_CHANGE,
	"FIRST_KERN_CRYPTO_MSG":    AUDIT_FIRST_KERN_CRYPTO_MSG,
	"FS_RELABEL":               AUDIT_FS_RELABEL,
	"GET":                      AUDIT_GET,
	"GET_FEATURE":              AUDIT_GET_FEATURE,
	"GRP_AUTH":                 AUDIT_GRP_AUTH,
	"GRP_CHAUTHTOK":            AUDIT_GRP_CHAUTHTOK,
	"GRP_MGMT":                 AUDIT_GRP_MGMT,
	"INTEGRITY_DATA":           AUDIT_INTEGRITY_DATA,
	"INTEGRITY_HASH":           AUDIT_INTEGRITY_HASH,
	"INTEGRITY_LAST_MSG":       AUDIT_INTEGRITY_LAST_MSG,
	"INTEGRITY_METADATA":       AUDIT_INTEGRITY_METADATA,
	"INTEGRITY_PCR":            AUDIT_INTEGRITY_PCR,
	"INTEGRITY_RULE":           AUDIT_INTEGRITY_RULE,
	"INTEGRITY_STATUS":         AUDIT_INTEGRITY_STATUS,
	"IPC":                      AUDIT_IPC,
	"IPC_SET_PERM":             AUDIT_IPC_SET_PERM,
	"KERNEL":                   AUDIT_KERNEL,
	"KERNEL_OTHER":             AUDIT_KERNEL_OTHER,
	"KERN_MODULE":              AUDIT_KERN_MODULE,
	"LABEL_LEVEL_CHANGE":       AUDIT_LABEL_LEVEL_CHANGE,
	"LABEL_OVERRIDE":           AUDIT_LABEL_OVERRIDE,
	"LAST_ANOM_MSG":            AUDIT_LAST_ANOM_MSG,
	"LAST_ANOM_RESP":           AUDIT_LAST_ANOM_RESP,
	"LAST_APPARMOR":            AUDIT_LAST_APPARMOR,
	"LAST_CRYPTO_MSG":          AUDIT_LAST_CRYPTO_MSG,
	"LAST_DAEMON":              AUDIT_LAST_DAEMON,
	"LAST_EVENT":               AUDIT_LAST_EVENT,
	"LAST_KERN_ANOM_MSG":       AUDIT_LAST_KERN_ANOM_MSG,
	"LAST_KERN_CRYPTO_MSG":     AUDIT_LAST_KERN_CRYPTO_MSG,
	"LAST_SELINUX":             AUDIT_LAST_SELINUX,
	"LAST_USER_LSPP_MSG":       AUDIT_LAST_USER_LSPP_MSG,
	"LAST_USER_MSG":            AUDIT_LAST_USER_MSG,
	"LAST_USER_MSG2":           AUDIT_LAST_USER_MSG2,
	"LAST_VIRT_MSG":            AUDIT_LAST_VIRT_MSG,
	"LIST":                     AUDIT_LIST,
	"LIST_RULES":               AUDIT_LIST_RULES,
	"LOGIN":                    AUDIT_LOGIN,
	"MAC_CALIPSO_ADD":          AUDIT_MAC_CALIPSO_ADD,
	"MAC_CALIPSO_DEL":          AUDIT_MAC_CALIPSO_DEL,
	"MAC_CHECK":                AUDIT_MAC_CHECK,
	"MAC_CIPSOV4_ADD":          AUDIT_MAC_CIPSOV4_ADD,
	"MAC_CIPSOV4_DEL":          AUDIT_MAC_CIPSOV4_DEL,
	"MAC_CONFIG_CHANGE":        AUDIT_MAC_CONFIG_CHANGE,
	"MAC_IPSEC_ADDSA":          AUDIT_MAC_IPSEC_ADDSA,
	"MAC_IPSEC_ADDSPD":         AUDIT_MAC_IPSEC_ADDSPD,
	"MAC_IPSEC_DELSA":          AUDIT_MAC_IPSEC_DELSA,
	"MAC_IPSEC_DELSPD":         AUDIT_MAC_IPSEC_DELSPD,
	"MAC_IPSEC_EVENT":          AUDIT_MAC_IPSEC_EVENT,
	"MAC_MAP_ADD":              AUDIT_MAC_MAP_ADD,
	"MAC_MAP_DEL":              AUDIT_MAC_MAP_DEL,
	"MAC_POLICY_LOAD":          AUDIT_MAC_POLICY_LOAD,
	"MAC_STATUS":               AUDIT_MAC_STATUS,
	"MAC_UNLBL_ALLOW":          AUDIT_MAC_UNLBL_ALLOW,
	"MAC_UNLBL_STCADD":         AUDIT_MAC_UNLBL_STCADD,
	"MAC_UNLBL_STCDEL":         AUDIT_MAC_UNLBL_STCDEL,
	"MAKE_EQUIV":               AUDIT_MAKE_EQUIV,
	"MMAP":                     AUDIT_MMAP,
	"MQ_GETSETATTR":            AUDIT_MQ_GETSETATTR,
	"MQ_NOTIFY":                AUDIT_MQ_NOTIFY,
	"MQ_OPEN":                  AUDIT_MQ_OPEN,
	"MQ_SENDRECV":              AUDIT_MQ_SENDRECV,
	"NETFILTER_CFG":            AUDIT_NETFILTER_CFG,
	"NETFILTER_PKT":            AUDIT_NETFILTER_PKT,
	"OBJ_PID":                  AUDIT_OBJ_PID,
	"PATH":                     AUDIT_PATH,
	"PROCTITLE":                AUDIT_PROCTITLE,
	"REPLACE":                  AUDIT_REPLACE,
	"RESP_ACCT_LOCK":           AUDIT_RESP_ACCT_LOCK,
	"RESP_ACCT_LOCK_TIMED":     AUDIT_RESP_ACCT_LOCK_TIMED,
	"RESP_ACCT_REMOTE":         AUDIT_RESP_ACCT_REMOTE,
	"RESP_ACCT_UNLOCK_TIMED":   AUDIT_RESP_ACCT_UNLOCK_TIMED,
	"RESP_ALERT":               AUDIT_RESP_ALERT,
	"RESP_ANOMALY":             AUDIT_RESP_ANOMALY,
	"RESP_EXEC":                AUDIT_RESP_EXEC,
	"RESP_HALT":                AUDIT_RESP_HALT,
	"RESP_KILL_PROC":           AUDIT_RESP_KILL_PROC,
	"RESP_ORIGIN_BLOCK":        AUDIT_RESP_ORIGIN_BLOCK,
	"RESP_ORIGIN_BLOCK_TIMED":  AUDIT_RESP_ORIGIN_BLOCK_TIMED,
	"RESP_SEBOOL":              AUDIT_RESP_SEBOOL,
	"RESP_SINGLE":              AUDIT_RESP_SINGLE,
	"RESP_TERM_ACCESS":         AUDIT_RESP_TERM_ACCESS,
	"RESP_TERM_LOCK":           AUDIT_RESP_TERM_LOCK,
	"ROLE_ASSIGN":              AUDIT_ROLE_ASSIGN,
	"ROLE_MODIFY":              AUDIT_ROLE_MODIFY,
	"ROLE_REMOVE":              AUDIT_ROLE_REMOVE,
	"SECCOMP":                  AUDIT_SECCOMP,
	"SELINUX_ERR":              AUDIT_SELINUX_ERR,
	"SERVICE_START":            AUDIT_SERVICE_START,
	"SERVICE_STOP":             AUDIT_SERVICE_STOP,
	"SET":                      AUDIT_SET,
	"SET_FEATURE":              AUDIT_SET_FEATURE,
	"SIGNAL_INFO":              AUDIT_SIGNAL_INFO,
	"SOCKADDR":                 AUDIT_SOCKADDR,
	"SOCKETCALL":               AUDIT_SOCKETCALL,
	"SOFTWARE_UPDATE":          AUDIT_SOFTWARE_UPDATE,
	"SYSCALL":                  AUDIT_SYSCALL,
	"SYSTEM_BOOT":              AUDIT_SYSTEM_BOOT,
	"SYSTEM_RUNLEVEL":          AUDIT_SYSTEM_RUNLEVEL,
	"SYSTEM_SHUTDOWN":          AUDIT_SYSTEM_SHUTDOWN,
	"TEST":                     AUDIT_TEST,
	"TRIM":                     AUDIT_TRIM,
	"TRUSTED_APP":              AUDIT_TRUSTED_APP,
	"TTY":                      AUDIT_TTY,
	"TTY_GET":                  AUDIT_TTY_GET,
	"TTY_SET":                  AUDIT_TTY_SET,
	"USER":                     AUDIT_USER,
	"USER_ACCT":                AUDIT_USER_ACCT,
	"USER_AUTH":                AUDIT_USER_AUTH,
	"USER_AVC":                 AUDIT_USER_AVC,
	"USER_CHAUTHTOK":           AUDIT_USER_CHAUTHTOK,
	"USER_CMD":                 AUDIT_USER_CMD,
	"USER_DEVICE":              AUDIT_USER_DEVICE,
	"USER_END":                 AUDIT_USER_END,
	"USER_ERR":                 AUDIT_USER_ERR,
	"USER_LABELED_EXPORT":      AUDIT_USER_LABELED_EXPORT,
	"USER_LOGIN":               AUDIT_USER_LOGIN,
	"USER_LOGOUT":              AUDIT_USER_LOGOUT,
	"USER_MAC_CONFIG_CHANGE":   AUDIT_USER_MAC_CONFIG_CHANGE,
	"USER_MAC_POLICY_LOAD":     AUDIT_USER_MAC_POLICY_LOAD,
	"USER_MGMT":                AUDIT_USER_MGMT,
	"USER_ROLE_CHANGE":         AUDIT_USER_ROLE_CHANGE,
	"USER_SELINUX_ERR":         AUDIT_USER_SELINUX_ERR,
	"USER_START":               AUDIT_USER_START,
	"USER_TTY":                 AUDIT_USER_TTY,
	"USER_UNLABELED_EXPORT":    AUDIT_USER_UNLABELED_EXPORT,
	"USYS_CONFIG":              AUDIT_USYS_CONFIG,
	"VIRT_CONTROL":             AUDIT_VIRT_CONTROL,
	"VIRT_CREATE":              AUDIT_VIRT_CREATE,
	"VIRT_DESTROY":             AUDIT_VIRT_DESTROY,
	"VIRT_INTEGRITY_CHECK":     AUDIT_VIRT_INTEGRITY_CHECK,
	"VIRT_MACHINE_ID":          AUDIT_VIRT_MACHINE_ID,
	"VIRT_MIGRATE_IN":          AUDIT_VIRT_MIGRATE_IN,
	"VIRT_MIGRATE_OUT":         AUDIT_VIRT_MIGRATE_OUT,
	"VIRT_RESOURCE":            AUDIT_VIRT_RESOURCE,
	"WATCH_INS":                AUDIT_WATCH_INS,
	"WATCH_LIST":               AUDIT_WATCH_LIST,
	"WATCH_REM":                AUDIT_WATCH_REM,
}

// GetAuditMessageType accepts a type name and returns its numerical
// representation. If the name is unknown and error is returned.
func GetAuditMessageType(name string) (AuditMessageType, error) {
	name = strings.ToUpper(name)

	typ, found := auditMessageNameToType[name]
	if found {
		return typ, nil
	}

	// Parse type from UNKNOWN[1329].
	start := strings.IndexByte(name, '[')
	if start == -1 {
		return 0, errInvalidAuditMessageTypName
	}
	name = name[start+1:]

	end := strings.IndexByte(name, ']')
	if end == -1 {
		return 0, errInvalidAuditMessageTypName
	}
	name = name[:end]

	num, err := strconv.ParseUint(name, 10, 16)
	if err != nil {
		return 0, errInvalidAuditMessageTypName
	}

	return AuditMessageType(num), nil
}
