//
// Copyright 2021 The Sigstore Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package cli

import (
	"context"
	"flag"
	"fmt"
	"io"
	"os"

	"github.com/google/go-containerregistry/pkg/authn"
	"github.com/google/go-containerregistry/pkg/name"
	"github.com/google/go-containerregistry/pkg/v1/remote"
	"github.com/peterbourgon/ff/v3/ffcli"

	"github.com/sigstore/sigstore/pkg/signature/payload"
)

func Generate() *ffcli.Command {
	var (
		flagset     = flag.NewFlagSet("cosign generate", flag.ExitOnError)
		annotations = annotationsMap{}
	)
	flagset.Var(&annotations, "a", "extra key=value pairs to sign")

	return &ffcli.Command{
		Name:       "generate",
		ShortUsage: "cosign generate [-a key=value] <image uri>",
		ShortHelp:  "Generates (unsigned) signature payloads from the supplied container image",
		LongHelp: `Generates an unsigned payload from the supplied container image and flags.
This payload matches the one generated by the "cosign sign" command and can be used if you need
to sign payloads with your own tooling or algorithms.

EXAMPLES
	# Generage a simple payload for an image
	cosign generate <IMAGE>

	# Generate a payload with specific annotations
	cosign generate -a foo=bar <IMAGE>

	# Use this payload in another tool
	gpg --output image.sig --detach-sig <(cosign generate <IMAGE>)`,
		FlagSet: flagset,
		Exec: func(ctx context.Context, args []string) error {
			if len(args) != 1 {
				return flag.ErrHelp
			}
			return GenerateCmd(ctx, args[0], annotations.annotations, os.Stdout)
		},
	}
}

func GenerateCmd(_ context.Context, imageRef string, annotations map[string]interface{}, w io.Writer) error {
	ref, err := name.ParseReference(imageRef)
	if err != nil {
		return err
	}

	get, err := remote.Get(ref, remote.WithAuthFromKeychain(authn.DefaultKeychain))
	if err != nil {
		return err
	}
	repo := ref.Context()
	img := repo.Digest(get.Digest.String())

	payload, err := (&payload.Cosign{Image: img, Annotations: annotations}).MarshalJSON()
	if err != nil {
		return err
	}
	fmt.Fprintln(w, string(payload))
	return nil
}
