package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/format"
	"io/ioutil"
	"os"
	"text/template"
)

var (
	outputTemplate = template.Must(template.New("output").Funcs(template.FuncMap{
		"printMultiLineString": printMultiLineString,
	}).Parse(`// Code generated by github.com/twpayne/chezmoi/internal/generate-assets. DO NOT EDIT.
{{- if .Tags}}
// +build {{ .Tags }}
{{- end }}

package cmd

func init() {
{{- range $key, $value := .Assets }}
	assets[{{ printf "%q" $key }}] = []byte("" +
		{{ printMultiLineString $value }})
{{- end }}
}`))

	output = flag.String("o", "/dev/stdout", "output")
	tags   = flag.String("tags", "", "tags")
)

func printMultiLineString(s []byte) string {
	b := &bytes.Buffer{}
	for i, line := range bytes.Split(s, []byte{'\n'}) {
		if i != 0 {
			b.WriteString(" +\n")
		}
		b.WriteString(fmt.Sprintf("%q", append(line, '\n')))
	}
	return b.String()
}

func run() error {
	flag.Parse()

	assets := make(map[string][]byte)
	for _, arg := range flag.Args() {
		var err error
		assets[arg], err = ioutil.ReadFile(arg)
		if err != nil {
			return err
		}
	}

	source := &bytes.Buffer{}
	if err := outputTemplate.Execute(source, struct {
		Tags   string
		Assets map[string][]byte
	}{
		Tags:   *tags,
		Assets: assets,
	}); err != nil {
		return err
	}

	formattedSource, err := format.Source(source.Bytes())
	if err != nil {
		return err
	}

	return ioutil.WriteFile(*output, formattedSource, 0666)
}

func main() {
	if err := run(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}
