// +build codegen

package aws

import (
	"bytes"
	"fmt"
	"path/filepath"
	"strings"
	"text/template"

	"github.com/apex/log"
	"github.com/aws/aws-sdk-go/private/model/api"
	"github.com/jckuester/awsls/gen/util"
)

type Service struct {
	Name                   string
	TerraformResourceTypes []ResourceType
}

type ResourceType struct {
	Name         string
	Tags         bool
	CreationTime bool
	Owner        bool
}

func GenerateListFunctions(outputPath string, services []Service, resourceIDs map[string]string,
	resourceTypesWithTags []string, apis api.APIs) []Service {

	resourcesWithRequiredFieldsCount := 0
	noOutputFieldNameFoundCount := 0
	noListOpCandidatesFoundCount := 0
	noResourceIDFoundCount := 0

	var servicesResult []Service

	for _, service := range services {
		fmt.Println()
		fmt.Printf("service: %s\n---\n", service.Name)

		var rTypesResult []ResourceType

		for _, rType := range service.TerraformResourceTypes {
			_, ok := ExcludedResourceTypes[rType.Name]
			if ok {
				log.WithField("resource", rType.Name).Info("exclude")
				continue
			}

			listOpCandidates := FindListOperationCandidates(rType.Name, service.Name, apis)
			if len(listOpCandidates) == 0 {
				noListOpCandidatesFoundCount++
				log.WithField("resource", rType.Name).Errorf("no list operation candidate found")

				continue
			}

			outputFieldType := "structure"
			outputFieldName, op, err := findOutputField(rType.Name, listOpCandidates, outputFieldType)
			if err != nil {
				outputFieldType = "string"
				outputFieldName, op, err = findOutputField(rType.Name, listOpCandidates, outputFieldType)
				if err != nil {
					noOutputFieldNameFoundCount++
					log.WithError(err).WithField("resource", rType.Name).Errorf("unable to find output field name")

					continue
				}

				log.WithField("resource", rType.Name).Infof("found output field of type string")
			}

			outputField := op.OutputRef.Shape.MemberRefs[outputFieldName]

			if len(op.InputRef.Shape.Required) > 0 {
				resourcesWithRequiredFieldsCount++
				log.WithField("resource", rType.Name).
					Errorf("required input fields: %s", op.InputRef.Shape.Required)

				continue
			}

			resourceID, err := findResourceID(rType.Name, resourceIDs, outputField)
			if err != nil && outputFieldType != "string" {
				noResourceIDFoundCount++
				log.WithField("resource", rType.Name).Errorf("no resource ID found")

				continue
			}

			for k, _ := range op.InputRef.Shape.MemberRefs {
				if strings.Contains(strings.ToLower(k), "owner") {
					log.Infof("input; found owner field for %s: %s", rType.Name, k)
				}
			}

			op.Service = op.API.PackageName()
			serviceV2, ok := AWSServicesV1toV2[service.Name]
			if ok {
				op.Service = serviceV2
			}
			_, noPaginator := missingPaginatorAPI[rType.Name]
			if noPaginator {
				op.Paginator = nil
			}
			op.OutputFieldName = outputFieldName
			op.OutputFieldType = outputFieldType
			op.TerraformType = rType.Name
			op.ResourceID = resourceID
			op.OpName = rType.ListFunctionName()
			op.Inputs = Inputs[rType.Name]

			rType.CreationTime = op.GetCreationTimeGoCode() != ""
			rType.Owner = op.GetOwnerGoCode() != ""
			rType.Tags = util.Contains(resourceTypesWithTags, rType.Name)

			if rType.Name != "aws_instance" {
				// note: code is manually added for "aws_instance"
				writeListFunction(outputPath, &op)
			} else {
				rType.CreationTime = true
			}

			rTypesResult = append(rTypesResult, rType)
		}

		if len(rTypesResult) > 0 {
			servicesResult = append(servicesResult, Service{
				Name:                   service.Name,
				TerraformResourceTypes: rTypesResult,
			})
		}
	}

	log.Infof("list functions with required fields: %d", resourcesWithRequiredFieldsCount)
	log.Infof("unable to find output field name: %d", noOutputFieldNameFoundCount)
	log.Infof("resources without list operation candidate: %d", noListOpCandidatesFoundCount)
	log.Infof("no resource ID found: %d", noResourceIDFoundCount)

	return servicesResult
}

func writeListFunction(outputPath string, op *ListOperation) {
	err := util.WriteGoFile(
		filepath.Join(outputPath, op.TerraformType+".go"),
		util.CodeLayout,
		"",
		"aws",
		op.GoCode(),
	)

	if err != nil {
		panic(err)
	}
}

type ListOperation struct {
	api.Operation

	Service         string
	TerraformType   string
	ResourceID      string
	OutputListName  string
	OpName          string
	Inputs          string
	OutputFieldName string
	OutputFieldType string
}

func (o *ListOperation) GoCode() string {
	var buf bytes.Buffer
	err := listResourcesOperationTmpl.Execute(&buf, o)
	if err != nil {
		panic(err)
	}

	return strings.TrimSpace(buf.String())
}

func (o ListOperation) GetCreationTimeGoCode() string {
	outputField := o.OutputRef.Shape.MemberRefs[o.OutputFieldName]

	creationTimeFieldNames := []string{
		"LaunchTime",
		"CreateTime",
		"CreateDate",
		"CreatedTime",
		"CreationDate",
		"CreationTime",
		"CreationTimestamp",
		"StartTime",
		"InstanceCreateTime",
	}

	for k, v := range outputField.Shape.MemberRef.Shape.MemberRefs {
		for _, name := range creationTimeFieldNames {
			if k == name {
				if v.Shape.Type == "string" {
					return `t, err := time.Parse("2006-01-02T15:04:05.000Z0700", *r.` + k + `)
							if err != nil {
								return nil, err
							}`
				}

				if v.Shape.Type == "timestamp" {
					return `t := ` + fmt.Sprintf("*r.%s", k)
				}

				if v.Shape.Type == "long" {
					return fmt.Sprintf("t := time.Unix(0, *r.%s * 1000000).UTC()", k)
				}

				log.Warnf("uncovered creation time type: %s", v.Shape.Type)
			}
		}
	}

	return ""
}

func (o ListOperation) GetOwnerGoCode() string {
	outputField := o.OutputRef.Shape.MemberRefs[o.OutputFieldName]

	for k, _ := range outputField.Shape.MemberRef.Shape.MemberRefs {
		if k == "OwnerId" {
			return `if *r.OwnerId != client.AccountID {
						continue
					}`
		}
		if strings.Contains(strings.ToLower(k), "owner") {
			log.Infof("output; found owner field: %s", k)
		}
	}

	return ""
}

var listResourcesOperationTmpl = template.Must(template.New("listResourcesOperation").Funcs(
	template.FuncMap{
		"Title": strings.Title,
	}).Parse(`
import(
	"context"

	"github.com/jckuester/awstools-lib/aws"
	"github.com/jckuester/awstools-lib/terraform"
	"github.com/aws/aws-sdk-go-v2/service/{{ .Service }}"
)

{{ $pagerType := printf "%sPaginator" .ExportedName -}}

func {{.OpName}}(ctx context.Context, client *aws.Client) ([]terraform.Resource, error) {
	var result []terraform.Resource

	{{ if .Paginator }}
    p := {{ .Service }}.New{{ $pagerType }}(client.{{ .Service | Title }}conn, &{{ .Service }}.{{ .InputRef.ShapeName }}{ {{ if ne .Inputs "" }}{{ .Inputs }}{{ end }} })
	for p.HasMorePages() {
		resp, err := p.NextPage(ctx)
		if err != nil {
			return nil, err
		}
	{{ else }}
    resp, err := client.{{ .Service | Title }}conn.{{ .ExportedName }}(ctx, &{{ .Service }}.{{ .InputRef.ShapeName }}{ {{ if ne .Inputs "" }}{{ .Inputs }}{{ end }} })
	if err != nil {
		return nil, err
	}

	if len(resp.{{ .OutputListName }}) > 0 {
	{{ end }}
		for _, r := range resp.{{ .OutputListName }}{
			{{ if ne .GetOwnerGoCode "" }}{{ .GetOwnerGoCode }}{{ end }}
			{{ if ne .GetCreationTimeGoCode "" }}{{ .GetCreationTimeGoCode }}{{ end }}
			result = append(result, terraform.Resource{
				Type: "{{ .TerraformType }}",
				{{ if ne .OutputFieldType "string" }}ID: *r.{{ .ResourceID }},{{ else }}ID: r,{{ end }}
				Profile: client.Profile,
				Region: client.Region,
				AccountID: client.AccountID,
				{{ if ne .GetCreationTimeGoCode "" }}CreatedAt: &t,{{ end }}
			})
		}
	}

	return result, nil
}
`))
