package extractors

import (
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"

	"github.com/iawia002/annie/downloader"
	"github.com/iawia002/annie/request"
	"github.com/iawia002/annie/utils"
)

type qqVideoInfo struct {
	Fl struct {
		Fi []struct {
			ID    int    `json:"id"`
			Name  string `json:"name"`
			Cname string `json:"cname"`
			Fs    int    `json:"fs"`
		} `json:"fi"`
	} `json:"fl"`
	Vl struct {
		Vi []struct {
			Fn    string `json:"fn"`
			Ti    string `json:"ti"`
			Fvkey string `json:"fvkey"`
			Cl    struct {
				Fc int `json:"fc"`
				Ci []struct {
					Idx int `json:"idx"`
				} `json:"ci"`
			} `json:"cl"`
			Ul struct {
				UI []struct {
					URL string `json:"url"`
				} `json:"ui"`
			} `json:"ul"`
		} `json:"vi"`
	} `json:"vl"`
	Msg string `json:"msg"`
}

type qqKeyInfo struct {
	Key string `json:"key"`
}

const qqPlayerVersion string = "3.2.19.333"

func qqGenFormat(vid, cdn string, data qqVideoInfo) (map[string]downloader.FormatData, error) {
	format := map[string]downloader.FormatData{}
	var vkey string
	// number of fragments
	clips := data.Vl.Vi[0].Cl.Fc
	if clips == 0 {
		clips = 1
	}

	for _, fi := range data.Fl.Fi {
		var fmtIDPrefix string
		fns := strings.Split(data.Vl.Vi[0].Fn, ".")
		if fi.ID > 100000 {
			fmtIDPrefix = "m"
		} else if fi.ID > 10000 {
			fmtIDPrefix = "p"
		}
		if fmtIDPrefix != "" {
			fmtIDName := fmt.Sprintf("%s%d", fmtIDPrefix, fi.ID%10000)
			if len(fns) < 3 {
				// v0739eolv38.mp4 -> v0739eolv38.m701.mp4
				fns = append(fns[:1], append([]string{fmtIDName}, fns[1:]...)...)
			} else {
				// n0687peq62x.p709.mp4 -> n0687peq62x.m709.mp4
				fns[1] = fmtIDName
			}
		} else if len(fns) >= 3 {
			// delete ID part
			// e0765r4mwcr.2.mp4 -> e0765r4mwcr.mp4
			fns = append(fns[:1], fns[2:]...)
		}

		var urls []downloader.URLData
		var totalSize int64
		var filename string
		for part := 1; part < clips+1; part++ {
			// Multiple fragments per format
			if fmtIDPrefix == "p" {
				if len(fns) < 4 {
					// If the number of fragments > 0, the filename needs to add the number of fragments
					// n0687peq62x.p709.mp4 -> n0687peq62x.p709.1.mp4
					fns = append(fns[:2], append([]string{strconv.Itoa(part)}, fns[2:]...)...)
				} else {
					fns[2] = strconv.Itoa(part)
				}
			}
			filename = strings.Join(fns, ".")
			html, err := request.Get(
				fmt.Sprintf(
					"http://vv.video.qq.com/getkey?otype=json&platform=11&appver=%s&filename=%s&format=%d&vid=%s", qqPlayerVersion, filename, fi.ID, vid,
				), cdn, nil,
			)
			if err != nil {
				return nil, err
			}
			jsonString := utils.MatchOneOf(html, `QZOutputJson=(.+);$`)[1]
			var keyData qqKeyInfo
			json.Unmarshal([]byte(jsonString), &keyData)
			vkey = keyData.Key
			if vkey == "" {
				vkey = data.Vl.Vi[0].Fvkey
			}
			realURL := fmt.Sprintf("%s%s?vkey=%s", cdn, filename, vkey)
			size, err := request.Size(realURL, cdn)
			if err != nil {
				return nil, err
			}
			urlData := downloader.URLData{
				URL:  realURL,
				Size: size,
				Ext:  "mp4",
			}
			urls = append(urls, urlData)
			totalSize += size
		}
		format[fi.Name] = downloader.FormatData{
			URLs:    urls,
			Size:    totalSize,
			Quality: fi.Cname,
		}
	}
	return format, nil
}

// QQ download function
func QQ(url string) (downloader.VideoData, error) {
	vid := utils.MatchOneOf(url, `vid=(\w+)`, `/(\w+)\.html`)[1]
	if len(vid) != 11 {
		u, err := request.Get(url, url, nil)
		if err != nil {
			return downloader.VideoData{}, err
		}
		vid = utils.MatchOneOf(
			u, `vid=(\w+)`, `vid:\s*["'](\w+)`, `vid\s*=\s*["']\s*(\w+)`,
		)[1]
	}
	html, err := request.Get(
		fmt.Sprintf(
			"http://vv.video.qq.com/getinfo?otype=json&platform=11&defnpayver=1&appver=%s&defn=shd&vid=%s", qqPlayerVersion, vid,
		), url, nil,
	)
	if err != nil {
		return downloader.VideoData{}, err
	}
	jsonString := utils.MatchOneOf(html, `QZOutputJson=(.+);$`)[1]
	var data qqVideoInfo
	json.Unmarshal([]byte(jsonString), &data)
	// API request error
	if data.Msg != "" {
		return downloader.VideoData{}, errors.New(data.Msg)
	}
	cdn := data.Vl.Vi[0].Ul.UI[len(data.Vl.Vi[0].Ul.UI)-1].URL
	format, err := qqGenFormat(vid, cdn, data)
	if err != nil {
		return downloader.VideoData{}, err
	}
	extractedData := downloader.VideoData{
		Site:    "腾讯视频 v.qq.com",
		Title:   utils.FileName(data.Vl.Vi[0].Ti),
		Type:    "video",
		Formats: format,
	}
	err = extractedData.Download(url)
	if err != nil {
		return downloader.VideoData{}, err
	}
	return extractedData, nil
}
