package arg

import (
	"fmt"
	"os"
)

// This example demonstrates basic usage
func Example() {
	// These are the args you would pass in on the command line
	os.Args = []string{"./example", "--foo=hello", "--bar"}

	var args struct {
		Foo string
		Bar bool
	}
	MustParse(&args)
	fmt.Println(args.Foo, args.Bar)
}

// This example demonstrates arguments that have default values
func Example_defaultValues() {
	// These are the args you would pass in on the command line
	os.Args = []string{"--help"}

	var args struct {
		Foo string
		Bar bool
	}
	args.Foo = "default value"
	MustParse(&args)
	fmt.Println(args.Foo, args.Bar)
}

// This example demonstrates arguments that are required
func Example_requiredArguments() {
	// These are the args you would pass in on the command line
	os.Args = []string{"--foo=1", "--bar"}

	var args struct {
		Foo string `arg:"required"`
		Bar bool
	}
	MustParse(&args)
}

// This example demonstrates positional arguments
func Example_positionalArguments() {
	// These are the args you would pass in on the command line
	os.Args = []string{"./example", "in", "out1", "out2", "out3"}

	var args struct {
		Input  string   `arg:"positional"`
		Output []string `arg:"positional"`
	}
	MustParse(&args)
	fmt.Println("Input:", args.Input)
	fmt.Println("Output:", args.Output)
}

// This example demonstrates arguments that have multiple values
func Example_multipleValues() {
	// The args you would pass in on the command line
	os.Args = []string{"--help"}

	var args struct {
		Database string
		IDs      []int64
	}
	MustParse(&args)
	fmt.Printf("Fetching the following IDs from %s: %q", args.Database, args.IDs)
}

// This eample demonstrates multiple value arguments that can be mixed with
// other arguments.
func Example_multipleMixed() {
	os.Args = []string{"./example", "-c", "cmd1", "db1", "-f", "file1", "db2", "-c", "cmd2", "-f", "file2", "-f", "file3", "db3", "-c", "cmd3"}
	var args struct {
		Commands  []string `arg:"-c,separate"`
		Files     []string `arg:"-f,separate"`
		Databases []string `arg:"positional"`
	}
	MustParse(&args)
	fmt.Println("Commands:", args.Commands)
	fmt.Println("Files", args.Files)
	fmt.Println("Databases", args.Databases)
}

// This example shows the usage string generated by go-arg
func Example_usageString() {
	// These are the args you would pass in on the command line
	os.Args = []string{"--help"}

	var args struct {
		Input    string   `arg:"positional"`
		Output   []string `arg:"positional"`
		Verbose  bool     `arg:"-v,help:verbosity level"`
		Dataset  string   `arg:"help:dataset to use"`
		Optimize int      `arg:"-O,help:optimization level"`
	}
	MustParse(&args)
}
