package seed

import (
	"testing"

	. "github.com/RyuaNerin/go-krypto/testingutil"
)

func Test_SEED128_Encrypt(t *testing.T) { BTE(t, BIW(NewCipher), CE, testCases128, false) }
func Test_SEED128_Decrypt(t *testing.T) { BTD(t, BIW(NewCipher), CD, testCases128, false) }

var testCases128 = []BlockTestCase{
	// TTAS.KO-12.0004/R1
	// p. 21
	// Ⅰ.1. 참조구현값 1
	{
		Key:    `00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00`,
		Plain:  `00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F`,
		Secure: `5E BA C6 E0 05 4E 16 68 19 AF F1 CC 6D 34 6C DB`,
	},
	// p. 21
	// Ⅰ.2. 참조구현값 2
	{
		Key:    `00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F`,
		Plain:  `00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00`,
		Secure: `C1 1F 22 F2 01 40 50 50 84 48 35 97 E4 37 0F 43`,
	},
	// p. 22
	// Ⅰ.3. 참조구현값 3
	{
		Key:    `47 06 48 08 51 E6 1B E8 5D 74 BF B3 FD 95 61 85`,
		Plain:  `83 A2 F8 A2 88 64 1F B9 A4 E9 A5 CC 2F 13 1C 7D`,
		Secure: `EE 54 D1 3E BC AE 70 6D 22 6B C3 14 2C D4 0D 4A`,
	},
	// p. 24
	// Ⅰ.4. 참조구현값 4
	{
		Key:    `28 DB C3 BC 49 FF D8 7D CF A5 09 B1 1D 42 2B E7`,
		Plain:  `B4 1E 6B E2 EB A8 4A 14 8E 2E ED 84 59 3C 5E C7`,
		Secure: `9B 9B 7B FC D1 81 3C B9 5D 0B 36 18 F4 0F 51 22`,
	},
	//////////////////////////////////////////////////
	// 암호알고리즘 검증기준 V3.0
	// 테스트 벡터
	// SEED128(ECB)KAT.txt
	{
		Key:    `9F3879B964CB88490000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `9B48304CAAAE2AB1C5303CAA47069C4D`,
	},
	{
		Key:    `A33C7DBD68CF8C4D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `D62612F69BD9177C71E9EFBFD408078A`,
	},
	{
		Key:    `A74081C16CD390510000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `1DCC2369C3966E7FC3CB46A6AE7CA0F0`,
	},
	{
		Key:    `AB4485C570D794550000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `34A36E53DED33DB658E040DCE87651FC`,
	},
	{
		Key:    `AF4889C974DB98590000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `E07F74EB9F2D0E1ACDE4F949BE3C7530`,
	},
	{
		Key:    `B34C8DCD78DF9C5D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `F6AAA2475FAB60F4D408A7FC958B3F4D`,
	},
	{
		Key:    `B75091D17CE3A0610000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `AE36480DC49CEBA24F5CAEBD4DCF67BF`,
	},
	{
		Key:    `BB5495D580E7A4650000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `E36BB5038713CAA9F45752742D8C27FA`,
	},
	{
		Key:    `BF5899D984EBA8690000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `46DB5766314C932EE36EAE31FF4C53D6`,
	},
	{
		Key:    `C35C9DDD88EFAC6D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `72EC25F302E3DB2925F2B77B237B0A0C`,
	},
	{
		Key:    `C760A1E18CF3B0710000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `AF9FE3272AABAC918FB3521718157B54`,
	},
	{
		Key:    `CB64A5E590F7B4750000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `66D4CC726AF71312CF65A3336A74BF26`,
	},
	{
		Key:    `CF68A9E994FBB8790000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `8A11AAF9B950EF930327FFCD8938DC9B`,
	},
	{
		Key:    `D36CADED98FFBC7D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `F70A99C49FAA40E1DADF670FE0C9D011`,
	},
	{
		Key:    `D770B1F19D03C0810000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `7BD77F78856A3F77A23B99BD8C15BB3D`,
	},
	{
		Key:    `DB74B5F5A107C4850000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `6974FD886125199E13A4262CA9565EBF`,
	},
	{
		Key:    `DF78B9F9A50BC8890000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `070B20B8AA5AE97164B9F668BBC1A1C5`,
	},
	{
		Key:    `E37CBDFDA90FCC8D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `C8AB93883AEAFF1E8CEE851E5F61BE76`,
	},
	{
		Key:    `E780C201AD13D0910000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `8D3AAE34F0793E9452651922CE886737`,
	},
	{
		Key:    `EB84C605B117D4950000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3B5B42196B3AD9883A81CEA9047D06C6`,
	},
	{
		Key:    `EF88CA09B51BD8990000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `69CF0DE9E55FA3B1496CEFAC9F3A8D08`,
	},
	{
		Key:    `F38CCE0DB91FDC9D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `0151FCBCCAE3870A2AD7ACFEADBD2981`,
	},
	{
		Key:    `F790D211BD23E0A10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `EFFADC7754AF279B2B07F3507A97C49B`,
	},
	{
		Key:    `FB94D615C127E4A50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `CAA6069D0326055D4BBB4AC3B2AEA1F1`,
	},
	{
		Key:    `FF98DA19C52BE8A90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `366AF003E6492C28C8F27890B8BDBAE0`,
	},
	{
		Key:    `039CDE1DC92FECAD0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `AEF552AAFCDCF74B4E55107CF91B5003`,
	},
	{
		Key:    `07A0E221CD33F0B10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3CDD39973403935FCDC418D8B0827DEA`,
	},
	{
		Key:    `0BA4E625D137F4B50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B3F58989F69FAA2F01719236A413A198`,
	},
	{
		Key:    `0FA8EA29D53BF8B90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `217EF4F8AFBAD3529694D8AAD6C27821`,
	},
	{
		Key:    `13ACEE2DD93FFCBD0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `C412FEE42B21D45964363A1CD451BEED`,
	},
	{
		Key:    `17B0F231DD4400C10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `5301FEF2FDB215290136EB14F272AD64`,
	},
	{
		Key:    `1BB4F635E14804C50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `4B72BAFD7FF5628FD42899F50CD1C993`,
	},
	{
		Key:    `1FB8FA39E54C08C90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `EDD74D7C185DF959BB0468E37D88A6DD`,
	},
	{
		Key:    `23BCFE3DE9500CCD0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3530F691A5B8C5667936FBFCAA147FF0`,
	},
	{
		Key:    `27C10241ED5410D10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `14CD526D978FAD99E8F85D7395494B80`,
	},
	{
		Key:    `2BC50645F15814D50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `093DB1F948B65A3C5B2115424CD5F865`,
	},
	{
		Key:    `2FC90A49F55C18D90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `24F389DA66F475541F1FD0C9A3ECA20B`,
	},
	{
		Key:    `33CD0E4DF9601CDD0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `84DF098F012E03462462747F58D3A135`,
	},
	{
		Key:    `37D11251FD6420E10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `E0E7A3647B9E5C56F9E030032F26C3DC`,
	},
	{
		Key:    `3BD51655016824E50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3F8EDD349A5DDEB853FBFAF6CCCD9152`,
	},
	{
		Key:    `3FD91A59056C28E90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3A42E97FA1B5CD1C5848374F69F6DAE7`,
	},
	{
		Key:    `43DD1E5D09702CED0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `2AFEF8A73D0823E5FC5F09680835AF2C`,
	},
	{
		Key:    `47E122610D7430F10000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `2E54DEF3BBBD779CE2A941F6267E2734`,
	},
	{
		Key:    `4BE52665117834F50000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `3DF843061004C97B315233C143ED70A0`,
	},
	{
		Key:    `4FE92A69157C38F90000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B35DDF498AB1BEA1DF6312336C6AB541`,
	},
	{
		Key:    `53ED2E6D19803CFD0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `63A232BB15A9A3A87E42834DCB3EE653`,
	},
	{
		Key:    `57F132711D8441010000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `7312841FA862B15A31B0C59DC24DC102`,
	},
	{
		Key:    `5BF53675218845050000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `FC5DA4D15F3F701AF26AE8696B12B1CD`,
	},
	{
		Key:    `5FF93A79258C49090000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `954B2CE56F22DD3C96D7445B6D3E6E8B`,
	},
	{
		Key:    `63FD3E7D29904D0D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B276FE028279DB354FB5570BCDC49A81`,
	},
	{
		Key:    `680142812D9451110000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `800349F0665D84237D265580A40D484A`,
	},
	{
		Key:    `6C054685319855150000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `916F34ACB0450923FEDDEF607A8E3A21`,
	},
	{
		Key:    `70094A89359C59190000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `C6342EA96AC4BDF85D45DC0E9267B464`,
	},
	{
		Key:    `740D4E8D39A05D1D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `398521ABCFED032707748FB3F5CAFA33`,
	},
	{
		Key:    `781152913DA461210000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B6F0A38FCB0CD5A48A555AB8B843D0EA`,
	},
	{
		Key:    `7C15569541A865250000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `EA9C40A30A6CB5558CABC8C907125794`,
	},
	{
		Key:    `80195A9945AC69290000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `92A7D6586459F66F80CC20C85F31AD89`,
	},
	{
		Key:    `841D5E9D49B06D2D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `ACF11BFC8AEBB0BDF9971D15894A101F`,
	},
	{
		Key:    `882162A14DB471310000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `F3397ED554EC4899258039B0B79C989C`,
	},
	{
		Key:    `8C2566A551B875350000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `0C21E75A4C72943A0D9EC2B7CDE5F92E`,
	},
	{
		Key:    `90296AA955BC79390000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `202EC3E131B3DC0EEDA328A4D8DDBF94`,
	},
	{
		Key:    `942D6EAD59C07D3D0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `9A01282362DE906FEC3B253417513F6A`,
	},
	{
		Key:    `983172B15DC481410000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `CE6DB3FAD9F78986FFA5B62D25771F01`,
	},
	{
		Key:    `9C3576B561C885450000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `F0B987B25FBC129DA98784BFE98C4EFB`,
	},
	{
		Key:    `9E3779BA61C8864A0000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `642C99E8227DAE632A680EF74107C7AD`,
	},
	{
		Key:    `9E3779C061C886560000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `BFA0A0F08983FAE01B5BF05EDD11BB25`,
	},
	{
		Key:    `9E3779D861C886860000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `2C163F1F4BA6625D94B6F95B2B838133`,
	},
	{
		Key:    `9E377A3861C887460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `96512A6D93863A3346ADEBD03CD76543`,
	},
	{
		Key:    `9E377BB861C88A460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `0FD4503CBD571D8E6ECFAA9D0C43A397`,
	},
	{
		Key:    `9E3781B861C896460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `053BB69F94E8FF3C89F6ED361A6E7D71`,
	},
	{
		Key:    `9E3799B861C8C6460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `7698F5F3692BA3A46E1414417D6FD845`,
	},
	{
		Key:    `9E37F9B861C986460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `4AA9642C17315B1996AA47D994660276`,
	},
	{
		Key:    `9E3979B861CC86460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `E170DB1AC840A2FA153D83D0169C38AE`,
	},
	{
		Key:    `9E3F79B861D886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `6F6A05DB45B50FB04D6AA1C1271FC6F1`,
	},
	{
		Key:    `9E5779B8620886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `5FCA48C007BFC08E13D6EA3275EB9AB9`,
	},
	{
		Key:    `9EB779B862C886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `050FF057293130A50437E67287A299A9`,
	},
	{
		Key:    `A03779B865C886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `8C0B1A06DB5CDEFFD390D91010A11F1C`,
	},
	{
		Key:    `A63779B871C886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `DA8B3C0D78FCEE8F0A229150B1C05DB3`,
	},
	{
		Key:    `BE3779B8A1C886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `C99CD4DB5D1811514D6708AB0892A2F2`,
	},
	{
		Key:    `1E3779B861C886460000000000000000`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `7BC1F3CD2759AEA15B4CD74831041FDA`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000FC8F8C7EF757124C`,
		Secure: `981ECD2D6647F552A949A80B3C4179B0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000003C8F8C7E3757124C`,
		Secure: `9BDFB7B291E178E511AC0BF6D87A271C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000005C8F8C7E5757124C`,
		Secure: `6888580FB19F8A8D62C51566F530E1ED`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000006C8F8C7E6757124C`,
		Secure: `9777879E62D420D6016B050775D4DD96`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000748F8C7E7F57124C`,
		Secure: `4C6F7DF11193EC3764493DDCC03E2125`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000788F8C7E7357124C`,
		Secure: `91084EE847B207967E01A2CF94C7EE91`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007E8F8C7E7557124C`,
		Secure: `41D52CFD36166AB5CDCD360EE8A63F88`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007D8F8C7E7657124C`,
		Secure: `CF5957DD557BDF903512F23B64D87388`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C0F8C7E77D7124C`,
		Secure: `BCE54FEA16C02D61D8DDDF66C39B00B7`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007CCF8C7E7717124C`,
		Secure: `74082A258D4D1BECD0BCF83DD94B1D65`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007CAF8C7E7777124C`,
		Secure: `EB2FD9C47EED1294B7A0F4C37F14DEA3`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C9F8C7E7747124C`,
		Secure: `02777338260280E1CEA0E6670EA894AF`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C878C7E775F124C`,
		Secure: `0D7B528B18DFF6B712C4C61921C41384`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8B8C7E7753124C`,
		Secure: `EAF6D013B0EDC9C55B1DC2D2F010301D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8D8C7E7755124C`,
		Secure: `8D546DDC569CF029DE648FCE32C7F630`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8E8C7E7756124C`,
		Secure: `9916B02C86BC81C7D5C4DC7097BD093B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F0C7E7757924C`,
		Secure: `CFC8BD3B83D39DAD477FCD139BD1E7C2`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8FCC7E7757524C`,
		Secure: `5C172A3F96DFBC962845532624EC5FBA`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8FAC7E7757324C`,
		Secure: `9058B3B92968B24B8C415B6D27483E29`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F9C7E7757024C`,
		Secure: `E55A7034FA366C898DBF273EA8E863C7`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F847E77571A4C`,
		Secure: `A8453729D91141083257A51230F169F4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F887E7757164C`,
		Secure: `CDE7C71E6C3575C7697D8BDDBBF8242B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8E7E7757104C`,
		Secure: `4821E1DC6D885D9771A02334EA7791FC`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8D7E7757134C`,
		Secure: `B2C725AB31E0A4CC4FF42C028AD02FF3`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8CFE775712CC`,
		Secure: `9FF6F07089DD4DBFAC46F23A96D786E0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C3E7757120C`,
		Secure: `23696BE63AE6E09B696284FB531775B4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C5E7757126C`,
		Secure: `EA4B109980DAF9A9FB2771954AA573F9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C6E7757125C`,
		Secure: `76E494D70CCAD3F9BA71CD507D9601D9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C7677571244`,
		Secure: `E70E575C79B2E3E2612372108FBBD66D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C7A77571248`,
		Secure: `5B25EF4A103C75A51635E52BBB0090C7`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C7C7757124E`,
		Secure: `78E9674D107951C006B6B6E7678F7372`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007C8F8C7F7757124D`,
		Secure: `577D4F462D290EF77E02B14495169C27`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000000001`,
		Secure: `A809864319CBBE1AF28FDFB20B22C6D6`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000000003`,
		Secure: `91CB9C42BC26D979D637FE87178F6FF6`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000000007`,
		Secure: `B9D0C835424EF313C99AE7295EB053B0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000000000F`,
		Secure: `B3CC36E0AFE4FC8B4FA2CCE7A4CB0058`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000000001F`,
		Secure: `E7E1B6CBCFB99543D7D792C9C6164D5C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000000003F`,
		Secure: `E2CA217104A6CC02DB606BA11924EC1B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000000007F`,
		Secure: `37ADA3F8D41198F6BB34463CD29056D4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000000000FF`,
		Secure: `588114A92A9F612C2F406B4DCEBD6C7B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000000001FF`,
		Secure: `48AC8CD56F001B37ED2430069D35FA16`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000000003FF`,
		Secure: `919B95217480F2F2D2E36043975727F0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000000007FF`,
		Secure: `5BEE38F3EEE7488426D66F4FAE8E772C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000000FFF`,
		Secure: `6246DDA2DBDF30ACA0559E575E8FCFE1`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000001FFF`,
		Secure: `963A9E403A50E6B22DF020A8BC64E8FA`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000003FFF`,
		Secure: `CCA19F4A2ED86F38E51F76365C0C175A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000007FFF`,
		Secure: `B5ECD88BDE5841497A18A92E7AC650B6`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000000FFFF`,
		Secure: `3223EF4AECA6D5DF1F4CC0C871EC469C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000001FFFF`,
		Secure: `F65499DBE71487B32EECE9DC2111C0D9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000003FFFF`,
		Secure: `CDAFC5750F8354AF9CFFB699FDB18488`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000007FFFF`,
		Secure: `B10AC6DE2238656E657A9FACA6EFEFA4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000000FFFFF`,
		Secure: `DED6A7A73EEEC24D6475189A99EBE161`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000001FFFFF`,
		Secure: `43C289063A1E5C8D84865F6FF756551A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000003FFFFF`,
		Secure: `E005EFF0215D3D150F24D6A42A6C7A27`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000007FFFFF`,
		Secure: `3ADA185A97414BE354835A694A5993BE`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000000FFFFFF`,
		Secure: `3F170289429460642F6F857C8E8821D0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000001FFFFFF`,
		Secure: `CD88471E833DB6BA7900AA64E72CBD6A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000003FFFFFF`,
		Secure: `B0A78D345A62BCD48A3E225B55CB404B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000007FFFFFF`,
		Secure: `1B7295721405BF5F984C2F8FE01E43D4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000000FFFFFFF`,
		Secure: `1EC6902B25217CB0C886BABD858FAE39`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000001FFFFFFF`,
		Secure: `54480198C3A9A5F2EEE6DC7880366BC0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000003FFFFFFF`,
		Secure: `7EF52F5BE48A03913A93BFA402298793`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000007FFFFFFF`,
		Secure: `95F88F7845FC6D9988B01FB9C6AEF708`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000000FFFFFFFF`,
		Secure: `3E9C74F77AEFB4BD258554E36BC5C498`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000001FFFFFFFF`,
		Secure: `183F4A21EB88BB248626C56A93E7917A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000003FFFFFFFF`,
		Secure: `30B361338D4030A2CDB6FB47C6116A27`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000007FFFFFFFF`,
		Secure: `B85A8A31F4A92EB3DD64156647F6A885`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000000FFFFFFFFF`,
		Secure: `180DA8C15B7AEF727AB2DFB22D961508`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000001FFFFFFFFF`,
		Secure: `6EC665B67F1D9C685861DC8BF412DBCE`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000003FFFFFFFFF`,
		Secure: `F2E0DB6F8AEE54F7B2A8F00672261854`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000007FFFFFFFFF`,
		Secure: `F02906FF88E1AAA6212E53E3F2F3F75A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000000FFFFFFFFFF`,
		Secure: `2B2AEF2E2921C19C12E42F3E2E8FD964`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000001FFFFFFFFFF`,
		Secure: `B27EDC0C8A1BA8325BC53BF81F61F6E0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000003FFFFFFFFFF`,
		Secure: `CB1D7D89537FE211C6A9321254858CBD`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000007FFFFFFFFFF`,
		Secure: `14B4E9BDF19D244BA459C4B9253968C0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000000FFFFFFFFFFF`,
		Secure: `4891B68BA901D4CA2445A6AA7DE8AA59`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000001FFFFFFFFFFF`,
		Secure: `52D857C9CB95E18F851F11565C958062`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000003FFFFFFFFFFF`,
		Secure: `2E108E7AB44E635E11F20EF02E2D8F79`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000007FFFFFFFFFFF`,
		Secure: `24BF7BED4549C53870EC7652DEC9CA7E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000000FFFFFFFFFFFF`,
		Secure: `51081A62B1D9B924A7119B8BAC4F5071`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000001FFFFFFFFFFFF`,
		Secure: `5EE506A44EAA2CE468A0E6DECDCDB466`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000003FFFFFFFFFFFF`,
		Secure: `BCF55E86707ED659167113FBBB5ED2A2`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000007FFFFFFFFFFFF`,
		Secure: `72B67F99DA327328BFD6CBD008B2E826`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000000FFFFFFFFFFFFF`,
		Secure: `76C80CB00408625B24A051A475159D12`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000001FFFFFFFFFFFFF`,
		Secure: `1093A85433D8A4BC7EAAA08B00AD28C4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000003FFFFFFFFFFFFF`,
		Secure: `43124B9BDAB9406CE416B2ACFFB1C295`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000007FFFFFFFFFFFFF`,
		Secure: `8732751384172C4446A159BB13951D50`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000000FFFFFFFFFFFFFF`,
		Secure: `A492F676EF725BD9E65FD9EA6EDD245E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000001FFFFFFFFFFFFFF`,
		Secure: `30E5D8AEF9821365EA3A621FFEEA4A77`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000003FFFFFFFFFFFFFF`,
		Secure: `07C5BF26B984A1720E009FACFDE932FF`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000007FFFFFFFFFFFFFF`,
		Secure: `31F9D60576BBD2A7235005E8472D5D2F`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000000FFFFFFFFFFFFFFF`,
		Secure: `1E7BB72941C6BF5EE8AA83178CF29D73`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000001FFFFFFFFFFFFFFF`,
		Secure: `7B7F647E81C851E8A1BF5CC1ADCCC5AB`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000003FFFFFFFFFFFFFFF`,
		Secure: `6879D144F37D6C0853C7137513D5D4F9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000007FFFFFFFFFFFFFFF`,
		Secure: `EC566C142DFFC66CB53BEEFE27B6C4DB`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000000FFFFFFFFFFFFFFFF`,
		Secure: `9CB5914212C2CA7834E0FFD319726907`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000001FFFFFFFFFFFFFFFF`,
		Secure: `07BB805169B718982E18CA8E83DC5464`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000003FFFFFFFFFFFFFFFF`,
		Secure: `55F6F5FE4B88416C0073F11D33C8F04F`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000007FFFFFFFFFFFFFFFF`,
		Secure: `7A1A4AB083A2906808E9C758B816E62D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000000FFFFFFFFFFFFFFFFF`,
		Secure: `7C7A630E08D924FA45F476780B49724B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000001FFFFFFFFFFFFFFFFF`,
		Secure: `1AC46126540D689F6D0C3FC012A479CA`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000003FFFFFFFFFFFFFFFFF`,
		Secure: `F3ECBF775F210EA0BC8D43AC8DB1AD1A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000007FFFFFFFFFFFFFFFFF`,
		Secure: `87741B219400770F5C443C5760BC94FF`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000000FFFFFFFFFFFFFFFFFF`,
		Secure: `E33AB532BC8ADCD244652052F54D2EC9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000001FFFFFFFFFFFFFFFFFF`,
		Secure: `5D5C4EAC123D5A198D0E7CA4FDAD838C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000003FFFFFFFFFFFFFFFFFF`,
		Secure: `97A1E5D2AB8FC28EC542ECF9FDD16AA5`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000007FFFFFFFFFFFFFFFFFF`,
		Secure: `D7F60C2A64420C01D937791A6DB7B642`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000000FFFFFFFFFFFFFFFFFFF`,
		Secure: `A8A85B08F089F6C2092D12F73E69C9A7`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000001FFFFFFFFFFFFFFFFFFF`,
		Secure: `22CAE48EBCF0E3CEA663649F7F9918C7`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000003FFFFFFFFFFFFFFFFFFF`,
		Secure: `D9ADF33C8C024FACCCBD9A22B03E1746`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000007FFFFFFFFFFFFFFFFFFF`,
		Secure: `AB96BE2A017FD9E6469D6B1A13736F45`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000000FFFFFFFFFFFFFFFFFFFF`,
		Secure: `6918F6D79103D568BA4F00775A9A7296`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000001FFFFFFFFFFFFFFFFFFFF`,
		Secure: `6A71EB6D08FEB4E5A14DBCF2EDA1DDBE`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000003FFFFFFFFFFFFFFFFFFFF`,
		Secure: `1961A7CA2F867B61E24A6C7B2E76DC88`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000007FFFFFFFFFFFFFFFFFFFF`,
		Secure: `CC1ABECAADA13B11017045F6A35FCBD5`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000000FFFFFFFFFFFFFFFFFFFFF`,
		Secure: `03F78EABD54D81FEBC563DD3D988CA43`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000001FFFFFFFFFFFFFFFFFFFFF`,
		Secure: `68940DC91BD6C994F4123D303A16F92D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000003FFFFFFFFFFFFFFFFFFFFF`,
		Secure: `10CED1BDC102D3B068E63043F43BF540`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000007FFFFFFFFFFFFFFFFFFFFF`,
		Secure: `4D946E4EC2E7590EADCB9C7ED28BBA35`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000000FFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `C4CBD39CF3938F4EA6270555D268D007`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000001FFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `BC0925BD1AF5DD2D6E4BF49A96AE454E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000003FFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `14265FDE1475A2F50056F39079410875`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000007FFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `8CAE1BDDC2C1C22201ECD7B09CF9FA51`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000000FFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `B37F569AD3548093D01BC9124A1C01E9`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000001FFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `734E7857ED187F6C6586B263A81969F0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000003FFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `C68ED44D4835F5EC475E073997C4868E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000007FFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `3C3DF8E67B88A071A1E0D0DFB5B649A1`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000000FFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `6886A2A255BAC982F6A404CD822C30AB`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000001FFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `C16E0D9AF3368BC23FAB6ABE4D55642A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000003FFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `4A002AC6F904CBC7B5CEBDF7F1C29964`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000007FFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `F1F253FA255A7B5858DA42D84993DB12`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000000FFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `6464743DECD5794D474BF16E5BCD697E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000001FFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `12FF0DF560C6674E8B5FCD54C0CBF4DE`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000003FFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `08360F0C931907B40C4DB429F9C2983D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000007FFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `E31C5A915D71EF26F0151D25D77E68D3`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000000FFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `1F01F4C070EC47683080494F41ADB2F0`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000001FFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `D85EA5473B1C185BE46CF89C6B45B212`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000003FFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `E4DDFA531A52E25AE7628DDF6DD9C595`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000007FFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `FB9203B168ED42ADF3EF821667D7AB39`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00000FFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `C4C9D398A5D1B6E46BEA2E8823C5C79A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00001FFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `4EB3768B5AF7E2FFC830081EA89D271E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00003FFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `265E1C93521E21E3C2B30399C807AB6D`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00007FFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `B09263BB6AFC7F0917E668D25B49B610`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0000FFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `8665B8F03C1EB73D1EC3568C28734416`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0001FFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `5D6137CFBB1C18EE13F7CD16BF695811`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0003FFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `05599246E5EE9ADF1768B62D875B1E58`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0007FFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `C223D6228C1420EB136004BCAEB3AA01`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `000FFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `A2B70C3FA22310E571B42BBEB8A65D19`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `001FFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `1989F31D645F50CCA2488D0045D5B1F4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `003FFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `A99B07BB96B191DDFB1330A3FBA8ADA4`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `007FFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `DF3C2CC4042B858F574D157F28E28BDE`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `F73DA52FC7D16CC854992E61FF0F1935`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `A8B79D01C00FA50E84B3F08CCA3DA7BD`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `03FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `819825DADE0149E736FE6E8F0E976E3C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `07FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `66BB57E0AC2A7ADB3A515037C8D883E6`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `F26F0F03936B9823807557B0D25ACA2B`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `E924D5BA3488549C8B899DF4C875472A`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `6D3F517AD1C3CE3D8DBC2D52C791A98C`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `AE376DB7C01D439B3EE9B1ECCDA5254E`,
	},
	{
		Key:    `00000000000000000000000000000000`,
		Plain:  `FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`,
		Secure: `845F78166A0BE9C9450C09C36F8A38BB`,
	},
	{
		Key:    `0000000100000001FFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `8DA44D65BF9B7E3DDFBA3894A9434A15`,
	},
	{
		Key:    `0000000300000003FFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `E66196AF9B042069F300EA0433CF5F36`,
	},
	{
		Key:    `0000000700000007FFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `BE1AC5A85E2431016310370377B0F123`,
	},
	{
		Key:    `0000000F0000000FFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `EF9B324ABFB15408FAA5A440E9317094`,
	},
	{
		Key:    `0000001F0000001FFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `C0D7DDE25CFDF56453B28659B7E2E4B8`,
	},
	{
		Key:    `0000003F0000003FFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `35F95AAE8381C987B666172A4168BCB4`,
	},
	{
		Key:    `0000007F0000007FFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `2044A591069160B13CD7E7F6F9DC4797`,
	},
	{
		Key:    `000000FF000000FFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `AD67ECE1D0C4C7E0F8CAE13D24BE491B`,
	},
	{
		Key:    `000001FF000001FFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `A6233A68300B764D8D467F5EEFC3F930`,
	},
	{
		Key:    `000003FF000003FFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `AA9B1C1840FC14A68CB543E035554EA1`,
	},
	{
		Key:    `000007FF000007FFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `A1EA935E9601058C8988C6B67229EC99`,
	},
	{
		Key:    `00000FFF00000FFFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `0C13BEF31F1CD2CB59711927AF66D6E9`,
	},
	{
		Key:    `00001FFF00001FFFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `2239D0A88230EF7F83B1D8F22DD0948A`,
	},
	{
		Key:    `00003FFF00003FFFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `F16E2A6559D053FD24BDFCAA7769034A`,
	},
	{
		Key:    `00007FFF00007FFFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `98727FE965B7AC32F6DCC345FDC34784`,
	},
	{
		Key:    `0000FFFF0000FFFFFFFFFFFFFFFFFFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `66EC7A98C5A93F66CDCA64C4AECAC157`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000000100000001`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `07A6F5001097358DD1040735625C5DC2`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000000300000003`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `60716B59BB955915C93A9CEF9F6BD762`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000000700000007`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `EB9303B4280B740963C0AEFB45752AD6`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000000F0000000F`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `FA73DCB9A1E5D3E64B7BD81AAAB1F268`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000001F0000001F`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B4F125041E3C2A4D2DF0B20AA353590F`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000003F0000003F`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `166CFE9F67454669413EEF5AE8287B91`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000007F0000007F`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `69B34E9E94E65A33C81BA28EE11CC35C`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF000000FF000000FF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `4FAF931A13F186010F775B34747289AE`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF000001FF000001FF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `6C6E3B46FBE128D93DA07227EDFB9DEF`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF000003FF000003FF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `73A96448F9D7811DA2E90941F33B20F8`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF000007FF000007FF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `558D7F4C0EE48AC5503EF3E711126DDF`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF00000FFF00000FFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `ED4175EAA0EE97CB9E8743C1C165B421`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF00001FFF00001FFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `8CCF942694002B6FA913F55A7E24932C`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF00003FFF00003FFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `6019A32956E014305B823BE5F11916CE`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF00007FFF00007FFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `B00F2082FA636566C3FFB084167C1261`,
	},
	{
		Key:    `FFFFFFFFFFFFFFFF0000FFFF0000FFFF`,
		Plain:  `00000000000000000000000000000000`,
		Secure: `DA8B0D7E841CDEF38E8806C13FF63D8B`,
	},
}
