package lsh256

import (
	"testing"

	. "github.com/RyuaNerin/go-krypto/testingutil"
)

func Test_LSH256_Go(t *testing.T) { HT(t, newContextGo(Size), testCases256, false) }

var testCases256 = []HashTestCase{
	// 암호알고리즘 검증기준 V3.0
	// 테스트 벡터
	// LSH(256-256)ShortMsg.txt
	{
		Msg: ``,
		MD:  `F3CD416A03818217726CB47F4E4D2881C9C29FD445C18B66FB19DEA1A81007C1`,
	},
	{
		Msg: `5A`,
		MD:  `7F5DCDBFE357041971CB978EC659A898AF203C0AA012F238ABED9C2E70C39DD1`,
	},
	{
		Msg: `693C`,
		MD:  `72C817828D19E733D76C48A6F09C4B398B82196A149E0850AD6B99D4C3FA945F`,
	},
	{
		Msg: `AFBFC4`,
		MD:  `B59C7CBB30D91E21948B30C9C80329FB7D40049A6FF5F09A3BEC675E33568FC2`,
	},
	{
		Msg: `C58532D1`,
		MD:  `EF72980DF84AD5326C30FE6E41C333901391ED50FE99F159F3FD1E33AE4EBB96`,
	},
	{
		Msg: `AFD45576D5`,
		MD:  `9E32DC6E43FAD67D6A1761D1D929AD661FC82FA36FA18110A6805B39EBF73D98`,
	},
	{
		Msg: `64E353E86C44`,
		MD:  `2D1BB00D19404422DB0DA4220FA258B2C85764A753194EF602DDF6A45AD4AA2E`,
	},
	{
		Msg: `02A9FC5E9D5C98`,
		MD:  `7C5302ADB494E1D372CA19D451EBCDE66A0B9831098CA85A48702C5A2B819E01`,
	},
	{
		Msg: `AD8179259652A40A`,
		MD:  `691FF9CDEF0BE984C9490B85061E9011EBFB54812CF865C1EB14B2B4D464BC11`,
	},
	{
		Msg: `F840DF8F0588FF8850`,
		MD:  `21B9A21280F74105C734DC7D04CDF8F1D1D441878714BB9BFE3A7C9BE9D4146A`,
	},
	{
		Msg: `3D5C382648CD7022AED4`,
		MD:  `0F932F0A5C2D628507ED880BD20715E08C3B06B1256ABEC71CD777D9241228DD`,
	},
	{
		Msg: `F7FC11152EA27A0DF48A55`,
		MD:  `F4174F78FD9B7356EDDD85DEBCED58F8E1199E088D7FD91A70D529BF0B6119F6`,
	},
	{
		Msg: `00C846BD332D59FA5C534948`,
		MD:  `45530F9472C9EBC08137D73536A8656AD2D2C0C25420355887F16F973AC3CA77`,
	},
	{
		Msg: `57F3F5DA1A7B291FEB6B7044D1`,
		MD:  `9A40F3720AB664CFA869290C7DCCFA37495B446F01C2D30B79F5E947E96F50DD`,
	},
	{
		Msg: `FF29FA480EDD793832D0C52829D6`,
		MD:  `50028C93A985EB41D734E8C9A047CB55C313E20E9B54D8F7F48324C5B6A934B4`,
	},
	{
		Msg: `30AF7B68202C5DF712B7D680DF5820`,
		MD:  `2EBB2CAD43F119634FA35969057DD0CFF0DF7F26A26B6E26810A79243D377A66`,
	},
	{
		Msg: `613E76516B8B923B38C73590BFDF264A`,
		MD:  `A3EC03F36B102B972303516C268D73CB989B7277F20349BBB91A0358884136AA`,
	},
	{
		Msg: `2531E149D74C453ED5F78DC9517681DE78`,
		MD:  `FF5A9111E428737D4D3ABEAE6E13D0ECFD42B89BC342101F92FFFE57F2B57DC6`,
	},
	{
		Msg: `08ED6D90B453424345DD2A3A949557EE05D8`,
		MD:  `FACDCEC006D2A058B2A09086F984DD06A879935FCD422D347D8A2FB1488B9F41`,
	},
	{
		Msg: `05BB54085E3792829BC16AD06D5DA91F9BBFFE`,
		MD:  `2CC094A576EB99A5C4B5FF56CA505763C6B9F470596C6187B71378D525175692`,
	},
	{
		Msg: `F3ECA5FD19476A6ADE19D8B57EF73FC1DF85C3E9`,
		MD:  `AB51464A7F3149C63BC07CC19D668C2D0F421615A85C88525693D78AC2E8193A`,
	},
	{
		Msg: `7F96FA45AB94B729F32320EFDA0107F7D23077EB67`,
		MD:  `01BF0704D20F26BC7CE3136A050A6550D24C0DBB147AD400DBC85A4D0F23584F`,
	},
	{
		Msg: `588C48BA67881A317A24076D00248E7C1C2E91504CDC`,
		MD:  `47862511A719B1E74EC4692DC8DDA83C5EAF53E4F40E5824673BB77E4114CC0A`,
	},
	{
		Msg: `117049103AEE2E889F99DBB2C6739CC1E7660600F13222`,
		MD:  `F14E23B3D4D2B55BF0EA6EB18C9F3192E8D371BC79F906D4D20A6ACE983422F2`,
	},
	{
		Msg: `66CEAACE6BE8205F1BEC1E8B69CE5787C50DDC6CF13893DF`,
		MD:  `8F46E10E2F1CA957F896530E8338B4C6702AEA636F539E1A0F695386AEFE315F`,
	},
	{
		Msg: `10B1E6F100B1B844E98C9D7016364AA404064A91B9BBF38357`,
		MD:  `2E49C03E76E75B5D81E94E588F45BF4671BA8ECA3C3FE8CE7499FCA897C80729`,
	},
	{
		Msg: `BD99D1F1A501789DFE1803CB4DE5AED550BDCAD784B663C0E95E`,
		MD:  `9EE836B3DDBA0A1061FA35799935ABF71DC44689D3DF72B6EC4F96E455A82B82`,
	},
	{
		Msg: `21A909AC6C663736D35B15A5AF915966030AD0A3A751A372A654D9`,
		MD:  `1712C31CCFA8A81D2C57855AFBFA5CD1B70D4B1F18003C68BA79EC5D2B272146`,
	},
	{
		Msg: `4CE4761DC004E1C1688819041D3AD79067CB144737BB643FD978850B`,
		MD:  `EA17F1F2FD487AD2A5491EBF2E6AC30DA22988F2A93816E8C3EA5B7EAF010269`,
	},
	{
		Msg: `24CA383DF46E4F28DA0208EB112199ABB4CE0C1F2F164D52F4F182A46F`,
		MD:  `FBD3F12CEDA7764022501340BB49B9D211C23D397AA854B529EDD865EC9EF385`,
	},
	{
		Msg: `6A9A123BFF29B8964B739412E2C68D8E7D0EB46557A3C9E5D66501036E0B`,
		MD:  `9F85BD89BFA04E91D521F1F35B6DA3B47A4CB509E9F01BEA664C6AE2B95A4C7F`,
	},
	{
		Msg: `064CC3B8401F0DCF531AC109E9D26EC5B1C7E49FA8979C4397715484DCCDFA`,
		MD:  `FFAAE86134FC68FB4BC2AE475511279E0E03E701CDC37D5849A3AF68FA74839C`,
	},
	{
		Msg: `A86E5ACB061C38BCD41568193AAD816CAA5EA4A0740DB166958FBBD8D06CBA5C`,
		MD:  `7A1025FCA63077A415CB2A11C7BEB9A5F9FBC91DC376A386BE32DF1DDEAB544A`,
	},
	{
		Msg: `01E380CCAD2EEFD4DD07689B2D6A86A5209D0DF8B66E1D7D41BCACAA7438335EBD`,
		MD:  `9441201BF49E04BC0B8D9B6576036C00108FFB37EB3BB983DBAADC112C2043DD`,
	},
	{
		Msg: `6DA60CD5FEAA1C952F038E0BA74BE1CD75EA338D5042A9EE232A1196C99EC68414B4`,
		MD:  `2DD99B225DBFE5D3943648489621CA593A377521A3BE7EAC7224DE3DC75803EA`,
	},
	{
		Msg: `D4A415D25DA1971F3CFBED37E759EE239A203D88B244CE1E575FA5645D00E1936C45FA`,
		MD:  `84EC1189536672A314E4DBD3FEBD4BABB89471874D0A50DB8BB1405EDDD71D81`,
	},
	{
		Msg: `D05FAC47A5EB26C3B2816D10A07598BC575FEDC471C2F656EEEBD7DB291CDE350AEA5C45`,
		MD:  `E02FE3FE76E9F4FD4C71742459C99637566D10BBD57CE3A2B2C58ED0F395319C`,
	},
	{
		Msg: `7EE4B3B12475B1E44A0C1C5617798E0D099B8364A338B10B5F2C984FEAC6599B906FCB3FF5`,
		MD:  `9C6F31232F277379EB204F3C0CF592E292CCE780C5104C8CAF11333E8DF3CA09`,
	},
	{
		Msg: `8CCC7107A6D59130C8D2B2D0293055EA0B37771EE39ACFD7BA4C51C5D396F68E8E67BFC08CEB`,
		MD:  `7939ADA636725DC46F19F11D40DE5BDE0A1F92DC6E0E774934BF1B66826FCEF8`,
	},
	{
		Msg: `F80B145EBAADB623B32D2E10294C8B0D3D3990DD6D045A3B5D554329D75E0785100E25DD2D8788`,
		MD:  `58E816D88E08C0C1DBE68B92B6BA92BB3A54382F79D14296E731F5834E42DD69`,
	},
	{
		Msg: `8405D367E529D26A1822770EB3F9344AE3FDB373BBC0C75DCB63A2C3F51348E87CD4447F5E7D99C8`,
		MD:  `C68A496F368F69D59EDFD02E63BBDC592E9A36D5CA344C3FB77F9C148ED6FC58`,
	},
	{
		Msg: `E404FBFD77FC3FECB55C11E5C482E5D6E67830B54E75D71E33A7E73E222F88EF986E2C24D07F0C88DA`,
		MD:  `028825055DDB478B100CBDC29F3496F62F57220D2FF1FFDF3AE7C38D82EC9ECB`,
	},
	{
		Msg: `04D562ED269EBCFED71EC4915A1C72D006266BDA4786265DE5724D67898366797D69B2DEB3DC553A09D5`,
		MD:  `D9B77E4CA7FFE9D2CF1A815C853DC9526410C85C2A114E6B9638B6D70F0E5E2F`,
	},
	{
		Msg: `B866CBFF2C7B7968D02DCB7DB45BF3353EE3228CCDBB4D339D3E3186EBC395BE5974CF68935E891633C4C5`,
		MD:  `A4AF76C3A4D46D7C4AE358270030F622BC8225EFF634127306469DE2F3C2EA98`,
	},
	{
		Msg: `54E2B91FDDC0EE520F07C8FE9DEBB5A984C4C103406E646E3FF09EBD965D463CF3F8A0F1D29B2D22F45AC718`,
		MD:  `91BADE1E1A2D98AB17D710C23CE10671C0DFAC3EAE2C47A566804918F4A49AC8`,
	},
	{
		Msg: `0D63ED22BE285BDCC2C1DDB06C8B669E51165C39423E37F27D974A99230FE021EDCE8BA41B61B9A6AE18889ED7`,
		MD:  `F3637A856BCA2358B4867BE7C34C6CE95FC475E7BE79CF233118F0B26B526A2A`,
	},
	{
		Msg: `ACDCB3AF25AEC328FBA7A9447CEDC0856D3DA3A4F43A3786B604EC6B3C0D3F699B28841954E3E29079179F9011BD`,
		MD:  `22AEA8B71E1DBD4B4BB364C2777E7350FB0511B96687F68A4646CBC8CE0B5F5C`,
	},
	{
		Msg: `9D00BE03259A9DA0E4E46E643361F5139C089F7399F569A49E5FFEB05ACA1F633B7D485D30379165B21ED20005F80A`,
		MD:  `D124419594865CC02C1747697FEC2EDA46956783D79B816F67A594DF3E4B25B8`,
	},
	{
		Msg: `42BB9D02E57BAC2D98007ED39964586E869FE555714BE39732AA1C2A01E9984544A6110AA5B1A768680FD540FB45A4C6`,
		MD:  `103B51F6451359D12F0CADA3C4817B745A1659BBE79570461FF8820F6E795EB5`,
	},
	{
		Msg: `CFD31E2B67FE0679AFD2C6BE8E68A96CBE654A00DCC34DE2BBD44C8CFE8314EFE6DE121268D108FD833404D5D00993485B`,
		MD:  `8DBFF1B741A349DD41F52F0FB47405229DF1B90EB63CFAF1C41D6A2CDE5AFF08`,
	},
	{
		Msg: `FE322CB13A05623A55CC01C9A812B072C8A7549C4D51EC56D8BFB8EF54BEA1C07BB0D54D9C39B28D9C3BE5853C05D31C99C7`,
		MD:  `3F3465795DBE17C8B613D95D8DB10224298922CAC82CFC44533B5B1AFF056566`,
	},
	{
		Msg: `382AF9490DD2C4DDB012D1933F98E3AB44D77450985BB1838159C5551E8B5A59053A5D592A9D024803610978E9893F0DC27847`,
		MD:  `8672DF124CD963BA401D09EE4427E7282FF09071EFCD70D5C6E6004A257C9B5F`,
	},
	{
		Msg: `B2DD4C9A8A1D6ECD695A67134504F602E7095F9AE6F90583D6693B96BD95589FC64909613A01ECDBAB4F5ACA737D846F5D73C192`,
		MD:  `8722358ECB256AFF66EAADFCE597DB16F3269D87549008AD0166F02A09EFF702`,
	},
	{
		Msg: `A1DC26C7168558F0A3059E03F6A33A0E7974FA7882C40A89CBB70C2C21306AF460F2EB5E9E068556A75C73FCAAB587A731BFC02A56`,
		MD:  `3ABE369BD185A0CC6AB59BDE76104B50A2EF125D2864788CA2CFA74AF08322DA`,
	},
	{
		Msg: `821F5734FCB7C29D0804C80B857DCFD8B36CCEB1E0FE1148459636492B677C9052423D9992ACB837C3341489DB04A22A56B563A9DB68`,
		MD:  `9F7C52CDE5D1300E606BFCFB25484E3ED9479F158217111EC03F9744B3A6BE40`,
	},
	{
		Msg: `CA391C12C2A9FE17F396FA1B41DFC214965928A4E2C43CA760FEB5081A7793D8AFE502A1543980F719D142516EEFD417E56A01FC39A3F2`,
		MD:  `F77409E041904E0ED5D39400894EB2354803B3CB05C8E1A24B778A841CA142FF`,
	},
	{
		Msg: `8E90BB9A0BFD03517749B4BEEBF3E1FB3075A9F48DC9B110DC2CD214E3220E44E2BAC9CB6913355C6C1F0F1A99A789B5E98F87D6765C0DBE`,
		MD:  `5315E695461B9F08C225886C0AA946842AAABE2F8EB22236DABE6CF9BA24675B`,
	},
	{
		Msg: `8CF3F82A02A3ED79CB669010C864F6FEE0B95CDC537819F74BF0D1B2B52EEC254EC3460A01C728DA701D7833D1F92C7DFBA004CDC06F79097E`,
		MD:  `686B756EE4A2E6DC041D568BE157DCDFC75C3CD2C51D6ED8A81143546A4170A8`,
	},
	{
		Msg: `1FF14EF147348E4B22237E8793A8AFFA66E9FA2BD9C122E45B70B589939133A9A44D72D3B0CF41AB9BF74C3C11534090CA4B2F7E0C9B35C47207`,
		MD:  `3811E689A4D22AA98218E0AAB98DD310E641268CD6EB1C62D1EB2F28F3FDB80E`,
	},
	{
		Msg: `5C56ED49BDF5084FE80270A38695B32B3DCA4812D45BD00698749BA4FEC26E3089CE838AEE89D25D77278F9AA531F42519D640CEB4830082FEB38D`,
		MD:  `98B8EC0A9FD27798AAA96A81393AF7567303C8E82A40AC00D2143CEFDAB4618C`,
	},
	{
		Msg: `C2C407D65506995F3A108E8AE1E6EA169BE5A8E54EA80B026D292FE22728E3EADFD1BB32DD99608943692BB942FEF20C8D90220E0EB10E95D97987FF`,
		MD:  `B5CA57C2FC7BFC873205C450E9F174594727ECE156E922D4DFB7BE19362D3A6D`,
	},
	{
		Msg: `5EE0ADBBE26B040B300BCB078B4289E6E887E8795AECC6F8CEFE8A01F8ECB4DE8DECB907B3DB0DE7B62381E3FCE2CF993B497365E942BD2AF455254FB4`,
		MD:  `33BFD97170E4322D87601D4ABA3FCACB69CC07786A81911EB39B6049BA8BFD92`,
	},
	{
		Msg: `CCDE071E15AED63A69FC51D7AB2B257D93265AA30B3CD922D92F2CEA7F8786F35F2C03772334305F5CFACEFB6970D3F59BB6B28897EBDCEDFB4387164459`,
		MD:  `FA1C5A5EA13A94D3423F9F665590D2E9D561002726A2FE4E3770117418EF1A1F`,
	},
	{
		Msg: `AE9FEFC75902DEF51E0AFE2EE88A27F9F74E43A62852B3277732EE21C2E26B43D09FC2D791250E96A72395C126ECAB6D986FD31F45692898B1D9FBF265B5BE`,
		MD:  `C7E8544213D6DB5149C5AB75A6B7BB3A6D5225CE1FFBA183946ED734B6627691`,
	},
	{
		Msg: `ECF2DB2083344A7AC2D4926ADA1089BD75BF4093488F35BDE792CA6ADEF9F603FD1F3B53F3C9D8EB1FAC3FC5A22FC927B973A9CA4A901F3CBC22EE830C050F03`,
		MD:  `F18B29ADD26B5B2F0E34E9828A24A25C8C9A0D17489CE2B4886F594853E35B06`,
	},
	{
		Msg: `1D5CA7A250D83729553712ED410A15B0E0BC92C24CD08D6BBE6A2B8EA5116786F5B30754C2E9CEFC0D7DF42CDE2AD31F93A2E3C177D6E81496716AD2D0A566542D`,
		MD:  `789A0961278632A735B49656D6188E31BF792926282BB1E055B9D742591482AA`,
	},
	{
		Msg: `2B43B27085072CEF87CBC8A1156D3D792E8410F22A99FDEC3F9498B64A74346CCC430483334980A25C810AB35549D23B675DF7E65406EF67DD8F287E87F9EF5A9FF0`,
		MD:  `D306D949EB6D48094A0AA2CA21D488BC7A2210ED357DEB30680258167CA31CC0`,
	},
	{
		Msg: `92227CFDC1CF04FC3A3C72D1A6805657BD7BDC99B1AF5245CA51E55D4DA0AB30A376E789B71D8826A063F26B80F10158ACD84BB90D6DA69CC657EB0816E6F9EC80A7FC`,
		MD:  `E43D782DA5AA23DD79BB714D9CCCA88836C300C4287FA87EF79004584EB40F2C`,
	},
	{
		Msg: `562A35EFE584515F81293572490A2629CDC949E27E63B7216312D38F0D72DC43CCE2BE41232D2E407A6AF5B113208CFC3FAEEA0DD2EA9A0FD409107779BF3D4553A66186`,
		MD:  `9986A80A727013E0F6BAEECF77E030B36688E25237E40B717DFAB4290CF675C9`,
	},
	{
		Msg: `C7E4211EBE09D9C49C36334E7684F92F80A3273EC4C245B91AAFF3895440C6C7B9F15280B5CE88B5CE9443A6DA131D516FD687359C06195BD68C26973576940CE19CEAEEC7`,
		MD:  `C94C1ACECAE16F9A5FFE06C1EF7138E9B0DFD630F9464516189AEACB28675ADE`,
	},
	{
		Msg: `83AB8C9081E38DD322FF7989CBE17CC29D3467979CFDF86011735B272D11F05CEEE94FC9473B8AC97B46B2B5CA5E9CA04F3FE44429A1333BD90DBE73EBEE0FDCD9EB08632C74`,
		MD:  `A1FECB7972A9139A5037E7A20DD81DE4EE8C9BF3266F8589ADB68886B71E669D`,
	},
	{
		Msg: `2524E49B01DB1A261C9CE4E2DBC5793AD03095073CAD1FAAD2AF0A59BE4579B6E64C26A670E40460842023C45302D4EF58A60F0CDDBD116CC23D49DA7A57752856AA090E06B6D0`,
		MD:  `27C42146AF9C7584767C817D8F48A36AE9B07D53DA40DEDAC3D71E40DE3A6093`,
	},
	{
		Msg: `355D93D58E54E284E852050DC2AA8B095D65EE601643816AF3FCA4547CCB3AC6CC2D5DC2E7E750D04E46B458D1AAFF4FA5462B15B621AB9E3E60A3D197C8582D4E7FE6B2781231FF`,
		MD:  `F81322298AC69EB3236912775FD205D951179CB901DDC350A943897960650204`,
	},
	{
		Msg: `49BC1DB8E2E6022A6BEA5A32621CBD5A320B10460804F94FE2D8956DE9DF6E5A003DA7F49D647AE4D7DBE57DC13D9FE6F784517C76870372B1C2D4BAE156EF299C50BAD6A92E51C313`,
		MD:  `AF0C2273C23B3E03CA36D661D864C1E2543F7185E7D51FD819948E42F7D49FA0`,
	},
	{
		Msg: `A032BF7C8B3A8AB453D03B8A5A60DFE2C40B63087E9CB4447EBE24AF4C6D1044C1ABBF3A0E748E9201F34AEC84F6DE9E67A02E075F5D42214E1BA756C4B81295C6B647A4CF3639B81524`,
		MD:  `E0BB25382BD0031D3EC31E0E4EF401F2F459D611A64DC61670B55CF932A935C2`,
	},
	{
		Msg: `73430DAFB60D31FCE486011B1C5B483A2747A018132C38BD1D62D569D54E17CA7C37FE71B156AC874D2C6DBC8B19F4816C29E87759319F270A1B29F3A87E904C2E725E2615A260A930A2AA`,
		MD:  `81DA96F1898AA00F31456D77FBBFE6E4943858E590BB7B04CCF5A62226A92E25`,
	},
	{
		Msg: `2AD583DB593DA5A2BC2DCD815CD677D1BEFB6E0714519935B52B18D4CB5690C04B42C2842B6363D8CFD72768C44920D7B015460489AD578C063BE19053889CB8091CFE775DA70A91989F69DB`,
		MD:  `E4ADF9BC4E65FC9DCC09E2C7739F261353158376A4025CC8DE7E82D286BBCEBD`,
	},
	{
		Msg: `C72BBB42B3402F0F1309516A018D241CAB8C71994BA95E3D59CF42D6FECDB863CD43F31BAEC251B11A2F32D78E680A01392AB56BF8330B5D85A3B5AEECA83F873E5A6D6988E4D37106E5A0DB95`,
		MD:  `8043D0B0A975F0AB05EFDC67721BCF051D299462B8172DEDE26F5C46060965E1`,
	},
	{
		Msg: `16BBC6672C1136FA002D5F4BED0921477923CF8238D4342BEEDDD840F4437FFEDD5EF39C582C033858EBCC1273E637F8888FC301A5CE857E28996D2E24C5CAB1061B103F78AFD99A856D2B41FEF5`,
		MD:  `58C00926B6A91EDC82A280972443002288A4DD4EF5453E10A69A645F121ED36D`,
	},
	{
		Msg: `F9E71DA9EC2369036369D92DB8393CCBC596D81BA6BA5241C108CEDC044256DD6EFC8099D1514B1DD6D44BC141C9060076969D863C74AF8BC26A7A60E07C43479E20F888FAF115C1E6A7C47E914748`,
		MD:  `994C764E94E9C9396BF067C7310B4D8139993FC68769EE0646CB79BAE97CC876`,
	},
	{
		Msg: `09E7979B845BD55634E4A8D22045578DDB9239CF9EF06D8D999CCA8D8B2F19CB9120821C3B8A4511E52FAB1E45BE3BFB4051304B023A2B7C4DFF26EE11A3A9573E274066D6512CA646AC59A6F6D09057`,
		MD:  `C9C761D3CCE5CAD6026942075B8A99217611EE0CDADFA7DDB16C257BB01771F7`,
	},
	{
		Msg: `1FE3EEB5DD6095A98517A12E38765EB4B806B8C17B38F2DEFE629BB15C05BE671F515A5556EB11DEE8ED2497272F9FA9A8C5D17753C8FF81B65A5A889EE42DF4E9A25B72966209E11E88986F0756D6FFB6`,
		MD:  `A26340493E6642DA08CDF83E36174E432E5C4A5048634D319D3DC290B16EBB5A`,
	},
	{
		Msg: `6A75CF8588BC57159A7C74E6F546C524DEDB55C44DE47BD0F1E3FFFF736A7E89FF8B1039018F27C42CEABB59C16FA4790EC14709350A74A62EEF8F380FC3EECA8BFA8399B9E1BB6A178DA521CF11579039F5`,
		MD:  `1F34CC14C4FFE07716DD93BB73681D5CC02712C2D3E8D8C8CD1DBBFEBAE99EAD`,
	},
	{
		Msg: `8E8DA0E3944749A267DAB149172017AC70CA3A1BD9B0644C7C66A795A710675E727719D7A35C49E6CE0CE264C134AA881FF70CA34A3E1A0E864FD2615CA2A0E63DEF254E688C37A20EF6297CB3AE4C76D746B5`,
		MD:  `8A813BA33B3830E86E4F2385EC180AF089BE8778E7102801F7C905A74E52E709`,
	},
	{
		Msg: `E3D6BB41BD0D05D7DF3EEDED74351F4EB0AC801ABE6DC10EF9B635055EE1DFBF4144D0E24057B03E76149A55F1A42D1B6537DFFA21349519439271DE2885EBB2275E24047FAC76908DA850E143D04114D5D152E0`,
		MD:  `B69B0A72AB45BE753941DBC97384F42A9D45AFF55494CDA429A4D8D00B99151C`,
	},
	{
		Msg: `B5AA99D3C20F0BDEBD0B9214FCEA60AF625682548EC0B898530284C2684281D26119F6C3DEF4AB0E03BBFB6F28B12BEBD19CFA12E025798209811DAC875113DB0DC7D372000BF874E3DD4F4A137B2EC1B4C50753EB`,
		MD:  `62208B342C7CF4C02198E7BC8B97FA69D7E86DB25DFCDD2E63F4128A6FEF38D9`,
	},
	{
		Msg: `9F3CA74D97355B98C8B100A1FD59668E1A9B1A45B8B52E25101038F31AD09D17F6A910088AF455056D024B0603B029F2EA3B08129556A2C6377A5976E37E7BCCF167C420A381243409AFC32221FF439243814530BC5A`,
		MD:  `F902B0838D41A8951AFC3656E6B598243DC710626ACEE05C434CDEC2785E9476`,
	},
	{
		Msg: `2E5EBA142B2568387272591FCB7F2A054F8F635F70DE9F6E3AD8ED15D98975E85A555F85B6E9F0270C71F6782E0B67A75F7C45BB5B23C427982BAF9ECC040828AF13EF7602A8C472D031901BF4AEA190DB65756A8BD7DA`,
		MD:  `5F3546EC39EFD7A996CE53012F88B42D01B46F41FB3C2CCF72403046C406F3B0`,
	},
	{
		Msg: `4FE4F252C52688BC43B400318E184C2D43461BEFAF254F22E74BB1BC1508E32711A7979F31F0FFDB61F284EF76FFBB71BFF795DADDCAD630D538C8E264DB30F49368815B8A3250ABA9EF08EFD5BFDFD8106A081B4DE92A14`,
		MD:  `F9D6C0364801FD7A38415869F34E43CED048855798043E84041C47557295A967`,
	},
	{
		Msg: `822912AFFE0BA588FDAF18517AF263754435F2FA54619EAC5E27DD78A68D351615A6F7EB8CA0292D7805C448A73A20805A4CACD89BEBE233D0B77ABA91BBDE7DB737DF5DB5A3A6292CF0ED07B9116F1612BA9C51516892B78F`,
		MD:  `2D265285D4C9922D7C65ECFC9ABCC9BF75C9B6BEE26FD1DFE0801253D8660318`,
	},
	{
		Msg: `A149A9B57D0D938F6532CE221B03C540FAFB575980387D94092C2D1EDD967EC26DB00CBBDAAA9EC6B758067B102849DCAB56002F8D63FE09C0C333AC54D6D092A419AC4ADB9EE783B6B9CD678BFDFD2A8A66CAE243D8E19968C8`,
		MD:  `35B69BC18DC24E2182281E6BF81BCF6A6268253CE987377B57BF8700BDAD4B89`,
	},
	{
		Msg: `2CA0D439F727CDEB520B86CFB6EFE8BDEB8E9269284F1F947D1BA77F6E6B88A5E1A566A93AA8708EC4A5E5265332D7CD279930571476F246E081C340EDA8A8C11761110EE03E166ACBC1008B05730B4F4A306AAC07C58CE5BCBD9E`,
		MD:  `60A476DD7D1D5CC0E502B20330E20117550A7630BF7765B7E809439B514E72F5`,
	},
	{
		Msg: `04A5775184997CDD4E321046E72E07DDE715DA0BAD1CFA4A514A6474DAD97640B1DF0109562E9FD2527847886DB71E8620CB905F4D65A0D72E257C328CFF55B231E36D5B9F9D72F902FA0A64642CAF03206FBD8F99EDC9192BDF354F`,
		MD:  `B17B73689F31882C5862B1800AAF9280DDB9055BA372F58ADD436D84CAEC36FA`,
	},
	{
		Msg: `62F601FB6AF9E4CCC7AF9C5946FF82633456615ACD498823F77C1E1931C2B652C685232346698EE6362CC6FA07D1D60C45FDDD9511A11AA4D1D947DE261428C31F766333AC7A923055DC97E52BB62FE7FFE9CBBCD3BE9B9A8ECDEB70F4`,
		MD:  `253E1C3B155DACF3474AF5311FEAF2B07010762917FEFBF6D3F8D07D475A1736`,
	},
	{
		Msg: `1EB22BD48F6832837519328A86B8219CF7B57DCBE1D1D5FBE9328B9620B2D881287D104F99DB28B5A24A753EF4D0B640DB8F550FAE9AEA706A0701519A461DB77AB1FDF75B631D29494E80DAEC88AD01ACE6F83418CC021C89438327024C`,
		MD:  `F79473FC539674A16420683FFCBF790CD3DF705FC04762DC08819EC3D1B45514`,
	},
	{
		Msg: `477C5B7577349FA91AC7B24CDBC437880DC6B0E1ECA3E71026E68E0E9A80226F5FBA15F1086E2B552677256D0F669D77DE7631A9BBC3E30D5DB6C08FCCC4C0556C58EE4E370DC80E4A21E9D9A327E5FD29FB7565ECFBD46CAF7468EFA45BA5`,
		MD:  `B8DBF54B749AD4AA61133D88EA20D1F129B0AA497A6429D997A66B2DE956BD3A`,
	},
	{
		Msg: `A6763E2FB48C793E7C8DAD47B71F6ED9C3F53CC53E08B581D42408CE55A7AD990B86541F4A901B5548F703872F0CA53F714A21685C9658B3EEC55B3AEA5F22642BA2820839CA59D51A900CF16BF133E0AF3092A8E61E4C32629953F4D65323CD`,
		MD:  `E1E524B7D861AD0A50D02B5F228A28A0460B0EE676E96EF43E1414E807A8EDCD`,
	},
	{
		Msg: `B0D5B030A8C39C859961AECC9056808423B077CB0529F8C6D7A8D943400B169D3F24C6E0C0014D2E6E71D262C9C3F4F278D1B81AE03EC637F5FBD6715010659C8399AC60A90ED29841C861F93CC1F7F4549C3EDDA0E51D4C659930C430E9769221`,
		MD:  `BFF8AC30BD3410EB86810D82F7B0BB0B53A75CB0B965F8E9CFADFA7A5B988584`,
	},
	{
		Msg: `3B4AF88A72958BB96F41DA11D7F1506017EF5C6CA665CECF8AD185051EB347C41288F08BB3FCCF7E6D5C4D718E4ACF434CB4DE6D0A0786DAD79FFBFA38838312274F8FD65B712B39CA9899846D6A39A54376ADCA9869392F87847034BE4D0DB38185`,
		MD:  `AB988A3A1F6CDC198BC49661E478E2BF36CCF28373BF835606FCC04477D625EB`,
	},
	{
		Msg: `1F90F25B69928FDF0871FCB95630031B0FACD9F4B276A2672ADE387BAE502814E6E4FE224F30C318E061C3A0EE8A948C4D3B5310B856312F721E2E742E9125E22DF8092DA6BC6E2B6A7B4B85C85BFEB2AE9E902612773BC6901C29A0551D7FD93BF49A`,
		MD:  `D2A3BAAC237F5E8AED28C373387DA4486606E6F6E42B0CDCEC8F4E8FF4BE59C6`,
	},
	{
		Msg: `E635DD98FFEEE81BD1AD4C49747CFE189C8982C813CC02B167EA9D3EE05A013AB4C63C4A7EB4CFBBED65047EB9FD8271DCE51FE1FE6EF54C31B7B86BDED0476FFF1F808C8F0359B43FB82CA56B7B53254CEEF58C6FBD1F846682170DBDE6EDC430095264`,
		MD:  `AE46E82B87C9FAD0DBA23852F0B16F7D6236276730CD27AA48D73C11F0B9983F`,
	},
	{
		Msg: `1A25AD8D316E6CC4713C98BC566BADD34E02A8E06E14612C64E41BBEAD4C5ADC0860F5CE2A4E6BE5B3897ABC6126EEBE8386B8E8B1353FD0BB12FBE7DFB03FFB3FFD56009221D20F19A921AFD4E56154F78E752598AC5902AAB7B2C62EFDAAC9A843FAAAD0`,
		MD:  `0D53D10A48D787777A315E18DBD515E9D7B9FEF3A5A57ADD3E8B020ABB77DC99`,
	},
	{
		Msg: `AF484B8BE6B41C1971AE9D90650A1E894356C9191D6BE303FA424F2B7C09544EC076A0F1865C8C97927CA137529D5BEDC0DF2EF08A4CC7C470B094B1EEAA86731C041633D24086B60F7369D59C57652DEC9B3817477DF9DB289BA020E306C9A78A99B5391289`,
		MD:  `62AF8D3C1A1AB0CA9D21045B2A06139F672EC737DC59E9C4F6F81B583EC19FF5`,
	},
	{
		Msg: `92DEB23CFC508C5FC3AF8F33EF769C2701AEE7AC7C62A145A452D4DD86EB8E5E3877417F62926ADEFDDD714E5BF6D07268A7788B2F5D708EA0DCC07A90328DE801E985E3A4817EFB9C1E5995857D2D85B52CA2DFC1FB9A0943AE1715DBF4594E749D4611C9070F`,
		MD:  `76B9C45615498562F8A62EAE356DB94495265565520B8C9371E8DAEF3C40B228`,
	},
	{
		Msg: `DF27D911AF74F4F98970A72C503B2266E0182EA059701B38B91AF94CB1D90B66B126785F4630447DA5227FDF88A4BC1B38A4C525EB568D735179665E6ADD74CC9A22620CFFC2530102521C832C04B0023C71DB6C769534513D1B4D46E4DB2C87C2A2FFE084310BCD`,
		MD:  `196060FC9A90607B276583151D14B7D67EB97EB95715D0560C07AC57C5F5205C`,
	},
	{
		Msg: `C788E6E0E764E6631106D93FA036F61CA68814CAC8DE1ADF502EB7A27D8553281D4C45BD5ECB4686E2EB25641140BD1AB34625B3BEEB261E6C4EF21C91C7125C2C27A1E33C41B8B2D5A929A372BB70958DDFE6C86D0C388A408B181469FE32AF1BA0621B27CE243012`,
		MD:  `5E761634D303778D864F7455B162DF743B29823E1A4596342DF16F6CF6E3B56D`,
	},
	{
		Msg: `C74CC66BB0F7D35BDBC1D37E48386DF16C9153AD329A6A1340F02BC77B7249C2CDF593A3263A1D46C01D9A98A80E0DAA7AC5A1046B32D43B8DB3DBC64F8A420D438ED4C32520BA761861E76432A3B3913B1DC516B2E237BC1114F5C95354661AB27E4F72B593F4126AF7`,
		MD:  `0CE5F5C5C44643377883210342DB2987E434B99EE0E9F560F1DA41D8D580F51C`,
	},
	{
		Msg: `808AF0DDA6CE39FF898C0E92D9BD54793F7B8DB9FC12A22DCD59DF7B11433F462DB1EAA5308AFCD4B30E882A0467DD4B8381079F4B1C99031520AABE5A301EB6B88127E497ED0A050C432792FEE234F38786C9F09C612168F16E1F4517028FAFE7A4AF80241EEDF85CD4F3`,
		MD:  `1D0FB5816CCAA07924A08760D786EB0309138FEAB83DEE0AEA4C82C598F2E43F`,
	},
	{
		Msg: `B30F55E1D71430D9923709C1C427EB44F890E01A9984BF8A631F462D176B68F47381E86C61F127416F643B027F804023BF0CF348DC557DC173D08C50C900A833BB9AC55EF8E814096D6B1F6A01C415C483D562A0E023835A9A7E19E075E53CB774A4113F03BF9F377693DE83`,
		MD:  `238354CD98BB65B1DD0B961647034DD89E3E58B80CC02B71A5CDC1C413A26667`,
	},
	{
		Msg: `B2BB112D2A3156AB0EB57C60BE73146585E0CB8884E5F795EC3F6FD57126A0D5FB2CDD689E69F7EFCCD50ED8C9F5A66560BAFDAFD5A69DEA1A302DEA456A8CE642E1217DC5C46CF68349E1CCA24DF76A6E3B7C334D2E2BC5F55A807FDCA8CB41F0A08472898CDE489CDA9E4331`,
		MD:  `28E55EF3B9AEB280F5A3D35B71BD2573B5F37252F1C75FB3A6D525C27DB511B9`,
	},
	{
		Msg: `C32EFB172AED1CD6C7CAAFE584119B1D27846B6E32AE51D107746B7D50ED465E9E74A1C350FB7357B7A22F52788846F031E0ED235F4822847F57E9907375DC1128F0E2DA764C3FB747DACDA4FC8C883D0C075076EEFAEB8C3AAE4A3C0BDF0D32A7CB49C4628888E0BF9A1143D9C5`,
		MD:  `81C3EFB9641D73BC4D28B76618CAF608747302B391D97E8550826CE1658D4688`,
	},
	{
		Msg: `6AEC01201CD1A96BB6D081A214E8DFD2E1A9642C5BE9DD5856B9670C8498D7DCF3E65854D79B56448A389B5BD1FEB8F7615EA73CA2B2AE64B22A8655AD09BD792C1159CA0B3BBE2C70ACD20008E55EF6943AB822A5859F53A31B66A014FF55431B564CAB3F1BA9CC6E146CBA420599`,
		MD:  `535F5E542215E3144F210B01D816EAC0C45890D64B2CEF1F2F0E13B376FC9C90`,
	},
	{
		Msg: `768323DE07ECCB76CD32744D1342E365E3CA9E781EFCA2DFCC8F58B2B006C6311F3402621300EDFC7C661D84D6BE1BB60041CF264CC93827D3D861EC0BDC90ED9BB1C08C6D6851F90547FF53121011104387D29E5C2E454A89531562F24CCB968F0243A43EAC613513DD0001EFC12B3A`,
		MD:  `04BB63CF7E19884A472006DDBFAD1B86281A0A2A344AD2B37B491A0924732AB7`,
	},
	{
		Msg: `54A3902877B3311197075D5B8EDFDFF4A62D75F2AD5584774C3109D89458D3FEDAA0F00CC5E16541A5FC2221061C34CF863C3DBEE3EA2188584A031E2A5D70227E83D9916D3F47A55A3B4AFA838C32EFD736824F08F3A231E030D41CD859696FE7563500985640045FCBBE2F2DB5421C42`,
		MD:  `F4C6C26A1447F3A6B0821DAECABE9A5F798167DF310A0E55EE85C3635E509D9A`,
	},
	{
		Msg: `242EEE43F86E388CEC661AC7D328ADBD21EBA16DACB9255BB467072B21CA1D6E0D28E3BC82ED6503859F601E5314A4AAEAE3BC15CB18D1F7FF2CCCA65834BB05F958769C188782EE0369350661951B54BB0599B73BC6D0DBB3F4C4C5E0E6BEACF1A6EF0B43A455F46FABE3A78613DC639B39`,
		MD:  `157BF749EC950D10FCE54FB795C81302B6E1BAD68720FD40BBE1B56A211F467B`,
	},
	{
		Msg: `807C822CEAAED8A939B1434AEE0DFBFA95308AACA8BED04DB02849F6AD809367B88C2AB1E7197732719109C612D0223CB134750741F32E36E5B233A75753A9778A0EBBBC8EC39677B58EB8C4F49D6B548B3E8C1DE83C9150C96CCB8CA283E7513E2ABBB7A85540604CD0DAAB5A1AE02A6B7758`,
		MD:  `6D06DBEB9DA23C1CBA582780721179E7B9F5A4EB87C1C924B13AF50238080406`,
	},
	{
		Msg: `5767410282A67F990E55B955DD1C0AB409FD767AD5807EF6E30DF288A79907601B85A988B4C21225953A9CFA046BD32D1A55CF4EBE97C195D4D5CB6BB15C152CD2091D2CCE674EF88A17D5DEAC75B0AF527807C303DAC03EE6794E9231FEC945E5BC44F5B9DB5B9953B50B09BE80C7C1192EC7E1`,
		MD:  `D2D75B48DF987A41D927536E8847B55FA41DF6496C78B1A4F7E61E25553B8E05`,
	},
	{
		Msg: `8C91A47981C3220410C54C32BA66C7EB5B6660905ADCE78423D540EBD73EB7B1410494CE0D79EFC934FAC67BBBCF4D169D52856606531056C1BC44B5F4717E3A0B7AB9C6F2645FEA87E45EA94FBE7F7F67E5F4A2BEE42F87827CFBA8D88FC585D03D3AE8984C186EF0CD3D43DDFC052DA269D2CCEC`,
		MD:  `1DCB2FBCE7B5E55560D2B497EDD839D78DAA5CF13F643B4C1B6AA1A763DDF292`,
	},
	{
		Msg: `55EC204D5F8C4BEB6033987A9340BD67F9C25D56FB8BD1CF52862834244A6CF76FEE2709AB1F2696AB717DEFC5F3CCC83206C0A3B76FDC8C85AA79075F810B47FADEA7A0A9DEC23F3A263C191A7A9D21C08C9EBE4911EB7C601A270261A23044BE63A5F9A1EFA650D3FBDBDEDCFEB8DB6CFBFF8FF99A`,
		MD:  `2ADDFEC398FE8F79F7A2C65ABDADBC131A118AE3B65A5A478458058A4FECDAA8`,
	},
	{
		Msg: `8607AA7E3A0C74D3645683262BFCDB7A5A2F3F98642851A6A65FFCFBEBFE0F3235952A7DBB0FDE3C2EE510CF40919EA8674646A4DEDEC39BEE1438366BF3D200818EA9C0B1F2E80716592E05EDA034A054FBAFC852E878041861EA5186568B2DF15AEFEF249DDA59A41AA5703911E38BA9B844ABC72EB9`,
		MD:  `7B35C370A67FACADF12C10F3C4F2C8CCC0FB9EED2ED4DABF19781F3F8713000E`,
	},
	{
		Msg: `0AACB0D8F7450F83E00E61C4B11CF7FC5D61903DB92CCCF491EFE4CBF64597CDD53368CBCEAE06FF425E0CF0AEA2EA9538D423BD88CEBAAFD9ADE60D50D50DFE8C35F08B7ECC244A7B6BC13A8FBD5EEF15863CEF8428C15B8C6B28D2627D78925838B2E0B0A44FABB2BDE96CFC7639501AFD41995641CA41`,
		MD:  `A4D914A0F02DBD6B014A48BC7F58116D65909925A876A44F1289773D1E93378B`,
	},
	{
		Msg: `D1B211E1D347BED260C615920C40706910DAED4C9CD464016FC39C84A621BEAAD10809044C3BC806CD66816839A770CDBC4C33A318DFC74A3D60B3781BD183DCB4F8780175607E76C6A15E2CE0E1351FDE9B2538179F4A72AC5AF03D5D580320A107C6A1F5BEC86974FD15A4A8F7A9A7A04D2AD5421DD6C3E3`,
		MD:  `35B1659D9DD1823279F0CEC04B990E5CC04F117010AC41F5D42EF72CD5E5D069`,
	},
	{
		Msg: `3BBA9A15BE24BDC0BF5CAC755D8F29FE040C993AF0B3AC28ECA940416051830B8A7858EB0CD0FD3AA7A0EA9C46B9FC8D3EACD23DD43A789E000C42BB70A5FB4BD0943F8649EE263747FB231380302B49325D195AAB28512B0FBAA35956B873CC9475178F7D6681DE28D61E8113DAFD59688834397E6480F7A31B`,
		MD:  `1FE0E0896662D9E7B011D1093F5116B86FC20BE08F5B5DF1B774FFF9923DDA19`,
	},
	{
		Msg: `2BC2197433B55E83A8ADB0205658D197A48F08A1BCE66893EC8154D64C294397A29EC1BF822FB726BA89EB03EDCDCAA87DE5666CAE17BBE809DA3666FFE97896CF975783AA6225C614CA35F98D649C6F712A35221D15E8ED53EE2BFCE061C8807B53E1A7F17532CB7F1997EA7F45CE5648067DAD4F07770C92623B`,
		MD:  `7FD5A28E2018399AA98748B3702FB16B0B4AD23C079625D5F03D545EE7447FBD`,
	},
	{
		Msg: `8036B6BEBD6E7A6A13A091669DC31E18BC0C99E8C3C92347F48D16F0AC8DE16EDA4A13DFB570B5F7AFFEDD48A4A880C0587A887320BD269ADF1E6ADC340D11CDF2649EB5D149AC926355826049F6E897A61139F6D4ECBA7F564C2AC30EDA9A5FB0CB7FC777C0064217EBA736F274FDA68E1BD5F9471E85FD415D7F2B`,
		MD:  `190062CAA46640A9904F8C5C79D0D7C446A9F35A08FA2E6804E90F875F15B1D5`,
	},
	{
		Msg: `A1BC252DFDC9DF9D109F6B10B30A755309EB68F4F9DE8304E11B2F1C889A909E557B5A3DBFA4E6C005AC85021FF8143EF423D089CDA36CA00AAA61E4144B86A392DEE099E0A5BBB7BCB0F92343FDAB7F1DE4DE6EF4B2B7542A95C35ECBB888C3C0AE60494614870A9F7272780763629FEAF63AAA54538DA9A9A2FC9196`,
		MD:  `1EB3EF9133AE94D0E1300FCC0265EC303D6DA38FC4A8F62B76AF4451827F4D2D`,
	},
	{
		Msg: `28BF8447B38153DB39714215BC960B54706225C5208B45D7CE1626EF9AB5865EE5527190B5ECC1855BA2E685540DE3106ED399152B7948646A79536444954309356A0F017C489B2EC4417E5903800825A3DD8441E0AEBAE271DFB8716A45D280612FE803CF3EE7BDBA7792F89197837958A66CB304CFA33B296FDE7CD26C`,
		MD:  `A7B3544310176B08AF6DB4C32187274E9773CB675F7DD0AAC48736874FF6A613`,
	},
	{
		Msg: `3BACB0C883FF7876AF1A866CABCAC6F95BF6D459DCD262870C56DB524CEF45B384746C5E685B3D97FA2D704750376A4F9883611C5906C17BFC1BCEAF8A308BC2B7BFDB182081E74AD1DEBF310AA74440430745176DB7E8567B724CAF21A96C40DB77471F32D73E363B15BD5160821D75626CC9749F4DA0BE91439B7A98C0B0`,
		MD:  `D82D16C7109A1C5B12F9EE1C4A4A92F6EA13A48C3259942460F0AF6536801B97`,
	},
	{
		Msg: `D5F983F69A26C7090440BAB0D0DA30BD95009935587730B15B4776043777DF1E2A18CB31B285B7125BE8B5E1D50019EC492276D1ED7EC9E3D7A4F3CF0F476D80E740373F8FD5111FE43CEE98895BA67239D2EBE45B3B7EFED7B0A244298ED29CB479AF6A58A13CF946434D1D13723D160C17E0B8CA37B8C906746CABAED7753A`,
		MD:  `2049E9305ED6E0D9E0CFBA6A267F7983FA4886CBE544499E8C21E8FFD7A08DAA`,
	},
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 암호알고리즘 검증기준 V3.0
	// 테스트 벡터
	// LSH(256-256)LongMsg.txt
	// 용량 문제로 일부만 추가
	{
		Msg: `CB8CDE5AFD7AC4D740CCC149EA0D80C4910DE2B1CD92E91F3E4BD55AD1058CBF40213FCED8464B817F535DDCF598DAC65A457E17F12243F223818455C77262EEA035673127F569A78BB3B189F7805D8D13EF0300CDCA971559048F7D76EF4437F4AF9CAA73EFE6F16DD20576772CD1C2C3F3BA93EAF0AF2FE8BA3EF968E5FC4243772761439B8A062C9AA1CBFD06FE0C7339EBF9CFFE51E9D98DD8F225BD50B69575507F1F631C8FD0F1CCB11175AB142AA8DE59045A46EB3364A78F33DD093CF2915E2C8FAE663362C1D0503CE1A282F02ADADC126E55DCFE26F1F91CAFEF7B63B39B`,
		MD:  `EAC621CD6495672EF57F82D2FBE852DB5AB7A7A8775188EEED5B39CEC355044E`,
	},
	{
		Msg: `8F1CE52E9BE9F1F3CF05B2A9FDCDA629AA80A0486543BDFF576899C91CEFC34DD14D6E3D6E6D697E56F551892DCA0511EAD65AE62D091019D29D0561C69FA9BD19ABB25B55F711B90F92240ABBEE2DDBC59D03F6DD58078692455A7D227A4D3390BD194FF88DD1EB206794F34F4207CE625C02411C388A90BEE700E6D78897F65E0B0CE3FE3A915DB1F90A0E90CC7B34EA9BC18F93B2817D7E0C5F648ED17050AA1DF1DD10033958EBD2EC212796725384E2A7A7E9CDD496FC3BDFBEEF5EBE879C7C3106D5F2B122F579A3F5BBA8D27559BA1C53C7926C235DFCE8BEA1366BE45EAF3526F60EE103F8769852F50885E090AA8859D5072F6CCCD7E22A4D615DAF153E64051A5FB1451C5232FF7DC071DC523B5482BA3606B86F5535CB9AE87E30ECB2256AEAEE0A2F8894D8C6FAD880B3C7F3E6961F27D9516EFD496931D2B4AF0991E21F0DC2`,
		MD:  `2D489D0644EEDE572068335E12E59881ECC7EE8560D4A95716005A8EF1B26803`,
	},
	{
		Msg: `D20865C4F46D155699AB165DA75DABE1917DF25886CCB927E973946602EA2CE3F319DA6AEDBD7544A30E68FEFF9F076C148523EC55BBDBEF04C5B6B6ED94EF5A53AA100F2B7EC2A98822E4605723DAD04CB128441CFF3F33EDED960B0AE5FD2EDC022F3A1678A68E2F30B30A15B2CD1520482AE544608CFB8C335EECC3A4FF67746B5E741672C7FA807FFC84A31488447794505936A66B4DC9DE33E27F9B9C0E052BC5449336CDF56359E8576711B2653BDBC327599983338C383F74A7907D2A768B8D32F58A6188C0049F0ACB70F37E5366AAD714017DB4BD87AA2AA14C4AC3AFD3DEC6A4372BB97FC9482535FBF599A77D2EF1D1A500D844149C8CD697AAE2984BDF890805D29288F00B300F785DBD1F6777FEF74EB5A809273C23AE38458D193BB90289BBFF1BC3C111B4C0B0D6F3A36EAC86EEC4432BF408B27692F9C4CF1AEA93BA9023595B19848239B06B06421BD8F50F1DCF546AEEFE270DE8A0CFAD83A196378303895A70808645477195B26FEE7B23EE378D6BDD52C37119F60D313CA8EA03CC243620B2FE4362ED892C4B87F86649C8C39838AB4DAD298EC327622D47B6A6D14E09B6C5`,
		MD:  `16F0A2AC89980AD05D83C74AB12ECB03ADFCAB43B3FA20972856910DC1AE85E5`,
	},
	{
		Msg: `46E4170A7D73164B3EDE0A133D1DD93E340EBDAECFC4483FC622A6FC48A92393A08FEC0613111AA4070AEC386BC35480520DB6E0E18DEC586C588DB4DCC06F02546D6A4914AF5F789D12799D1533B357EED39B8EC32A5561827DE260D0F3A3FCA9A5183B47F4EEB1461F39AB0515FEAD4F87F41F3B428B4350BC2E6AEDECC568E85F7F457689CE354B59B3CA02103B69BE524AFB127D97E41E3BF839FB53E02D5AC525CD671507EEF4E76F63D3CC7473835C238A1085802E3422C835C2CFF93544FC943DE341A2C288F1F5DC41F2B0B3E6CC0A33FCFF4E08D99A26C609091A67F12F54FBB2B5A69B51A1CE9F142AF8112FCA84619F960A5957CB9B549AA94AABA884EB719C181B60951D8112141B5375A5801B79C2F0BB0BF4DCAE473B5692E9B023E30569120AF99D8D969F086EC9C7911456E42BB56905F9F5E807B5B8F8095235C421E04D7A4DB11A95ADBACA62EF3BAEBD0AA38E1C2FAE40D0FCD07887F4D6E2142CCB6E734618EC07A3F0D827D5EB77D953F123DD715AE2EE8F533E449084515D8EF020FDCB1C2A74EB73401F61E9963128DF1BB3B34606CA2608B139C6A3AB26AE170921C503FC62EC0BA9527C7C344EEF331EA7DDBBD2ED2BB57A6D7F7D17F8F60AD1E61A168B5B0E7FBBC90CEE79B612B6D6C0D7FF6EDE042341E8A158BE5ACD902155B39DFFE6B9991F8BFA858CF3F730E806895A03251B1AADB3157DC8D49B70A17478E0808C55`,
		MD:  `AFC0372EC10B5F59AF5C7C3EE46EFBA87603206EAAEBBD7CB0D957C1A0A00461`,
	},
}
