package kcdsa

import (
	"testing"

	"github.com/RyuaNerin/go-krypto/internal"
)

func Test_Verify_2048_224_SHA224(t *testing.T) { testVerify(t, testCase_2048_224_SHA224) }

var testCase_2048_224_SHA224 = []testCase{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// KCDSA_(2048)(224)(SHA-224)_SGT.txt
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`99811DA76C8218A64F16C765831F140AA833095E3D3126DAC600CAAE8BDDC68E364AB88DDD4AFC56BFFADF17E76DA84C515D41B8A291CF536CB7E087060EC6CEEFE80D1B58A33108C35FD13E897B7B88B51BE194BD693DA5754EC008EBC712C96A4A3BC24B67AABFD2FD9DA28FA1F9EC8032A8E96F6B89AC2AC941AB66E83B8E1BEC0DEE9241110003071BE1B028FC29C7DD046E829D43046014D1BF30FDF2C68C4F60CBE42DCC1E83A9A3E0F9E745BADA12BC7629E801D1AFC244BDEBD1274CE415CA476A807CADA8F2C42E8E43E1F41E66DB4C4389814301DEFFD48B24270EF46A1A387EE8F188FAE13A4C4CDD16FB93212D41F7F6FEB48FC9C41771FBFB97`),
		Q:     internal.HI(`9361B81F5CACD41B174CA3ECE9A6D00C2659025F4E971AD576C011EF`),
		G:     internal.HI(`85433986A2AD22D6966D09ADC7FEA96C819F3EA05D48BB4F6DA7320338C1F15FCA3E0A0997E0F3E534147560B3BF001364080E5EFFEB22D741BB6B847E510EBC5760E7DB680D4B9D5BA056C67FECFD43E5BA603BE1F45E033DC54DF46BA15BA1DB7B03FC8622CF8D7FF722790ADB1A3C7A6F3C6A45BAFCD4B334E37E542A07E8981B1A9B662287B79EC23503DEECE39323EF41113B1CE7B6B1F97DD616CEFBEB7E6E6DA548E5707D51C0B30147D280284E68E939DC3B3F372BDD8DBAE6E81E6D81C311FE425D21D9911FC62D9FC9F75C692E93C035A80092E6FE1C9B4EDFE3EDBA2BA8B73AA981F134A1994E70412ED1342207F991610EF9CA9739FCB322F324`),
		M:     internal.HB(`6118C4BADFA0BEC25E09F8DFC4D47AF90F319CC3C857D6833E49BF82C0CB0EC0137F6962461073DBA83BBC6513B5203CABB92FA56E37AFCF12744F7E2214929857DA1E0C8B266918D55A70782FD9F3FD22356993E9AA93B1842D7FC524F7D020F57C92C7A6A005E8B51C153167F4FE4CEA5DBD5582F84971D5D152ECA874717C`),
		X:     internal.HI(`B5D1ADCEA743768BF532EB0A98FEEEB4B80E8FF40FE508DDE9449754`),
		Y:     internal.HI(`57222DEB1FF792B0E14F4DB4499D4A4F668E6407A85D69E0EF3E9063FC1B637F87C9A5FB8CF304D34C33B5B8C87486DC42BC98B79F754774CF6D79D0D90076B56B4C1E5E93C9AFAF755518C111C9DBD6FC954CE02FD5CCC158DB89A5E96142A6F8C271C52D1F5DE033832AFFC44F2606BCB4DC44F7C882C51D7127D1D157D689780BB2C833C196C36F20F27E5B9C205A33FF225B4A104EB6D2EB9DA09F2724B9D30025F8B344E7F77B80884BE5EB7B71D4EEB6EDF105C3CA12AD3410471B246474FBE697C395256D1BA77277E62068B6827E02FD1883C399F562FBE76C49B2D6FAD25C5AF66D1204F9E64FAD797F57DFD6647C2D7BD70CC848744FD490B92A1`),
		R:     internal.HI(`7A6D051717CED32EDC9A43A6B6DBF6800FF5FEF1B2066C996ECF817`),
		S:     internal.HI(`85AD4A43FB7971E336CE6E727B35084CE610C2174727FD4E2E5FBC0C`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`87C7CC1148C5CE5CB839BB27A8A15330A27BE9F99C80000DEE4790DF95BEE0E164807AD109CDBCD3E9B235AE6C1845C47718D2547A24FDB90B93318501C9487D43992C634D19AFA368F256BC75832D503F8DEC3B3F57ADC40C3098EF6F072C30FEA3DE6482BFA3C2762F28A0E67DBA0EE1CB07E8B19976BC8ABD4D6B096B5490E2D4FFD50F586D15CC6BCFA4C4B9BE073AD1C8C1C7B98F9341CF99DEFC7B354B48A9E0123676A2DB1DF714B8402A7D1024ED737EAB1E344640A84DF721E5CB0961D0AB88357B273797477C5BA01B3F2A08CE5C24A1FC70B43F7482A8EAFDE7928F5F3D9C28EC94FDB8E23DF05ADA929F2D07B8148DC801A3496EE9CF8FA9A117`),
		Q:     internal.HI(`811F50F2207676A6FD0DA5A28D844AD5D5EDEDD9E7460723FF8D6BCB`),
		G:     internal.HI(`6DCC0503FDA95C0CD23B5790FF9A5878019A14D27E9DA8334BB10B2F07643A65BA583026F7E23BAE464BAA2B89BCB68A7368E3191E993FD4F135BBE75600F57030470F6A6F733954C6CF37E920358F45F74FD0E1314D8CF0E73D19FFA42FDA7BCAB34D8FBD53CCC1FBE0525ED17B320F14A6F6E00332EA932F9EC7060233D4FD422ABADD4262B3CB4A3C2F11EE85F1757706992C955CA7A4ABEE268CF45943AE852975FDB191F026A97261F56441F5BC480F12275048F0032F15ABF8700FFCF38ECC785B5E8DD2B96BF985EE8B0BC504A7E04CD07370CB6CF7F53041DC24060EB31A4307D3EBCD0EA8C513F0C39E5F81156B7C6E79CB7BB9D744670DEE87BEA`),
		M:     internal.HB(`96067C16B0D06FF1F9083021304E0A675F3E2226257E5C102B15CF7C321A628B477093EEE71F960B39FBCB491EC023C5BFBC9338612B1CBF8A7D85F2EA0ED05B798715BAAD832CC06DA87521EC3D9C10300C8D2DEA839B6562660040A00BB3B002EA41D7880EA3D57591BF157C450EA53737169E9920B11D837ED549425C4102`),
		X:     internal.HI(`19C6D332D0FBB688CCB54FADC9AB8AEA9238601EE96DAAA7BC202F695`),
		Y:     internal.HI(`29914630C87AAAA7167EDC589FC7FC2157A4DAB579563BE4C0BBE90BE3B0C54124EC4F0EFCC11FBDD8C9F4C5BD86626F0116BE51026A618A228AB9E625DC68B7879DCA6C575710D3790555C0771053BA1B4BD44AD4E84630A5637BFC32F1A70E2A02C643E637EA16E6C4E23C317DA29CBFA4505D6FB02B8AC2A6B87805A51421F86DD05DC610253776D5AD7F53CFEA3ED51F1F0B61D759A7FDBEACA1AFD31E8E164A5C8864E4FC29B023848524D5BFA0EDFE7F18FFA6D95AA66C73A32B0519F520B8B86910D791FAAA53BE201AA0667D854CE6F7223026C8707ACF44C51CFEA3607E97ECFC40C0A50DEE0DDE6FD7940148D45CDFEAE6CE4AF13E463E3A1147B2`),
		R:     internal.HI(`79F8966A756C6683FA7A91B5E886A48D24BA747FD21000B7DA5CDF37`),
		S:     internal.HI(`2CE1D9E021F479E42ECAC7A9DB62F1E62EB2A039AC729EB6EF337205`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`EC1C873E596CC8537C4473F474D3116D7F0B3FD3FA7ACD6F7FEEB7D27636059BAD32971AA6DA022F5F618F731D1C4180FDBD3F0C7D1F374744620BDC8D643FC768BBE7711A67CF4E19CF89DE48AA9FB64F533D7ACEE4518A7CB472F017F5BB2E659E935CF4B7F74DA2D38A9ED38B5B91F4918AA14C903211C89B11355817D0F46CDD1B7B4214D47E4861DDC81D67CB60D2CA0E9541FC748EED592141CEEE7B3C38B7C66E625140CF1FA176405373B11151FE9508709F51CD2EA39C42255B1AB63BF0BB4E9AAFF2A6787FD24B670B399F1D0E067A8586E94C831886F4477C1511BD23BDCB0EB595CDCADD8C52AE25AA5FF3A4EA6A5048FF9095193FB9E7A4EF27`),
		Q:     internal.HI(`E67B2502EE1752A619A5406C2FCB167DB93B96D92A6009F59C876CC3`),
		G:     internal.HI(`D2BA0E8C1A2CBA475C55530F3EA965F4DD1AEAC82B73121778C7D10F4DC8D07D124F7067FC88CE8E4FBD89D88B785BB56D9532F3B48136099868C60E93257ECAC576278BBA8DAA5774EED117DCD8DA50A26B7D7D502CA75606BC9D52365B5CDA8DE29778569064CDB9DFD5DA4783C05746153BBBBE116B95048C397AFCA508B04412DCBA609436BB5AF519454157C45F3111B8A1414061A3E6E589D7FFD7AB0D1F4272747DD5B36F1EDE8AB4D6F63E6033DA086EB180BE088CCD7836F9C2ABC330B46B5BD90F3E3E7993C9EEB9A73E9579B291D846C57DF2DAF1F6933D2DFC3CC9A63FC0722CF4CDE2651068810E06D1B2C1A63B2ECD8D8DB8943ADB13CF2724`),
		M:     internal.HB(`9A0D5D972260B73C93913E824E704F16C09B301869ADF9B520481DE766E4F57B603732C1C8B0649EED538CBC8AEA00A79B6BE998FBEB5430D29BF0E8C419927B55068B8F6EC8EB588FD8C60C7AF38154FB2B4F96122EE1441721728C3B021B175EE5BA2E2809D279E5F6102A0FDC7A49F8B6F3F7155E8B5CCDD9C488C93E5795`),
		X:     internal.HI(`13A09EFF585A1ECA21FD7BE8345B6F1DA2CC1043FC0FD1224219BEEE9`),
		Y:     internal.HI(`5B7E89F4D0F018FA9716E1F760D243913B12391BF0DB77FCAACE13E41929B90B11934D2B63C766E472208627BB19F5C53CA9D6531E4DBD7713587B7D0546B2791AFA7308208C4DB0884B7004CAFAE88140E437B78E9BA93ED69EFEE78A65260C6BDFDCA0A180C427CABF723B01F630511C7549DD653C03CA284EA79F7D4456F23D801748B0F21477F8FA169129E0A2D3314C18E5FE2BF77C6CE19B1B4F26062F02689AF28EFAC1778D95CE0698E385BC1227C7D0C29147042179781B8A531602881932AA1FBFBBDC46964461D80008B37CFD02964919F78DC41C9535FF1CB16EF421FA09F8EA5D6AE2B7096B8A7363D03554F2696FF2C037811ACD1D8EB97A21`),
		R:     internal.HI(`88452F0FC1D6986E0582E495C6D7826ADDB326D3A10D234EC0269BDA`),
		S:     internal.HI(`903E9CC7D40BE8590DA548BADA9C08EE7F91195BFE89F6E348F875A8`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`E116EC9C01DF34A5408C8D61B97E031344B95B0E29EA83A55405A7A7EAA5E2CAAF91492F7FB71280591A1E95C251E4E21FFCE82379677CD58C77360001D54B9E3BD0CF297027585A5E51B41BF4EA1FB5D4817FDCF4B2171694E5F61C4338D8828C15873826E82B7F6B42CE6BA39D5D7DE2DBF1D1BF3CE952EFE83CD62F26A2F3283D6550CC1D74FC19D69D693F4502E5D3760F44B1EF0EEC28523FC7B0EC6B104677925F4F67A65FE09100C8C9022C2A7D979425D978B27ECACA526CCCFE4EBF906B5BCA69918FABA1EFAB50AFD7AC2334BD094830DE4C0D3A884B27B602D47D4D3F6BB29460095382EB8EF936D8BB67DF44E4AD763A680D4AF4686D7A94E177`),
		Q:     internal.HI(`CCBE1FC2E2F547ABD9697389E70C7E35C454AF6794B4DEFB11F674BB`),
		G:     internal.HI(`172DE2ABEB6A37C237C719130635F389ED4F29EACC57BA5D1A78C098F3B40B53BCAB26A9B7E25468C41992F488DA9893998622764D53DC2D5A75C4947B31E05C24C7CB123FA9965AB25E945A4407260C1BF4C16F67262145484F832C70F0AF669F89ACC9E71DBC6F38EF5A0E4D95E82E28B4CAC8E8C461D08A82DD8EE015831161E735CE29D4FD889C0B7AB251713A69B747D214368B9AF23B17811435C74CB960EDB66EC06FB03B9753A13ACDA59536D2DDC3A62AB944C96A89DDC6F126B262034154F425A4028A4FC721817FDE84613C16011146D5ABE04B2A161B6C6466F34B5D6834A8E1FB7FDA0731FE16967F53B2FFA5E56928484BBD903E717AFFAF78`),
		M:     internal.HB(`451E1341B0E0E1C644560D46591054670DB96F1912F77B1639960F06967DD2F037C56F5DE458A6B9617267041206BE4499375BCAB68BA1C405007DA16906605BC6C6AA8D4809F1FD561ECDF9134F2B2B600456D0D83D316EBB8E3FECAF3111B56762264F7E215A7422FDF033DA0FCE18894B04E2EFC5204FCD9309EB7AEB3E1A`),
		X:     internal.HI(`10A70B2BD677740C473657BD5C3DF0712225FDB0A58A03BF56D8C0885`),
		Y:     internal.HI(`D91A0D97C1EA001A986CFEDA8995C3E95343BA00CA67892221019AD2A68C20CF17D4E8E1C570CE920B0ABB4A875B8F99B3277576712A9622FDA50A40C7BD388776CEE2690A82E2D82AD751FB11AB2E7CD95D3C85987E3697E200D2F0962970A068B5F1D5B0C350A65AFF4382304FAC07C32A0FC916AAAD385C19D500ED00DDB1C0436D39861A969C11BB94B167E9A7E52C0664B24C024A40BC64F7C4225B66C110742E52A344B34D1D1C423C6BB24375B1A63A4A59BE59E845B1854CE60E6E14372B2D1333EE43C370F621158481926D046DB11844E821FEB3EF0B15FA994C6DD6F98F7AEE03229DFAFB1FF9ADB4A1B2F329AD6FEB8E53647CADE9126F259A0E`),
		R:     internal.HI(`704C040B69A5F83C997E4212596B74F223B103A75EBEE449C344E43`),
		S:     internal.HI(`A8A43D25C406C906F9F9C394FB88044F4CB0521011E64B5E2D04F6C0`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`C748086250F4C7F5C1DDE5CF1BBC199FF6712DA2ED589956DF2D7135F311A44756F5C168B1D63A94C1293CDE59E3AB07051415BDAF68F8B216720DC4F465C355E6790F999809521F4C309C03EA80D9D1944CADF29907726C0FF99780A05357E8F5FB7FCFF12A160F4DA01CD1EBE6036DF5E34C0B982BC720F72AE08F500932B1A0EAA5DF2397EF935706D03F800FA06CDB6B4CED8E50B9AADF12AAC9B0604DF9214F2952CACA1A2E6EA74090F6F26DA85786CCCE804C93432AF744951FA1186374BB5EE856FF6DF5B78027BE339C15E9C8DB90020A840DFAE8DE792222D9E36D374907ED6CD0826A72C3BCA803BEA328DB67A4DE6F321CBFBD1BDC62D685884B`),
		Q:     internal.HI(`B5C30C21D727F8789D1B3CB3D1593C2BF2B081035539E46BE52BC753`),
		G:     internal.HI(`37BB6074B75D2FF541430735D15D78232CAA77F17EA632C15C8DC3DD7CE407E00D03334F811636EAB0A50A8555FF8890FD2C5D19BE8070771C9C7192A7B312522C6018C5FAD436AE02CFE75AF56114B1DC7DFE4677F5AEEDBAC65B932541138AD1FDBA1A5A442F6F9CBC3AA03F4DFE344B55E4742452817E8E9DB30DDBF582FFDFF99FF5A25424A72DED8E42FBD7FC87973257265F04064C67CBEB239B3FCAE06ECCC70DDAADF57A57D06D758B84AA158C93EAAA2C99C7C8C40B3648F824C849E5D83BBCF978D385C977373F64EBA6C0F1F0235DCB923CF6923EF184753F6E7F3CC996297D20BEDACBCD6C8F6047D94CE361782E1E6ECF7B46FB8E4D3EFEA28`),
		M:     internal.HB(`F3AC4715D3E337B12808072E0DFB253D200A892D96EF2AD310309198FDB684D0A88CF4C335A6A6FDAD8CC8DE70E26B7F939213CF0E9DCE1FBADB94DE93A8435C283E9D37B4D30AD55CAAF42C6F2124F8B789CDDE3A42D1076AE051A138691F6EF651303C866A066AC4D6CB7199AB95F544E86D61A2E6B8189FDE0C340CB680EF`),
		X:     internal.HI(`114FD6D2708FA2492F4953C4A5D22E8DCEAC243EF77E87A72734800B2`),
		Y:     internal.HI(`32AECB4AA5742E86DCF83C18F280AAD2ADDD12636797DEFA2D68F6DA11AEC5D2D4C114EDA1C20DDC63F01867EA0DF78D6C6C75A0F9AC7D731766FCC91EB57AC23C58F70559B82A3D46F2A130A67FDD6F49743A56DD6B12ABD3C93519D522E20D6AECCAF451EF50AC15D5E45533C1E377B629E3E20B11E88AEFEB3ACC237CDCCE13D9BD646783BF088B6387E9CB124F29F4CD08CEF56FF64284CB0927FC21A9AE62EAB01E80471ABE252884D0EADC3E6ADB42972CF1943BBB511662B188BC0E3E84374BBE0D8E051BDB41311325F47C707A684394A1D55335D3F10B1E21ED8C12B867FEDDEC1094867867BC2D0B9670043528571E474A84217673FDC2C66CAA6D`),
		R:     internal.HI(`70652F14B6D03FA452408BCEE3123A59EBE484C8BDA4FF1738E55364`),
		S:     internal.HI(`224F5B5DA1426E48113F15C868505A76AA28A1E4AAF256131B57FD81`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`B71D892A9DD10BACBB8F3358574A83831BB1565EE6084C1F23EBFBF67724C87FF78252689C34E227CDF624FD22C6BA3D56FC61D86E98310E29854CF9694B3BAFFCEE8BCAE85AB0F6CDBACBB0268DC91CC8299B38D8CE2E1D13BD24AC166BDEED48FE1A0CF51350B6366DFE10C5D0BB99F21854ABC700C7D6C8D780DA5AA499F004A05F528730EE3F082DA12D51A9CF1E1D0DB11867014EF48EBD95B915D8364D4B83545F2F2DD5AFD31E6DA0A8F07E62F1DBDF255B6FA74AA91F9700F27B6782CCEB4702C416ABE12D0645793E41A218A3ACB8ACAB544BEDD89CA7A86FC09079FDAF3C6C1E0DF2C35A66571E6F644EA1CD638745DF805866E552DA75FE1C04B7`),
		Q:     internal.HI(`A935B8B087204D8BC36FF5662CA4A95595A096D90B53031FAD407EA5`),
		G:     internal.HI(`50E3AC18BA0092CF2888D665A8B8D68D4C972D60A70F3878EDE9F5F96352A0F5BB2D3BDDB6952CCD9EA8DE124F878835DADE26B5A8F8BC0E009F7DA4430E1BE9C6F9DED3EAC59F8B07F16A9BB453FC3B71CD734512AB5727207C0E4FA0108C60F2AF5293F8C867FE065E6AFCF4A3693C3DFF7ACD5F8F9833CED3937C09A9BFA06CD45367381809ACACBB0C8C64A66A3DF308EAF44FE19E4822A002D63C0EDE2B3F0B7D23EDBC5152AAAC5B93544DA0309CD939F93E8EAFD88B23AA9EE55F47339E060F222BD8408E6E606A730545915853BA8050180F5374699DB3F61834033773C96A3BEF6809710B1F63F9191F16921DF29DE6E72977624CABECEDF4237AE`),
		M:     internal.HB(`FD29A29407FA031FD75A94FB218849F9D2002955F226CF8F3E480D61556213FB8C7EEC76332E2F8D6C5297905E540E39646F3A29FAB2A55F8EE21F637CD047E052DC8C0DAC3C7C013BAF24E64A3CF51D5C2C60C12F4F0EAF3ECA12ED8D7B5024652480753AF41E7BE5360BA70585D7440181DCF6275420D9DBED39A53A7128FA`),
		X:     internal.HI(`18EF4F404B451A8DC1B69E0912CD02B90FEAF9E6BE6B5D900E69A8CD6`),
		Y:     internal.HI(`45E6C71EF425C5B9482855B760B21DA6399FBDAA7DCEA61F7790153A73D2E224C48BA3FFCEF14CB7D549A5BFFE7405A2251B706DAC306B265DC5920D97B9CE7B1E957568FC8E39F68D9C0B5AAF8588EC92F966B7D0EC918CABCDE8B2206EC067B91C1BDC6C0780FBBAC59F9F49DE3EB6DF9201E36A2C0EDDCDE945F5ADF48E573F6B753818BB6F19B8E7E4B25AA1E073862A7BD20D1EE330AAF21D0509C4F8FA361E7CD1C29F917B1C4227D1B77AF083EDF2BE636B259BE2D332E85A939879A3F91069D32F1DC9F16719848CD0A980EF3A62E4BAF3CD9DE18145B2B4B3F9E5A49F470C8823C61B68903CD2B9872B07FE485903D334A7A3834EF1A77D5497A585`),
		R:     internal.HI(`B02CA3830A6824E7AC34D8C18EECCD41FB0512877A81A7B2E732A2B`),
		S:     internal.HI(`42E64053B94EFD1B9D3EE67712D9E21F39FB0336271D09E376195652`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`AC32E42A0CE2C7E403CA40C2661DF3AAACF31769E2F1F51D49DD251225CFE0E909B981A7112CEAD4DEA9CB08506A05651B64C648BA4FD01C5F17EF6DE7625DB2E5F72ECF73ED8EDF57CB890EFDA6B7E4345CA2DDDE44B5933757424C31B83AA4612F5A384AEA9600C4B3120874DDA2975771BEB325FB74126A979F7B1326504A6F9C3C50C8DA8FFED2E83D9298914E34574B91A5D840D76764B560651A1E959AC26ED422AF7FB866E2A6F315F3A2A88B49DE742668F491431D67AE440D900DF0B559BB3D482B428E0AFB0BF629D4D9B03714CF0DFEC61271AC578AF154522A3E2982E117ABBA4320ACBEBC1FAF3D924AF8DA2641017D4668EAA9CB56394916C3`),
		Q:     internal.HI(`A58A83F5E4B9BF0B3158E975339F471E7F592AF24AE4111E6EED5D2F`),
		G:     internal.HI(`10C592682957A7103048B64FEA9CF3636CC9A31B89A607FD2BF6AFCD7B220A1D29C654C2B8F379EA13AE40A5CFAD555D3AA736367DADF5B81B8F53ADAC4E08B6B6CBCB24C806628DDF593A5F49FADF1A81D6D48DBC3DCA5649D8D44E081856D2492AF98216D5B438200E3A4F207EBFCEC49BA5E3B19064CD3C269C4DADF4F42BE8C968EA2DC9AB313A4421CAA5A3CC4BBC14DA93819DBB88A41ED24112BB4ABE0F77CC99FC9D124564D8F846159AFB989864A6D191434421427CCF8C5A53BE53D3B617B4B7685A2C7BA1C19370DD23C2852097A2E7C74D84C1F6157E3864A5C7FB239EB2D9F156A4A17C668536DCAB2BC9F822DA97C1E58316680B0CA1B61151`),
		M:     internal.HB(`3F874F42C4B98EB06CFE2173D1864C7F7F8CF7141F2EB6EB5D10ECA1D7D10AD33D8E81F7D9010A0A36F7415C952BB2D3E4BFFAD977DCF02A1A4788F0DD52754B2095A2122CD413A28EDFC76ADD722C7C27DEB57C32F5B10A4F7DEB15EABBAEB60ECCC37F92D46ECBA14C1817586F20E41C85F923F8209FB51D737A003FEDBE1D`),
		X:     internal.HI(`1852338A75C2CDCA0807EE63226906945A92F4DD3D2B0985C85B1E79A`),
		Y:     internal.HI(`4AE7DAA8469466EE5C0F6E8CAA5A36A7ECB52837CF55E082C0F065C53371E82A6E0E54F69A64665902CBC78D76486FAA4AF26B078E42F91F022E781838E62206366DCF344645049C7F5A69D08EB415CFB7F64449D8D49C9C21726A0ED1525BD938A5B1C0146E8B9CB1C65D247F8823A085C5EF1F2D34F63DE9D90F13C2EC7F4B75374567DFEAC6A513FF6A0E7222808675566DE083D4A7DDD56AEF402A99EA0DF10ADD49F6B150B94E25416D809F7A668D3CA8B8012A381C5C3D958E9E0790014CA5161F5E4BC3402CE881FBB6F90C6336C44A67AC54264D87335C7C6B491BE29664D1FF775E377191B7097E14CC84DE03B21C41D13FEF106AAF0361001698E2`),
		R:     internal.HI(`BFCA5C42301C19BA7BFF3892E41B2E4911B4FC5DA7C0995CB5835840`),
		S:     internal.HI(`7C8EFD8C957FB6A472CBA2B06337B2A93BA76040500E35CD4963899C`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`B7544568DA85E4D52B25D3AF6D6F17F1F55EB96F246FBDA1CA5CEA3AEB06167829B2E2CF4F2DD20C8A56A57CCEFF6DB43F949D1C4475D515A4C5576E84ACCF06268A61C2679B39BE6BBBE6CA2727B9D53EA738CED863C16DD30893BA917FD642D531165693C1728F297A13B822FB3781EE6C1CD02765F03DE7530B125C5C5589E3BC48052E011F1752B6BABB5222E1959E4817EAB18D515EC3F4F6CEDC8483D380E0BDB8D67D94B530AA241EA1A5FF3DF5F1F9FC94C17607ED0329D333C858057D957E277B6FB803E0817BBE56F9833A786B2C8FC070669C15DD9B2C8D1BAD843A4BF2EBE2B5D4D6BDFAB40D4FB5D28CCB0151F5BA2AE928D35693DE0DCE524F`),
		Q:     internal.HI(`A719193936A10FA91741D456991423729DA012C02C7F1A38719A1ACB`),
		G:     internal.HI(`578E5B373B7FE22DA9BAC269B3AD426AB8E6714D82A3F45DDEEE147E11F7855EBBBA60334727122C9312B86DE19D21C284C6E1057AAEE84D9DEB5B2D81AC2F21EAB4C8DF4C5BAF9876050B0FFD1BD214B40EEFA352285B5E327CFDC168DC81C50413AD312FC7F06394D0638BE965C2A17C6184605593CACACD8CB8045271BCCA9F828231604E47EA6BDE0C469A13D873D7A70D1D5ADECA5B170FC04C3268BEC6F422C40D5B6CE30F344561DC4DEB93BBFBBF8F6ADF7FC20AC86C377A9D9A3724AB091A44F80EDAFC0889A3F86DA765FD969EBF49313774921DEA7CB43ED5948147AA1EAADD6117AB7E2B873BEEA8C31FD13D07140148DC552B074C364D169B8C`),
		M:     internal.HB(`11B7791F05312489822716D3564937AF7EA2206A9798288C083A1B9D3FD671B9152D5CC923318195282CAC81D13BE033EFF47F637D2F78A0F6B9BAC9737DD6FC6B5A89492DAF985BED6C49F76315CFF572937910BDC80438B82B48560A7B4389CB3E21D98A99BE7B8F4CDC824B3AF7B96D69706991DC82CD7C9FB707537D4FB6`),
		X:     internal.HI(`14738F6497BB319E37093B4CD3CD234A05F33AD1B235D093BDF5CA4F9`),
		Y:     internal.HI(`A9B9D5F6337D7762FC677BC52BC5262D69F28BC7F035908FC13ACD222F0C20E61C7DF8F483D1A0F26135D74AD8FA7E0FB20E9A4C0A08E623D404CC1CEE81D886C39281BFA6D3C03A42CF776D6E65A9550FDEF88B60378126C78F2350CAE4C001287C8222A701AC84E07C585E6E062DCAA205AD1699E795357E34B6A1A6BCFF36FE32D2EEF7A4AAD74AA8E343786DEB11BF4106C28157642DE906DD0ACFB181B4E36F1501DD01A5A1826536F28C2D73547427BB639F3B55D9443E26A5D6D515A5612EC005DF70491D1E8D9C174A43C939650128FA857AFC1C8E4061892B94D61F7CAF4DDE0597D8471C020E951BF965E0A3567AD836E2CE23438E8ABD4BF553FC`),
		R:     internal.HI(`71B5905037976EF078E1DADD22D39C50358C953E424AAEE7DDAAC154`),
		S:     internal.HI(`753B2DAC65F0C677A220625A405B43B6F6938A445015A49C19ED0BD4`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`A92ED90C8C6D83C3EBB71F816742A759648A333761FDB15179529141AD0420B2BF39F7DDF02263740C59E0A92C9A96F46A8A37F0E977C1ECEB5517B3C1807269F1E5CEB9027B15C2C800B2382B8A263CB3996807C27E8C939912EB4B0D57F86181A0D9D51AF73967F56FD44C2D11228D45EE17DCC9B9327BC9FA43DE003D98B6642AB09E7A7EC78D8E15764F2DF36866D7F82CE11CD5DC613D355C0A61B4FFF26B60AA10A94431DD09DCBF652E9F444E2B029D7D9C269B9C669773CE1ADCB3ED6C9889F7830DA0F848B3F89CAA292A07C39430CD7094DA568E9C034EC1E59AEAA2FEC3E83E5B7BEFEA812A6E7D0D1326BBE0464FB69485C96429099FDF83C48B`),
		Q:     internal.HI(`A5038E4A4AE2E034027E0438013E38464C0FBCFA4FFC631636ADB6A9`),
		G:     internal.HI(`23C9B7B700E25870331E72E2B29B691B1C691EC1B542FC1C9009C03E6F17F059D660D1D1DCEC4F5B2A9AA77BD193D4A6F81810D7A00119B01997D37C285EBB55A8A64D809692BEFD7027E2EE19EC70B3D805EBA4C42BABEC8799D90B5518566C65BA8451D19652133CB97DDD8EFF2F694F455360E877E4087EFBC213B4D5A0839D7C412593EBE99E6A360B08A0365D2A0609D3DBC99F4C4EDC488B7FBCB72BC5EDBA252BA1C5C18E84291C7946933F80FF45552720E5531F7A1D16EC47EE7B9121DF82968049CE5B516043D05B436469F50654EF65C5E7715058B1AB41D9A0D87F2526F4EECF4B90BF9676B47B1F72FEFFE93609AC066ACA9D1A18A37F728B67`),
		M:     internal.HB(`4CACC82F44F20C4BB2055E606BA195EC2C324A5CD477EE11D6F700954444D5916D4D286F87D0DE525A23C5444BD52536DE01F1C7063A086341EC1E2FF5A577D78F1C6BB2285DD74DF307925495F66AEE193DD67FCAD8505C950791F4A40C98FDF56945091BD75BADCC67442C1938E925D01CE9CB6A9B914416255C7BB1F2622C`),
		X:     internal.HI(`1116544889AE70F8B3F262E83DE17CA4EC1AE13733A9BF998C4315DCA`),
		Y:     internal.HI(`8676CB0B85F90E7001996F1FB1EAFC79BF9211A157C78CDF238C72ABB3D48AFE700F29DC05BE5658F8CA7F9A7AEAE7A6A120FC806C387BBB44175059B955D7B1967F95C05290BFFF2D8C2A99391158596C84CA9A12A1386F9968B486FA6627DC9A0251E1E211B99D5F58CAB6770DF5652ECE76DF5D816FAA95836315899409DEED2EC0DC0FFC8FC555180074FEAFA3C9936D433C3C876F6E453934AFF073808C2F4746ED8348F87771BCD6E3A3DD502A46CEE0018EA25109FD37DABF965671E75AC5FA2ECCCD07AF6274019EBA10098F05339B03D3EA4DA67A1FEE08759FF7BCAA0B799C2E2B6817279B838DAFD0E43414A0E194BF8F48430DB902F8034174D3`),
		R:     internal.HI(`4D40FFE746E7B597867D04DA2ADF2BDE942CD47CDE1A855CFD270925`),
		S:     internal.HI(`81C7BC918676EBACABCDCAD0B2489DB53ADD8B79B3062341740408E4`),
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`DBF0035414F968FD804701035BC45F09371B7CF5081B8664DD3FBBA25E217B9828416EB342811A0F3D45C3E865B35DCEEECDE1E6B295B409DCF24499CF02A4467B2C26FA3764AFC2637363C0DAD0A6AFB95475EE6320029B81A8C41B4C4625A4086C0D826072140025C5D25A01A32673F4F9CF44BB9C12F70AE8B98856A4ED6BF67694D04C644DE7E4F2FF153C71DEDED59B018B855A43D2C51D2871A3C9328B81D8F6CAC0028BB2EAFD9C7496E69C381DCC8D827CDEDDD1B0B65FB31CDD2D6BF1F06FCAF01F6491D468FEF36440754A2915A38B40565A5ACBB85BEF1A7CEAEEB337EE5B69D46DE396C9F94799C1277F7C7FD5C423F82F1FB42CFA205537AB87`),
		Q:     internal.HI(`CDED6E70EA33B2CAA7357D79A3C0A3EE602E9A8F599AF96BE371968F`),
		G:     internal.HI(`D45240D3BCDCD0AF5F7152417BB84FD26B7E9FE3A279F2EB1463240780B6186D6F4E1D756419E5708D405A7DF5F5B46E7D6ACA6947540E3B4ABF11DE386E4F90059AD7EE1001522C822A93A0DCC30682BD684EE73E92BFF11AC2A1695D1D1691CBFA583CE70647EFD96DCB795C766D022E075D6BAA846062367B409EB11FF5BE7E3268F9FB17A2B040517ECEB87100744E434C723BDE4E0E3EE67977790E0A9E97890CE4916FF9C83A2508B8062F251E0EA5346CCE366776EFA0DB3A7BE61BCB641C354053FAB6285B9EDC053D7AA189A80BBD457EAF77FBB24F43635313E602BEDF0BA45E655E81E62FFF00BCB12CB6A7FDF0D30D2A37307F4DF6BD1FD0D93E`),
		M:     internal.HB(`4DD767737A11131795CA625B49E06F98E130A26950DED41D667C084C23EBBE3C21600E6883716C62678E76E43DCC88C00CD7FB870D91E89610911F639F1DDFBE63CDF2D098F0191B3BE30DBF2E698647734BF4CBD3B9E118FDC0B134F4BEB8F669BFD78D3F9F8C83EFCEB755FB3C00871E928F4CFCEE963C02B7D2A0151F02DC`),
		X:     internal.HI(`5BBB1F350F0E98843654EA197D29BCB5FD40ABB5A1ED7939BED05C7F`),
		Y:     internal.HI(`4B8DD6B714873F29B5A8957F14A55714C339E382A1A0E8BD06C4F9E195F5D3211ADF913E864E282D6D178E31286B7E30AB0BAAAF03D323DC6FE3046B9718336F10FE64CB0D9F01187EC76A562B0CC352471C8BDCC95A70174E2E536DC8DD5756AF6A55A49B0A85F6AB4E77115AF4DC4CB378F388113942F561B2E54AE5136C690CD8544F51E9995573FF3B9EC9848BD7445CFA2E8DB27E31485144EB8D0CFCEDB74AE39A3C278276446571BD97465C6FC161EE3BC9E78D18F88E7D1E5D7146D3764D807337D60E84F4F1663EB496B9E97A599BC090F8EB1920893DD01CD58A31B4DA3424218A1D1BB231B97878BC1343EC3C40790C6F26764EA621DA51437187`),
		R:     internal.HI(`102BBABE4C8A5832F96705025994EDADC73D71FA8A21EC0D3FD113B9`),
		S:     internal.HI(`6A7E4C2CFC1A7EF0DC122D326803CABFB6FB38F5ABED4370359C9535`),
	},
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// KCDSA_(2048)(224)(SHA-224)_SVT.txt
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`DF7ED9AB6884F692A31752B549E8B3AD479A87B3F0B42869C307CDB20E4DC98B724E4486A04A11E033FBD6B923EF1682A281611CDB443FB7751CFDFA42D01A73FE615CE694EF5176C166533081CFBE1037857EF1F1079F82A8CD03C1F9CDEDFE27067C03E4D5D047B45CF5DB63C637F471A818BCD553855107556E99B8F671DD1EF1C3B83747F16EFC4FAE20D3F654F3425236AD336679D1C75894BF0EE11FE798401B8C3D755C2D12143A10E36D32F92456A49FBE6648DAA3E095178F0FB61ED27634ACAC7B03EFF7EC1A62AC1D351B19EBF7133162076AE260572BD8E93EA9958287EE445A2147327C9A754FE40794ACB38B33504F14A74EB4892F8FBE08DA2EB956D0CDFD37EED4D4CC51F5859ECBF0B786D0E60C15E13B1EA20061C5AA2973D7A87615350FC8756B8BA6D0AC344B7E69D463C65EF98D884AE11E03210537C2024FAFF5BDBBDE35217BCB19ED4DC879A02A0FCCF4F44B964B4DB833D740CC03C6E4F04A369565BD3B06BF12C5B6218B9F059DDB6484E0549BB3A1B6D9CCD9A31AC657FE2B3EB53D6A5E2B03978258E60EAA7EDF2B703C351EAC545A006076985920A9FE0EA0526EC47F5E3EB00E9B45FE25CCC9B0C575351F96F4F308FBE36149E2534239EDE690C92C4F1D43FB3EEAE94C489740DACADA51984E5F9CE68B0516C46CCAEDA0456A5FEE9E006D38BE9FAFBA5D`),
		X:     internal.HI(`6EE602B8815ABE058E731CE10552FAD89E07C623D1898F4B2CC7C316`),
		Y:     internal.HI(`4BF94F782B0ACA739D31830B798C589B62037EB3C216C9E0F795B9DF044545B3AC0871926BD4F707018A33584E1145F6BB596B3239562EA38E9DCA7FCCA5439D1EBE7AE87316232E922AD4DA792DC8540E6ED4D782051AE9C86ACB587D915608EB534B411266FCCBEB3AACC6CB1A8D47D29B349DF5680B4C32AF6F4F97CB7109EA0895F9F1737939E7F0DCA6BF52888961A4CFCED17E72508F7AC6D6D6FFB309A9DEEBA3C7F7270CC32E926F339A2EA0847237D369F7AD329498051B9221E151A272F0EA38D2F77BC2292B2604B7C646FB080DD11FBEB8590803A3A654BAC52CF4A82CE0496D0918958F95C567E31DF1FF92171F7EFF75B4520F3B9A7F79D23E`),
		R:     internal.HI(`3610CF67C006E65A449C298DEA7BE30DC199BDB7226EE60D4FB82BE7`),
		S:     internal.HI(`4C42687E08185574FDCDDF21854A51C56B5D98A5789B31E098351C09`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`A02ECFA0D4844DB1FC115749B19D5579674ED51A925132F7BFB79AD61BF046FE87C71C65A34E9133F09B08B787C984827113E4C91C817DB40D0C5ACB3D9BDE2236C123677EB3C88C47438E21FD68720EAC1AA8DFD20A23339F01098F2365CB2C8AC037ECD0B03C240A2FB65EC69703B68B4CAB06F56DE0808F7F27DFD873E2DA8BF01402AD4A1CAB680CB510427E005CA918DAB26B3358EF23F89F3E0CF0E1326606DB82D0907F18BC1129A0764D1B2AC8778D1CC5ED171BC668C0FB1A0E6F82733F19089E9567AC85FC163F0F3AEE91D2EA8249393592E80F514429040A185F305DC1FD2179BAEE6C14E9E46386FC4CDB77E202A8AC2383B9BC4CA6712754A844AE2C900E5E4AAE422878526F77AE5B1DB13BDB99FC0F5EA83D6117B5B081827D0620B6BF74CE04FE8C0010D95C560AFBAF842E3CD68336EAED74E7C9F8E45AD1C2C62E39F0E74FC3202571EE8E2FEAFE111C1E68F2920FB36FDE4D4E5BACE760C5B280260E1C38D74AF48BA56B5BD4DA00CA969C1338345FED61458D3BEE27707CDEFBDB14DDAEACF9E4419A5AE4EB692CBD4901025A39731925937905A95F6FDBAEB5544F82E9DBA2D03B14CBBB98AECE8DB1658FC4FA9C2EBBC5AC2CC31EF45FED8C32154C692344FEEA0036BB8DD3A538124195299BAEEE1C2F652A0A38BE0CCE29C3C463F66E651B87F119A6C42BFA2676`),
		X:     internal.HI(`4F66BE9B4454DA97218B6929C63D097E166D3262FD4A862659C665CD`),
		Y:     internal.HI(`7F40671CE445D5DDD63C1110F66341B3B304DCAE63B493250C02971531ACC1F2708C94DF9CC1A832B343AAA43E3C9CC748C6D5364DF7EC5B0B9DB58CC8E1F5DA48EB99A5AF7CB399C86AA8420F20B7107D56A501140E5F34616DF76C699EF1713446728C02F141EFC2928A7DE95F1D87860E78414A46E4FC18B28E28634990CAA2C09BE72158714842C14524AD76F348CB52CAD44F17E4DA7A3EB42D8694B19D8A9E081FE66135BDA245E8692345C31466F31F97B5095B8A98A572AB5C01282EDC6F5F2F8A645DBFC907CAF7347A34D517759052FAD97067E43BB6D715E8B7A0519061F6C4D9724F757425CCC73C99D371F6CAEBA2F2A44ABE65D6A3E7B38622`),
		R:     internal.HI(`B86D6E95B1EF20236BD2CAA4705469E49D971F9135E308B0E1C34F49`),
		S:     internal.HI(`43261961E9A1620CA79E88951A89EC8E60822B2B30E8C47A25909A69`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`87A6A4AAEF908433CCB36EECF9FABFD79FCC133D1229FC267F309E25B086973E73A8212EE8BDC4DE40E1994A3372759FBD75E5DF548B6ECC4785E86C5A256EDE5A01C0D8C8277E8F41802A53D76F5B9CEECAFAE7DFA40714EFF5B99BD9AA725A8025634103AF4DB6C6ED45DAABDCEE961C55F7A20BAC655F907279B955D5A6ED71B1756CD7DDD4A80BB0971A3D4EB33C75C40AC6F87E30F78B171935C30C11BD016AE8C049E5C0A3987753B7E5FE392872F1EB6F90A2180F8C2511E6DD5FC6D54B3B360F249FC2CD391837BBC2D213D9D3DDD7228443D4BF8309610A2687DD29B43B6292FE8D97350492869ABA55E0B79EAF95CD4B38230CAA539249E9E17995E6A5F6E933D901D0580C0C2E7CBA451224B973C42800CCDD84C0B3B23877C2DDD6DD031FE954CD7CDAD41FBA7FDDEF22FE7347C523BD9E06DA315EBDEEF8EAADC17125A40D78CC0078619AEA014193080A7E70F40C3F7241BDB1B247AFBA2A9A28157E515266DC0A6751E1D10A12EFCA70A2D3DF7DFB262F88735899E4BFC41FD9A5B86836E8DE2F266BE1E96923C95BA1CFE07BD60FA25BDDD18061C1AC00A0E7260690FE6FBFEE7B9E0F16B3F1EC91531F8D244040D436A44DE2B641D3306AAEC521DCB71571AD720066A3499E2F2C88D257A1ACFCB51C1193BF182629ADB1D2D64CC4359CF0BA63CE6B4350F66FD58650461D`),
		X:     internal.HI(`06606F6B5574A44DE8D2A503428369351DA551E715E593E40421087D`),
		Y:     internal.HI(`13694FE8E67072FC43201818FA2761143B36C2344E26221F8909F44999BCC5808CA51E369F48F22F285C4FD83A4681802887850A2F2E641BE80BEA22E814B954633AB853C3796EAFB435D38DF988B06FC31CF1BAC582BBB121EB2ED078BBF0FE50E0B87ADF842E37826E12CCADFB2F05009E9CA32EE0CD2ABF24BBABEAF975503B394B62812E6D6B31201EB950EE4916ACB4790E1C919DDDF177553173B8C60FE58C5B89288261DB8F373E25EE96ECC2B3AA8F7169910A5D6DD1B8C9233B87F6AC6A7927997FE43D41F3454ADBF4037A946D1A4B5065E957AFB9B6A7A4644B9A97515EFD30D6523F90F73E1789A8ED4CDB4DC9BB34B3CFFD42CD37457FC135DC`),
		R:     internal.HI(`455A9631F86690BB4DA3E9173BB11844D366B6790588B26D35B1F984`),
		S:     internal.HI(`3807B8B6A2A08EB6825DDA6A30D412BDC413B18BAE8388F8E7821A64`),
		Fail:  true,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`0EE762F19633A81CA9A048E32A2219357F72ADAE5E4DA82E2DF37E58A5DB643FDE2EFB1B6C1A8B2DB36D3CC1189D2D7546ADE29E1401E01352AFE4220948A4EF72A9791B31DEDDE20D962889CE3A70605828F617BF1E03C17759AD353577532CC99A50D383E0BA1F4AA9A5B41DC3C59417FBE16EA418F4D012D275001EC797CAABEE00094CA9CA31C2DC52A41F2B9319AEE5A5DF4D8C1E425FA5E0D801BE1C02A5380B6EBAEA3BCD960BD6A3378DCB5E114E4B8E8E3F7480630399FA660B15790FB4009A467AC310AEBEE0E30D2CE539FA45E587841C6E5AECC6538E1A853F390745750DB05AA080BC20287E9370E0EBEB828BBF933A100B8D6EC9A2322ADEB67576052FFEB4990A38096D7704EF47193165601065D3E332A426BF2B0C20BFCA077C585081D7FC0163F577B8E16128D3DDF48C3FF04EFBDA55BF00074FB637B833311AA9D23C9F25460A1411F4A91C91CAE041F96E36F1728EB35FFDE764202C3A1A0059CF83E199B2151C3F4FD344309B81B8D1643FE8D50323B7BA0BC6E0392162C868A076A8EA3F8C6EE24C1047F9BBD532439D478A4330D9EDD639A5635AB7DC37C7B605620F4E8EF2858CBB579A5D84F9B42D530A655B47EBCC36EE1D7193041A4DC748066308DE9799F9572B2A7CDE5E70708D214C9185A50510B70BCA15FE47BBD58011AC5FEB5479C68F0329D9DABAAA`),
		X:     internal.HI(`6011034B58110A5A15A5D80332014C26ABFEE0857B72D3455C11F596`),
		Y:     internal.HI(`0D44798BE9D50928A8C63E00F2222874B230DB0813605B8DDCD8A3E3724111A9FD3A4D70BCBE4FF92958E433A3B973D56963101A5F0FF8F6365A341B6F542546E49083762F43E1DC838F97D41B3C7086B6CE33C189840BC1E70D0A1EA4BDE91300F63D0BFF3D4C3ACA70BD4D870CA0548104A759B4002E9FB722AFEE812A190473EEBA7920BC14C40625CE95BE0F63E2C51F13772788A78BFA93BE6D5249C7A279DD6C1898DFADBE5910F2382BAF5F8C56E018BA61314CBA8DB59364EDCD1050B176CCD8096C9906A7F99A786641B50444271C7C2527E8FB85C9356CC863BD3D09444F2BD774B7A92D6BFC5DA97806A22EA472248B9DBC90DC0DEA9519F62C3C`),
		R:     internal.HI(`0AF8AD2BB06B37865681F9AFA256451C43C9214E5BEF517D4CE45A55`),
		S:     internal.HI(`324E94D9FA6E9AB2FB12BC06116E82716261F0D00F6DFF08D873E025`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`72A65615CD1D41B11862959BB826158B170ACA29676B34A97D01196F7EE60CA8B73723405CFBF01A76693DFBD54B9801398A391C8DAB416C7647E1E6F239713565AAFE30CE8F54DD67E21155891BC28A0F299E3EFD90DA25CAC32290F07E48922D29A0F92A082414D64CD59E36B673C4FEF706A5B26B4833995B74E74C79852327AC6BB65FDC3D6417ECF4D32284B3D6D3CD26718CD6FE8069B8166BC43645D1D4EE688B41C7A6FAE3D282F97E35B46DC34C5CC956B2967C2B4DEFA7FB09CD0F1F7449A290D08F895FD53B1E63C0CEC16CDDAEDDC32BD41E3455902B7D8D88D7578B6730F1454D4E15938258CF66818FD2B0CAE36CB1A2E744D33091BDA70DAC3648C56FF2BB5F0EF97463C3ABAE761D63BE041BD30012E18292AF7A19811698DE870BA20A116B1565A59184C8687D3AF3C85ACD63165E997D249390D5918A2DD6EE8B16966B41361C1F68CADCAC8F3DC05773476D40EB2B2CE50D831B91748510EA3F1CDD39D6CF4A9FEBC98AD9CE046EBC9BE12C0F4134EEF6F40D01850843E6AEC9120D2F4AC382ADC6BF579880F40B0FC8FBC15C14DE8A44C8F082BAA491173871593B4CE380BF974CF1B5D817FD0B3197FB374A3EDA3080B0F482C4CC22CE4D295E65D50FF863747AACFBD22B944DCC4F518042C25E75267CECCE7D2D2F9B798C93735965A93A24F2466A047EB8144FDD75`),
		X:     internal.HI(`299543CD19C36576039DD9F29641AC1AB1951975454ED2B02C31FD82`),
		Y:     internal.HI(`67BE80F3ABE8601996F6DC765F82126646FEE0F705543741A857D3D722144174F5D08BDC6C34CB3E00F53DDA59FD24EF196FE6BE63F8E8C1A46745D091772D2FCD98A10461A77C644FB69CFE60EFB69698B3170EE595F3AA7C0A692FBBC3CBEB6ABB320A4CA512FD0490D163A18F8CFF7EC4584CA3D6AAF31CFCFB70CD749940197AC7D62D884B43E35D31D62F556C4A38658E4F6276885E7E0636BB1D9CA927CA8825C16CAF127F29DA8EE4F3CF47F254E4FA6156F1B2E6E0BA03B8DAEE15B498C1D1B4C0CA329C014CB9BEBBD84EE1F22641E1C9A00F6D8877FF1DD3C8D6B5B46C92E9A956A8FA884BEFD914030797BE9EE7113EE91435380ECCA572BD0A5B`),
		R:     internal.HI(`7529C478311EBDDD81BDA4EE22A9269A7CB24A37BAB8FE0D0ED69EC2`),
		S:     internal.HI(`2B1CD462D6D6100CD228219E1562E8DC79AB6139B65D5DB3B097B520`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`2D5B78A3AF1A0A9B7C7812DA7531ACA497774C011C2936F6EF0EF4D6E4DA60A98D4682462101D61351C534FD8556EDB14EB55B9993497AA64851BC772858C2D95008152A290896B9834CC4875081829DE3F7895619C225C559586FA8CF80A5FDCED5CDC1BA9B45D96A3D6EB16936189CF1A9D47C2274D9B53F099EF470CEDCE0E64F0D22CC0D27E9EA32A85D6AE4A60F9E9C03F72D267CFDA3B2A392ACA38073027B000C619CC8866C2BAF14F9E2687A950CA65ABF8A404EB1069D5F2848F2D311C997E6826AFC72E3928708BF6CE68E0A9D11E064369B81202575E094F1DE408C0F8EC14085DE9CCA39FC1370AAEC20BB59656EB2AB4D962E91DD44A9B83427748E6954B6DFD3172257A5B5C7AA902FEBB4898E47525FB7A2394D60249E2E1850EB71FE03568521758EDD19896031E267882A74C87A1F33CA7107B1CE8D4FD03037BAC752ACE91DD7E7CA117FAC74878319C1FCE15B26837CF7125D76576030AEE71E5FD48F3C99E2D357157D5447951B118BA94915544717F04032F2B87243EADC401DC2930049B72B3B476005DFAA94848FA8BDB0D14981E829A4234FE03C8B5C8B015D3300090130F2720956BC8DD9EC9DCB021D0B7929D52ED3EFA94C74C21631B2EFD353DEF38CC10564C5A12C602E4C8EE534BCEF0313798247369F7048212F80C9BF52F4464FB41C65B79E9E2593F914`),
		X:     internal.HI(`2599B3BC2466461D8FD3F881620BCA5ECCAD01554CCAE3369357C13D`),
		Y:     internal.HI(`03F351C9EFDBD7FD9C7786FE9643EEA287E0B7ADDA60CC3A6F7FAB4D54FC32124207B7463A9F424EF46A9E122D72E96457CF3DB43854274312B06D2527F9686658AA3EAF1307E86A95A0A9942EC66AE593C8752907652D013BC70ED9F98E542940996E3EAACB5FB07326F6B1DBD96F3062DD9EA35BFE216FAE62B29FE53122D8DC7A1EAC44121DAC79333BD1FAEE12BDDEFC85AA7B7863B0B61E72271F821A137BA76385BD92CA6FA2C5BC2D5E54D90F67CC373D6AB31AB61451212E313790320695DDE430AEE14839D7021938FAEF884527CCC8CA31223C2E85BF84036D1388645B0521A5B9EAF05A82B09BB3F35771F7553EB5CD6C466BDB5B4A159A90DD22`),
		R:     internal.HI(`99AD47AA215B38E347396C3E98BECA5A1026D8F58D64C65ABD56647C`),
		S:     internal.HI(`823B000C00F26B7256A84C1D10AA2E4363EEF7B07BF92780717B89DA`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`EB9069FA2020510BD85CFD4562442DA2A13EF2A077077C0722ABCFCC2AE445C7D7D50369C37B3C0A81C79000F8C2372E61A45723804D487C1E00FDB342C279F844607548275F215D64DA2AB28AABE12BD8CAC7A049433AE121B6A06679C4C650E848BEDFB3DB120FA46C817507721914D38BF2458BFB5864E0ECA46392DE8CECA62C413F740709728BDEB52E0690B1AD464AF107A892907C3C455AF1F80F15358630CD421F01E81E8D194D85C88566024EF047A2AA2AB9E63AED7F1FBF5B90DBB705968713F179F6438C37EF33DCDC672FF1DB9B41EE8FA7099934C3A3D014D594DF3A7854076F257030CBA6D8259F7757450040C813BA0F0085077E0684A3649C7EBC45917A621B0085C8ADEDFA231759706BA43FD3C7B29C76E8B4F693230F78288BE71E89D792039455CE53FCC471F37C42A552722D6E85B73495242266A6F8AB7C20FA7B358CB5EC019D92D4C7FB08FB0CE7503B4969891DAE19EE602341165DCB77C99FD05479A7C375D734576FA607BAD7348262119F0480FD5681FB3FF11C1E3EDA4EE27AD864FB77FCD588D10144A5ABA0A0A51CE5513EC807C37449D250828D23E68CD9874C6E8F8079566E75DA8E5FDCFB2987A270E3CA576A004E2AE66B4640CED9925E0F4D7189E8A6DA877D9FB7DBFDEB994555D1193FC5F7869154B7117777BB1061E62D95E9F543EFE3666740`),
		X:     internal.HI(`1814E0975FBF39BDF48B4C125BFBD0B6291E428FFDD52E7004BC461D`),
		Y:     internal.HI(`7658DCD82D8F226130240BE9C42601ADB452A3FC8EDC408D4C36A5014729C64C0D585F0A87CCDFA049AA50EA902CB95AA21525D8CAC12CFBE7E241FB2CAD90B0A047F594503AC6D6D5A7D9E82FA8D0970B261869570B999E9658F8A2E38BECDB116BB319AE67C79B078DAB1796649B0AF08FE3D9F11ABF65E0BFCDE0D25C8B055DDFE2E397C21B66C85EDA8901781AFA0331F29CA03D345CD6BDF5EAE5DEB3A9084A437E1806381A1223395CACA98385950A67402D57CAB69171DBDA447A458D321E5030269FFBC6087C38250BD551E9296297344901E8518D056E43BBBC2759D287CC8A8F58157CADB4C54F43C3BE39812F45B04B76729488DB54DA2E33B679`),
		R:     internal.HI(`21697C891BF87A95C83B01DC8AA0EA1325F5866895CE8E4248DDE79A`),
		S:     internal.HI(`2906C5204062C6B60FBFAFAC563C6FB447811FDF28A9DD3AD019AB08`),
		Fail:  true,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`B8705A54FA90E41FD9B2C41CB4B300F5E10D17B116D3D97D9C18A4E071C99369C04A3A9AD5CD4523FEC5C4227259698EB6497D689923E1F1423ACC785D3AA6F7EED0BA16D4430AD684670E6DFAD9A7957FE507E72E04E46EDB4EAF2055DC2055896DDCFD9D447B9F7BA2C0E3DBC43CC258CF2C058D8BDF3DCF6561DF7C88BBA082918CCBFCA949A99D8AE0B1CC38139306FB4DE2985737702995DB091D9E94B96FB5A042E5D68FEA4C3A5C4DAADA804EF268B2E6578FBADF9FFD0037AA06324B905CD26F75B5CB1F92D50B757CD53D04321A8BC1F9E19C2A90C6974BBE3186CACD0DC8A6F1B7E9CB1F87A92F71DE6C8A801FF06AD98379BB001E556F1B17E570B65B0C82C4D8393D4E83DEC9E0309B80418CE122763359C29E3DD414AB370F3F81DF13E785997389200537D94791BA4C7F7F46707938A739BF6295F5829C2B030F3A3A02EE05B98B3F4F2A3B4D4D271DEE1CEF24B25F38E060D50213DAD4C84FE715C445E5AE95E8FBAE1617BF38A2EAEB9295561B004A3F26E46EB815F8DD7D3822DD6D77AFF60E507340DA96AE59727715D665D3F981A961E71275BEA9CAB0DB1A997ED8A93731DDFBD739EE94DD3C40E23CFA23B0EB34043D1226880F54D44EBFF5C364C8184DEEAAF13310552995974037057C13FDC3AE4E75EB4DD9AC99B9D9D4D0A0BCC22973E98B0D69E7A2967A7B1EBC`),
		X:     internal.HI(`5C0EAF94E2369C84A492BD73779D375CEA5C06A7EDA0995D03E5FE6B`),
		Y:     internal.HI(`1B42DD4EA51692E8C35A5D550F8C3861E9C24A120BACA9A1EFB166C549ACBF084A613920EFD0FB00874842F0D6063C03B9B95D5E49A5966F9ED8924135632D389473CEE704E3A4E60C6366625F3E9B19D3484C9B2E7F49146759913D8B32EE4BA13AAFED1B86B1A3DE11D6A57176850C3508947D34C36A7359676DD9ED84778D9B01BD2CD10318465F507170D24B6A047F950F34919EFBE3C988A38E1C4BD9590E9E7A0BF9981BB98B8496F7C445A7573F2112AE7F87EFC8E9B978ADBBFB484829711184FE700805CE6F955E5DA0154EDE191F3155DE1E1AA024CB61D86661538D4C0D982AD9A7F627804AADF82F5F7CA9BC5FBDC63143A0141D88962148C9F4`),
		R:     internal.HI(`D9DEB2A7A5429AAE058AD121420245F8CD057016D191D2DB54925F27`),
		S:     internal.HI(`2A069CF184F2453900C0E5F895F9C458090E6EBED6CC4780088A6F82`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`FEA589302E0F41687DED3A038138C2C650062D8D5690C5111D8CE931A0D23FF456D5651BA1FA0787C15FBB35FA029B1E17E8F1C3E243F4AAD018E69979798CCB04D462F93E60737170313F07A7FA912A9B04B8E4D06A01091F11D966567A8F2F08E0067447BB55F55D29ADFD3B4376B4C2EA0E8D8612002F41142F8F8F0FDE9B2CDFE21F45AE064822B8F782B22CA6EC37D58646B1738A8834645CF6C59997F5005D8C720B04660C24F417EB5028026C6FA5BA7C48EAC8E8BEECDF61DF9F618BDC92A4D0B2AFDD468B9D0D74A1D4F535A8E54F8587FE658870403C810CD26911E359D1839BC95B664C1BE44179F570AFE7C5F0A0F65F940D9F9C01EEC50B66A4FC39C3BE72955A43227DAD5EF376EFACF72050F161E212826BE3C328C94A97CAD95F309A277FD81C8E7B80BF746E73666E772A88DE862359BAA0209721B9C16FF1A1D81AF5175F97DB757F41A617877CA9F4425BCC72946D3B08BC8C1CA633E9877D83265C1A00C51D72D4A67FD73DF8CC676346D0F4B90265F5463D518CC4F4A31700932869511C2F21B09C393B304BC44C6112D8826EC276EC71CCA10BD1B5153E2419690E737AB2DA4BB658F6814F47C3166B1ABC18C17718FD4033EB42BB7767D25A7B3E1028139DE870A9E6DC43D09566E71368A37A344CAE8A781C83F929B0EFE1FD5256D58312D02F400F72D2A5353C06`),
		X:     internal.HI(`8A4BB905BB83270F3174861E1C8C1989EC688BB9CF2BF4A223F6228E`),
		Y:     internal.HI(`5A2E47BD0BCC8E8A2FAF1AB44D75305F3450D3D7ABE9C6714E4E41491BD54D27DC51FEE96D58CE4087CB8BA642D757F2275048AA3C16BB2B0988788492DEB7DF74C382D6281A6ADA3871F0429755F6D791C4A660EAD985B96E3C11DC61F859AA1C8C848F1CB297EF7D6121F938672AF4763FD5C259AD49986995009F54E68B42A76ACB137AAB9B232589259F96A77FC41D6DB3ADD1462848AFAE42947179682BE99A6AF1021CEFA726C60D8F1492615560ED84E1884DB02098105BCFB90BE932188570CDDDD7EC299A296BD1FA839ACC90F58FD081F8931674F10F974B05B41D01952BB68541425C38E8209DDB2E5C8EE7BA6BA591CCFAE64A407D19017DFBB2`),
		R:     internal.HI(`DC9282A1F8690526B2610991FB6452188F6527FD5DF6AEE3E47BBFF7`),
		S:     internal.HI(`39C6472B4F8BD571B1AC6463B2F648EF0B99B856D1D69127EC0EE06D`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`CF4605518127B88CCD54DE6B20DBCC0096FE87533E779EFE0CD3BD8C2C8BF7B50D0A65BBE63FFEDBFBE95F3D7146584BEC46F4CCD2F3E7C16B2EEC50A86E30BDE5A938CB60095876B2A1476516B8C701BDE6FA9C079F6687CF532E8DAFCF85A67310F265801557EC86FD56EA38F315735932DF2BE0B0C1A281A7934D619584073AF6AE76BD3AB1527344F1DBEB006479774393EC4AA1C4EC3356D6A5C3EEE5FA26DB30F479994948E089244F2731FE7235FDAE57193763C7FE5757AEC34F86248804E3DDFA9826F399A4A666D02156481B0A6FEE097CBB1E6364218D37756DB11D82DC3571E87A02D336D447AFB0076A17DDBE3BBEC410624A04E46CDD68C857072AD50AF7809CAA2CA7B3227609D4D281B22ACEC3ADD08F0583FB805C74F050D19B32718AA10EACA829F22DBF9EA6FE198AEB408E188E264DF66803413061626F3D008813D2784CB4B4E6A90D2890F80431E1347A3308324339D2390279C3D83B3EEF7363E3AA5826098CDCC9AACD50D23A9252CB70214570F28D6CFE75E487F9955E6030EC9E293AB08A15466CC01E7AFE2F4CAD8CE77AC58D92F17A93BBCCD4014F851C4C739B96FC2EADBD02F2025BA28FDB358B8D739B3F8223A587668A5E086E1FAFAC721447046C01514318263D1032C15EB87159B305A665944F2C2F93F90F7258FB0C84C1A9E47B0C540A384EFB40C6`),
		X:     internal.HI(`4BA3C77369F2EE37EE5A86837BBB229BF10466096662760594AFE8F2`),
		Y:     internal.HI(`54D799FE583E9A0EBFEABFDC5DCA8AA3F1FEB6D22F919286CF74D6026171AB2C2AD663E2E411521BA6F4F1470241CF542E2FE64876EFDFC91D0580B8B87D4CD740ECC0EE07839B4F3A7E65783F7501399B7503FF435E953C768C10121B08A26D2F7996D76CEFD36A862C8504322629C5D652418535525395569781D1D3649C6C22BCF9975DE5E2F13ECDC66F89A68FF2F048CDA35CCC2ADA4C4007B5070F508C4FF7C08D90D55E736D23AEBA2C1B94DD43B49D811E1C65F5CE8D75568B67E008FE0E912CC170BE4D46C6AE6BA129E763560B3918D18DD172942D99A7EF6877D92A32AB0D851D757EF6005FFA528CCBAB17EF24F21F46F12BC667C732F8BFF1FF`),
		R:     internal.HI(`286E4330C0ADF4E0AB8382087E42A59B3FDE0A0A9286E116179A5780`),
		S:     internal.HI(`6E3B271E6F4232F556FB8E3152257F3C6A5273628DB194D39A264DDC`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`E037A3F22146327BADD5EC2CCD6F72D961E294D988DF9DCC7225DD87BC0D342C42B7A8FDCDA1BDEB82F0BE6B85358F9AA9EF373AA0A5CF8893C1FA7F7C128383412C6609E273C639B2181715F6D11A38E195C99D4320E692E48A5B49E8BA001B27F8FC01E5726A4CC9E6661F875425C219E94B3EF8FA2262364FD5ABE9875E17C7E44D76603C1133DB5EE125A5B60572E86887044EC50E9DE146021570E599C37B1F08A0E756662386EC8869C4D75CA96AF70D7CDDFB7B0CC40E449E772AF5952545A0631530627DEC6120D8627E11EE46E436DB43FE83A249B0C6070090FE29305552478D1F40C6BBFA3A05055C53F2ACE223002919897A5F997CB7103D88448DB9237DFB6186AD27592B2B3A0A9B8E5010BC6F3C7C35D67FA120BDBE3BAED1B842E1E0131D0409EB8A122F9606A8C272F2B35736437920A90635D1207FD5E6B2291FF28E5FCED74D00D79BB7BA83B7D776808CD56F90EC6671055258E6A7C0070F3BDB32204340199328A441747BBECFF0648DE0EBFCF4C6F0A44890331AC2C7FC5A6CC83B0790A38A7B24E26E274F2F1E687E2688851B61F9EC63F812687A8E6367212578093FC78E119F50C7680A55245D2F80013E6A576D80FACB17169B7D1C191A238580EBE9B3EF42488865B92890DB13CCF780155192CC0D48BFEF034068EC21A7F8E75CF774E4DF8EA08F4B17564349`),
		X:     internal.HI(`6D92225FAA866D025F1DDBF82EF30E6C66D35B32AB413F7F4C2BB380`),
		Y:     internal.HI(`880809A5DA4957A5C806D9CB5CF3B39CB913B58D708121237F54513143CF14E3F8C04F84E5EA4FCD1CF7C114ECDF29DA24B526F9AB182F596D47A26A97F90072103567657CB07507B290E52D76F9AF4C2C38366929CE37570AA051C33D80B1D358CCAA352173E42E25FE4F5B1F62D3E619DB02D95A02D4D6E9FD0399F2FF8592210B0421AE810CD429FD4CFF77FBEA7CFB14D7804D63E266FC0898422FA93443F6843452D48E884B537D0F71797F29A24AE62D9512F1DE751A9AA68D37FA25BF2ADF1DB7FEF1F9EFB06D2B9964C16A0741164F48D784697B044D7E754CE8E6F029647649D81FC556795654F1851A2B840BF7EFDD9731615F8569E3CFA9AA12BE`),
		R:     internal.HI(`31AEA02513CD91EECC414AEEB99555169885D0CA22536BDFE294709E`),
		S:     internal.HI(`2E17103EC06ADA041F9ECB0E7F9579503FA63637DAFF5D4EA73D3A77`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`767F180143BB6D07B608F124A79A4E067E64B487F2F1E79DDA17D5BF96E06D721099141AEC736EBAE190C6CFC7F1EEEA6A2CBD389F461E8EA6A7D03E678FB93D8DE107DB00CBF5ACD71A63C140A725EA5DD4431CB069EDAC17CE757AD4FD1AC8C01DD8CE1D241FC91770A6977EC02BA2BE6962239B41E384BE185AF49832BBD6A39ED5468CE6F04444E5DAB692451DF81C7E9B037ABA79F0539E7AF4193DAF71593EB55E3B8453992A53B04B354C001B2D2FDB3B11CA0F10B8FD3B8B6636EEE92D5B98F9C4761D8CBF1C404BC8EFC481D0227313CB6FEC4AF45B6B92333B5CDB91DE05C1653E0A291F2B0C7256513EE60C822098BCAD424F39673EAADE75C1261F38EB290565BEC48FF1FD45909E2C510E0404A2A0922716DE57513B6D12D1F49B61A06C357DC5F87A6A6411CF0934102EE4ABCFDA339DE065E9A9778D4A26B9EFD9E58D2C1E93AA7618FAEB15CDE5B8E9E6068775AD8B347664B457945C402C2CA9DF57C9EB85044004DFB00C2EB428E85773F72626C1E2E196479A7F8F8B518F611E5C948CDD7CBBC19E0405770085F70BB31847CCF702A0D69DCAF4335971791899F8BDB3C9CCF56A2654FAFC0E3C0F5FF1A7DDD2CDF4D2015E38419FE31E7570B04D1B195CF921A1D2D58C170C054C36C12C9378CC5FC07D95FC5B324C32369128452E7F924E9D906285052C12DCF7FD1EF5`),
		X:     internal.HI(`642E96AA4E72181BA90AE8B2BAE130B46FA851F64D65FB8C283E9A00`),
		Y:     internal.HI(`7362EC4D8E631B97078D033ACFD0AE324AA1B8A925638D64302F63E4C72001973F216F087C5D8E0A9E3F133CA664E7B0468A321138633373FE77F957D68FB8DA9633B29E6D4AF22892E355BA67F527A24F7422C539034A2A95A806814972DAFA04764C65FCA0ABCB28F16D7FED0E047DC5ED74AC31AB8AD8BCADFFD0F7165B3A2148E63B74732F37FAFC357D8652F1A0D5A78F95C46C374DCDEC4C6A8C002FD6FEE37547AC82B87E40EA07A61362A682B66D66B6E32676FEB30D47C22BA3F9E1C3BBF273C43F73769AC0B281D8025B849722A863FA98D950DCA9DEB265F9D212166CABF569F7B5FBDB1EBA6FCEA3FDBFF0135D704B1305BCB819CD6AFF04E4A2`),
		R:     internal.HI(`FB0C03BC4ED157DC1A317E239ABD9B5979C1F2329FBA131A4F08648B`),
		S:     internal.HI(`6E2682C9993766942579D1D31B7BD8242CEDA1C5405A6E27591A193F`),
		Fail:  true,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`53F4E22CFD3F912DA8CBA0E7B9EFE7C6A8A104F1693C9DB7CBA3B5AFBB8362AB0045B0347779FBA2562E78937B2750D15441F459620815AFE8D0E71EC2471314F7E40A3A0B18107E8FACC19E0B38577BB70C7BC3118BE529E147A2C6A6D72DDDFBCB3E96F8A084E82024244ED3292B7BE601D9A8478DF5AF56680193E6FD90A4750081E5A61F73CE7B793134638960539F2739CA005E93948F9D09B8240C86CF778AF20CA2295EE8BC9B5F27746EF34C488CAD325ED776F27A58A6B12FDEBF89BA809839A4D932B4A87D0F47E6764975EE492F32AB58BFAEB011AB42FAD655C8A0FC5FE389D5417AA91D89FDC1C72FA8467DA2655AC9F270714AD677A2DDC94832221FC759484547D381FDF73710DA87AE4FCEAC069B01ADA38CC9A46B65038F221C95EB42E664F2E1B6832D9F87E8782CEF673071C5429FD7691264C36A55E8C91E679C9AED261B38D11BE07BEB5DAF6E93066486C973494379239C3D6E79692A642471E0208029E2EFAD977281A722C87B2E0157AEBC76C85F5A76967A8EEFED314147BBCBCD499336092257189A9338EF48091F06ADB8EDC3F968B3221E1E64CF1C44F9C4CF73A8D3E7992007738C653DA6C741E73B6AE1562E2CA17F1A648993FCD69166B16722FCE85BEF2BD85C9BBD83CB46F3C6B07DD30DC79334DBF5FED70EB3A19607A9AFED93110CEBD61CD0CF00F0`),
		X:     internal.HI(`4B9432152F68AEE1B25C01CA2D2247FE676C39FF1F2ECB6BA6BD16AE`),
		Y:     internal.HI(`59F8310CC28D69F3BB3BE07A1F106F3B476A9B4D97664F9EB4B8E19D7DF70F5BBF9B8D71D8CFE1B5013FF486FB1E4F119F7B692FEB72F9ACE9DDDD7FB53CE3F457AFBAC6E3E6508733FC7E4EA465EA59CC10996FB98942B0D6E05961AF0AEE10C9FF7A95376A990FEF6690F2899A5EA9A698962E828B732D0AD8D2A113414E1BA1BD39AE7CBEF6BF371DFF2E25DF724F191D0648B7B7673675B46643C4C14A11F58B51EA886D7D6A52DCC873CAC654E97F0E6B844CE9A704B9AA3E0E355CFE99CA1450E48E0FE574A65D32FE07F9EB430F18515AD1275F1FDC4587620A58DEFF9750CF4D57C0F7F05D4173B995A3BAFE94C8F0BF2517AAA7773226859EE89E92`),
		R:     internal.HI(`EF6B83ABCFFCD9476890FAB40112A37371CF8FF3634C4E6E42FFBA1A`),
		S:     internal.HI(`7CA218AE9AA82DA8494B7D94800C2A11CE3D5429674F45E0B80C9E69`),
		Fail:  true,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`5104111CFD0661FE6ED16E27313FBC5B98F6EA7D2C4EA8E14E7CD00EA7658B7F660CA9B1CDFC0D032A8E66A6AA8708371FE2263B3F3D2D125DC2A4953007330A9ABB3D8D198FB691C34C98BD7A69ACA24122F3A14E63817BDA7806DC36F0CF6E308D91A14DB7B7DBE34A852F993FDA4054F9CA811CBCF0730221A80C85FB06F18F8AD4A57F7252595D521488A669AEFDD32F8D7A134BB2983CCBA712910E6924483F991B6EC8B1CE2BB39319E8512E0C641884F0431BE3D0170984A6751870DE13C3DF4C7EC8E6447045BA004F6944E9D28E610F66408A4849F82C46F60C7A40D2B40948BD89EB0C776AA71B7329C45913F43CCEDED69576B13CF0397DE8D1B08C38E6E7E02CA3C0B208E21893126C66433398E7B400DB1855018D8E20B2A3DF73FEAACC46D6D642BB925A6A97AFDF64A6BE5CE19AE7193564FD261B997609C4DE3DF25EF3B838BC56FE664B0E91A8F1AA8FD907EAC0F518356A44804C4B03A04FB1C4CE9508619F6CCDC4BE31503DEEE229CA70A5C8FD5A460EDC23454D79FA6DA38C168307D6A611089C8DDD8FF9890B954FF7743FA8D73D34493337A23CA20ADF1FF6B8FBFED27E3F7E4C9CF620340B67F042AA7353B4EAB14DA77E7703B01FBDBBF6DC342E6D178A5F559C36DFACECB8A0BC3FB7456142E2143E1E026E84CA1A05693B089D096689A0CBB50A48F5E42BB79C`),
		X:     internal.HI(`0B829C0B75255F98CF3ACB10B38245C027E68B3BA61E7CCA5E1BE566`),
		Y:     internal.HI(`285EC7392BDA8564A5741AE4FBA6C7E45B60A97A007E2F727E323828512D26450E95E40384568B5A4F1FE05505EECA87B9AEA2DC4BC84DCDF6C5EDB07DD897F1E64FC75294D5CC827B0BF5FB7311794F4C34A02461E7AAFB3258EEA363E6289B9ED1351A9F2A7E10D845AB7A79434147CFFBB33A85FFB91BAD010B52C88B3C0A89186F4600AC2AE7B3652FE54131B8451975B0661F16AD00EA4D2A83F5C6257B31CB84A128C1C43284C9FBE49F98D844D5D81CA1E51059F93466658A59E655B438293067032BE97CBC1475EF0541DA5D6AD53F5BFD918DB41B0D98003D0CAD426698F3E191F0548AE53DD543E0B8FCD580A6B3E129E18CCCE8FAF20A5FC0D743`),
		R:     internal.HI(`960FADFEABB0FC5EFFC8F4A8E76223C381E29D36A44BCF1DEA8845CB`),
		S:     internal.HI(`3C535B306631E00622ED93DC4FE61090EA86517EBE590EBA3257E75D`),
		Fail:  false,
	},
	{
		Sizes: L2048N224SHA224,
		P:     internal.HI(`8EBAF0658E1C4D432BFC2835FEE088CC9E5EDB8FD06AEB48177594F9B2DC31E37517ECEDFB018F827380A24DABA5D916AEDEAA9D7DC767DC760E81E635BD7610EFAB9C199C03F6A6AE2B1FC825E9824E08C756F69AE4356A0DCC7BC46C227B6FABEDCC1CD61D04720A1B790C254BFCE0811BB657761A57D3764E438CE2C0B33D04BA28EF043A575FE8A062727FA4BB122C2EADEB1EB30B81E36702A25A4AAD20A96A9EE28594B232385ED740D8291FD6CB97B25AC18207C1AF33E2E86B16DDAD55BF4C49938F3F31B1DFD806DC9A637220F693005D33863655920EB5A0FF2E4922C138D12BA99112F9312CCEB2EE55DBC079E3D6A3C5D1CBDDDEC6B6E7BDD19F`),
		Q:     internal.HI(`8D492391758CDFAFDF2F1DED9C7C1A6B28A3FAF259DA6FF1878EEE07`),
		G:     internal.HI(`6EC9C5EA61CDAA4163DCC4DA8C5FEC6CC5469CDC3D86169A9E022DA08DC218C2757183DB9F7E9B49CF9E860C528E92F771EF4ED6C2D796A903D588BC0514219B239AB94084B7AB83F90ABEF19BC012347C0D2ECFD0EE80D65C08861D54272FB648759E1ADE2D55BAD3C050E4F066670B36468398BE6CBA75FD518B8DAE123773C95CC81D789776C62CBFBA0A8CC7CE0A2EC1040D87320D0E91A851C36F4BD1D1C23140B548F9D994B8E4ECC3C73356B709B47563F56CEBBE1048BE3FCCF3E0217B996A1C022ED9026131F00677A17387D884336D28BC0F386B07B30C3EA36118D057299D24685E9BE7B1652508AB3ABA36D03E37C8F9EE0579BBAEDB58AFF7DC`),
		M:     internal.HB(`9264AA307FC17F05C5575D0965CBD8707F1E6507996732595A50EAF6DEB7E09C4698769DB7BC34396333753ECD2A0BAFF219CFC372DA78F670B4A88747D49516C34A3ABE00096F6196F9D77E499D0F4DCB59FD6BD1DCACEA11BD9EF962C8F7FE9513735D354E84BEDE6982114A125D23BD0AB200F7E6A9E9623E8F4787D3802494B025AF00D4D177036CFCA3091E73F5C07D34E5F1165091CE35C6D2D0E0B53CDC09DC4DCE91BC9D141153498700D96B172E55EC91358A640AC71462247F20E5D42BAD3CD61EB1286A921F7F8B9B1E194BBF6C4D72B248CE1240D4292EEA55A5294D33E618C126F8A44F7C2CA87EDA772FFA5AA4F7A583252813E9BF6303F0EAD2CE931D5A6498D4AAD1129C36DFABE9DCD388FB4ACD3DA3D9DFBC260254920B0B357A1E2A9B8D6EABC80F84559C3AB8B463E5BF5E92806DF96840C70D0DE9455C2F1C8CE1A1915C200D2902F038351660EDEAC8EE035C90A399EF725209A6BD919434709B5B3B1FC8A09E9AB5E4541DD3DC97557DD8EBFE3C88CC6266C98684C162083F42532720ACAA35391F7256D047C3750D56704D956E716036A6764C8D48BF64D482BEFBAD1B7C3C356E6302183D5B93008BD4AE182FBABFF936E3C2B8CFFB9B71D3766400AC8D894AABDA28C77E888BA2F7B23D33B9ED821A0488BDCB408C8BC27300173840807B9EA6A69E991E547CD5`),
		X:     internal.HI(`0CA9D7E19AF718447C1C8C22B2C1250FDE500E8D981F31BDB23E0BE1`),
		Y:     internal.HI(`8B33F735D63F0854A6BDED76FA034C98BABC0AA90FC1E2654AC88760A3FCD6CD66F0B6780B3CACD686EC94C68C1E4F8F8A6E91578BC47AA53C9E5426A3BF41A7CF48DC1C4D9FD6A4DD49B5F887CF182055F5DDFAC94CBD0320D763C44A097695764802D3A327E322974136C9239F474CC670680E58A15617683DCB784184B72DAD7FB289F26518D80C1DED61611DC0F1823CD33B2E7A6280FE7E28A7C55BFBEA18F4BEA31B1EFD9601CEBFA232FD0C80ED25C08782FB52B53FCF752B395F773F4CFADE7929D187B5E8A623F8720161AD7F817594A768945C98FB905945C6797B27C6695C27B86DA6F5CF070522A50FA63FD27E95DF85189ABF062B491D95A21A`),
		R:     internal.HI(`F6E46410DD66D7CF8EF46D451A25429101D331FB6666E9B4837E6ACC`),
		S:     internal.HI(`88E920BDC73C49A729FC5C82CF21BF106CED85CA76280191AA0C1FBB`),
		Fail:  true,
	},
}
