// Copyright 2018 The gVisor Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package linux

import (
	"github.com/metacubex/gvisor/pkg/abi/linux"
	"github.com/metacubex/gvisor/pkg/errors/linuxerr"
	"github.com/metacubex/gvisor/pkg/hostarch"
	"github.com/metacubex/gvisor/pkg/sentry/kernel"
)

// copyInSigSet copies in a sigset_t, checks its size, and ensures that KILL and
// STOP are clear.
func copyInSigSet(t *kernel.Task, sigSetAddr hostarch.Addr, size uint) (linux.SignalSet, error) {
	if size != linux.SignalSetSize {
		return 0, linuxerr.EINVAL
	}
	b := t.CopyScratchBuffer(8)
	if _, err := t.CopyInBytes(sigSetAddr, b); err != nil {
		return 0, err
	}
	mask := hostarch.ByteOrder.Uint64(b[:])
	return linux.SignalSet(mask) &^ kernel.UnblockableSignals, nil
}

// copyOutSigSet copies out a sigset_t.
func copyOutSigSet(t *kernel.Task, sigSetAddr hostarch.Addr, mask linux.SignalSet) error {
	b := t.CopyScratchBuffer(8)
	hostarch.ByteOrder.PutUint64(b, uint64(mask))
	_, err := t.CopyOutBytes(sigSetAddr, b)
	return err
}

// copyInSigSetWithSize copies in a structure as below
//
//	struct {
//	    sigset_t* sigset_addr;
//	    size_t sizeof_sigset;
//	};
//
// and returns sigset_addr and size.
func copyInSigSetWithSize(t *kernel.Task, addr hostarch.Addr) (hostarch.Addr, uint, error) {
	switch t.Arch().Width() {
	case 8:
		in := t.CopyScratchBuffer(16)
		if _, err := t.CopyInBytes(addr, in); err != nil {
			return 0, 0, err
		}
		maskAddr := hostarch.Addr(hostarch.ByteOrder.Uint64(in[0:]))
		maskSize := uint(hostarch.ByteOrder.Uint64(in[8:]))
		return maskAddr, maskSize, nil
	default:
		return 0, 0, linuxerr.ENOSYS
	}
}
