// Copyright 2012, 2018 GoPacket Authors. All rights reserved.
//
// Use of this source code is governed by a BSD-style license
// that can be found in the LICENSE file in the root of the source
// tree.

package layers

import (
	"bytes"
	"net"
	"strings"
	"testing"

	"github.com/metacubex/gopacket"
)

func FuzzDecodeFromBytes(f *testing.F) {
	f.Fuzz(func(t *testing.T, bytes []byte) {
		dns := DNS{}
		dns.DecodeFromBytes(bytes, gopacket.NilDecodeFeedback)
	})
}

// it have a layer like that:
//    name: xxx.com
//    type: CNAME
//    class: 254 (QCLASS None)   # [RFC 2136]
//    ttl: 0
//    data-length: 0
//    data: nil
var testPacketDNSNilRdata = []byte{
	0x96, 0x99, 0x28, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x00, 0x00, 0x02, 0x6a, 0x79, 0x09,
	0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x03, 0x63, 0x6f, 0x6d, 0x00, 0x00, 0x06,
	0x00, 0x01, 0x0c, 0x50, 0x46, 0x32, 0x45, 0x41, 0x5a, 0x43, 0x4c, 0x2d, 0x42, 0x49, 0x5a, 0x02,
	0x6a, 0x79, 0x09, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x03, 0x63, 0x6f, 0x6d,
	0x00, 0x00, 0x05, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x22, 0x00, 0x1c, 0x00,
	0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x22, 0x00, 0x01, 0x00, 0xff, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xc0, 0x22, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x04, 0xb0, 0x00, 0x04, 0x0a,
	0x54, 0x4e, 0xf1,
}

func TestPacketDNSNilRdata(t *testing.T) {
	p := gopacket.NewPacket(testPacketDNSNilRdata, LayerTypeDNS, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeDNS}, t)
}

// testPacketDNSRegression is the packet:
//   11:08:05.708342 IP 109.194.160.4.57766 > 95.211.92.14.53: 63000% [1au] A? picslife.ru. (40)
//      0x0000:  0022 19b6 7e22 000f 35bb 0b40 0800 4500  ."..~"..5..@..E.
//      0x0010:  0044 89c4 0000 3811 2f3d 6dc2 a004 5fd3  .D....8./=m..._.
//      0x0020:  5c0e e1a6 0035 0030 a597 f618 0010 0001  \....5.0........
//      0x0030:  0000 0000 0001 0870 6963 736c 6966 6502  .......picslife.
//      0x0040:  7275 0000 0100 0100 0029 1000 0000 8000  ru.......)......
//      0x0050:  0000                                     ..
var testPacketDNSRegression = []byte{
	0x00, 0x22, 0x19, 0xb6, 0x7e, 0x22, 0x00, 0x0f, 0x35, 0xbb, 0x0b, 0x40, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x44, 0x89, 0xc4, 0x00, 0x00, 0x38, 0x11, 0x2f, 0x3d, 0x6d, 0xc2, 0xa0, 0x04, 0x5f, 0xd3,
	0x5c, 0x0e, 0xe1, 0xa6, 0x00, 0x35, 0x00, 0x30, 0xa5, 0x97, 0xf6, 0x18, 0x00, 0x10, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x08, 0x70, 0x69, 0x63, 0x73, 0x6c, 0x69, 0x66, 0x65, 0x02,
	0x72, 0x75, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x00,
}

func TestPacketDNSRegression(t *testing.T) {
	p := gopacket.NewPacket(testPacketDNSRegression, LinkTypeEthernet, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeEthernet, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
}
func BenchmarkDecodePacketDNSRegression(b *testing.B) {
	for i := 0; i < b.N; i++ {
		gopacket.NewPacket(testPacketDNSRegression, LinkTypeEthernet, gopacket.NoCopy)
	}
}

// response to `dig TXT google.com` over IPv4 link:
var testParseDNSTypeTXTValue = `v=spf1 include:_spf.google.com ~all`
var testParseDNSTypeTXT = []byte{
	0x02, 0x00, 0x00, 0x00, // PF_INET
	0x45, 0x00, 0x00, 0x73, 0x00, 0x00, 0x40, 0x00, 0x39, 0x11, 0x64, 0x98, 0xd0, 0x43, 0xde, 0xde,
	0x0a, 0xba, 0x23, 0x06, 0x00, 0x35, 0x81, 0xb2, 0x00, 0x5f, 0xdc, 0xb5, 0x98, 0x71, 0x81, 0x80,
	0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03,
	0x63, 0x6f, 0x6d, 0x00, 0x00, 0x10, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00,
	0x0e, 0x10, 0x00, 0x24, 0x23, 0x76, 0x3d, 0x73, 0x70, 0x66, 0x31, 0x20, 0x69, 0x6e, 0x63, 0x6c,
	0x75, 0x64, 0x65, 0x3a, 0x5f, 0x73, 0x70, 0x66, 0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e,
	0x63, 0x6f, 0x6d, 0x20, 0x7e, 0x61, 0x6c, 0x6c, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
}

func TestParseDNSTypeTXT(t *testing.T) {
	p := gopacket.NewPacket(testParseDNSTypeTXT, LinkTypeNull, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeLoopback, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
	answers := p.Layer(LayerTypeDNS).(*DNS).Answers
	if len(answers) != 1 {
		t.Error("Failed to parse 1 DNS answer")
	}
	if len(answers[0].TXTs) != 1 {
		t.Error("Failed to parse 1 TXT record")
	}
	txt := string(answers[0].TXTs[0])
	if txt != testParseDNSTypeTXTValue {
		t.Errorf("Incorrect TXT value, expected %q, got %q", testParseDNSTypeTXTValue, txt)
	}
}

var testParseDNSBadVers = []byte{
	0x02, 0x00, 0x00, 0x00, // PF_INET
	0x45, 0x00, 0x00, 0x38, 0xa5, 0xa0, 0x40, 0x00, 0x38, 0x11, 0x00, 0xbd, 0xc0, 0x05, 0x05, 0xf1,
	0xac, 0x1e, 0x2a, 0x43, 0x00, 0x35, 0xfd, 0x78, 0x00, 0x24, 0x40, 0xc1, 0x8f, 0xb3, 0x81, 0x00,
	0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x29,
	0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
}
var testParseDNSBadVersResponseCode = DNSResponseCodeBadVers

func TestParseDNSBadVers(t *testing.T) {
	p := gopacket.NewPacket(testParseDNSBadVers, LinkTypeNull, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeLoopback, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
	questions := p.Layer(LayerTypeDNS).(*DNS).Questions
	if len(questions) != 1 {
		t.Error("Failed to parse 1 DNS question")
	}
	answers := p.Layer(LayerTypeDNS).(*DNS).Answers
	if len(answers) != 0 {
		t.Error("Failed to parse 0 DNS answer")
	}
	additionals := p.Layer(LayerTypeDNS).(*DNS).Additionals
	if len(additionals) != 1 {
		t.Error("Failed to parse 1 DNS additional")
	}

	optAll := additionals[0].OPT
	if len(optAll) != 0 {
		t.Errorf("Parsed %d OPTs, expected 0", len(optAll))
	}
	responseCode := p.Layer(LayerTypeDNS).(*DNS).ResponseCode
	if responseCode != testParseDNSBadVersResponseCode {
		t.Errorf("Incorrect extended response code, expected %q, got %q", testParseDNSBadVersResponseCode, responseCode)
	}
}

var testParseDNSBadCookie = []byte{
	0x02, 0x00, 0x00, 0x00, // PF_INET
	0x45, 0x00, 0x00, 0x54, 0xfe, 0xaa, 0x00, 0x00, 0x40, 0x11, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x01,
	0x7f, 0x00, 0x00, 0x01, 0x00, 0x35, 0xd6, 0xaa, 0x00, 0x40, 0xfe, 0x53, 0xf6, 0xab, 0x81, 0x87,
	0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00, 0x00, 0x29,
	0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x0a, 0x00, 0x18, 0x36, 0xbf, 0x11, 0x1f,
	0xef, 0x2e, 0x01, 0x09, 0x7d, 0x8f, 0xfe, 0x06, 0x5c, 0x63, 0x6f, 0xfb, 0x14, 0x2d, 0x76, 0x74,
	0x94, 0x40, 0x7a, 0x73,
}
var testParseDNSBadCookieResponseCode = DNSResponseCodeBadCookie

func TestParseDNSBadCookie(t *testing.T) {
	p := gopacket.NewPacket(testParseDNSBadCookie, LinkTypeNull, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeLoopback, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
	questions := p.Layer(LayerTypeDNS).(*DNS).Questions
	if len(questions) != 1 {
		t.Error("Failed to parse 1 DNS question")
	}
	answers := p.Layer(LayerTypeDNS).(*DNS).Answers
	if len(answers) != 0 {
		t.Error("Failed to parse 0 DNS answer")
	}
	additionals := p.Layer(LayerTypeDNS).(*DNS).Additionals
	if len(additionals) != 1 {
		t.Error("Failed to parse 1 DNS additional")
	}

	optAll := additionals[0].OPT
	if len(optAll) != 1 {
		t.Errorf("Parsed %d OPTs, expected 1", len(optAll))
	}
	responseCode := p.Layer(LayerTypeDNS).(*DNS).ResponseCode
	if responseCode != testParseDNSBadCookieResponseCode {
		t.Errorf("Incorrect extended response code, expected %q, got %q", testParseDNSBadCookieResponseCode, responseCode)
	}
}

var testParseDNSTypeURI = []byte{
	0x02, 0x00, 0x00, 0x00, // PF_INET
	0x45, 0x00, 0x00, 0x6f, 0x3e, 0x65, 0x00, 0x00, 0x40, 0x11, 0x3e, 0x17, 0x7f, 0x00, 0x00, 0x01,
	0x7f, 0x00, 0x00, 0x01, 0x00, 0x35, 0xe7, 0xd3, 0x00, 0x5b, 0xfe, 0x6e, 0xaf, 0x2d, 0x85, 0x80,
	0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x05, 0x5f, 0x68, 0x74, 0x74, 0x70, 0x03, 0x64,
	0x6e, 0x73, 0x04, 0x74, 0x65, 0x73, 0x74, 0x00, 0x01, 0x00, 0x00, 0x01, 0xc0, 0x0c, 0x01, 0x00,
	0x00, 0x01, 0x00, 0x00, 0x2a, 0x30, 0x00, 0x1c, 0x00, 0x0a, 0x00, 0x05, 0x68, 0x74, 0x74, 0x70,
	0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64, 0x6e, 0x73, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x3a,
	0x38, 0x30, 0x30, 0x30, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
}
var testParseDNSTypeURITarget = "http://www.dns.test:8000"
var testParseDNSTypeURIPriority = uint16(10)
var testParseDNSTypeURIWeight = uint16(5)

func TestParseDNSTypeURI(t *testing.T) {
	p := gopacket.NewPacket(testParseDNSTypeURI, LinkTypeNull, testDecodeOptions)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeLoopback, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
	answers := p.Layer(LayerTypeDNS).(*DNS).Answers
	if len(answers) != 1 {
		t.Error("Failed to parse 1 DNS answer")
	}
	if len(answers[0].URI.Target) < 1 {
		t.Error("Failed to parse 1 URI record")
	}
	target := string(answers[0].URI.Target)
	if target != testParseDNSTypeURITarget {
		t.Errorf("Incorrect URI target value, expected %q, got %q", testParseDNSTypeURITarget, target)
	}
	priority := answers[0].URI.Priority
	if priority != testParseDNSTypeURIPriority {
		t.Errorf("Incorrect URI priority value, expected %q, got %q", testParseDNSTypeURIPriority, priority)
	}
	weight := answers[0].URI.Weight
	if weight != testParseDNSTypeURIWeight {
		t.Errorf("Incorrect URI weight value, expected %q, got %q", testParseDNSTypeURIWeight, weight)
	}
}

var testParseDNSTypeOPT = []byte{
	0x00, 0x90, 0x0b, 0x12, 0x91, 0xc1, 0x00, 0x1c, 0xc0, 0x93, 0x33, 0xfb, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x5A, 0xce, 0x58, 0x00, 0x00, 0x40, 0x11, 0x67, 0xe2, 0xac, 0x10, 0x01, 0xc7, 0x4b, 0x4b,
	0x4b, 0x4b, 0xd6, 0x00, 0x00, 0x35, 0x00, 0x46, 0x44, 0xb0, 0x50, 0x12, 0x01, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x77, 0x77, 0x77, 0x04, 0x69, 0x65, 0x74, 0x66, 0x03,
	0x6f, 0x72, 0x67, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x80,
	0x00, 0x00, 0x13, 0x69, 0x42, 0x00, 0x0F, 0x4F, 0x70, 0x65, 0x6E, 0x44, 0x4E, 0x53, 0x01, 0x23,
	0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
}

func TestParseDNSTypeOPT(t *testing.T) {
	p := gopacket.NewPacket(testParseDNSTypeOPT, LinkTypeEthernet, testDecodeOptions)

	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeEthernet, LayerTypeIPv4, LayerTypeUDP, LayerTypeDNS}, t)
	questions := p.Layer(LayerTypeDNS).(*DNS).Questions
	if len(questions) != 1 {
		t.Error("Failed to parse 1 DNS question")
	}
	additionals := p.Layer(LayerTypeDNS).(*DNS).Additionals
	if len(additionals) != 1 {
		t.Error("Failed to parse 1 DNS additional")
	}

	optAll := additionals[0].OPT
	if len(optAll) != 1 {
		t.Errorf("Parsed %d OPTs, expected 1", len(optAll))
	}

	if additionals[0].OPT[0].Code != DNSOptionCodeDeviceID {
		t.Error("Failed to parse the OPT Code")
	}
	if string(additionals[0].OPT[0].Data[:7]) != "OpenDNS" {
		t.Error("Failed to parse the Data Part 1")
	}
	if !bytes.Equal(additionals[0].OPT[0].Data[7:], []byte{0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF}) {
		t.Error("Failed to parse the Data Part 2")
	}
}

func testQuestionEqual(t *testing.T, i int, exp, got DNSQuestion) {
	if !bytes.Equal(exp.Name, got.Name) {
		t.Errorf("expected Questions[%d].Name = %v, got %v", i, string(exp.Name), string(got.Name))
	}
	if exp.Type != got.Type {
		t.Errorf("expected Questions[%d].Type = %v, got %v", i, exp.Type, got.Type)
	}
	if exp.Class != got.Class {
		t.Errorf("expected Questions[%d].Class = %v, got %v", i, exp.Class, got.Class)
	}
}

func testResourceEqual(t *testing.T, i int, name string, exp, got DNSResourceRecord) {
	if !bytes.Equal(exp.Name, got.Name) {
		t.Errorf("expected %s[%d].Name = %v, got %v", name, i, string(exp.Name), string(got.Name))
	}

	if exp.Type != got.Type {
		t.Errorf("expected %s[%d].Type = %v, got %v", name, i, exp.Type, got.Type)
	}

	if exp.Class != got.Class {
		t.Errorf("expected %s[%d].Class = %v, got %v", name, i, exp.Class, got.Class)
	}

	if exp.TTL != got.TTL {
		t.Errorf("expected %s[%d].TTL = %v, got %v", name, i, exp.TTL, got.TTL)
	}
	if exp.DataLength != got.DataLength {
		t.Errorf("expected %s[%d].DataLength = %v, got %v", name, i, exp.DataLength, got.DataLength)
	}

	// we don't check .Data

	if !exp.IP.Equal(got.IP) {
		t.Errorf("expected %s[%d].IP = %v, got %v", name, i, exp.IP, got.IP)
	}
	if !bytes.Equal(exp.NS, got.NS) {
		t.Errorf("expected %s[%d].NS = %v, got %v", name, i, exp.NS, got.NS)
	}
	if !bytes.Equal(exp.CNAME, got.CNAME) {
		t.Errorf("expected %s[%d].CNAME = %v, got %v", name, i, exp.CNAME, got.CNAME)
	}
	if !bytes.Equal(exp.PTR, got.PTR) {
		t.Errorf("expected %s[%d].PTR = %v, got %v", name, i, exp.PTR, got.PTR)
	}
	if len(exp.TXTs) != len(got.TXTs) {
		t.Errorf("expected %s[%d].TXTs = %v, got %v", name, i, exp.TXTs, got.TXTs)
	}
	for j := range exp.TXTs {
		if !bytes.Equal(exp.TXTs[j], got.TXTs[j]) {
			t.Errorf("expected %s[%d].TXTs[%d] = %v, got %v", name, i, j, exp.TXTs[j], got.TXTs[j])
		}
	}

	// SOA
	if !bytes.Equal(exp.SOA.MName, got.SOA.MName) {
		t.Errorf("expected %s[%d].SOA.MName = %v, got %v", name, i, exp.SOA.MName, got.SOA.MName)
	}
	if !bytes.Equal(exp.SOA.RName, got.SOA.RName) {
		t.Errorf("expected %s[%d].SOA.RName = %v, got %v", name, i, exp.SOA.RName, got.SOA.RName)
	}
	if exp.SOA.Serial != got.SOA.Serial {
		t.Errorf("expected %s[%d].SOA.Serial = %v, got %v", name, i, exp.SOA.Serial, got.SOA.Serial)
	}
	if exp.SOA.Refresh != got.SOA.Refresh {
		t.Errorf("expected %s[%d].SOA.Refresh = %v, got %v", name, i, exp.SOA.Refresh, got.SOA.Refresh)
	}
	if exp.SOA.Retry != got.SOA.Retry {
		t.Errorf("expected %s[%d].SOA.Retry = %v, got %v", name, i, exp.SOA.Retry, got.SOA.Retry)
	}
	if exp.SOA.Expire != got.SOA.Expire {
		t.Errorf("expected %s[%d].SOA.Expire = %v, got %v", name, i, exp.SOA.Expire, got.SOA.Expire)
	}
	if exp.SOA.Minimum != got.SOA.Minimum {
		t.Errorf("expected %s[%d].SOA.Minimum = %v, got %v", name, i, exp.SOA.Minimum, got.SOA.Minimum)
	}

	// SRV
	if !bytes.Equal(exp.SRV.Name, got.SRV.Name) {
		t.Errorf("expected %s[%d].SRV.Name = %v, got %v", name, i, exp.SRV.Name, got.SRV.Name)
	}
	if exp.SRV.Weight != got.SRV.Weight {
		t.Errorf("expected %s[%d].SRV.Weight = %v, got %v", name, i, exp.SRV.Weight, got.SRV.Weight)
	}
	if exp.SRV.Port != got.SRV.Port {
		t.Errorf("expected %s[%d].SRV.Port = %v, got %v", name, i, exp.SRV.Port, got.SRV.Port)
	}
	// MX
	if !bytes.Equal(exp.MX.Name, got.MX.Name) {
		t.Errorf("expected %s[%d].MX.Name = %v, got %v", name, i, exp.MX.Name, got.MX.Name)
	}
	if exp.MX.Preference != got.MX.Preference {
		t.Errorf("expected %s[%d].MX.Preference = %v, got %v", name, i, exp.MX.Preference, got.MX.Preference)
	}
	// URI
	if !bytes.Equal(exp.URI.Target, got.URI.Target) {
		t.Errorf("expected %s[%d].URI.Target = %v, got %v", name, i, exp.URI.Target, got.URI.Target)
	}
	if exp.URI.Weight != got.URI.Weight {
		t.Errorf("expected %s[%d].URI.Weight = %v, got %v", name, i, exp.URI.Weight, got.URI.Weight)
	}
	if exp.URI.Priority != got.URI.Priority {
		t.Errorf("expected %s[%d].URI.Priority = %v, got %v", name, i, exp.URI.Priority, got.URI.Priority)
	}

	// OPT
	if len(exp.OPT) != len(got.OPT) {
		t.Errorf("expected len(%s[%d].OPT) = %v, got %v", name, i, len(exp.OPT), len(got.OPT))
	}

	for j := range exp.OPT {
		if exp.OPT[j].Code != got.OPT[j].Code {
			t.Errorf("expected %s[%d].OPT[%d].Code = %v, got %v", name, i, j, exp.OPT[j].Code, got.OPT[j].Code)
		}
		if !bytes.Equal(exp.OPT[j].Data, got.OPT[j].Data) {
			t.Errorf("expected %s[%d].OPT[%d].Data = %v, got %v", name, i, j, exp.OPT[j].Data, got.OPT[j].Data)
		}
	}
}

func testDNSEqual(t *testing.T, exp, got *DNS) {
	if exp.ID != got.ID {
		t.Errorf("expected ID = %v, got %v", exp.ID, got.ID)
	}
	if exp.AA != got.AA {
		t.Errorf("expected AA = %v, got %v", exp.AA, got.AA)
	}
	if exp.OpCode != got.OpCode {
		t.Errorf("expected OpCode = %v, got %v", exp.OpCode, got.OpCode)
	}
	if exp.AA != got.AA {
		t.Errorf("expected AA = %v, got %v", exp.AA, got.AA)
	}
	if exp.TC != got.TC {
		t.Errorf("expected TC = %v, got %v", exp.TC, got.TC)
	}
	if exp.RD != got.RD {
		t.Errorf("expected RD = %v, got %v", exp.RD, got.RD)
	}
	if exp.RA != got.RA {
		t.Errorf("expected RA = %v, got %v", exp.RA, got.RA)
	}
	if exp.Z != got.Z {
		t.Errorf("expected Z = %v, got %v", exp.Z, got.Z)
	}
	if exp.ResponseCode != got.ResponseCode {
		t.Errorf("expected ResponseCode = %v, got %v", exp.ResponseCode, got.ResponseCode)
	}
	if exp.QDCount != got.QDCount {
		t.Errorf("expected QDCount = %v, got %v", exp.QDCount, got.QDCount)
	}
	if exp.ANCount != got.ANCount {
		t.Errorf("expected ANCount = %v, got %v", exp.ANCount, got.ANCount)
	}
	if exp.ANCount != got.ANCount {
		t.Errorf("expected ANCount = %v, got %v", exp.ANCount, got.ANCount)
	}
	if exp.NSCount != got.NSCount {
		t.Errorf("expected NSCount = %v, got %v", exp.NSCount, got.NSCount)
	}
	if exp.ARCount != got.ARCount {
		t.Errorf("expected ARCount = %v, got %v", exp.ARCount, got.ARCount)
	}

	if len(exp.Questions) != len(got.Questions) {
		t.Errorf("expected %d Questions, got %d", len(exp.Questions), len(got.Questions))
	}
	for i := range exp.Questions {
		testQuestionEqual(t, i, exp.Questions[i], got.Questions[i])
	}

	if len(exp.Answers) != len(got.Answers) {
		t.Errorf("expected %d Answers, got %d", len(exp.Answers), len(got.Answers))
	}
	for i := range exp.Answers {
		testResourceEqual(t, i, "Answers", exp.Answers[i], got.Answers[i])
	}

	if len(exp.Authorities) != len(got.Authorities) {
		t.Errorf("expected %d Answers, got %d", len(exp.Authorities), len(got.Authorities))
	}
	for i := range exp.Authorities {
		testResourceEqual(t, i, "Authorities", exp.Authorities[i], got.Authorities[i])
	}

	if len(exp.Additionals) != len(got.Additionals) {
		t.Errorf("expected %d Additionals, got %d", len(exp.Additionals), len(got.Additionals))
	}
	for i := range exp.Additionals {
		testResourceEqual(t, i, "Additionals", exp.Additionals[i], got.Additionals[i])
	}
}

func TestDNSEncodeQuery(t *testing.T) {
	dns := &DNS{ID: 1234, OpCode: DNSOpCodeQuery, RD: true}
	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("example1.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
		})

	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("example2.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
		})

	buf := gopacket.NewSerializeBuffer()
	opts := gopacket.SerializeOptions{FixLengths: true}
	err := gopacket.SerializeLayers(buf, opts, dns)
	if err != nil {
		t.Fatal(err)
	}
	if int(dns.QDCount) != len(dns.Questions) {
		t.Errorf("fix lengths did not adjust QDCount, expected %d got %d", len(dns.Questions), dns.QDCount)
	}

	p2 := gopacket.NewPacket(buf.Bytes(), LayerTypeDNS, testDecodeOptions)
	dns2 := p2.Layer(LayerTypeDNS).(*DNS)
	testDNSEqual(t, dns, dns2)
}

func TestDNSEncodeQueryWithOPT(t *testing.T) {
	dns := &DNS{ID: 1234, OpCode: DNSOpCodeQuery, RD: true}
	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("example1.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
		})

	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("example2.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
		})
	dns.Additionals = append(dns.Additionals,
		DNSResourceRecord{
			Type:  DNSTypeOPT,
			Class: 4096,
			OPT: []DNSOPT{
				DNSOPT{
					Code: DNSOptionCodeDeviceID,
					Data: []byte("OpenDNS"),
				},
			},
		})

	buf := gopacket.NewSerializeBuffer()
	opts := gopacket.SerializeOptions{FixLengths: true}
	err := gopacket.SerializeLayers(buf, opts, dns)
	if err != nil {
		t.Fatal(err)
	}
	if int(dns.QDCount) != len(dns.Questions) {
		t.Errorf("fix lengths did not adjust QDCount, expected %d got %d", len(dns.Questions), dns.QDCount)
	}
	if int(dns.ARCount) != len(dns.Additionals) {
		t.Errorf("fix lengths did not adjust ARCount, expected %d got %d", len(dns.Additionals), dns.ARCount)
	}

	p2 := gopacket.NewPacket(buf.Bytes(), LayerTypeDNS, testDecodeOptions)
	dns2 := p2.Layer(LayerTypeDNS).(*DNS)
	testDNSEqual(t, dns, dns2)
}

func TestDNSEncodeResponse(t *testing.T) {
	dns := &DNS{ID: 1234, QR: true, OpCode: DNSOpCodeQuery,
		AA: true, RD: true, RA: true}
	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("example1.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
		})
	dns.Questions = append(dns.Questions,
		DNSQuestion{
			Name:  []byte("www.example2.com"),
			Type:  DNSTypeAAAA,
			Class: DNSClassIN,
		})

	dns.Answers = append(dns.Answers,
		DNSResourceRecord{
			Name:  []byte("example1.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
			TTL:   1024,
			IP:    net.IP([]byte{1, 2, 3, 4}),
		})

	dns.Answers = append(dns.Answers,
		DNSResourceRecord{
			Name:  []byte("example1.com"),
			Type:  DNSTypeURI,
			Class: DNSClassIN,
			TTL:   1024,
			URI: DNSURI{
				Target:   []byte("http://www.example2.com/"),
				Priority: 10240,
				Weight:   1,
			},
		})

	dns.Answers = append(dns.Answers,
		DNSResourceRecord{
			Name:  []byte("www.example2.com"),
			Type:  DNSTypeAAAA,
			Class: DNSClassIN,
			TTL:   1024,
			IP:    net.IP([]byte{5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4}),
		})

	dns.Answers = append(dns.Answers,
		DNSResourceRecord{
			Name:  []byte("www.example2.com"),
			Type:  DNSTypeCNAME,
			Class: DNSClassIN,
			TTL:   1024,
			CNAME: []byte("example2.com"),
		})

	buf := gopacket.NewSerializeBuffer()
	opts := gopacket.SerializeOptions{FixLengths: true}
	err := gopacket.SerializeLayers(buf, opts, dns)
	if err != nil {
		t.Fatal(err)
	}
	if int(dns.ANCount) != len(dns.Answers) {
		t.Errorf("fix lengths did not adjust ANCount, expected %d got %d", len(dns.Answers), dns.ANCount)
	}
	for i, a := range dns.Answers {
		if a.DataLength == 0 {
			t.Errorf("fix lengths did not adjust Answers[%d].DataLength", i)
		}
	}

	p2 := gopacket.NewPacket(buf.Bytes(), LayerTypeDNS, testDecodeOptions)
	dns2 := p2.Layer(LayerTypeDNS).(*DNS)
	testDNSEqual(t, dns, dns2)
}

// testDNSMalformedPacket is the packet:
//   10:30:00.389666 IP 10.77.43.131.60718 > 10.1.0.17.53: 18245 updateD [b2&3=0x5420] [18516a] [12064q] [21584n] [12081au][|domain]
//   	0x0000:  0000 0101 0000 4e96 1476 afa1 0800 4500  ......N..v....E.
//   	0x0010:  0039 d431 0000 f311 b3a0 0a4d 2b83 0a01  .9.1.......M+...
//   	0x0020:  0011 ed2e 0035 0025 0832 4745 5420 2f20  .....5.%.2GET./.
//   	0x0030:  4854 5450 2f31 2e31 0d0a 486f 7374 3a20  HTTP/1.1..Host:.
//   	0x0040:  7777 770d 0a0d 0a                        www....
var testDNSMalformedPacket = []byte{
	0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x4e, 0x96, 0x14, 0x76, 0xaf, 0xa1, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x39, 0xd4, 0x31, 0x00, 0x00, 0xf3, 0x11, 0xb3, 0xa0, 0x0a, 0x4d, 0x2b, 0x83, 0x0a, 0x01,
	0x00, 0x11, 0xed, 0x2e, 0x00, 0x35, 0x00, 0x25, 0x08, 0x32, 0x47, 0x45, 0x54, 0x20, 0x2f, 0x20,
	0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d, 0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20,
	0x77, 0x77, 0x77, 0x0d, 0x0a, 0x0d, 0x0a,
}

func TestDNSMalformedPacket(t *testing.T) {
	p := gopacket.NewPacket(testDNSMalformedPacket, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "invalid index") {
		t.Errorf("unexpected error message: %v", err)
	}
}

// testDNSMalformedPacket2 is the packet:
//   15:14:42.056054 IP 10.77.0.245.53 > 10.1.0.45.38769: 12625 zoneInit YXRRSet- [49833q],[|domain]
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  0079 3767 4000 3911 f49d 0a4d 00f5 0a01  .y7g@.9....M....
//   	0x0020:  002d 0035 9771 0065 6377 3151 f057 c2a9  .-.5.q.ecw1Q.W..
//   	0x0030:  fc6e e86a beb0 f7d4 8599 373e b5f8 9db2  .n.j......7>....
//   	0x0040:  a399 21a1 9762 def1 def4 f5ab 5675 023e  ..!..b......Vu.>
//   	0x0050:  c9ca 304f 178a c2ad f2fc 677a 0e4c b892  ..0O......gz.L..
//   	0x0060:  ab71 09bb 1ea4 f7c4 fe47 7a39 868b 29a0  .q.......Gz9..).
//   	0x0070:  62c4 d184 5b4e 8817 4cc0 d1d0 d430 11d3  b...[N..L....0..
//   	0x0080:  d147 543f afc7 1a                        .GT?...
var testDNSMalformedPacket2 = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x79, 0x37, 0x67, 0x40, 0x00, 0x39, 0x11, 0xf4, 0x9d, 0x0a, 0x4d, 0x00, 0xf5, 0x0a, 0x01,
	0x00, 0x2d, 0x00, 0x35, 0x97, 0x71, 0x00, 0x65, 0x63, 0x77, 0x31, 0x51, 0xf0, 0x57, 0xc2, 0xa9,
	0xfc, 0x6e, 0xe8, 0x6a, 0xbe, 0xb0, 0xf7, 0xd4, 0x85, 0x99, 0x37, 0x3e, 0xb5, 0xf8, 0x9d, 0xb2,
	0xa3, 0x99, 0x21, 0xa1, 0x97, 0x62, 0xde, 0xf1, 0xde, 0xf4, 0xf5, 0xab, 0x56, 0x75, 0x02, 0x3e,
	0xc9, 0xca, 0x30, 0x4f, 0x17, 0x8a, 0xc2, 0xad, 0xf2, 0xfc, 0x67, 0x7a, 0x0e, 0x4c, 0xb8, 0x92,
	0xab, 0x71, 0x09, 0xbb, 0x1e, 0xa4, 0xf7, 0xc4, 0xfe, 0x47, 0x7a, 0x39, 0x86, 0x8b, 0x29, 0xa0,
	0x62, 0xc4, 0xd1, 0x84, 0x5b, 0x4e, 0x88, 0x17, 0x4c, 0xc0, 0xd1, 0xd0, 0xd4, 0x30, 0x11, 0xd3,
	0xd1, 0x47, 0x54, 0x3f, 0xaf, 0xc7, 0x1a,
}

func TestDNSMalformedPacket2(t *testing.T) {
	p := gopacket.NewPacket(testDNSMalformedPacket2, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "offset pointer too high") {
		t.Errorf("unexpected error message: %v", err)
	}
}

// testBlankNameRootQuery is the packet:
//   08:31:18.143065 IP 10.77.0.26.53 > 10.1.0.233.65071: 59508- 0/13/3 (508)
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  0218 76b2 4000 7211 7ad2 0a4d 001a 0a01  ..v.@.r.z..M....
//   	0x0020:  00e9 0035 fe2f 0204 b8f5 e874 8100 0001  ...5./.....t....
//   	0x0030:  0000 000d 0003 0c61 786b 7663 6863 7063  .......axkvchcpc
//   	0x0040:  7073 6c0a 7878 7878 7878 7878 7878 036e  psl.xxxxxxxxxx.n
//   	0x0050:  6574 0000 0100 0100 0002 0001 0000 0e10  et..............
//   	0x0060:  0014 016d 0c72 6f6f 742d 7365 7276 6572  ...m.root-server
//   	0x0070:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0080:  0014 0161 0c72 6f6f 742d 7365 7276 6572  ...a.root-server
//   	0x0090:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x00a0:  0014 0169 0c72 6f6f 742d 7365 7276 6572  ...i.root-server
//   	0x00b0:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x00c0:  0014 0162 0c72 6f6f 742d 7365 7276 6572  ...b.root-server
//   	0x00d0:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x00e0:  0014 016c 0c72 6f6f 742d 7365 7276 6572  ...l.root-server
//   	0x00f0:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0100:  0014 0166 0c72 6f6f 742d 7365 7276 6572  ...f.root-server
//   	0x0110:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0120:  0014 0167 0c72 6f6f 742d 7365 7276 6572  ...g.root-server
//   	0x0130:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0140:  0014 0164 0c72 6f6f 742d 7365 7276 6572  ...d.root-server
//   	0x0150:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0160:  0014 0168 0c72 6f6f 742d 7365 7276 6572  ...h.root-server
//   	0x0170:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x0180:  0014 0165 0c72 6f6f 742d 7365 7276 6572  ...e.root-server
//   	0x0190:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x01a0:  0014 016a 0c72 6f6f 742d 7365 7276 6572  ...j.root-server
//   	0x01b0:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x01c0:  0014 016b 0c72 6f6f 742d 7365 7276 6572  ...k.root-server
//   	0x01d0:  7303 6e65 7400 c02d 0002 0001 0000 0e10  s.net..-........
//   	0x01e0:  0014 0163 0c72 6f6f 742d 7365 7276 6572  ...c.root-server
//   	0x01f0:  7303 6e65 7400 c038 0001 0001 0000 0e10  s.net..8........
//   	0x0200:  0004 ca0c 1b21 c058 0001 0001 0000 0e10  .....!.X........
//   	0x0210:  0004 c629 0004 c078 0001 0001 0000 0e10  ...)...x........
//   	0x0220:  0004 c024 9411                           ...$..
var testBlankNameRootQuery = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x02, 0x18, 0x76, 0xb2, 0x40, 0x00, 0x72, 0x11, 0x7a, 0xd2, 0x0a, 0x4d, 0x00, 0x1a, 0x0a, 0x01,
	0x00, 0xe9, 0x00, 0x35, 0xfe, 0x2f, 0x02, 0x04, 0xb8, 0xf5, 0xe8, 0x74, 0x81, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x0d, 0x00, 0x03, 0x0c, 0x61, 0x78, 0x6b, 0x76, 0x63, 0x68, 0x63, 0x70, 0x63,
	0x70, 0x73, 0x6c, 0x0a, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x03, 0x6e,
	0x65, 0x74, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x6d, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x61, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x69, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x62, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x6c, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x66, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x67, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x64, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x68, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x65, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x6a, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x6b, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x2d, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x14, 0x01, 0x63, 0x0c, 0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	0x73, 0x03, 0x6e, 0x65, 0x74, 0x00, 0xc0, 0x38, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x04, 0xca, 0x0c, 0x1b, 0x21, 0xc0, 0x58, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x04, 0xc6, 0x29, 0x00, 0x04, 0xc0, 0x78, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10,
	0x00, 0x04, 0xc0, 0x24, 0x94, 0x11,
}

func TestBlankNameRootQuery(t *testing.T) {
	p := gopacket.NewPacket(testBlankNameRootQuery, LinkTypeEthernet, testDecodeOptions)
	if err := p.ErrorLayer(); err != nil {
		t.Error("Error layer on blank DNS name field:", err)
	}
}

// testAnotherMalformedDNS is the packet:
//   10:52:13.690904 IP 10.77.0.29.53 > 10.1.0.6.42280: 13491 op6+% [b2&3=0x3313] [11720a] [23583q] [29742n] [52087au] Type22277 (Class 43688)? M- M-<.VM-^KM-wQM-s"M-^E^]M-+^Wx^P^@M-^\^\M-oM-FM-U^F^E7M-tM-^VM-^[M-F^H>G^FM-uM-^KM-_6GM-[M-jM-bM-^H]hM-^J.[|domain]
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  05c1 2eea 4000 3611 fbd1 0a4d 001d 0a01  ....@.6....M....
//   	0x0020:  0006 0035 a528 05ad 00a2 34b3 3313 5c1f  ...5.(....4.3.\.
//   	0x0030:  2dc8 742e cb77 2da0 bc2e 568b f751 f322  -.t..w-...V..Q."
//   	0x0040:  851d ab17 7810 009c 1cef c6d5 0605 37f4  ....x.........7.
//   	0x0050:  969b c65e 483e 4706 f58b df36 47db eae2  ...^H>G....6G...
//   	0x0060:  885d 688a c5a5 5705 aaa8 95eb 93a4 d85a  .]h...W........Z
//   	0x0070:  c9af 261f 7816 a354 2d23 d84a 579c 4876  ..&.x..T-#.JW.Hv
//   	0x0080:  a391 43db 5c41 191a 92b8 dcdd 6839 eef5  ..C.\A......h9..
//   	0x0090:  728e 13e0 0679 6f47 88a0 25b9 44d8 f8e7  r....yoG..%.D...
//   	0x00a0:  8afe 0bfa f811 8da5 f8a3 1f8e d23b fe12  .............;..
//   	0x00b0:  d943 9327 92ad 4410 183e 688d b06d 5391  .C.'..D..>h..mS.
//   	0x00c0:  695b e49f 8f1e c075 d043 afe0 1174 9db0  i[.....u.C...t..
//   	0x00d0:  06b0 f01e b85b 3c84 945e 06d0 b20f 9eaa  .....[<..^......
//   	0x00e0:  123d 0ab0 2a55 309c 0ee9 3e5e db2f f377  .=..*U0...>^./.w
//   	0x00f0:  d7f1 9bae 373d 3316 0796 b80e dd18 5173  ....7=3.......Qs
//   	0x0100:  b28d 84fd 1812 d87b 42c8 5f11 4db6 b269  .......{B._.M..i
//   	0x0110:  1c42 4aea d5a4 644b 6c00 f0c0 fcee 71a7  .BJ...dKl.....q.
//   	0x0120:  e7f0 719c a207 dc5c a6fa f005 a338 7ff0  ..q....\.....8..
//   	0x0130:  5beb 3b4d 8952 2a46 d47b a5a2 e1fb 9e76  [.;M.R*F.{.....v
//   	0x0140:  c815 6258 50f4 6997 bad5 b479 2d06 ebbb  ..bXP.i....y-...
//   	0x0150:  2cac 2ecc e4f0 1f94 ce9f 186c 61da 9681  ,..........la...
//   	0x0160:  345c 4d88 efc7 037b fbe3 4402 ea06 2e5d  4\M....{..D....]
//   	0x0170:  2e6e 4860 e180 3ef7 c006 0ad1 ebb9 c4ff  .nH`..>.........
//   	0x0180:  dee2 f21c 02c2 751a ded8 ae2e 13a9 3fa2  ......u.......?.
//   	0x0190:  392a 8b54 11b2 2b4e 2bf1 4780 db9f 8c10  9*.T..+N+.G.....
//   	0x01a0:  ac6f 61b0 7b19 423f 07e5 4628 b870 f75d  .oa.{.B?..F(.p.]
//   	0x01b0:  09a3 63b2 77af 5985 a0ae 51d8 243f a7c8  ..c.w.Y...Q.$?..
//   	0x01c0:  ab08 7fc6 0217 c09f c412 0c45 e6aa 96bf  ...........E....
//   	0x01d0:  184c 4307 1f1f c4f4 7734 da31 2088 662b  .LC.....w4.1..f+
//   	0x01e0:  44c5 096f 1d1d 2dc5 ffd6 867d 9fc5 7b45  D..o..-....}..{E
//   	0x01f0:  f949 7dd9 38de 0d51 ac2a 32fc f50b 1bbe  .I}.8..Q.*2.....
//   	0x0200:  1c4b 5441 fbf3 0821 6c28 4530 5676 1d27  .KTA...!l(E0Vv.'
//   	0x0210:  5087 466c 3d5b 45a6 af7f 917a 6d43 66c2  P.Fl=[E....zmCf.
//   	0x0220:  036a 8bef ca60 9b13 8d29 9fda 82fa 01b1  .j...`...)......
//   	0x0230:  df8f 1f83 c71d 630f 349e 508c 9f7a e3da  ......c.4.P..z..
//   	0x0240:  a114 3622 9df8 9926 4dac 4150 d505 7b3a  ..6"...&M.AP..{:
//   	0x0250:  6fed fc75 6b4f 2d60 8a89 767d 9af0 896e  o..ukO-`..v}...n
//   	0x0260:  907d 1ada def3 345c 0d81 283c a24f fcbb  .}....4\..(<.O..
//   	0x0270:  bbdd b7b3 e3bb 9f1b d966 51b7 8217 7fa0  .........fQ.....
//   	0x0280:  e828 d3ca a6f1 532f 164e e405 bb3b 0de3  .(....S/.N...;..
//   	0x0290:  985d 6e89 d825 ebc6 d8ba 5190 a114 c6a3  .]n..%....Q.....
//   	0x02a0:  18b4 8aa7 181a 01ac cdc0 8048 ea72 a5e3  ...........H.r..
//   	0x02b0:  e37a dc57 65cd b787 39e6 c39e 317b 45d8  .z.We...9...1{E.
//   	0x02c0:  475c 05ba e8f8 8224 5a85 27b8 1584 8d78  G\.....$Z.'....x
//   	0x02d0:  62b6 6495 ac10 338f 1122 f2ff 043e 9e2a  b.d...3.."...>.*
//   	0x02e0:  1058 a910 5792 6fcd 9a96 6183 6708 8f70  .X..W.o...a.g..p
//   	0x02f0:  edc6 a67c 64ff 50fa 520b de94 c82c c4d6  ...|d.P.R....,..
//   	0x0300:  7d8f 0fd5 2f0d 9833 7c6c be10 a4e5 dc99  }.../..3|l......
//   	0x0310:  a467 ef5f b35b c11c e23c 131a 48b2 9cef  .g._.[...<..H...
//   	0x0320:  5a2f fece dd9e 2aea 0db9 faf3 a6ef b29d  Z/....*.........
//   	0x0330:  e85d a410 dd6a 6806 3fc6 1694 179f cb4b  .]...jh.?......K
//   	0x0340:  08c4 86b2 0713 cddb b257 d56b fe82 7d82  .........W.k..}.
//   	0x0350:  0d1f 6dc9 67b2 d2a1 6791 4f38 edf9 491f  ..m.g...g.O8..I.
//   	0x0360:  2c02 35f5 8165 ecc3 bc6a b631 3c7e 1ad4  ,.5..e...j.1<~..
//   	0x0370:  8e27 f962 f942 11b5 1b45 9bac b474 3c6e  .'.b.B...E...t<n
//   	0x0380:  6832 3075 be6d ac0d a8a0 7d47 a6ef 4e43  h20u.m....}G..NC
//   	0x0390:  6b9a 3097 8a8b 82a3 9515 362c f7d6 a37f  k.0.......6,....
//   	0x03a0:  7313 1199 a5f3 03dc bcc9 fb10 c23d eeb9  s............=..
//   	0x03b0:  78ff c8f3 0d38 9f74 ceec b7ae 63e3 3424  x....8.t....c.4$
//   	0x03c0:  b783 f106 011f 666b bf2d abc8 ea10 57a1  ......fk.-....W.
//   	0x03d0:  7cf2 4a3f 57ca 1386 bfba 27e5 4662 81c8  |.J?W.....'.Fb..
//   	0x03e0:  041e 1820 b3d5 c399 cd4d 222f 29f0 b994  .........M"/)...
//   	0x03f0:  865a e6e2 1686 3261 b0cd caaf 07ec d0bc  .Z....2a........
//   	0x0400:  afb8 3cf0 51c1 6c7a 6383 6b3a ff47 9551  ..<.Q.lzc.k:.G.Q
//   	0x0410:  1099 525f 355e 4684 bd34 ec12 88c9 dcc2  ..R_5^F..4......
//   	0x0420:  d11c 826d f1df 37e6 f08f 6ce8 817d bdc3  ...m..7...l..}..
//   	0x0430:  20b9 a274 c645 c67d f299 fef9 287f 09ee  ...t.E.}....(...
//   	0x0440:  ac67 6872 a126 b1d3 922c 4c2a 0ec9 b6d4  .ghr.&...,L*....
//   	0x0450:  fb59 6163 d1c4 1708 8d94 bc3d be5e ae29  .Yac.......=.^.)
//   	0x0460:  51ff a765 9df6 ae35 ed6b 0555 933f 3ed6  Q..e...5.k.U.?>.
//   	0x0470:  259b d93e f86f 6088 0c4e 357b 5c67 7d93  %..>.o`..N5{\g}.
//   	0x0480:  a695 1a42 e1e1 ef91 14d7 b7b7 0ca4 2dda  ...B..........-.
//   	0x0490:  6ac1 771e 25c1 a578 4ca8 6fd8 de04 1c09  j.w.%..xL.o.....
//   	0x04a0:  df49 f179 6a58 2b45 7231 307f bc67 e5e7  .I.yjX+Er10..g..
//   	0x04b0:  c5cd fec0 b021 508e 4fc5 f821 f734 90bc  .....!P.O..!.4..
//   	0x04c0:  c87f 14f1 2e5c d17b 1818 5b4a 6b68 0212  .....\.{..[Jkh..
//   	0x04d0:  1791 4a30 8518 99a9 b516 67e7 ed56 d1d1  ..J0......g..V..
//   	0x04e0:  239d dfda 11c5 0afe e58a b6e0 fb66 ab5c  #............f.\
//   	0x04f0:  f590 dcd6 457d 01d1 83f5 a9f0 cdb2 9c14  ....E}..........
//   	0x0500:  ff66 f10c d428 a07b 34e3 d600 91f2 aca7  .f...(.{4.......
//   	0x0510:  4e1f f3ac a96e 2aa3 ec9b 448c 748d f858  N....n*...D.t..X
//   	0x0520:  131c d496 af9b f5f0 d2f5 57ac 0b64 55a1  ..........W..dU.
//   	0x0530:  860e 5ad0 3e62 26b5 9e17 f51f 88c1 02e3  ..Z.>b&.........
//   	0x0540:  4a38 de70 3216 6f88 5d1e f429 ee19 4121  J8.p2.o.]..)..A!
//   	0x0550:  f571 84ac 3789 141f 1798 90b1 8373 2499  .q..7........s$.
//   	0x0560:  c131 b13f f3a3 9a07 aef1 bfe8 8cd7 8c2e  .1.?............
//   	0x0570:  ba35 dfc5 eb07 013c 7621 6481 bdfb 6233  .5.....<v!d...b3
//   	0x0580:  22e2 0f05 7e15 0417 67e4 2632 5207 28a6  "...~...g.&2R.(.
//   	0x0590:  8e88 9423 de54 5412 b53e fd8d d47a de58  ...#.TT..>...z.X
//   	0x05a0:  a1b2 6e08 d06d dc21 1eda 14a0 a2f7 1701  ..n..m.!........
//   	0x05b0:  a5e0 dfd7 871f 595d db43 70f5 bab3 b732  ......Y].Cp....2
//   	0x05c0:  6275 da15 b203 dac7 321f 8d61 11bd 30    bu......2..a..0
var testAnotherMalformedDNS = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x05, 0xc1, 0x2e, 0xea, 0x40, 0x00, 0x36, 0x11, 0xfb, 0xd1, 0x0a, 0x4d, 0x00, 0x1d, 0x0a, 0x01,
	0x00, 0x06, 0x00, 0x35, 0xa5, 0x28, 0x05, 0xad, 0x00, 0xa2, 0x34, 0xb3, 0x33, 0x13, 0x5c, 0x1f,
	0x2d, 0xc8, 0x74, 0x2e, 0xcb, 0x77, 0x2d, 0xa0, 0xbc, 0x2e, 0x56, 0x8b, 0xf7, 0x51, 0xf3, 0x22,
	0x85, 0x1d, 0xab, 0x17, 0x78, 0x10, 0x00, 0x9c, 0x1c, 0xef, 0xc6, 0xd5, 0x06, 0x05, 0x37, 0xf4,
	0x96, 0x9b, 0xc6, 0x5e, 0x48, 0x3e, 0x47, 0x06, 0xf5, 0x8b, 0xdf, 0x36, 0x47, 0xdb, 0xea, 0xe2,
	0x88, 0x5d, 0x68, 0x8a, 0xc5, 0xa5, 0x57, 0x05, 0xaa, 0xa8, 0x95, 0xeb, 0x93, 0xa4, 0xd8, 0x5a,
	0xc9, 0xaf, 0x26, 0x1f, 0x78, 0x16, 0xa3, 0x54, 0x2d, 0x23, 0xd8, 0x4a, 0x57, 0x9c, 0x48, 0x76,
	0xa3, 0x91, 0x43, 0xdb, 0x5c, 0x41, 0x19, 0x1a, 0x92, 0xb8, 0xdc, 0xdd, 0x68, 0x39, 0xee, 0xf5,
	0x72, 0x8e, 0x13, 0xe0, 0x06, 0x79, 0x6f, 0x47, 0x88, 0xa0, 0x25, 0xb9, 0x44, 0xd8, 0xf8, 0xe7,
	0x8a, 0xfe, 0x0b, 0xfa, 0xf8, 0x11, 0x8d, 0xa5, 0xf8, 0xa3, 0x1f, 0x8e, 0xd2, 0x3b, 0xfe, 0x12,
	0xd9, 0x43, 0x93, 0x27, 0x92, 0xad, 0x44, 0x10, 0x18, 0x3e, 0x68, 0x8d, 0xb0, 0x6d, 0x53, 0x91,
	0x69, 0x5b, 0xe4, 0x9f, 0x8f, 0x1e, 0xc0, 0x75, 0xd0, 0x43, 0xaf, 0xe0, 0x11, 0x74, 0x9d, 0xb0,
	0x06, 0xb0, 0xf0, 0x1e, 0xb8, 0x5b, 0x3c, 0x84, 0x94, 0x5e, 0x06, 0xd0, 0xb2, 0x0f, 0x9e, 0xaa,
	0x12, 0x3d, 0x0a, 0xb0, 0x2a, 0x55, 0x30, 0x9c, 0x0e, 0xe9, 0x3e, 0x5e, 0xdb, 0x2f, 0xf3, 0x77,
	0xd7, 0xf1, 0x9b, 0xae, 0x37, 0x3d, 0x33, 0x16, 0x07, 0x96, 0xb8, 0x0e, 0xdd, 0x18, 0x51, 0x73,
	0xb2, 0x8d, 0x84, 0xfd, 0x18, 0x12, 0xd8, 0x7b, 0x42, 0xc8, 0x5f, 0x11, 0x4d, 0xb6, 0xb2, 0x69,
	0x1c, 0x42, 0x4a, 0xea, 0xd5, 0xa4, 0x64, 0x4b, 0x6c, 0x00, 0xf0, 0xc0, 0xfc, 0xee, 0x71, 0xa7,
	0xe7, 0xf0, 0x71, 0x9c, 0xa2, 0x07, 0xdc, 0x5c, 0xa6, 0xfa, 0xf0, 0x05, 0xa3, 0x38, 0x7f, 0xf0,
	0x5b, 0xeb, 0x3b, 0x4d, 0x89, 0x52, 0x2a, 0x46, 0xd4, 0x7b, 0xa5, 0xa2, 0xe1, 0xfb, 0x9e, 0x76,
	0xc8, 0x15, 0x62, 0x58, 0x50, 0xf4, 0x69, 0x97, 0xba, 0xd5, 0xb4, 0x79, 0x2d, 0x06, 0xeb, 0xbb,
	0x2c, 0xac, 0x2e, 0xcc, 0xe4, 0xf0, 0x1f, 0x94, 0xce, 0x9f, 0x18, 0x6c, 0x61, 0xda, 0x96, 0x81,
	0x34, 0x5c, 0x4d, 0x88, 0xef, 0xc7, 0x03, 0x7b, 0xfb, 0xe3, 0x44, 0x02, 0xea, 0x06, 0x2e, 0x5d,
	0x2e, 0x6e, 0x48, 0x60, 0xe1, 0x80, 0x3e, 0xf7, 0xc0, 0x06, 0x0a, 0xd1, 0xeb, 0xb9, 0xc4, 0xff,
	0xde, 0xe2, 0xf2, 0x1c, 0x02, 0xc2, 0x75, 0x1a, 0xde, 0xd8, 0xae, 0x2e, 0x13, 0xa9, 0x3f, 0xa2,
	0x39, 0x2a, 0x8b, 0x54, 0x11, 0xb2, 0x2b, 0x4e, 0x2b, 0xf1, 0x47, 0x80, 0xdb, 0x9f, 0x8c, 0x10,
	0xac, 0x6f, 0x61, 0xb0, 0x7b, 0x19, 0x42, 0x3f, 0x07, 0xe5, 0x46, 0x28, 0xb8, 0x70, 0xf7, 0x5d,
	0x09, 0xa3, 0x63, 0xb2, 0x77, 0xaf, 0x59, 0x85, 0xa0, 0xae, 0x51, 0xd8, 0x24, 0x3f, 0xa7, 0xc8,
	0xab, 0x08, 0x7f, 0xc6, 0x02, 0x17, 0xc0, 0x9f, 0xc4, 0x12, 0x0c, 0x45, 0xe6, 0xaa, 0x96, 0xbf,
	0x18, 0x4c, 0x43, 0x07, 0x1f, 0x1f, 0xc4, 0xf4, 0x77, 0x34, 0xda, 0x31, 0x20, 0x88, 0x66, 0x2b,
	0x44, 0xc5, 0x09, 0x6f, 0x1d, 0x1d, 0x2d, 0xc5, 0xff, 0xd6, 0x86, 0x7d, 0x9f, 0xc5, 0x7b, 0x45,
	0xf9, 0x49, 0x7d, 0xd9, 0x38, 0xde, 0x0d, 0x51, 0xac, 0x2a, 0x32, 0xfc, 0xf5, 0x0b, 0x1b, 0xbe,
	0x1c, 0x4b, 0x54, 0x41, 0xfb, 0xf3, 0x08, 0x21, 0x6c, 0x28, 0x45, 0x30, 0x56, 0x76, 0x1d, 0x27,
	0x50, 0x87, 0x46, 0x6c, 0x3d, 0x5b, 0x45, 0xa6, 0xaf, 0x7f, 0x91, 0x7a, 0x6d, 0x43, 0x66, 0xc2,
	0x03, 0x6a, 0x8b, 0xef, 0xca, 0x60, 0x9b, 0x13, 0x8d, 0x29, 0x9f, 0xda, 0x82, 0xfa, 0x01, 0xb1,
	0xdf, 0x8f, 0x1f, 0x83, 0xc7, 0x1d, 0x63, 0x0f, 0x34, 0x9e, 0x50, 0x8c, 0x9f, 0x7a, 0xe3, 0xda,
	0xa1, 0x14, 0x36, 0x22, 0x9d, 0xf8, 0x99, 0x26, 0x4d, 0xac, 0x41, 0x50, 0xd5, 0x05, 0x7b, 0x3a,
	0x6f, 0xed, 0xfc, 0x75, 0x6b, 0x4f, 0x2d, 0x60, 0x8a, 0x89, 0x76, 0x7d, 0x9a, 0xf0, 0x89, 0x6e,
	0x90, 0x7d, 0x1a, 0xda, 0xde, 0xf3, 0x34, 0x5c, 0x0d, 0x81, 0x28, 0x3c, 0xa2, 0x4f, 0xfc, 0xbb,
	0xbb, 0xdd, 0xb7, 0xb3, 0xe3, 0xbb, 0x9f, 0x1b, 0xd9, 0x66, 0x51, 0xb7, 0x82, 0x17, 0x7f, 0xa0,
	0xe8, 0x28, 0xd3, 0xca, 0xa6, 0xf1, 0x53, 0x2f, 0x16, 0x4e, 0xe4, 0x05, 0xbb, 0x3b, 0x0d, 0xe3,
	0x98, 0x5d, 0x6e, 0x89, 0xd8, 0x25, 0xeb, 0xc6, 0xd8, 0xba, 0x51, 0x90, 0xa1, 0x14, 0xc6, 0xa3,
	0x18, 0xb4, 0x8a, 0xa7, 0x18, 0x1a, 0x01, 0xac, 0xcd, 0xc0, 0x80, 0x48, 0xea, 0x72, 0xa5, 0xe3,
	0xe3, 0x7a, 0xdc, 0x57, 0x65, 0xcd, 0xb7, 0x87, 0x39, 0xe6, 0xc3, 0x9e, 0x31, 0x7b, 0x45, 0xd8,
	0x47, 0x5c, 0x05, 0xba, 0xe8, 0xf8, 0x82, 0x24, 0x5a, 0x85, 0x27, 0xb8, 0x15, 0x84, 0x8d, 0x78,
	0x62, 0xb6, 0x64, 0x95, 0xac, 0x10, 0x33, 0x8f, 0x11, 0x22, 0xf2, 0xff, 0x04, 0x3e, 0x9e, 0x2a,
	0x10, 0x58, 0xa9, 0x10, 0x57, 0x92, 0x6f, 0xcd, 0x9a, 0x96, 0x61, 0x83, 0x67, 0x08, 0x8f, 0x70,
	0xed, 0xc6, 0xa6, 0x7c, 0x64, 0xff, 0x50, 0xfa, 0x52, 0x0b, 0xde, 0x94, 0xc8, 0x2c, 0xc4, 0xd6,
	0x7d, 0x8f, 0x0f, 0xd5, 0x2f, 0x0d, 0x98, 0x33, 0x7c, 0x6c, 0xbe, 0x10, 0xa4, 0xe5, 0xdc, 0x99,
	0xa4, 0x67, 0xef, 0x5f, 0xb3, 0x5b, 0xc1, 0x1c, 0xe2, 0x3c, 0x13, 0x1a, 0x48, 0xb2, 0x9c, 0xef,
	0x5a, 0x2f, 0xfe, 0xce, 0xdd, 0x9e, 0x2a, 0xea, 0x0d, 0xb9, 0xfa, 0xf3, 0xa6, 0xef, 0xb2, 0x9d,
	0xe8, 0x5d, 0xa4, 0x10, 0xdd, 0x6a, 0x68, 0x06, 0x3f, 0xc6, 0x16, 0x94, 0x17, 0x9f, 0xcb, 0x4b,
	0x08, 0xc4, 0x86, 0xb2, 0x07, 0x13, 0xcd, 0xdb, 0xb2, 0x57, 0xd5, 0x6b, 0xfe, 0x82, 0x7d, 0x82,
	0x0d, 0x1f, 0x6d, 0xc9, 0x67, 0xb2, 0xd2, 0xa1, 0x67, 0x91, 0x4f, 0x38, 0xed, 0xf9, 0x49, 0x1f,
	0x2c, 0x02, 0x35, 0xf5, 0x81, 0x65, 0xec, 0xc3, 0xbc, 0x6a, 0xb6, 0x31, 0x3c, 0x7e, 0x1a, 0xd4,
	0x8e, 0x27, 0xf9, 0x62, 0xf9, 0x42, 0x11, 0xb5, 0x1b, 0x45, 0x9b, 0xac, 0xb4, 0x74, 0x3c, 0x6e,
	0x68, 0x32, 0x30, 0x75, 0xbe, 0x6d, 0xac, 0x0d, 0xa8, 0xa0, 0x7d, 0x47, 0xa6, 0xef, 0x4e, 0x43,
	0x6b, 0x9a, 0x30, 0x97, 0x8a, 0x8b, 0x82, 0xa3, 0x95, 0x15, 0x36, 0x2c, 0xf7, 0xd6, 0xa3, 0x7f,
	0x73, 0x13, 0x11, 0x99, 0xa5, 0xf3, 0x03, 0xdc, 0xbc, 0xc9, 0xfb, 0x10, 0xc2, 0x3d, 0xee, 0xb9,
	0x78, 0xff, 0xc8, 0xf3, 0x0d, 0x38, 0x9f, 0x74, 0xce, 0xec, 0xb7, 0xae, 0x63, 0xe3, 0x34, 0x24,
	0xb7, 0x83, 0xf1, 0x06, 0x01, 0x1f, 0x66, 0x6b, 0xbf, 0x2d, 0xab, 0xc8, 0xea, 0x10, 0x57, 0xa1,
	0x7c, 0xf2, 0x4a, 0x3f, 0x57, 0xca, 0x13, 0x86, 0xbf, 0xba, 0x27, 0xe5, 0x46, 0x62, 0x81, 0xc8,
	0x04, 0x1e, 0x18, 0x20, 0xb3, 0xd5, 0xc3, 0x99, 0xcd, 0x4d, 0x22, 0x2f, 0x29, 0xf0, 0xb9, 0x94,
	0x86, 0x5a, 0xe6, 0xe2, 0x16, 0x86, 0x32, 0x61, 0xb0, 0xcd, 0xca, 0xaf, 0x07, 0xec, 0xd0, 0xbc,
	0xaf, 0xb8, 0x3c, 0xf0, 0x51, 0xc1, 0x6c, 0x7a, 0x63, 0x83, 0x6b, 0x3a, 0xff, 0x47, 0x95, 0x51,
	0x10, 0x99, 0x52, 0x5f, 0x35, 0x5e, 0x46, 0x84, 0xbd, 0x34, 0xec, 0x12, 0x88, 0xc9, 0xdc, 0xc2,
	0xd1, 0x1c, 0x82, 0x6d, 0xf1, 0xdf, 0x37, 0xe6, 0xf0, 0x8f, 0x6c, 0xe8, 0x81, 0x7d, 0xbd, 0xc3,
	0x20, 0xb9, 0xa2, 0x74, 0xc6, 0x45, 0xc6, 0x7d, 0xf2, 0x99, 0xfe, 0xf9, 0x28, 0x7f, 0x09, 0xee,
	0xac, 0x67, 0x68, 0x72, 0xa1, 0x26, 0xb1, 0xd3, 0x92, 0x2c, 0x4c, 0x2a, 0x0e, 0xc9, 0xb6, 0xd4,
	0xfb, 0x59, 0x61, 0x63, 0xd1, 0xc4, 0x17, 0x08, 0x8d, 0x94, 0xbc, 0x3d, 0xbe, 0x5e, 0xae, 0x29,
	0x51, 0xff, 0xa7, 0x65, 0x9d, 0xf6, 0xae, 0x35, 0xed, 0x6b, 0x05, 0x55, 0x93, 0x3f, 0x3e, 0xd6,
	0x25, 0x9b, 0xd9, 0x3e, 0xf8, 0x6f, 0x60, 0x88, 0x0c, 0x4e, 0x35, 0x7b, 0x5c, 0x67, 0x7d, 0x93,
	0xa6, 0x95, 0x1a, 0x42, 0xe1, 0xe1, 0xef, 0x91, 0x14, 0xd7, 0xb7, 0xb7, 0x0c, 0xa4, 0x2d, 0xda,
	0x6a, 0xc1, 0x77, 0x1e, 0x25, 0xc1, 0xa5, 0x78, 0x4c, 0xa8, 0x6f, 0xd8, 0xde, 0x04, 0x1c, 0x09,
	0xdf, 0x49, 0xf1, 0x79, 0x6a, 0x58, 0x2b, 0x45, 0x72, 0x31, 0x30, 0x7f, 0xbc, 0x67, 0xe5, 0xe7,
	0xc5, 0xcd, 0xfe, 0xc0, 0xb0, 0x21, 0x50, 0x8e, 0x4f, 0xc5, 0xf8, 0x21, 0xf7, 0x34, 0x90, 0xbc,
	0xc8, 0x7f, 0x14, 0xf1, 0x2e, 0x5c, 0xd1, 0x7b, 0x18, 0x18, 0x5b, 0x4a, 0x6b, 0x68, 0x02, 0x12,
	0x17, 0x91, 0x4a, 0x30, 0x85, 0x18, 0x99, 0xa9, 0xb5, 0x16, 0x67, 0xe7, 0xed, 0x56, 0xd1, 0xd1,
	0x23, 0x9d, 0xdf, 0xda, 0x11, 0xc5, 0x0a, 0xfe, 0xe5, 0x8a, 0xb6, 0xe0, 0xfb, 0x66, 0xab, 0x5c,
	0xf5, 0x90, 0xdc, 0xd6, 0x45, 0x7d, 0x01, 0xd1, 0x83, 0xf5, 0xa9, 0xf0, 0xcd, 0xb2, 0x9c, 0x14,
	0xff, 0x66, 0xf1, 0x0c, 0xd4, 0x28, 0xa0, 0x7b, 0x34, 0xe3, 0xd6, 0x00, 0x91, 0xf2, 0xac, 0xa7,
	0x4e, 0x1f, 0xf3, 0xac, 0xa9, 0x6e, 0x2a, 0xa3, 0xec, 0x9b, 0x44, 0x8c, 0x74, 0x8d, 0xf8, 0x58,
	0x13, 0x1c, 0xd4, 0x96, 0xaf, 0x9b, 0xf5, 0xf0, 0xd2, 0xf5, 0x57, 0xac, 0x0b, 0x64, 0x55, 0xa1,
	0x86, 0x0e, 0x5a, 0xd0, 0x3e, 0x62, 0x26, 0xb5, 0x9e, 0x17, 0xf5, 0x1f, 0x88, 0xc1, 0x02, 0xe3,
	0x4a, 0x38, 0xde, 0x70, 0x32, 0x16, 0x6f, 0x88, 0x5d, 0x1e, 0xf4, 0x29, 0xee, 0x19, 0x41, 0x21,
	0xf5, 0x71, 0x84, 0xac, 0x37, 0x89, 0x14, 0x1f, 0x17, 0x98, 0x90, 0xb1, 0x83, 0x73, 0x24, 0x99,
	0xc1, 0x31, 0xb1, 0x3f, 0xf3, 0xa3, 0x9a, 0x07, 0xae, 0xf1, 0xbf, 0xe8, 0x8c, 0xd7, 0x8c, 0x2e,
	0xba, 0x35, 0xdf, 0xc5, 0xeb, 0x07, 0x01, 0x3c, 0x76, 0x21, 0x64, 0x81, 0xbd, 0xfb, 0x62, 0x33,
	0x22, 0xe2, 0x0f, 0x05, 0x7e, 0x15, 0x04, 0x17, 0x67, 0xe4, 0x26, 0x32, 0x52, 0x07, 0x28, 0xa6,
	0x8e, 0x88, 0x94, 0x23, 0xde, 0x54, 0x54, 0x12, 0xb5, 0x3e, 0xfd, 0x8d, 0xd4, 0x7a, 0xde, 0x58,
	0xa1, 0xb2, 0x6e, 0x08, 0xd0, 0x6d, 0xdc, 0x21, 0x1e, 0xda, 0x14, 0xa0, 0xa2, 0xf7, 0x17, 0x01,
	0xa5, 0xe0, 0xdf, 0xd7, 0x87, 0x1f, 0x59, 0x5d, 0xdb, 0x43, 0x70, 0xf5, 0xba, 0xb3, 0xb7, 0x32,
	0x62, 0x75, 0xda, 0x15, 0xb2, 0x03, 0xda, 0xc7, 0x32, 0x1f, 0x8d, 0x61, 0x11, 0xbd, 0x30,
}

func TestAnotherMalformedDNS(t *testing.T) {
	p := gopacket.NewPacket(testAnotherMalformedDNS, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "offset too high") {
		t.Errorf("unexpected error message: %v", err)
	}
}

// testMalformedDNSAgain is the packet:
//   12:14:52.702061 IP 10.77.0.4.53 > 10.1.0.41.61610: 12529 updateDA [b2&3=0x5cad] [38274a] [61303q] [1718n] [14913au][|domain]
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  0091 2dff 0000 7811 ffe2 0a4d 0004 0a01  ..-...x....M....
//   	0x0020:  0029 0035 f0aa 007d 5b53 30f1 5cad ef77  .).5...}[S0.\..w
//   	0x0030:  9582 06b6 3a41 357a 8cef cdc0 a732 b800  ....:A5z.....2..
//   	0x0040:  466e 1c30 2e75 95ac c03d 1ed4 8635 2d09  Fn.0.u...=...5-.
//   	0x0050:  2fee 3a82 b4f0 427e 2b6b f870 cc7f c9a1  /.:...B~+k.p....
//   	0x0060:  e6f1 a761 97ec 2ff7 d248 4d95 321c 6e4e  ...a../..HM.2.nN
//   	0x0070:  57fa 6d3d 9ec0 fe3a 6f1e e634 4396 b494  W.m=...:o..4C...
//   	0x0080:  8b7a a929 d7e1 da7c c346 ca77 4890 6bf3  .z.)...|.F.wH.k.
//   	0x0090:  5ecb 7e97 c49d 3564 984f bf7c 8ac1 dd    ^.~...5d.O.|...
var testMalformedDNSAgain = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x91, 0x2d, 0xff, 0x00, 0x00, 0x78, 0x11, 0xff, 0xe2, 0x0a, 0x4d, 0x00, 0x04, 0x0a, 0x01,
	0x00, 0x29, 0x00, 0x35, 0xf0, 0xaa, 0x00, 0x7d, 0x5b, 0x53, 0x30, 0xf1, 0x5c, 0xad, 0xef, 0x77,
	0x95, 0x82, 0x06, 0xb6, 0x3a, 0x41, 0x35, 0x7a, 0x8c, 0xef, 0xcd, 0xc0, 0xa7, 0x32, 0xb8, 0x00,
	0x46, 0x6e, 0x1c, 0x30, 0x2e, 0x75, 0x95, 0xac, 0xc0, 0x3d, 0x1e, 0xd4, 0x86, 0x35, 0x2d, 0x09,
	0x2f, 0xee, 0x3a, 0x82, 0xb4, 0xf0, 0x42, 0x7e, 0x2b, 0x6b, 0xf8, 0x70, 0xcc, 0x7f, 0xc9, 0xa1,
	0xe6, 0xf1, 0xa7, 0x61, 0x97, 0xec, 0x2f, 0xf7, 0xd2, 0x48, 0x4d, 0x95, 0x32, 0x1c, 0x6e, 0x4e,
	0x57, 0xfa, 0x6d, 0x3d, 0x9e, 0xc0, 0xfe, 0x3a, 0x6f, 0x1e, 0xe6, 0x34, 0x43, 0x96, 0xb4, 0x94,
	0x8b, 0x7a, 0xa9, 0x29, 0xd7, 0xe1, 0xda, 0x7c, 0xc3, 0x46, 0xca, 0x77, 0x48, 0x90, 0x6b, 0xf3,
	0x5e, 0xcb, 0x7e, 0x97, 0xc4, 0x9d, 0x35, 0x64, 0x98, 0x4f, 0xbf, 0x7c, 0x8a, 0xc1, 0xdd,
}

func TestMalformedDNSAgain(t *testing.T) {
	p := gopacket.NewPacket(testMalformedDNSAgain, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "walked out of range") {
		t.Errorf("unexpected error message: %v", err)
	}
}

// testMalformedDNSOhGodMakeItStop is the packet:
//   15:08:24.430906 IP 10.77.0.19.53 > 10.1.0.19.50635: 12397 zoneInit% [b2&3=0x7232] [47729a] [46283q] [60247n] [61718au][|domain]
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  0079 c51c 4000 3511 6be4 0a4d 0013 0a01  .y..@.5.k..M....
//   	0x0020:  0013 0035 c5cb 0065 ef45 306d 7232 b4cb  ...5...e.E0mr2..
//   	0x0030:  ba71 eb57 f116 3994 e000 4626 0534 66cc  .q.W..9...F&.4f.
//   	0x0040:  7b32 24f2 eece bca7 20e2 9a2a e1ce e737  {2$........*...7
//   	0x0050:  ac39 5fae 72ec c3ec 284f ca4a 171f 466d  .9_.r...(O.J..Fm
//   	0x0060:  f6c6 84d7 e795 310f 26df 9b59 6db9 21cf  ......1.&..Ym.!.
//   	0x0070:  15cb 30a3 c4cf df23 805a ed1a 0584 4fc3  ..0....#.Z....O.
//   	0x0080:  7fa3 3cb4 e04f e9                        ..<..O.
var testMalformedDNSOhGodMakeItStop = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x79, 0xc5, 0x1c, 0x40, 0x00, 0x35, 0x11, 0x6b, 0xe4, 0x0a, 0x4d, 0x00, 0x13, 0x0a, 0x01,
	0x00, 0x13, 0x00, 0x35, 0xc5, 0xcb, 0x00, 0x65, 0xef, 0x45, 0x30, 0x6d, 0x72, 0x32, 0xb4, 0xcb,
	0xba, 0x71, 0xeb, 0x57, 0xf1, 0x16, 0x39, 0x94, 0xe0, 0x00, 0x46, 0x26, 0x05, 0x34, 0x66, 0xcc,
	0x7b, 0x32, 0x24, 0xf2, 0xee, 0xce, 0xbc, 0xa7, 0x20, 0xe2, 0x9a, 0x2a, 0xe1, 0xce, 0xe7, 0x37,
	0xac, 0x39, 0x5f, 0xae, 0x72, 0xec, 0xc3, 0xec, 0x28, 0x4f, 0xca, 0x4a, 0x17, 0x1f, 0x46, 0x6d,
	0xf6, 0xc6, 0x84, 0xd7, 0xe7, 0x95, 0x31, 0x0f, 0x26, 0xdf, 0x9b, 0x59, 0x6d, 0xb9, 0x21, 0xcf,
	0x15, 0xcb, 0x30, 0xa3, 0xc4, 0xcf, 0xdf, 0x23, 0x80, 0x5a, 0xed, 0x1a, 0x05, 0x84, 0x4f, 0xc3,
	0x7f, 0xa3, 0x3c, 0xb4, 0xe0, 0x4f, 0xe9,
}

func TestMalformedDNSOhGodMakeItStop(t *testing.T) {
	p := gopacket.NewPacket(testMalformedDNSOhGodMakeItStop, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "offset pointer too high") {
		t.Errorf("unexpected error message: %v", err)
	}
}

var testMalformedDNSOPT = []byte{
	0x00, 0x90, 0x0b, 0x12, 0x91, 0xc1, 0x00, 0x1c, 0xc0, 0x93, 0x33, 0xfb, 0x08, 0x00, 0x45, 0x00,
	0x00, 0x5A, 0xce, 0x58, 0x00, 0x00, 0x40, 0x11, 0x67, 0xe2, 0xac, 0x10, 0x01, 0xc7, 0x4b, 0x4b,
	0x4b, 0x4b, 0xd6, 0x00, 0x00, 0x35, 0x00, 0x46, 0x44, 0xb0, 0x50, 0x12, 0x01, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x77, 0x77, 0x77, 0x04, 0x69, 0x65, 0x74, 0x66, 0x03,
	0x6f, 0x72, 0x67, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x80,
	0x00, 0x00, 0x13, 0x69, 0x42, 0x00, 0x10, 0x4F, 0x70, 0x65, 0x6E, 0x44, 0x4E, 0x53, 0x01, 0x23,
	0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
}

func TestMalformedDNSOPT(t *testing.T) {
	p := gopacket.NewPacket(testMalformedDNSOPT, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "Malformed DNSOPT record") {
		t.Errorf("unexpected error message: %v", err)
	}
}

// testPacketDNSPanic7 is the packet:
//   07:56:25.174747 IP 10.77.0.11.53 > 10.1.0.67.55777: 41808*-| 3/7/0 TXT "google-site-verification=DC2uC-T8kD33lINhNzfo0bNBrw-vrCXs5BPF5BXY56g", TXT "v=spf1 include:spf-a.outlook.com include:spf-b.outlook.com ip4:157.55.9.128/25 include:spf.protection.outlook.com include:spf-a.hotmail.com include:_spf-ssg-b.microsoft.com include:_spf-ssg-c.microsoft.com ~all", TXT "google-site-verification=0iLWhIMhXEkeWwWfFU4ursTn-_OvoOjaA0Lr7Pg1sEM" (512)
//   	0x0000:  0055 22af c637 0022 55ac deac 0800 4500  .U"..7."U.....E.
//   	0x0010:  021c b5ca 4000 fa11 b46a 0a4d 000b 0a01  ....@....j.M....
//   	0x0020:  0043 0035 d9e1 0208 afd6 a350 8600 0001  .C.5.......P....
//   	0x0030:  0003 0007 0000 076f 7574 6c6f 6f6b 0363  .......outlook.c
//   	0x0040:  6f6d 0000 1000 01c0 0c00 1000 0100 0001  om..............
//   	0x0050:  2c00 4544 676f 6f67 6c65 2d73 6974 652d  ,.EDgoogle-site-
//   	0x0060:  7665 7269 6669 6361 7469 6f6e 3d44 4332  verification=DC2
//   	0x0070:  7543 2d54 386b 4433 336c 494e 684e 7a66  uC-T8kD33lINhNzf
//   	0x0080:  6f30 624e 4272 772d 7672 4358 7335 4250  o0bNBrw-vrCXs5BP
//   	0x0090:  4635 4258 5935 3667 c00c 0010 0001 0000  F5BXY56g........
//   	0x00a0:  012c 00d3 d276 3d73 7066 3120 696e 636c  .,...v=spf1.incl
//   	0x00b0:  7564 653a 7370 662d 612e 6f75 746c 6f6f  ude:spf-a.outloo
//   	0x00c0:  6b2e 636f 6d20 696e 636c 7564 653a 7370  k.com.include:sp
//   	0x00d0:  662d 622e 6f75 746c 6f6f 6b2e 636f 6d20  f-b.outlook.com.
//   	0x00e0:  6970 343a 3135 372e 3535 2e39 2e31 3238  ip4:157.55.9.128
//   	0x00f0:  2f32 3520 696e 636c 7564 653a 7370 662e  /25.include:spf.
//   	0x0100:  7072 6f74 6563 7469 6f6e 2e6f 7574 6c6f  protection.outlo
//   	0x0110:  6f6b 2e63 6f6d 2069 6e63 6c75 6465 3a73  ok.com.include:s
//   	0x0120:  7066 2d61 2e68 6f74 6d61 696c 2e63 6f6d  pf-a.hotmail.com
//   	0x0130:  2069 6e63 6c75 6465 3a5f 7370 662d 7373  .include:_spf-ss
//   	0x0140:  672d 622e 6d69 6372 6f73 6f66 742e 636f  g-b.microsoft.co
//   	0x0150:  6d20 696e 636c 7564 653a 5f73 7066 2d73  m.include:_spf-s
//   	0x0160:  7367 2d63 2e6d 6963 726f 736f 6674 2e63  sg-c.microsoft.c
//   	0x0170:  6f6d 207e 616c 6cc0 0c00 1000 0100 0001  om.~all.........
//   	0x0180:  2c00 4544 676f 6f67 6c65 2d73 6974 652d  ,.EDgoogle-site-
//   	0x0190:  7665 7269 6669 6361 7469 6f6e 3d30 694c  verification=0iL
//   	0x01a0:  5768 494d 6858 456b 6557 7757 6646 5534  WhIMhXEkeWwWfFU4
//   	0x01b0:  7572 7354 6e2d 5f4f 766f 4f6a 6141 304c  ursTn-_OvoOjaA0L
//   	0x01c0:  7237 5067 3173 454d c00c 0002 0001 0002  r7Pg1sEM........
//   	0x01d0:  a300 000e 036e 7332 046d 7366 7403 6e65  .....ns2.msft.ne
//   	0x01e0:  7400 c00c 0002 0001 0002 a300 0006 036e  t..............n
//   	0x01f0:  7334 c1ae c00c 0002 0001 0002 a300 0006  s4..............
//   	0x0200:  036e 7331 c1ae c00c 0002 0001 0002 a300  .ns1............
//   	0x0210:  0006 036e 7333 c1ae c00c 0002 0001 0002  ...ns3..........
//   	0x0220:  a300 0015 046e 7331 610d                 .....ns1a.
var testPacketDNSPanic7 = []byte{
	0x00, 0x55, 0x22, 0xaf, 0xc6, 0x37, 0x00, 0x22, 0x55, 0xac, 0xde, 0xac, 0x08, 0x00, 0x45, 0x00,
	0x02, 0x1c, 0xb5, 0xca, 0x40, 0x00, 0xfa, 0x11, 0xb4, 0x6a, 0x0a, 0x4d, 0x00, 0x0b, 0x0a, 0x01,
	0x00, 0x43, 0x00, 0x35, 0xd9, 0xe1, 0x02, 0x08, 0xaf, 0xd6, 0xa3, 0x50, 0x86, 0x00, 0x00, 0x01,
	0x00, 0x03, 0x00, 0x07, 0x00, 0x00, 0x07, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f, 0x6b, 0x03, 0x63,
	0x6f, 0x6d, 0x00, 0x00, 0x10, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x01,
	0x2c, 0x00, 0x45, 0x44, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2d, 0x73, 0x69, 0x74, 0x65, 0x2d,
	0x76, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x44, 0x43, 0x32,
	0x75, 0x43, 0x2d, 0x54, 0x38, 0x6b, 0x44, 0x33, 0x33, 0x6c, 0x49, 0x4e, 0x68, 0x4e, 0x7a, 0x66,
	0x6f, 0x30, 0x62, 0x4e, 0x42, 0x72, 0x77, 0x2d, 0x76, 0x72, 0x43, 0x58, 0x73, 0x35, 0x42, 0x50,
	0x46, 0x35, 0x42, 0x58, 0x59, 0x35, 0x36, 0x67, 0xc0, 0x0c, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00,
	0x01, 0x2c, 0x00, 0xd3, 0xd2, 0x76, 0x3d, 0x73, 0x70, 0x66, 0x31, 0x20, 0x69, 0x6e, 0x63, 0x6c,
	0x75, 0x64, 0x65, 0x3a, 0x73, 0x70, 0x66, 0x2d, 0x61, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f,
	0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x3a, 0x73, 0x70,
	0x66, 0x2d, 0x62, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x20,
	0x69, 0x70, 0x34, 0x3a, 0x31, 0x35, 0x37, 0x2e, 0x35, 0x35, 0x2e, 0x39, 0x2e, 0x31, 0x32, 0x38,
	0x2f, 0x32, 0x35, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x3a, 0x73, 0x70, 0x66, 0x2e,
	0x70, 0x72, 0x6f, 0x74, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f,
	0x6f, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x3a, 0x73,
	0x70, 0x66, 0x2d, 0x61, 0x2e, 0x68, 0x6f, 0x74, 0x6d, 0x61, 0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
	0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x3a, 0x5f, 0x73, 0x70, 0x66, 0x2d, 0x73, 0x73,
	0x67, 0x2d, 0x62, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f,
	0x6d, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x64, 0x65, 0x3a, 0x5f, 0x73, 0x70, 0x66, 0x2d, 0x73,
	0x73, 0x67, 0x2d, 0x63, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63,
	0x6f, 0x6d, 0x20, 0x7e, 0x61, 0x6c, 0x6c, 0xc0, 0x0c, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x01,
	0x2c, 0x00, 0x45, 0x44, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2d, 0x73, 0x69, 0x74, 0x65, 0x2d,
	0x76, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x30, 0x69, 0x4c,
	0x57, 0x68, 0x49, 0x4d, 0x68, 0x58, 0x45, 0x6b, 0x65, 0x57, 0x77, 0x57, 0x66, 0x46, 0x55, 0x34,
	0x75, 0x72, 0x73, 0x54, 0x6e, 0x2d, 0x5f, 0x4f, 0x76, 0x6f, 0x4f, 0x6a, 0x61, 0x41, 0x30, 0x4c,
	0x72, 0x37, 0x50, 0x67, 0x31, 0x73, 0x45, 0x4d, 0xc0, 0x0c, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02,
	0xa3, 0x00, 0x00, 0x0e, 0x03, 0x6e, 0x73, 0x32, 0x04, 0x6d, 0x73, 0x66, 0x74, 0x03, 0x6e, 0x65,
	0x74, 0x00, 0xc0, 0x0c, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02, 0xa3, 0x00, 0x00, 0x06, 0x03, 0x6e,
	0x73, 0x34, 0xc1, 0xae, 0xc0, 0x0c, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02, 0xa3, 0x00, 0x00, 0x06,
	0x03, 0x6e, 0x73, 0x31, 0xc1, 0xae, 0xc0, 0x0c, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02, 0xa3, 0x00,
	0x00, 0x06, 0x03, 0x6e, 0x73, 0x33, 0xc1, 0xae, 0xc0, 0x0c, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02,
	0xa3, 0x00, 0x00, 0x15, 0x04, 0x6e, 0x73, 0x31, 0x61, 0x0d,
}

func TestPacketDNSPanic7(t *testing.T) {
	p := gopacket.NewPacket(testPacketDNSPanic7, LinkTypeEthernet, testDecodeOptions)
	if errLayer := p.ErrorLayer(); errLayer == nil {
		t.Error("No error layer on invalid DNS name")
	} else if err := errLayer.Error(); !strings.Contains(err.Error(), "resource record length exceeds data") {
		t.Errorf("unexpected error message: %v", err)
	}
}

func TestDNSPacketWriteAnswer(t *testing.T) {
	dns := &DNS{ID: 0x1234, QR: true, OpCode: DNSOpCodeQuery, ResponseCode: DNSResponseCodeNoErr, Answers: []DNSResourceRecord{
		DNSResourceRecord{
			Name:  []byte("www.example.com"),
			Type:  DNSTypeA,
			Class: DNSClassIN,
			IP:    net.IPv4(127, 0, 0, 1),
		},
		DNSResourceRecord{
			Name:  []byte("www.example.com"),
			Type:  DNSTypeAAAA,
			Class: DNSClassIN,
			IP:    net.IP{15: 1},
		},
	}}
	buf := gopacket.NewSerializeBuffer()
	opts := gopacket.SerializeOptions{ComputeChecksums: true, FixLengths: true}
	if err := gopacket.SerializeLayers(buf, opts, dns); err != nil {
		t.Fatal(err)
	}
	dns2 := &DNS{}
	if err := dns2.DecodeFromBytes(buf.Bytes(), gopacket.NilDecodeFeedback); err != nil {
		t.Fatalf("could not decode: %v", err)
	}
	if want, got := 2, len(dns2.Answers); want != got {
		t.Fatalf("num answers, want %d got %d", want, got)
	} else if got, want := string(dns2.Answers[0].Name), "www.example.com"; got != want {
		t.Fatalf("unexpected first answer name %q, want %q", got, want)
	} else if got, want := string(dns2.Answers[1].Name), "www.example.com"; got != want {
		t.Fatalf("unexpected second answer name %q, want %q", got, want)
	}
	t.Log(gopacket.LayerString(dns2))
	if want, got := 86, len(buf.Bytes()); want != got {
		t.Fatalf("Encoded size, want %d got %d", want, got)
	}
}
