use errors::{bail, Result};
use minify_html::{with_friendly_error, Cfg};

pub fn html(html: String) -> Result<String> {
    let cfg = &Cfg { minify_js: false, minify_css: false };
    let mut input_bytes = html.as_bytes().to_vec();

    match with_friendly_error(&mut input_bytes, cfg) {
        Ok(len) => match std::str::from_utf8(&input_bytes[..len]) {
            Ok(result) => Ok(result.to_string()),
            Err(err) => bail!("Failed to convert bytes to string : {}", err),
        },
        Err(minify_error) => {
            bail!(
                "Failed to truncate html at character {}: {} \n {}",
                minify_error.position,
                minify_error.message,
                minify_error.code_context
            );
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    // https://github.com/getzola/zola/issues/1292
    #[test]
    fn can_minify_html() {
        let input = r#"
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
</head>
<body>


    <p>Example blog post</p>

  FOO BAR
</body>
</html>
"#;
        let expected = r#"<!doctype html><html><head><meta charset=utf-8><body><p>Example blog post</p> FOO BAR"#;
        let res = html(input.to_owned()).unwrap();
        assert_eq!(res, expected);
    }

    // https://github.com/getzola/zola/issues/1304
    #[test]
    fn can_minify_multibyte_characters() {
        let input = r#"
俺が好きなのはキツネの…ケツねｗ
ー丁寧なインタネット生活の人より
"#;
        let expected = r#"俺が好きなのはキツネの…ケツねｗ ー丁寧なインタネット生活の人より"#;
        let res = html(input.to_owned()).unwrap();
        assert_eq!(res, expected);
    }

    // https://github.com/getzola/zola/issues/1300
    #[test]
    fn can_minify_and_preserve_whitespace_in_pre_elements() {
        let input = r#"
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
</head>
<body>
  <pre><code>fn main() {
    println!("Hello, world!");
    <span>loop {
      println!("Hello, world!");
    }</span>
  }
  </code></pre>
</body>
</html>
"#;
        let expected = r#"<!doctype html><html><head><meta charset=utf-8><body><pre><code>fn main() {
    println!("Hello, world!");
    <span>loop {
      println!("Hello, world!");
    }</span>
  }
  </code></pre>"#;
        let res = html(input.to_owned()).unwrap();
        assert_eq!(res, expected);
    }
}
