+++
title = "Overview"
weight = 10
+++

Zola uses the [Tera](https://tera.netlify.com) template engine, which is very similar
to Jinja2, Liquid and Twig.

As this documentation will only talk about how templates work in Zola, please read
the [Tera template documentation](https://tera.netlify.com/docs#templates) if you want
to learn more about it first.

All templates live in the `templates` directory.  If you are not sure what variables are available in a template,
you can place `{{ __tera_context }}` in the template to print the whole context.

A few variables are available on all templates except feeds and the sitemap:

- `config`: the [configuration](@/documentation/getting-started/configuration.md) without any modifications
- `current_path`: the path (full URL without `base_url`) of the current page, always starting with a `/`
- `current_url`: the full URL for the current page
- `lang`: the language for the current page

Config variables can be accessed like `config.variable`, in HTML for example with `{{ config.base_url }}`.
The 404 template does not get `current_path` and `current_url` (this information cannot be determined).

## Standard templates
By default, Zola will look for three templates: `index.html`, which is applied
to the site homepage; `section.html`, which is applied to all sections (any HTML
page generated by creating a directory within your `content` directory); and
`page.html`, which is applied to all pages (any HTML page generated by creating an
`.md` file within your `content` directory).

The homepage is always a section (regardless of whether it contains other pages).
Thus, the `index.html` and `section.html` templates both have access to the
section variables.  The `page.html` template has access to the page variables.
The page and section variables are described in more detail in the next section.

## Built-in templates
Zola comes with four built-in templates: `atom.xml` and `rss.xml` (described in
[Feeds](@/documentation/templates/feeds/index.md)), `sitemap.xml` (described in [Sitemap](@/documentation/templates/sitemap.md)),
and `robots.txt` (described in [Robots.txt](@/documentation/templates/robots.md)).
Additionally, themes can add their own templates, which will be applied if not
overridden.  You can override built-in or theme templates by creating a template with
the same name in the correct path. For example, you can override the Atom template by
creating a `templates/atom.xml` file.

## Custom templates
In addition to the standard `index.html`, `section.html` and `page.html` templates,
you may also create custom templates by creating an `.html` file in the `templates`
directory.  These custom templates will not be used by default.  Instead, a custom template will _only_ be used if you apply it by setting the `template` front-matter variable to the path for that template (or if you `include` it in another template that is applied).  For example, if you created a custom template for your site's About page called `about.html`, you could apply it to your `about.md` page by including the following front matter in your `about.md` page:

```md
+++
title = "About Us"
template = "about.html"
+++
```

Custom templates are not required to live at the root of your `templates` directory.
For example, `product_pages/with_pictures.html` is a valid template.

## Built-in filters
Zola adds a few filters in addition to [those](https://tera.netlify.com/docs/#filters) already present
in Tera.

### markdown
Converts the given variable to HTML using Markdown. This doesn't apply any of the
features that Zola adds to Markdown; for example, internal links and shortcodes won't work.

By default, the filter will wrap all text in a paragraph. To disable this behaviour, you can
pass `true` to the inline argument:

```jinja2
{{ some_text | markdown(inline=true) }}
```

You do not need to use this filter with `page.content` or `section.content`, the content is already rendered.

### base64_encode
Encode the variable to base64.

### base64_decode
Decode the variable from base64.


## Built-in global functions

Zola adds a few global functions to [those in Tera](https://tera.netlify.com/docs#built-in-functions)
to make it easier to develop complex sites.


### `get_page`
Takes a path to an `.md` file and returns the associated page.

```jinja2
{% set page = get_page(path="blog/page2.md") %}
```

### `get_section`
Takes a path to an `_index.md` file and returns the associated section.

```jinja2
{% set section = get_section(path="blog/_index.md") %}
```

If you only need the metadata of the section, you can pass `metadata_only=true` to the function:

```jinja2
{% set section = get_section(path="blog/_index.md", metadata_only=true) %}
```

### ` get_url`
Gets the permalink for the given path.
If the path starts with `@/`, it will be treated as an internal
link like the ones used in Markdown, starting from the root `content` directory.

```jinja2
{% set url = get_url(path="@/blog/_index.md") %}
```

It accepts an optional parameter `lang` in order to compute a *language-aware URL* in multilingual websites. Assuming `config.base_url` is `"http://example.com"`, the following snippet will:

- return `"http://example.com/blog/"` if `config.default_language` is `"en"`
- return `"http://example.com/en/blog/"` if `config.default_language` is **not** `"en"` and `"en"` appears in `config.languages`
- fail otherwise, with the error message `"'en' is not an authorized language (check config.languages)."`

```jinja2
{% set url = get_url(path="@/blog/_index.md", lang="en") %}
```

This can also be used to get the permalinks for static assets, for example if
we want to link to the file that is located at `static/css/app.css`:

```jinja2
{{/* get_url(path="css/app.css") */}}
```

By default, assets will not have a trailing slash. You can force one by passing `trailing_slash=true` to the `get_url` function.
An example is:

```jinja2
{{/* get_url(path="css/app.css", trailing_slash=true) */}}
```

In the case of non-internal links, you can also add a cachebust of the format `?h=<sha256>` at the end of a URL
by passing `cachebust=true` to the `get_url` function.


### 'get_file_hash`

Gets the hash digest for a static file. Supported hashes are SHA-256, SHA-384 (default) and SHA-512. Requires `path`. The `sha_type` key is optional and must be one of 256, 384 or 512.

```jinja2
{{/* get_file_hash(path="js/app.js", sha_type=256) */}}
```

This can be used to implement subresource integrity. Do note that subresource integrity is typically used when using external scripts, which `get_file_hash` does not support.

```jinja2
<script src="{{/* get_url(path="js/app.js") */}}"
        integrity="sha384-{{/* get_file_hash(path="js/app.js", sha_type=384) */}}"></script>
```

Whenever hashing files, whether using `get_file_hash` or `get_url(..., cachebust=true)`, the file is searched for in three places: `static/`, `content/` and the output path (so e.g. compiled SASS can be hashed, too.)


### `get_image_metadata`
Gets metadata for an image. This supports common formats like JPEG, PNG, as well as SVG.
Currently, the only supported keys are `width` and `height`.

```jinja2
  {% set meta = get_image_metadata(path="...") %}
  Our image is {{ meta.width }}x{{ meta.height }}
```

### `get_taxonomy_url`
Gets the permalink for the taxonomy item found.

```jinja2
{% set url = get_taxonomy_url(kind="categories", name=page.taxonomies.category, lang=page.lang) %}
```

`name` will almost always come from a variable but in case you want to do it manually,
the value should be the same as the one in the front matter, not the slugified version.

`lang` (optional) default to `config.default_language` in config.toml

### `get_taxonomy`
Gets the whole taxonomy of a specific kind.

```jinja2
{% set categories = get_taxonomy(kind="categories") %}
```

The type of the output is:

```ts
kind: TaxonomyConfig;
items: Array<TaxonomyTerm>;
```

See the [Taxonomies documentation](@/documentation/templates/taxonomies.md) for a full documentation of those types.

### `load_data`
Loads data from a file or URL. Supported file types include *toml*, *json* and *csv*.
Any other file type will be loaded as plain text.

The `path` argument specifies the path to the data file relative to your base directory, where your `config.toml` is.
As a security precaution, if this file is outside the main site directory, your site will fail to build.

```jinja2
{% set data = load_data(path="content/blog/story/data.toml") %}
```

The optional `format` argument allows you to specify and override which data type is contained
within the file specified in the `path` argument. Valid entries are `toml`, `json`, `csv`
or `plain`. If the `format` argument isn't specified, then the path extension is used.

```jinja2
{% set data = load_data(path="content/blog/story/data.txt", format="json") %}
```

Use the `plain` format for when your file has a toml/json/csv extension but you want to load it as plain text.

For *toml* and *json*, the data is loaded into a structure matching the original data file;
however, for *csv* there is no native notion of such a structure. Instead, the data is separated
into a data structure containing *headers* and *records*. See the example below to see
how this works.

In the template:
```jinja2
{% set data = load_data(path="content/blog/story/data.csv") %}
```

In the *content/blog/story/data.csv* file:
```csv
Number, Title
1,Gutenberg
2,Printing
```

The equivalent json value of the parsed data would be stored in the `data` variable in the
template:
```json
{
    "headers": ["Number", "Title"],
    "records": [
        ["1", "Gutenberg"],
        ["2", "Printing"]
    ],
}
```

#### Remote content

Instead of using a file, you can load data from a remote URL. This can be done by specifying a `url` parameter
to `load_data` rather than `path`.

```jinja2
{% set response = load_data(url="https://api.github.com/repos/getzola/zola") %}
{{ response }}
```

By default, the response body will be returned with no parsing. This can be changed by using the `format` argument
as below.


```jinja2
{% set response = load_data(url="https://api.github.com/repos/getzola/zola", format="json") %}
{{ response }}
```

#### Data caching

Data file loading and remote requests are cached in memory during the build, so multiple requests aren't made
to the same endpoint.
URLs are cached based on the URL, and data files are cached based on the file modified time.
The format is also taken into account when caching, so a request will be sent twice if it's loaded with two
different formats.

### `trans`
Gets the translation of the given `key`, for the `default_language` or the `lang`uage given

```jinja2
{{/* trans(key="title") */}}
{{/* trans(key="title", lang="fr") */}}
```

### `resize_image`
Resizes an image file.
Please refer to [_Content / Image Processing_](@/documentation/content/image-processing/index.md) for complete documentation.
