use clap::{SubCommand, ArgMatches, Arg};
use chrono;
use commands::{BasicOptions, StaticSubcommand, ask};
use libpijul::Hash;
use libpijul::fs_representation::patches_dir;
use libpijul::patch::PatchFlags;
use commands::hooks::run_hook;
use std::mem::drop;
use std::collections::HashSet;

use error::Result;
use meta::{Global, Meta, load_global_or_local_signing_key};
use super::default_explain;
use super::record;

pub fn tag_args(sub: StaticSubcommand) -> StaticSubcommand {
    sub.arg(Arg::with_name("repository")
            .long("repository")
            .help("The repository where to record, defaults to the current directory.")
            .takes_value(true)
            .required(false))
        .arg(Arg::with_name("branch")
             .long("branch")
             .help("The branch where to record, defaults to the current branch.")
             .takes_value(true)
             .required(false))
        .arg(Arg::with_name("date")
             .long("date")
             .help("The date to use to record the patch, default is now.")
             .takes_value(true)
             .required(false))
        .arg(Arg::with_name("message")
             .short("m")
             .long("message")
             .help("The name of the patch to record")
             .takes_value(true))
        .arg(Arg::with_name("description")
             .short("d")
             .long("description")
             .help("The description of the patch to record")
             .takes_value(true))
        .arg(Arg::with_name("no-editor")
             .long("no-editor")
             .help("Do not use an editor to write the patch name and description, even if the variable is set in the configuration file")
             .takes_value(false))
        .arg(Arg::with_name("author")
             .short("A")
             .long("author")
             .help("Author of this patch (multiple occurrences allowed)")
             .takes_value(true))
}

pub fn invocation() -> StaticSubcommand {
    return SubCommand::with_name("tag")
        .about("create a patch (a \"tag\") with no changes, and all currently applied patches as dependencies")
        .arg(Arg::with_name("repository")
             .long("repository")
             .help("The repository where to record, defaults to the current directory.")
             .takes_value(true)
             .required(false))
        .arg(Arg::with_name("branch")
             .long("branch")
             .help("The branch where to record, defaults to the current branch.")
             .takes_value(true)
             .required(false))
        .arg(Arg::with_name("message")
             .short("m")
             .long("message")
             .help("The name of the patch to record")
             .takes_value(true))
        .arg(Arg::with_name("no-editor")
             .long("no-editor")
             .help("Do not use an editor to write the patch name and description, even if the variable is set in the configuration file")
             .takes_value(false))
        .arg(Arg::with_name("author")
             .short("A")
             .long("author")
             .help("Author of this patch (multiple occurrences allowed)")
             .takes_value(true))
}

pub fn run(args: &ArgMatches) -> Result<Option<Hash>> {
    let opts = BasicOptions::from_args(args)?;
    let patch_name_arg = args.value_of("message");
    let authors_arg = args.values_of("author").map(|x| x.collect::<Vec<_>>());
    let branch_name = opts.branch();

    let mut save_meta = false;
    let mut save_global = false;

    let mut global = Global::load().unwrap_or_else(|e| {
        info!("loading global key, error {:?}", e);
        save_global = true;
        Global::new()
    });

    let mut meta = match Meta::load(&opts.repo_root) {
        Ok(m) => m,
        Err(_) => {
            save_meta = true;
            Meta::new()
        }
    };

    let repo = opts.open_repo()?;
    let patch = {
        let txn = repo.txn_begin()?;
        debug!("meta:{:?}", meta);
        let authors: Vec<String> = if let Some(ref authors) = authors_arg {
            let authors: Vec<String> = authors.iter().map(|x| x.to_string()).collect();
            {
                if meta.authors.len() == 0 {
                    meta.authors = authors.clone();
                    save_meta = true
                }
                if global.author.len() == 0 {
                    global.author = authors[0].clone();
                    save_global = true
                }
            }
            authors
        } else {
            if meta.authors.len() > 0 {
                meta.authors.clone()
            } else if global.author.len() > 0 {
                vec![global.author.clone()]
            } else {
                let authors = ask::ask_authors()?;
                save_meta = true;
                meta.authors = authors.clone();
                save_global = true;
                global.author = authors[0].clone();
                authors
            }
        };

        debug!("authors:{:?}", authors);
        let (patch_name, description) = if let Some(ref m) = patch_name_arg {
            (m.to_string(), None)
        } else {
            // Sometimes, it can be handy to just write the patch name from
            // the CLI. We therefore provide a new flag, `no-editor`, that
            // forces pijul to fallback into the default behaviour even if
            // an editor is configured.
            let maybe_editor = if !args.is_present("no-editor") {
                // There are two places where one can configure an editor to
                // use: globally and locally. If it is configured in both
                // location, we prefer the local setting.
                if meta.editor.is_some() {
                    meta.editor.as_ref()
                } else {
                    global.editor.as_ref()
                }
            } else {
                None
            };

            ask::ask_patch_name(&opts.repo_root, maybe_editor)?
        };

        run_hook(&opts.repo_root, "patch-name", Some(&patch_name))?;

        debug!("patch_name:{:?}", patch_name);
        if save_meta {
            meta.save(&opts.repo_root)?
        }
        if save_global {
            global.save()?
        }
        debug!("new");
        let branch = txn.get_branch(&branch_name).unwrap();

        let mut included = HashSet::new();
        let mut patches = Vec::new();
        for (_, patch) in txn.rev_iter_applied(&branch, None) {
            // `patch` is already implied if a patch on the branch
            // depends on `patch`. Let's look at all patches known to
            // the repository that depend on `patch`, and see if a
            // patch on the branch (i.e. all patches in `included`,
            // since we're considering patches in reverse order of
            // application) depends on `patch`.
            let mut already_in = false;
            for (p, revdep) in txn.iter_revdep(Some((patch, None))) {
                if p == patch {
                    if included.contains(&revdep) {
                        already_in = true
                    }
                } else {
                    break
                }
            }
            if !already_in {
                let patch = txn.get_external(patch).unwrap();
                patches.push(patch.to_owned());
            }
            included.insert(patch.to_owned());
        }
        txn.new_patch(&branch,
                      authors,
                      patch_name,
                      description,
                      chrono::Utc::now(),
                      Vec::new(),
                      patches.into_iter(),
                      PatchFlags::TAG)
    };
    drop(repo);

    let dot_pijul = opts.repo_dir();
    let key = if let Ok(Some(key)) = meta.signing_key() {
        Some(key)
    } else {
        load_global_or_local_signing_key(Some(&dot_pijul)).ok()
    };
    debug!("key.is_some(): {:?}", key.is_some());
    let patches_dir = patches_dir(&opts.repo_root);
    let hash = patch.save(&patches_dir, key.as_ref())?;

    let pristine_dir = opts.pristine_dir();
    let mut increase = 40960;
    loop {
        match record::record_no_resize(&pristine_dir, &opts.repo_root, &branch_name, &hash, &patch,
                                       &HashSet::new(), increase) {
            Err(ref e) if e.lacks_space() => { increase *= 2 },
            _ => break
        }
    }
    println!("Recorded patch {}", hash.to_base58());
    Ok(Some(hash))
}

pub fn explain(res: Result<Option<Hash>>) {
    default_explain(res)
}
