use clap::{SubCommand, ArgMatches, Arg};
use chrono;
use commands::{BasicOptions, StaticSubcommand, ask};
use libpijul::{Repository, Hash, InodeUpdate, Patch, PatchId, MutTxn, RecordState, Key};
use libpijul::patch::{PatchFlags, Record};
use commands::hooks::run_hook;
use commands::tag::tag_args;
use libpijul::fs_representation::{patches_dir, untracked_files, ignore_file};
use std::mem::drop;
use error::{Result, ErrorKind};
use std::fs::canonicalize;
use std::collections::HashSet;
use libpijul;
use std::fs::{metadata, OpenOptions};
use std::path::{PathBuf, Path};
use meta::{Global, Meta, load_global_or_local_signing_key};
use super::ask::{ChangesDirection, ask_changes};
use super::default_explain;
use std::str::FromStr;
use rand;
use std::rc::Rc;
use std::cell::RefCell;
use std::io::Write;


pub fn invocation() -> StaticSubcommand {
    return tag_args(
        SubCommand::with_name("record")
            .about("record changes in the repository")
            .arg(
                Arg::with_name("all")
                    .short("a")
                    .long("all")
                    .help("Answer 'y' to all questions")
                    .takes_value(false),
            )
            .arg(
                Arg::with_name("add-new-files")
                    .short("n")
                    .long("add-new-files")
                    .help(
                        "Offer to add files that have been created since the last record",
                    )
                    .takes_value(false),
            )
            .arg(
                Arg::with_name("depends-on")
                    .help(
                        "Add a dependency to this patch (internal id or hash accepted)",
                    )
                    .long("depends-on")
                    .takes_value(true)
                    .multiple(true),
            )
            .arg(
                Arg::with_name("prefix")
                    .help("Prefix to start from")
                    .takes_value(true)
                    .multiple(true),
            ),
    );
}

fn add_untracked_files<T: rand::Rng>(
    txn: &mut MutTxn<T>,
    repo_root: &Path,
) -> Result<HashSet<PathBuf>> {
    let untracked = untracked_files(txn, repo_root);
    debug!("adding untracked_files at record time: {:?}", &untracked);
    for file in untracked.iter() {
        let m = metadata(&file)?;
        let file = file.strip_prefix(&repo_root)?;
        if let Err(e) = txn.add_file(&file, m.is_dir()) {
            if let &libpijul::ErrorKind::AlreadyAdded = e.kind() {
            } else {
                return Err(e.into())
            }
        }
    }
    Ok(untracked)
}

fn append_to_ignore_file(repo_root: &Path, lines: &Vec<String>) -> Result<()> {
    let ignore_file = ignore_file(repo_root);
    let mut file = OpenOptions::new().append(true).create(true).open(
        ignore_file,
    )?;
    for line in lines {
        file.write_all(line.as_ref())?;
        file.write_all(b"\n")?
    }
    Ok(())

}

fn select_changes(
    opts: &BasicOptions,
    add_new_files: bool,
    branch_name: &str,
    yes_to_all: bool,
    prefix: Option<HashSet<PathBuf>>,
) -> Result<(Vec<Record<Vec<Key<Option<Hash>>>>>, HashSet<InodeUpdate>)> {
    // Increase by 100 pages. The most things record can
    // write is one write in the branches table, affecting
    // at most O(log n) blocks.
    let repo = opts.open_and_grow_repo(409600)?;
    let mut txn = repo.mut_txn_begin(rand::thread_rng())?;
    let mut to_unadd = if add_new_files {
        add_untracked_files(&mut txn, &opts.repo_root)?
    } else {
        HashSet::<PathBuf>::new()
    };
    let (changes, syncs) =
        changes_from_prefixes(&opts.repo_root, &mut txn, &branch_name, prefix.as_ref())?;
    let changes: Vec<_> = changes
        .into_iter()
        .map(|x| txn.globalize_record(x))
        .collect();
    if !yes_to_all {
        let (c, i) = ask_changes(
            &txn,
            &opts.repo_root,
            &opts.cwd,
            &changes,
            ChangesDirection::Record,
            &mut to_unadd,
        )?;
        let selected = changes
            .into_iter()
            .enumerate()
            .filter(|&(i, _)| *(c.get(&i).unwrap_or(&false)))
            .map(|(_, x)| x)
            .collect();
        for file in to_unadd {
            txn.remove_file(&file)?
        }
        txn.commit()?;
        append_to_ignore_file(&opts.repo_root, &i)?;
        Ok((selected, syncs))
    } else {
        txn.commit()?;
        Ok((changes, syncs))
    }
}


pub fn run(args: &ArgMatches) -> Result<Option<Hash>> {
    let opts = BasicOptions::from_args(args)?;
    let yes_to_all = args.is_present("all");
    let patch_name_arg = args.value_of("message");
    let patch_descr_arg = args.value_of("description");
    let authors_arg = args.values_of("author").map(|x| x.collect::<Vec<_>>());
    let branch_name = opts.branch();
    let add_new_files = args.is_present("add-new-files");

    let patch_date = args.value_of("date").map_or(Ok(chrono::Utc::now()), |x| {
        chrono::DateTime::from_str(x).map_err(|_| ErrorKind::InvalidDate(String::from(x)))
    })?;

    let mut save_meta = false;
    let mut save_global = false;

    let mut global = Global::load().unwrap_or_else(|e| {
        info!("loading global key, error {:?}", e);
        save_global = true;
        Global::new()
    });

    let mut meta = match Meta::load(&opts.repo_root) {
        Ok(m) => m,
        Err(_) => {
            save_meta = true;
            Meta::new()
        }
    };

    run_hook(&opts.repo_root, "pre-record", None)?;

    debug!("prefix {:?}", args.value_of("prefix"));
    let prefix = prefix(args, &opts)?;

    let (changes, syncs) = select_changes(&opts, add_new_files, &branch_name, yes_to_all, prefix)?;


    if changes.is_empty() {
        println!("Nothing to record");
        Ok(None)
    } else {
        let repo = opts.open_repo()?;
        let patch = {
            let txn = repo.txn_begin()?;
            debug!("meta:{:?}", meta);
            let authors: Vec<String> = if let Some(ref authors) = authors_arg {
                let authors: Vec<String> = authors.iter().map(|x| x.to_string()).collect();
                {
                    if meta.authors.len() == 0 {
                        meta.authors = authors.clone();
                        save_meta = true
                    }
                    if global.author.len() == 0 {
                        global.author = authors[0].clone();
                        save_global = true
                    }
                }
                authors
            } else {
                if meta.authors.len() > 0 {
                    meta.authors.clone()
                } else if global.author.len() > 0 {
                    vec![global.author.clone()]
                } else {
                    let authors = ask::ask_authors()?;
                    save_meta = true;
                    meta.authors = authors.clone();
                    save_global = true;
                    global.author = authors[0].clone();
                    authors
                }
            };
            debug!("authors:{:?}", authors);

            let (patch_name, description) = if let Some(ref m) = patch_name_arg {
                (
                    m.to_string(),
                    patch_descr_arg.map(|x| String::from(x.trim())),
                )
            } else {
                // Sometimes, it can be handy to just write the patch name from
                // the CLI. We therefore provide a new flag, `no-editor`, that
                // forces pijul to fallback into the default behaviour even if
                // an editor is configured.
                let maybe_editor = if !args.is_present("no-editor") {
                    // There are two places where one can configure an editor to
                    // use: globally and locally. If it is configured in both
                    // location, we prefer the local setting.
                    if meta.editor.is_some() {
                        meta.editor.as_ref()
                    } else {
                        global.editor.as_ref()
                    }
                } else {
                    None
                };

                ask::ask_patch_name(&opts.repo_root, maybe_editor)?
            };

            run_hook(&opts.repo_root, "patch-name", Some(&patch_name))?;

            debug!("patch_name:{:?}", patch_name);
            if save_meta {
                meta.save(&opts.repo_root)?
            }
            if save_global {
                global.save().unwrap_or(())
            }
            debug!("new");
            let changes = changes.into_iter().flat_map(|x| x.into_iter()).collect();
            let branch = txn.get_branch(&branch_name).unwrap();

            let mut extra_deps = Vec::new();
            if let Some(deps) = args.values_of("depends-on") {
                for dep in deps {
                    if let Some(hash) = Hash::from_base58(dep) {
                        if let Some(internal) = txn.get_internal(hash.as_ref()) {
                            if txn.get_patch(&branch.patches, internal).is_some() {
                                extra_deps.push(hash)
                            } else {
                                return Err(ErrorKind::ExtraDepNotOnBranch(hash).into());
                            }
                        } else {
                            return Err(
                                ErrorKind::PatchNotFound(
                                    opts.repo_root().to_string_lossy().into_owned(),
                                    hash,
                                ).into(),
                            );
                        }
                    } else if let Some(internal) = PatchId::from_base58(dep) {
                        if let Some(hash) = txn.get_external(internal) {
                            if txn.get_patch(&branch.patches, internal).is_some() {
                                extra_deps.push(hash.to_owned())
                            } else {
                                return Err(ErrorKind::ExtraDepNotOnBranch(hash.to_owned()).into());
                            }
                        }
                    } else {
                        return Err(ErrorKind::WrongHash.into());
                    }
                }
            }
            txn.new_patch(
                &branch,
                authors,
                patch_name,
                description,
                patch_date,
                changes,
                extra_deps.into_iter(),
                PatchFlags::empty(),
            )
        };
        drop(repo);

        let dot_pijul = opts.repo_dir();
        let key = if let Ok(Some(key)) = meta.signing_key() {
            Some(key)
        } else {
            load_global_or_local_signing_key(Some(&dot_pijul)).ok()
        };
        debug!("key.is_some(): {:?}", key.is_some());
        let patches_dir = patches_dir(&opts.repo_root);
        let hash = patch.save(&patches_dir, key.as_ref())?;

        let pristine_dir = opts.pristine_dir();
        let mut increase = 409600;
        let res = loop {
            match record_no_resize(
                &pristine_dir,
                &opts.repo_root,
                &branch_name,
                &hash,
                &patch,
                &syncs,
                increase,
            ) {
                Err(ref e) if e.lacks_space() => increase *= 2,
                e => break e,
            }
        };

        run_hook(&opts.repo_root, "post-record", None)?;

        res
    }
}

pub fn record_no_resize(
    pristine_dir: &Path,
    r: &Path,
    branch_name: &str,
    hash: &Hash,
    patch: &Patch,
    syncs: &HashSet<InodeUpdate>,
    increase: u64,
) -> Result<Option<Hash>> {
    let size_increase = increase + patch.size_upper_bound() as u64;
    let repo = match Repository::open(&pristine_dir, Some(size_increase)) {
        Ok(repo) => repo,
        Err(x) => return Err(ErrorKind::Repository(x).into()),
    };
    let mut txn = try!(repo.mut_txn_begin(rand::thread_rng()));
    // save patch
    debug!("syncs: {:?}", syncs);
    txn.apply_local_patch(
        &branch_name,
        r,
        &hash,
        &patch,
        &syncs,
        false,
    )?;
    txn.commit()?;
    println!("Recorded patch {}", hash.to_base58());
    Ok(Some(hash.clone()))
}

pub fn explain(res: Result<Option<Hash>>) {
    default_explain(res)
}

pub fn changes_from_prefixes<T: rand::Rng>(
    repo_root: &Path,
    txn: &mut MutTxn<T>,
    branch_name: &str,
    prefix: Option<&HashSet<PathBuf>>,
) -> Result<
    (Vec<libpijul::patch::Record<Rc<RefCell<libpijul::patch::ChangeContext<PatchId>>>>>,
     HashSet<libpijul::InodeUpdate>),
> {
    let mut record = RecordState::new();
    if let Some(prefixes) = prefix {
        for prefix in prefixes {
            txn.record(
                &mut record,
                &branch_name,
                repo_root,
                Some(prefix.as_path()),
            )?;
        }
    } else {
        txn.record(&mut record, &branch_name, repo_root, None)?;
    }
    let (changes, updates) = record.finish();
    // let changes = changes.into_iter().map(|x| txn.globalize_change(x)).collect();
    Ok((changes, updates))
}

pub fn prefix(args: &ArgMatches, opts: &BasicOptions) -> Result<Option<HashSet<PathBuf>>> {
    if let Some(prefixes) = args.values_of("prefix") {
        let prefixes: Result<HashSet<_>> = prefixes
            .map(|prefix| {
                let p = opts.cwd.join(prefix);
                let p = if let Ok(p) = canonicalize(&p) { p } else { p };
                let file = p.strip_prefix(&opts.repo_root)?;
                debug!("prefix: {:?}", file);
                Ok(file.to_path_buf())
            })
            .collect();
        Ok(Some(prefixes?))
    } else {
        Ok(None)
    }
}
