use std::path;

use crate::commands::reload;
use crate::context::AppContext;
use crate::error::JoshutoResult;
use crate::history::DirectoryHistory;

// ChangeDirectory command
pub fn cd(path: &path::Path, context: &mut AppContext) -> std::io::Result<()> {
    std::env::set_current_dir(path)?;
    context.tab_context_mut().curr_tab_mut().set_cwd(path);
    Ok(())
}

pub fn change_directory(context: &mut AppContext, path: &path::Path) -> JoshutoResult {
    let new_cwd = if path.is_absolute() {
        path.to_path_buf()
    } else {
        let mut new_cwd = std::env::current_dir()?;
        new_cwd.push(path);
        new_cwd
    };

    cd(new_cwd.as_path(), context)?;
    let options = context.config_ref().display_options_ref().clone();
    let ui_context = context.ui_context_ref().clone();
    let tab_options = context
        .tab_context_ref()
        .curr_tab_ref()
        .option_ref()
        .clone();
    context
        .tab_context_mut()
        .curr_tab_mut()
        .history_mut()
        .populate_to_root(new_cwd.as_path(), &ui_context, &options, &tab_options)?;
    Ok(())
}

// ParentDirectory command
pub fn parent_directory(context: &mut AppContext) -> JoshutoResult {
    if let Some(parent) = context
        .tab_context_ref()
        .curr_tab_ref()
        .cwd()
        .parent()
        .map(|p| p.to_path_buf())
    {
        std::env::set_current_dir(&parent)?;
        context
            .tab_context_mut()
            .curr_tab_mut()
            .set_cwd(parent.as_path());
        reload::soft_reload_curr_tab(context)?;
    }
    Ok(())
}

// PreviousDirectory command
pub fn previous_directory(context: &mut AppContext) -> JoshutoResult {
    if let Some(path) = context.tab_context_ref().curr_tab_ref().previous_dir() {
        let path = path.to_path_buf();
        std::env::set_current_dir(&path)?;
        context
            .tab_context_mut()
            .curr_tab_mut()
            .set_cwd(path.as_path());
        reload::soft_reload_curr_tab(context)?;
    }
    Ok(())
}
