/*
 * Hurl (https://hurl.dev)
 * Copyright (C) 2023 Orange
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *          http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
use std::collections::HashMap;

use lazy_static::lazy_static;

// HTML5 named character references
//
// Generated from https://html.spec.whatwg.org/entities.json and
// https://html.spec.whatwg.org/multipage/named-characters.html.
// Map HTML5 named character references to the equivalent Unicode character(s).
lazy_static! {
    pub static ref HTML5_ENTITIES: HashMap<&'static str, &'static str> = {
        let mut m = HashMap::new();
        m.insert("AElig", "\u{C6}");
        m.insert("AElig;", "\u{C6}");
        m.insert("AMP", "\u{26}");
        m.insert("AMP;", "\u{26}");
        m.insert("Aacute", "\u{C1}");
        m.insert("Aacute;", "\u{C1}");
        m.insert("Abreve;", "\u{102}");
        m.insert("Acirc", "\u{C2}");
        m.insert("Acirc;", "\u{C2}");
        m.insert("Acy;", "\u{410}");
        m.insert("Afr;", "\u{1D504}");
        m.insert("Agrave", "\u{C0}");
        m.insert("Agrave;", "\u{C0}");
        m.insert("Alpha;", "\u{391}");
        m.insert("Amacr;", "\u{100}");
        m.insert("And;", "\u{2A53}");
        m.insert("Aogon;", "\u{104}");
        m.insert("Aopf;", "\u{1D538}");
        m.insert("ApplyFunction;", "\u{2061}");
        m.insert("Aring", "\u{C5}");
        m.insert("Aring;", "\u{C5}");
        m.insert("Ascr;", "\u{1D49C}");
        m.insert("Assign;", "\u{2254}");
        m.insert("Atilde", "\u{C3}");
        m.insert("Atilde;", "\u{C3}");
        m.insert("Auml", "\u{C4}");
        m.insert("Auml;", "\u{C4}");
        m.insert("Backslash;", "\u{2216}");
        m.insert("Barv;", "\u{2AE7}");
        m.insert("Barwed;", "\u{2306}");
        m.insert("Bcy;", "\u{411}");
        m.insert("Because;", "\u{2235}");
        m.insert("Bernoullis;", "\u{212C}");
        m.insert("Beta;", "\u{392}");
        m.insert("Bfr;", "\u{1D505}");
        m.insert("Bopf;", "\u{1D539}");
        m.insert("Breve;", "\u{2D8}");
        m.insert("Bscr;", "\u{212C}");
        m.insert("Bumpeq;", "\u{224E}");
        m.insert("CHcy;", "\u{427}");
        m.insert("COPY", "\u{A9}");
        m.insert("COPY;", "\u{A9}");
        m.insert("Cacute;", "\u{106}");
        m.insert("Cap;", "\u{22D2}");
        m.insert("CapitalDifferentialD;", "\u{2145}");
        m.insert("Cayleys;", "\u{212D}");
        m.insert("Ccaron;", "\u{10C}");
        m.insert("Ccedil", "\u{C7}");
        m.insert("Ccedil;", "\u{C7}");
        m.insert("Ccirc;", "\u{108}");
        m.insert("Cconint;", "\u{2230}");
        m.insert("Cdot;", "\u{10A}");
        m.insert("Cedilla;", "\u{B8}");
        m.insert("CenterDot;", "\u{B7}");
        m.insert("Cfr;", "\u{212D}");
        m.insert("Chi;", "\u{3A7}");
        m.insert("CircleDot;", "\u{2299}");
        m.insert("CircleMinus;", "\u{2296}");
        m.insert("CirclePlus;", "\u{2295}");
        m.insert("CircleTimes;", "\u{2297}");
        m.insert("ClockwiseContourIntegral;", "\u{2232}");
        m.insert("CloseCurlyDoubleQuote;", "\u{201D}");
        m.insert("CloseCurlyQuote;", "\u{2019}");
        m.insert("Colon;", "\u{2237}");
        m.insert("Colone;", "\u{2A74}");
        m.insert("Congruent;", "\u{2261}");
        m.insert("Conint;", "\u{222F}");
        m.insert("ContourIntegral;", "\u{222E}");
        m.insert("Copf;", "\u{2102}");
        m.insert("Coproduct;", "\u{2210}");
        m.insert("CounterClockwiseContourIntegral;", "\u{2233}");
        m.insert("Cross;", "\u{2A2F}");
        m.insert("Cscr;", "\u{1D49E}");
        m.insert("Cup;", "\u{22D3}");
        m.insert("CupCap;", "\u{224D}");
        m.insert("DD;", "\u{2145}");
        m.insert("DDotrahd;", "\u{2911}");
        m.insert("DJcy;", "\u{402}");
        m.insert("DScy;", "\u{405}");
        m.insert("DZcy;", "\u{40F}");
        m.insert("Dagger;", "\u{2021}");
        m.insert("Darr;", "\u{21A1}");
        m.insert("Dashv;", "\u{2AE4}");
        m.insert("Dcaron;", "\u{10E}");
        m.insert("Dcy;", "\u{414}");
        m.insert("Del;", "\u{2207}");
        m.insert("Delta;", "\u{394}");
        m.insert("Dfr;", "\u{1D507}");
        m.insert("DiacriticalAcute;", "\u{B4}");
        m.insert("DiacriticalDot;", "\u{2D9}");
        m.insert("DiacriticalDoubleAcute;", "\u{2DD}");
        m.insert("DiacriticalGrave;", "\u{60}");
        m.insert("DiacriticalTilde;", "\u{2DC}");
        m.insert("Diamond;", "\u{22C4}");
        m.insert("DifferentialD;", "\u{2146}");
        m.insert("Dopf;", "\u{1D53B}");
        m.insert("Dot;", "\u{A8}");
        m.insert("DotDot;", "\u{20DC}");
        m.insert("DotEqual;", "\u{2250}");
        m.insert("DoubleContourIntegral;", "\u{222F}");
        m.insert("DoubleDot;", "\u{A8}");
        m.insert("DoubleDownArrow;", "\u{21D3}");
        m.insert("DoubleLeftArrow;", "\u{21D0}");
        m.insert("DoubleLeftRightArrow;", "\u{21D4}");
        m.insert("DoubleLeftTee;", "\u{2AE4}");
        m.insert("DoubleLongLeftArrow;", "\u{27F8}");
        m.insert("DoubleLongLeftRightArrow;", "\u{27FA}");
        m.insert("DoubleLongRightArrow;", "\u{27F9}");
        m.insert("DoubleRightArrow;", "\u{21D2}");
        m.insert("DoubleRightTee;", "\u{22A8}");
        m.insert("DoubleUpArrow;", "\u{21D1}");
        m.insert("DoubleUpDownArrow;", "\u{21D5}");
        m.insert("DoubleVerticalBar;", "\u{2225}");
        m.insert("DownArrow;", "\u{2193}");
        m.insert("DownArrowBar;", "\u{2913}");
        m.insert("DownArrowUpArrow;", "\u{21F5}");
        m.insert("DownBreve;", "\u{311}");
        m.insert("DownLeftRightVector;", "\u{2950}");
        m.insert("DownLeftTeeVector;", "\u{295E}");
        m.insert("DownLeftVector;", "\u{21BD}");
        m.insert("DownLeftVectorBar;", "\u{2956}");
        m.insert("DownRightTeeVector;", "\u{295F}");
        m.insert("DownRightVector;", "\u{21C1}");
        m.insert("DownRightVectorBar;", "\u{2957}");
        m.insert("DownTee;", "\u{22A4}");
        m.insert("DownTeeArrow;", "\u{21A7}");
        m.insert("Downarrow;", "\u{21D3}");
        m.insert("Dscr;", "\u{1D49F}");
        m.insert("Dstrok;", "\u{110}");
        m.insert("ENG;", "\u{14A}");
        m.insert("ETH", "\u{D0}");
        m.insert("ETH;", "\u{D0}");
        m.insert("Eacute", "\u{C9}");
        m.insert("Eacute;", "\u{C9}");
        m.insert("Ecaron;", "\u{11A}");
        m.insert("Ecirc", "\u{CA}");
        m.insert("Ecirc;", "\u{CA}");
        m.insert("Ecy;", "\u{42D}");
        m.insert("Edot;", "\u{116}");
        m.insert("Efr;", "\u{1D508}");
        m.insert("Egrave", "\u{C8}");
        m.insert("Egrave;", "\u{C8}");
        m.insert("Element;", "\u{2208}");
        m.insert("Emacr;", "\u{112}");
        m.insert("EmptySmallSquare;", "\u{25FB}");
        m.insert("EmptyVerySmallSquare;", "\u{25AB}");
        m.insert("Eogon;", "\u{118}");
        m.insert("Eopf;", "\u{1D53C}");
        m.insert("Epsilon;", "\u{395}");
        m.insert("Equal;", "\u{2A75}");
        m.insert("EqualTilde;", "\u{2242}");
        m.insert("Equilibrium;", "\u{21CC}");
        m.insert("Escr;", "\u{2130}");
        m.insert("Esim;", "\u{2A73}");
        m.insert("Eta;", "\u{397}");
        m.insert("Euml", "\u{CB}");
        m.insert("Euml;", "\u{CB}");
        m.insert("Exists;", "\u{2203}");
        m.insert("ExponentialE;", "\u{2147}");
        m.insert("Fcy;", "\u{424}");
        m.insert("Ffr;", "\u{1D509}");
        m.insert("FilledSmallSquare;", "\u{25FC}");
        m.insert("FilledVerySmallSquare;", "\u{25AA}");
        m.insert("Fopf;", "\u{1D53D}");
        m.insert("ForAll;", "\u{2200}");
        m.insert("Fouriertrf;", "\u{2131}");
        m.insert("Fscr;", "\u{2131}");
        m.insert("GJcy;", "\u{403}");
        m.insert("GT", "\u{3E}");
        m.insert("GT;", "\u{3E}");
        m.insert("Gamma;", "\u{393}");
        m.insert("Gammad;", "\u{3DC}");
        m.insert("Gbreve;", "\u{11E}");
        m.insert("Gcedil;", "\u{122}");
        m.insert("Gcirc;", "\u{11C}");
        m.insert("Gcy;", "\u{413}");
        m.insert("Gdot;", "\u{120}");
        m.insert("Gfr;", "\u{1D50A}");
        m.insert("Gg;", "\u{22D9}");
        m.insert("Gopf;", "\u{1D53E}");
        m.insert("GreaterEqual;", "\u{2265}");
        m.insert("GreaterEqualLess;", "\u{22DB}");
        m.insert("GreaterFullEqual;", "\u{2267}");
        m.insert("GreaterGreater;", "\u{2AA2}");
        m.insert("GreaterLess;", "\u{2277}");
        m.insert("GreaterSlantEqual;", "\u{2A7E}");
        m.insert("GreaterTilde;", "\u{2273}");
        m.insert("Gscr;", "\u{1D4A2}");
        m.insert("Gt;", "\u{226B}");
        m.insert("HARDcy;", "\u{42A}");
        m.insert("Hacek;", "\u{2C7}");
        m.insert("Hat;", "\u{5E}");
        m.insert("Hcirc;", "\u{124}");
        m.insert("Hfr;", "\u{210C}");
        m.insert("HilbertSpace;", "\u{210B}");
        m.insert("Hopf;", "\u{210D}");
        m.insert("HorizontalLine;", "\u{2500}");
        m.insert("Hscr;", "\u{210B}");
        m.insert("Hstrok;", "\u{126}");
        m.insert("HumpDownHump;", "\u{224E}");
        m.insert("HumpEqual;", "\u{224F}");
        m.insert("IEcy;", "\u{415}");
        m.insert("IJlig;", "\u{132}");
        m.insert("IOcy;", "\u{401}");
        m.insert("Iacute", "\u{CD}");
        m.insert("Iacute;", "\u{CD}");
        m.insert("Icirc", "\u{CE}");
        m.insert("Icirc;", "\u{CE}");
        m.insert("Icy;", "\u{418}");
        m.insert("Idot;", "\u{130}");
        m.insert("Ifr;", "\u{2111}");
        m.insert("Igrave", "\u{CC}");
        m.insert("Igrave;", "\u{CC}");
        m.insert("Im;", "\u{2111}");
        m.insert("Imacr;", "\u{12A}");
        m.insert("ImaginaryI;", "\u{2148}");
        m.insert("Implies;", "\u{21D2}");
        m.insert("Int;", "\u{222C}");
        m.insert("Integral;", "\u{222B}");
        m.insert("Intersection;", "\u{22C2}");
        m.insert("InvisibleComma;", "\u{2063}");
        m.insert("InvisibleTimes;", "\u{2062}");
        m.insert("Iogon;", "\u{12E}");
        m.insert("Iopf;", "\u{1D540}");
        m.insert("Iota;", "\u{399}");
        m.insert("Iscr;", "\u{2110}");
        m.insert("Itilde;", "\u{128}");
        m.insert("Iukcy;", "\u{406}");
        m.insert("Iuml", "\u{CF}");
        m.insert("Iuml;", "\u{CF}");
        m.insert("Jcirc;", "\u{134}");
        m.insert("Jcy;", "\u{419}");
        m.insert("Jfr;", "\u{1D50D}");
        m.insert("Jopf;", "\u{1D541}");
        m.insert("Jscr;", "\u{1D4A5}");
        m.insert("Jsercy;", "\u{408}");
        m.insert("Jukcy;", "\u{404}");
        m.insert("KHcy;", "\u{425}");
        m.insert("KJcy;", "\u{40C}");
        m.insert("Kappa;", "\u{39A}");
        m.insert("Kcedil;", "\u{136}");
        m.insert("Kcy;", "\u{41A}");
        m.insert("Kfr;", "\u{1D50E}");
        m.insert("Kopf;", "\u{1D542}");
        m.insert("Kscr;", "\u{1D4A6}");
        m.insert("LJcy;", "\u{409}");
        m.insert("LT", "\u{3C}");
        m.insert("LT;", "\u{3C}");
        m.insert("Lacute;", "\u{139}");
        m.insert("Lambda;", "\u{39B}");
        m.insert("Lang;", "\u{27EA}");
        m.insert("Laplacetrf;", "\u{2112}");
        m.insert("Larr;", "\u{219E}");
        m.insert("Lcaron;", "\u{13D}");
        m.insert("Lcedil;", "\u{13B}");
        m.insert("Lcy;", "\u{41B}");
        m.insert("LeftAngleBracket;", "\u{27E8}");
        m.insert("LeftArrow;", "\u{2190}");
        m.insert("LeftArrowBar;", "\u{21E4}");
        m.insert("LeftArrowRightArrow;", "\u{21C6}");
        m.insert("LeftCeiling;", "\u{2308}");
        m.insert("LeftDoubleBracket;", "\u{27E6}");
        m.insert("LeftDownTeeVector;", "\u{2961}");
        m.insert("LeftDownVector;", "\u{21C3}");
        m.insert("LeftDownVectorBar;", "\u{2959}");
        m.insert("LeftFloor;", "\u{230A}");
        m.insert("LeftRightArrow;", "\u{2194}");
        m.insert("LeftRightVector;", "\u{294E}");
        m.insert("LeftTee;", "\u{22A3}");
        m.insert("LeftTeeArrow;", "\u{21A4}");
        m.insert("LeftTeeVector;", "\u{295A}");
        m.insert("LeftTriangle;", "\u{22B2}");
        m.insert("LeftTriangleBar;", "\u{29CF}");
        m.insert("LeftTriangleEqual;", "\u{22B4}");
        m.insert("LeftUpDownVector;", "\u{2951}");
        m.insert("LeftUpTeeVector;", "\u{2960}");
        m.insert("LeftUpVector;", "\u{21BF}");
        m.insert("LeftUpVectorBar;", "\u{2958}");
        m.insert("LeftVector;", "\u{21BC}");
        m.insert("LeftVectorBar;", "\u{2952}");
        m.insert("Leftarrow;", "\u{21D0}");
        m.insert("Leftrightarrow;", "\u{21D4}");
        m.insert("LessEqualGreater;", "\u{22DA}");
        m.insert("LessFullEqual;", "\u{2266}");
        m.insert("LessGreater;", "\u{2276}");
        m.insert("LessLess;", "\u{2AA1}");
        m.insert("LessSlantEqual;", "\u{2A7D}");
        m.insert("LessTilde;", "\u{2272}");
        m.insert("Lfr;", "\u{1D50F}");
        m.insert("Ll;", "\u{22D8}");
        m.insert("Lleftarrow;", "\u{21DA}");
        m.insert("Lmidot;", "\u{13F}");
        m.insert("LongLeftArrow;", "\u{27F5}");
        m.insert("LongLeftRightArrow;", "\u{27F7}");
        m.insert("LongRightArrow;", "\u{27F6}");
        m.insert("Longleftarrow;", "\u{27F8}");
        m.insert("Longleftrightarrow;", "\u{27FA}");
        m.insert("Longrightarrow;", "\u{27F9}");
        m.insert("Lopf;", "\u{1D543}");
        m.insert("LowerLeftArrow;", "\u{2199}");
        m.insert("LowerRightArrow;", "\u{2198}");
        m.insert("Lscr;", "\u{2112}");
        m.insert("Lsh;", "\u{21B0}");
        m.insert("Lstrok;", "\u{141}");
        m.insert("Lt;", "\u{226A}");
        m.insert("Map;", "\u{2905}");
        m.insert("Mcy;", "\u{41C}");
        m.insert("MediumSpace;", "\u{205F}");
        m.insert("Mellintrf;", "\u{2133}");
        m.insert("Mfr;", "\u{1D510}");
        m.insert("MinusPlus;", "\u{2213}");
        m.insert("Mopf;", "\u{1D544}");
        m.insert("Mscr;", "\u{2133}");
        m.insert("Mu;", "\u{39C}");
        m.insert("NJcy;", "\u{40A}");
        m.insert("Nacute;", "\u{143}");
        m.insert("Ncaron;", "\u{147}");
        m.insert("Ncedil;", "\u{145}");
        m.insert("Ncy;", "\u{41D}");
        m.insert("NegativeMediumSpace;", "\u{200B}");
        m.insert("NegativeThickSpace;", "\u{200B}");
        m.insert("NegativeThinSpace;", "\u{200B}");
        m.insert("NegativeVeryThinSpace;", "\u{200B}");
        m.insert("NestedGreaterGreater;", "\u{226B}");
        m.insert("NestedLessLess;", "\u{226A}");
        m.insert("NewLine;", "\u{A}");
        m.insert("Nfr;", "\u{1D511}");
        m.insert("NoBreak;", "\u{2060}");
        m.insert("NonBreakingSpace;", "\u{A0}");
        m.insert("Nopf;", "\u{2115}");
        m.insert("Not;", "\u{2AEC}");
        m.insert("NotCongruent;", "\u{2262}");
        m.insert("NotCupCap;", "\u{226D}");
        m.insert("NotDoubleVerticalBar;", "\u{2226}");
        m.insert("NotElement;", "\u{2209}");
        m.insert("NotEqual;", "\u{2260}");
        m.insert("NotEqualTilde;", "\u{2242}\u{338}");
        m.insert("NotExists;", "\u{2204}");
        m.insert("NotGreater;", "\u{226F}");
        m.insert("NotGreaterEqual;", "\u{2271}");
        m.insert("NotGreaterFullEqual;", "\u{2267}\u{338}");
        m.insert("NotGreaterGreater;", "\u{226B}\u{338}");
        m.insert("NotGreaterLess;", "\u{2279}");
        m.insert("NotGreaterSlantEqual;", "\u{2A7E}\u{338}");
        m.insert("NotGreaterTilde;", "\u{2275}");
        m.insert("NotHumpDownHump;", "\u{224E}\u{338}");
        m.insert("NotHumpEqual;", "\u{224F}\u{338}");
        m.insert("NotLeftTriangle;", "\u{22EA}");
        m.insert("NotLeftTriangleBar;", "\u{29CF}\u{338}");
        m.insert("NotLeftTriangleEqual;", "\u{22EC}");
        m.insert("NotLess;", "\u{226E}");
        m.insert("NotLessEqual;", "\u{2270}");
        m.insert("NotLessGreater;", "\u{2278}");
        m.insert("NotLessLess;", "\u{226A}\u{338}");
        m.insert("NotLessSlantEqual;", "\u{2A7D}\u{338}");
        m.insert("NotLessTilde;", "\u{2274}");
        m.insert("NotNestedGreaterGreater;", "\u{2AA2}\u{338}");
        m.insert("NotNestedLessLess;", "\u{2AA1}\u{338}");
        m.insert("NotPrecedes;", "\u{2280}");
        m.insert("NotPrecedesEqual;", "\u{2AAF}\u{338}");
        m.insert("NotPrecedesSlantEqual;", "\u{22E0}");
        m.insert("NotReverseElement;", "\u{220C}");
        m.insert("NotRightTriangle;", "\u{22EB}");
        m.insert("NotRightTriangleBar;", "\u{29D0}\u{338}");
        m.insert("NotRightTriangleEqual;", "\u{22ED}");
        m.insert("NotSquareSubset;", "\u{228F}\u{338}");
        m.insert("NotSquareSubsetEqual;", "\u{22E2}");
        m.insert("NotSquareSuperset;", "\u{2290}\u{338}");
        m.insert("NotSquareSupersetEqual;", "\u{22E3}");
        m.insert("NotSubset;", "\u{2282}\u{20D2}");
        m.insert("NotSubsetEqual;", "\u{2288}");
        m.insert("NotSucceeds;", "\u{2281}");
        m.insert("NotSucceedsEqual;", "\u{2AB0}\u{338}");
        m.insert("NotSucceedsSlantEqual;", "\u{22E1}");
        m.insert("NotSucceedsTilde;", "\u{227F}\u{338}");
        m.insert("NotSuperset;", "\u{2283}\u{20D2}");
        m.insert("NotSupersetEqual;", "\u{2289}");
        m.insert("NotTilde;", "\u{2241}");
        m.insert("NotTildeEqual;", "\u{2244}");
        m.insert("NotTildeFullEqual;", "\u{2247}");
        m.insert("NotTildeTilde;", "\u{2249}");
        m.insert("NotVerticalBar;", "\u{2224}");
        m.insert("Nscr;", "\u{1D4A9}");
        m.insert("Ntilde", "\u{D1}");
        m.insert("Ntilde;", "\u{D1}");
        m.insert("Nu;", "\u{39D}");
        m.insert("OElig;", "\u{152}");
        m.insert("Oacute", "\u{D3}");
        m.insert("Oacute;", "\u{D3}");
        m.insert("Ocirc", "\u{D4}");
        m.insert("Ocirc;", "\u{D4}");
        m.insert("Ocy;", "\u{41E}");
        m.insert("Odblac;", "\u{150}");
        m.insert("Ofr;", "\u{1D512}");
        m.insert("Ograve", "\u{D2}");
        m.insert("Ograve;", "\u{D2}");
        m.insert("Omacr;", "\u{14C}");
        m.insert("Omega;", "\u{3A9}");
        m.insert("Omicron;", "\u{39F}");
        m.insert("Oopf;", "\u{1D546}");
        m.insert("OpenCurlyDoubleQuote;", "\u{201C}");
        m.insert("OpenCurlyQuote;", "\u{2018}");
        m.insert("Or;", "\u{2A54}");
        m.insert("Oscr;", "\u{1D4AA}");
        m.insert("Oslash", "\u{D8}");
        m.insert("Oslash;", "\u{D8}");
        m.insert("Otilde", "\u{D5}");
        m.insert("Otilde;", "\u{D5}");
        m.insert("Otimes;", "\u{2A37}");
        m.insert("Ouml", "\u{D6}");
        m.insert("Ouml;", "\u{D6}");
        m.insert("OverBar;", "\u{203E}");
        m.insert("OverBrace;", "\u{23DE}");
        m.insert("OverBracket;", "\u{23B4}");
        m.insert("OverParenthesis;", "\u{23DC}");
        m.insert("PartialD;", "\u{2202}");
        m.insert("Pcy;", "\u{41F}");
        m.insert("Pfr;", "\u{1D513}");
        m.insert("Phi;", "\u{3A6}");
        m.insert("Pi;", "\u{3A0}");
        m.insert("PlusMinus;", "\u{B1}");
        m.insert("Poincareplane;", "\u{210C}");
        m.insert("Popf;", "\u{2119}");
        m.insert("Pr;", "\u{2ABB}");
        m.insert("Precedes;", "\u{227A}");
        m.insert("PrecedesEqual;", "\u{2AAF}");
        m.insert("PrecedesSlantEqual;", "\u{227C}");
        m.insert("PrecedesTilde;", "\u{227E}");
        m.insert("Prime;", "\u{2033}");
        m.insert("Product;", "\u{220F}");
        m.insert("Proportion;", "\u{2237}");
        m.insert("Proportional;", "\u{221D}");
        m.insert("Pscr;", "\u{1D4AB}");
        m.insert("Psi;", "\u{3A8}");
        m.insert("QUOT", "\u{22}");
        m.insert("QUOT;", "\u{22}");
        m.insert("Qfr;", "\u{1D514}");
        m.insert("Qopf;", "\u{211A}");
        m.insert("Qscr;", "\u{1D4AC}");
        m.insert("RBarr;", "\u{2910}");
        m.insert("REG", "\u{AE}");
        m.insert("REG;", "\u{AE}");
        m.insert("Racute;", "\u{154}");
        m.insert("Rang;", "\u{27EB}");
        m.insert("Rarr;", "\u{21A0}");
        m.insert("Rarrtl;", "\u{2916}");
        m.insert("Rcaron;", "\u{158}");
        m.insert("Rcedil;", "\u{156}");
        m.insert("Rcy;", "\u{420}");
        m.insert("Re;", "\u{211C}");
        m.insert("ReverseElement;", "\u{220B}");
        m.insert("ReverseEquilibrium;", "\u{21CB}");
        m.insert("ReverseUpEquilibrium;", "\u{296F}");
        m.insert("Rfr;", "\u{211C}");
        m.insert("Rho;", "\u{3A1}");
        m.insert("RightAngleBracket;", "\u{27E9}");
        m.insert("RightArrow;", "\u{2192}");
        m.insert("RightArrowBar;", "\u{21E5}");
        m.insert("RightArrowLeftArrow;", "\u{21C4}");
        m.insert("RightCeiling;", "\u{2309}");
        m.insert("RightDoubleBracket;", "\u{27E7}");
        m.insert("RightDownTeeVector;", "\u{295D}");
        m.insert("RightDownVector;", "\u{21C2}");
        m.insert("RightDownVectorBar;", "\u{2955}");
        m.insert("RightFloor;", "\u{230B}");
        m.insert("RightTee;", "\u{22A2}");
        m.insert("RightTeeArrow;", "\u{21A6}");
        m.insert("RightTeeVector;", "\u{295B}");
        m.insert("RightTriangle;", "\u{22B3}");
        m.insert("RightTriangleBar;", "\u{29D0}");
        m.insert("RightTriangleEqual;", "\u{22B5}");
        m.insert("RightUpDownVector;", "\u{294F}");
        m.insert("RightUpTeeVector;", "\u{295C}");
        m.insert("RightUpVector;", "\u{21BE}");
        m.insert("RightUpVectorBar;", "\u{2954}");
        m.insert("RightVector;", "\u{21C0}");
        m.insert("RightVectorBar;", "\u{2953}");
        m.insert("Rightarrow;", "\u{21D2}");
        m.insert("Ropf;", "\u{211D}");
        m.insert("RoundImplies;", "\u{2970}");
        m.insert("Rrightarrow;", "\u{21DB}");
        m.insert("Rscr;", "\u{211B}");
        m.insert("Rsh;", "\u{21B1}");
        m.insert("RuleDelayed;", "\u{29F4}");
        m.insert("SHCHcy;", "\u{429}");
        m.insert("SHcy;", "\u{428}");
        m.insert("SOFTcy;", "\u{42C}");
        m.insert("Sacute;", "\u{15A}");
        m.insert("Sc;", "\u{2ABC}");
        m.insert("Scaron;", "\u{160}");
        m.insert("Scedil;", "\u{15E}");
        m.insert("Scirc;", "\u{15C}");
        m.insert("Scy;", "\u{421}");
        m.insert("Sfr;", "\u{1D516}");
        m.insert("ShortDownArrow;", "\u{2193}");
        m.insert("ShortLeftArrow;", "\u{2190}");
        m.insert("ShortRightArrow;", "\u{2192}");
        m.insert("ShortUpArrow;", "\u{2191}");
        m.insert("Sigma;", "\u{3A3}");
        m.insert("SmallCircle;", "\u{2218}");
        m.insert("Sopf;", "\u{1D54A}");
        m.insert("Sqrt;", "\u{221A}");
        m.insert("Square;", "\u{25A1}");
        m.insert("SquareIntersection;", "\u{2293}");
        m.insert("SquareSubset;", "\u{228F}");
        m.insert("SquareSubsetEqual;", "\u{2291}");
        m.insert("SquareSuperset;", "\u{2290}");
        m.insert("SquareSupersetEqual;", "\u{2292}");
        m.insert("SquareUnion;", "\u{2294}");
        m.insert("Sscr;", "\u{1D4AE}");
        m.insert("Star;", "\u{22C6}");
        m.insert("Sub;", "\u{22D0}");
        m.insert("Subset;", "\u{22D0}");
        m.insert("SubsetEqual;", "\u{2286}");
        m.insert("Succeeds;", "\u{227B}");
        m.insert("SucceedsEqual;", "\u{2AB0}");
        m.insert("SucceedsSlantEqual;", "\u{227D}");
        m.insert("SucceedsTilde;", "\u{227F}");
        m.insert("SuchThat;", "\u{220B}");
        m.insert("Sum;", "\u{2211}");
        m.insert("Sup;", "\u{22D1}");
        m.insert("Superset;", "\u{2283}");
        m.insert("SupersetEqual;", "\u{2287}");
        m.insert("Supset;", "\u{22D1}");
        m.insert("THORN", "\u{DE}");
        m.insert("THORN;", "\u{DE}");
        m.insert("TRADE;", "\u{2122}");
        m.insert("TSHcy;", "\u{40B}");
        m.insert("TScy;", "\u{426}");
        m.insert("Tab;", "\u{9}");
        m.insert("Tau;", "\u{3A4}");
        m.insert("Tcaron;", "\u{164}");
        m.insert("Tcedil;", "\u{162}");
        m.insert("Tcy;", "\u{422}");
        m.insert("Tfr;", "\u{1D517}");
        m.insert("Therefore;", "\u{2234}");
        m.insert("Theta;", "\u{398}");
        m.insert("ThickSpace;", "\u{205F}\u{200A}");
        m.insert("ThinSpace;", "\u{2009}");
        m.insert("Tilde;", "\u{223C}");
        m.insert("TildeEqual;", "\u{2243}");
        m.insert("TildeFullEqual;", "\u{2245}");
        m.insert("TildeTilde;", "\u{2248}");
        m.insert("Topf;", "\u{1D54B}");
        m.insert("TripleDot;", "\u{20DB}");
        m.insert("Tscr;", "\u{1D4AF}");
        m.insert("Tstrok;", "\u{166}");
        m.insert("Uacute", "\u{DA}");
        m.insert("Uacute;", "\u{DA}");
        m.insert("Uarr;", "\u{219F}");
        m.insert("Uarrocir;", "\u{2949}");
        m.insert("Ubrcy;", "\u{40E}");
        m.insert("Ubreve;", "\u{16C}");
        m.insert("Ucirc", "\u{DB}");
        m.insert("Ucirc;", "\u{DB}");
        m.insert("Ucy;", "\u{423}");
        m.insert("Udblac;", "\u{170}");
        m.insert("Ufr;", "\u{1D518}");
        m.insert("Ugrave", "\u{D9}");
        m.insert("Ugrave;", "\u{D9}");
        m.insert("Umacr;", "\u{16A}");
        m.insert("UnderBar;", "\u{5F}");
        m.insert("UnderBrace;", "\u{23DF}");
        m.insert("UnderBracket;", "\u{23B5}");
        m.insert("UnderParenthesis;", "\u{23DD}");
        m.insert("Union;", "\u{22C3}");
        m.insert("UnionPlus;", "\u{228E}");
        m.insert("Uogon;", "\u{172}");
        m.insert("Uopf;", "\u{1D54C}");
        m.insert("UpArrow;", "\u{2191}");
        m.insert("UpArrowBar;", "\u{2912}");
        m.insert("UpArrowDownArrow;", "\u{21C5}");
        m.insert("UpDownArrow;", "\u{2195}");
        m.insert("UpEquilibrium;", "\u{296E}");
        m.insert("UpTee;", "\u{22A5}");
        m.insert("UpTeeArrow;", "\u{21A5}");
        m.insert("Uparrow;", "\u{21D1}");
        m.insert("Updownarrow;", "\u{21D5}");
        m.insert("UpperLeftArrow;", "\u{2196}");
        m.insert("UpperRightArrow;", "\u{2197}");
        m.insert("Upsi;", "\u{3D2}");
        m.insert("Upsilon;", "\u{3A5}");
        m.insert("Uring;", "\u{16E}");
        m.insert("Uscr;", "\u{1D4B0}");
        m.insert("Utilde;", "\u{168}");
        m.insert("Uuml", "\u{DC}");
        m.insert("Uuml;", "\u{DC}");
        m.insert("VDash;", "\u{22AB}");
        m.insert("Vbar;", "\u{2AEB}");
        m.insert("Vcy;", "\u{412}");
        m.insert("Vdash;", "\u{22A9}");
        m.insert("Vdashl;", "\u{2AE6}");
        m.insert("Vee;", "\u{22C1}");
        m.insert("Verbar;", "\u{2016}");
        m.insert("Vert;", "\u{2016}");
        m.insert("VerticalBar;", "\u{2223}");
        m.insert("VerticalLine;", "\u{7C}");
        m.insert("VerticalSeparator;", "\u{2758}");
        m.insert("VerticalTilde;", "\u{2240}");
        m.insert("VeryThinSpace;", "\u{200A}");
        m.insert("Vfr;", "\u{1D519}");
        m.insert("Vopf;", "\u{1D54D}");
        m.insert("Vscr;", "\u{1D4B1}");
        m.insert("Vvdash;", "\u{22AA}");
        m.insert("Wcirc;", "\u{174}");
        m.insert("Wedge;", "\u{22C0}");
        m.insert("Wfr;", "\u{1D51A}");
        m.insert("Wopf;", "\u{1D54E}");
        m.insert("Wscr;", "\u{1D4B2}");
        m.insert("Xfr;", "\u{1D51B}");
        m.insert("Xi;", "\u{39E}");
        m.insert("Xopf;", "\u{1D54F}");
        m.insert("Xscr;", "\u{1D4B3}");
        m.insert("YAcy;", "\u{42F}");
        m.insert("YIcy;", "\u{407}");
        m.insert("YUcy;", "\u{42E}");
        m.insert("Yacute", "\u{DD}");
        m.insert("Yacute;", "\u{DD}");
        m.insert("Ycirc;", "\u{176}");
        m.insert("Ycy;", "\u{42B}");
        m.insert("Yfr;", "\u{1D51C}");
        m.insert("Yopf;", "\u{1D550}");
        m.insert("Yscr;", "\u{1D4B4}");
        m.insert("Yuml;", "\u{178}");
        m.insert("ZHcy;", "\u{416}");
        m.insert("Zacute;", "\u{179}");
        m.insert("Zcaron;", "\u{17D}");
        m.insert("Zcy;", "\u{417}");
        m.insert("Zdot;", "\u{17B}");
        m.insert("ZeroWidthSpace;", "\u{200B}");
        m.insert("Zeta;", "\u{396}");
        m.insert("Zfr;", "\u{2128}");
        m.insert("Zopf;", "\u{2124}");
        m.insert("Zscr;", "\u{1D4B5}");
        m.insert("aacute", "\u{E1}");
        m.insert("aacute;", "\u{E1}");
        m.insert("abreve;", "\u{103}");
        m.insert("ac;", "\u{223E}");
        m.insert("acE;", "\u{223E}\u{333}");
        m.insert("acd;", "\u{223F}");
        m.insert("acirc", "\u{E2}");
        m.insert("acirc;", "\u{E2}");
        m.insert("acute", "\u{B4}");
        m.insert("acute;", "\u{B4}");
        m.insert("acy;", "\u{430}");
        m.insert("aelig", "\u{E6}");
        m.insert("aelig;", "\u{E6}");
        m.insert("af;", "\u{2061}");
        m.insert("afr;", "\u{1D51E}");
        m.insert("agrave", "\u{E0}");
        m.insert("agrave;", "\u{E0}");
        m.insert("alefsym;", "\u{2135}");
        m.insert("aleph;", "\u{2135}");
        m.insert("alpha;", "\u{3B1}");
        m.insert("amacr;", "\u{101}");
        m.insert("amalg;", "\u{2A3F}");
        m.insert("amp", "\u{26}");
        m.insert("amp;", "\u{26}");
        m.insert("and;", "\u{2227}");
        m.insert("andand;", "\u{2A55}");
        m.insert("andd;", "\u{2A5C}");
        m.insert("andslope;", "\u{2A58}");
        m.insert("andv;", "\u{2A5A}");
        m.insert("ang;", "\u{2220}");
        m.insert("ange;", "\u{29A4}");
        m.insert("angle;", "\u{2220}");
        m.insert("angmsd;", "\u{2221}");
        m.insert("angmsdaa;", "\u{29A8}");
        m.insert("angmsdab;", "\u{29A9}");
        m.insert("angmsdac;", "\u{29AA}");
        m.insert("angmsdad;", "\u{29AB}");
        m.insert("angmsdae;", "\u{29AC}");
        m.insert("angmsdaf;", "\u{29AD}");
        m.insert("angmsdag;", "\u{29AE}");
        m.insert("angmsdah;", "\u{29AF}");
        m.insert("angrt;", "\u{221F}");
        m.insert("angrtvb;", "\u{22BE}");
        m.insert("angrtvbd;", "\u{299D}");
        m.insert("angsph;", "\u{2222}");
        m.insert("angst;", "\u{C5}");
        m.insert("angzarr;", "\u{237C}");
        m.insert("aogon;", "\u{105}");
        m.insert("aopf;", "\u{1D552}");
        m.insert("ap;", "\u{2248}");
        m.insert("apE;", "\u{2A70}");
        m.insert("apacir;", "\u{2A6F}");
        m.insert("ape;", "\u{224A}");
        m.insert("apid;", "\u{224B}");
        m.insert("apos;", "\u{27}");
        m.insert("approx;", "\u{2248}");
        m.insert("approxeq;", "\u{224A}");
        m.insert("aring", "\u{E5}");
        m.insert("aring;", "\u{E5}");
        m.insert("ascr;", "\u{1D4B6}");
        m.insert("ast;", "\u{2A}");
        m.insert("asymp;", "\u{2248}");
        m.insert("asympeq;", "\u{224D}");
        m.insert("atilde", "\u{E3}");
        m.insert("atilde;", "\u{E3}");
        m.insert("auml", "\u{E4}");
        m.insert("auml;", "\u{E4}");
        m.insert("awconint;", "\u{2233}");
        m.insert("awint;", "\u{2A11}");
        m.insert("bNot;", "\u{2AED}");
        m.insert("backcong;", "\u{224C}");
        m.insert("backepsilon;", "\u{3F6}");
        m.insert("backprime;", "\u{2035}");
        m.insert("backsim;", "\u{223D}");
        m.insert("backsimeq;", "\u{22CD}");
        m.insert("barvee;", "\u{22BD}");
        m.insert("barwed;", "\u{2305}");
        m.insert("barwedge;", "\u{2305}");
        m.insert("bbrk;", "\u{23B5}");
        m.insert("bbrktbrk;", "\u{23B6}");
        m.insert("bcong;", "\u{224C}");
        m.insert("bcy;", "\u{431}");
        m.insert("bdquo;", "\u{201E}");
        m.insert("becaus;", "\u{2235}");
        m.insert("because;", "\u{2235}");
        m.insert("bemptyv;", "\u{29B0}");
        m.insert("bepsi;", "\u{3F6}");
        m.insert("bernou;", "\u{212C}");
        m.insert("beta;", "\u{3B2}");
        m.insert("beth;", "\u{2136}");
        m.insert("between;", "\u{226C}");
        m.insert("bfr;", "\u{1D51F}");
        m.insert("bigcap;", "\u{22C2}");
        m.insert("bigcirc;", "\u{25EF}");
        m.insert("bigcup;", "\u{22C3}");
        m.insert("bigodot;", "\u{2A00}");
        m.insert("bigoplus;", "\u{2A01}");
        m.insert("bigotimes;", "\u{2A02}");
        m.insert("bigsqcup;", "\u{2A06}");
        m.insert("bigstar;", "\u{2605}");
        m.insert("bigtriangledown;", "\u{25BD}");
        m.insert("bigtriangleup;", "\u{25B3}");
        m.insert("biguplus;", "\u{2A04}");
        m.insert("bigvee;", "\u{22C1}");
        m.insert("bigwedge;", "\u{22C0}");
        m.insert("bkarow;", "\u{290D}");
        m.insert("blacklozenge;", "\u{29EB}");
        m.insert("blacksquare;", "\u{25AA}");
        m.insert("blacktriangle;", "\u{25B4}");
        m.insert("blacktriangledown;", "\u{25BE}");
        m.insert("blacktriangleleft;", "\u{25C2}");
        m.insert("blacktriangleright;", "\u{25B8}");
        m.insert("blank;", "\u{2423}");
        m.insert("blk12;", "\u{2592}");
        m.insert("blk14;", "\u{2591}");
        m.insert("blk34;", "\u{2593}");
        m.insert("block;", "\u{2588}");
        m.insert("bne;", "\u{3D}\u{20E5}");
        m.insert("bnequiv;", "\u{2261}\u{20E5}");
        m.insert("bnot;", "\u{2310}");
        m.insert("bopf;", "\u{1D553}");
        m.insert("bot;", "\u{22A5}");
        m.insert("bottom;", "\u{22A5}");
        m.insert("bowtie;", "\u{22C8}");
        m.insert("boxDL;", "\u{2557}");
        m.insert("boxDR;", "\u{2554}");
        m.insert("boxDl;", "\u{2556}");
        m.insert("boxDr;", "\u{2553}");
        m.insert("boxH;", "\u{2550}");
        m.insert("boxHD;", "\u{2566}");
        m.insert("boxHU;", "\u{2569}");
        m.insert("boxHd;", "\u{2564}");
        m.insert("boxHu;", "\u{2567}");
        m.insert("boxUL;", "\u{255D}");
        m.insert("boxUR;", "\u{255A}");
        m.insert("boxUl;", "\u{255C}");
        m.insert("boxUr;", "\u{2559}");
        m.insert("boxV;", "\u{2551}");
        m.insert("boxVH;", "\u{256C}");
        m.insert("boxVL;", "\u{2563}");
        m.insert("boxVR;", "\u{2560}");
        m.insert("boxVh;", "\u{256B}");
        m.insert("boxVl;", "\u{2562}");
        m.insert("boxVr;", "\u{255F}");
        m.insert("boxbox;", "\u{29C9}");
        m.insert("boxdL;", "\u{2555}");
        m.insert("boxdR;", "\u{2552}");
        m.insert("boxdl;", "\u{2510}");
        m.insert("boxdr;", "\u{250C}");
        m.insert("boxh;", "\u{2500}");
        m.insert("boxhD;", "\u{2565}");
        m.insert("boxhU;", "\u{2568}");
        m.insert("boxhd;", "\u{252C}");
        m.insert("boxhu;", "\u{2534}");
        m.insert("boxminus;", "\u{229F}");
        m.insert("boxplus;", "\u{229E}");
        m.insert("boxtimes;", "\u{22A0}");
        m.insert("boxuL;", "\u{255B}");
        m.insert("boxuR;", "\u{2558}");
        m.insert("boxul;", "\u{2518}");
        m.insert("boxur;", "\u{2514}");
        m.insert("boxv;", "\u{2502}");
        m.insert("boxvH;", "\u{256A}");
        m.insert("boxvL;", "\u{2561}");
        m.insert("boxvR;", "\u{255E}");
        m.insert("boxvh;", "\u{253C}");
        m.insert("boxvl;", "\u{2524}");
        m.insert("boxvr;", "\u{251C}");
        m.insert("bprime;", "\u{2035}");
        m.insert("breve;", "\u{2D8}");
        m.insert("brvbar", "\u{A6}");
        m.insert("brvbar;", "\u{A6}");
        m.insert("bscr;", "\u{1D4B7}");
        m.insert("bsemi;", "\u{204F}");
        m.insert("bsim;", "\u{223D}");
        m.insert("bsime;", "\u{22CD}");
        m.insert("bsol;", "\u{5C}");
        m.insert("bsolb;", "\u{29C5}");
        m.insert("bsolhsub;", "\u{27C8}");
        m.insert("bull;", "\u{2022}");
        m.insert("bullet;", "\u{2022}");
        m.insert("bump;", "\u{224E}");
        m.insert("bumpE;", "\u{2AAE}");
        m.insert("bumpe;", "\u{224F}");
        m.insert("bumpeq;", "\u{224F}");
        m.insert("cacute;", "\u{107}");
        m.insert("cap;", "\u{2229}");
        m.insert("capand;", "\u{2A44}");
        m.insert("capbrcup;", "\u{2A49}");
        m.insert("capcap;", "\u{2A4B}");
        m.insert("capcup;", "\u{2A47}");
        m.insert("capdot;", "\u{2A40}");
        m.insert("caps;", "\u{2229}\u{FE00}");
        m.insert("caret;", "\u{2041}");
        m.insert("caron;", "\u{2C7}");
        m.insert("ccaps;", "\u{2A4D}");
        m.insert("ccaron;", "\u{10D}");
        m.insert("ccedil", "\u{E7}");
        m.insert("ccedil;", "\u{E7}");
        m.insert("ccirc;", "\u{109}");
        m.insert("ccups;", "\u{2A4C}");
        m.insert("ccupssm;", "\u{2A50}");
        m.insert("cdot;", "\u{10B}");
        m.insert("cedil", "\u{B8}");
        m.insert("cedil;", "\u{B8}");
        m.insert("cemptyv;", "\u{29B2}");
        m.insert("cent", "\u{A2}");
        m.insert("cent;", "\u{A2}");
        m.insert("centerdot;", "\u{B7}");
        m.insert("cfr;", "\u{1D520}");
        m.insert("chcy;", "\u{447}");
        m.insert("check;", "\u{2713}");
        m.insert("checkmark;", "\u{2713}");
        m.insert("chi;", "\u{3C7}");
        m.insert("cir;", "\u{25CB}");
        m.insert("cirE;", "\u{29C3}");
        m.insert("circ;", "\u{2C6}");
        m.insert("circeq;", "\u{2257}");
        m.insert("circlearrowleft;", "\u{21BA}");
        m.insert("circlearrowright;", "\u{21BB}");
        m.insert("circledR;", "\u{AE}");
        m.insert("circledS;", "\u{24C8}");
        m.insert("circledast;", "\u{229B}");
        m.insert("circledcirc;", "\u{229A}");
        m.insert("circleddash;", "\u{229D}");
        m.insert("cire;", "\u{2257}");
        m.insert("cirfnint;", "\u{2A10}");
        m.insert("cirmid;", "\u{2AEF}");
        m.insert("cirscir;", "\u{29C2}");
        m.insert("clubs;", "\u{2663}");
        m.insert("clubsuit;", "\u{2663}");
        m.insert("colon;", "\u{3A}");
        m.insert("colone;", "\u{2254}");
        m.insert("coloneq;", "\u{2254}");
        m.insert("comma;", "\u{2C}");
        m.insert("commat;", "\u{40}");
        m.insert("comp;", "\u{2201}");
        m.insert("compfn;", "\u{2218}");
        m.insert("complement;", "\u{2201}");
        m.insert("complexes;", "\u{2102}");
        m.insert("cong;", "\u{2245}");
        m.insert("congdot;", "\u{2A6D}");
        m.insert("conint;", "\u{222E}");
        m.insert("copf;", "\u{1D554}");
        m.insert("coprod;", "\u{2210}");
        m.insert("copy", "\u{A9}");
        m.insert("copy;", "\u{A9}");
        m.insert("copysr;", "\u{2117}");
        m.insert("crarr;", "\u{21B5}");
        m.insert("cross;", "\u{2717}");
        m.insert("cscr;", "\u{1D4B8}");
        m.insert("csub;", "\u{2ACF}");
        m.insert("csube;", "\u{2AD1}");
        m.insert("csup;", "\u{2AD0}");
        m.insert("csupe;", "\u{2AD2}");
        m.insert("ctdot;", "\u{22EF}");
        m.insert("cudarrl;", "\u{2938}");
        m.insert("cudarrr;", "\u{2935}");
        m.insert("cuepr;", "\u{22DE}");
        m.insert("cuesc;", "\u{22DF}");
        m.insert("cularr;", "\u{21B6}");
        m.insert("cularrp;", "\u{293D}");
        m.insert("cup;", "\u{222A}");
        m.insert("cupbrcap;", "\u{2A48}");
        m.insert("cupcap;", "\u{2A46}");
        m.insert("cupcup;", "\u{2A4A}");
        m.insert("cupdot;", "\u{228D}");
        m.insert("cupor;", "\u{2A45}");
        m.insert("cups;", "\u{222A}\u{FE00}");
        m.insert("curarr;", "\u{21B7}");
        m.insert("curarrm;", "\u{293C}");
        m.insert("curlyeqprec;", "\u{22DE}");
        m.insert("curlyeqsucc;", "\u{22DF}");
        m.insert("curlyvee;", "\u{22CE}");
        m.insert("curlywedge;", "\u{22CF}");
        m.insert("curren", "\u{A4}");
        m.insert("curren;", "\u{A4}");
        m.insert("curvearrowleft;", "\u{21B6}");
        m.insert("curvearrowright;", "\u{21B7}");
        m.insert("cuvee;", "\u{22CE}");
        m.insert("cuwed;", "\u{22CF}");
        m.insert("cwconint;", "\u{2232}");
        m.insert("cwint;", "\u{2231}");
        m.insert("cylcty;", "\u{232D}");
        m.insert("dArr;", "\u{21D3}");
        m.insert("dHar;", "\u{2965}");
        m.insert("dagger;", "\u{2020}");
        m.insert("daleth;", "\u{2138}");
        m.insert("darr;", "\u{2193}");
        m.insert("dash;", "\u{2010}");
        m.insert("dashv;", "\u{22A3}");
        m.insert("dbkarow;", "\u{290F}");
        m.insert("dblac;", "\u{2DD}");
        m.insert("dcaron;", "\u{10F}");
        m.insert("dcy;", "\u{434}");
        m.insert("dd;", "\u{2146}");
        m.insert("ddagger;", "\u{2021}");
        m.insert("ddarr;", "\u{21CA}");
        m.insert("ddotseq;", "\u{2A77}");
        m.insert("deg", "\u{B0}");
        m.insert("deg;", "\u{B0}");
        m.insert("delta;", "\u{3B4}");
        m.insert("demptyv;", "\u{29B1}");
        m.insert("dfisht;", "\u{297F}");
        m.insert("dfr;", "\u{1D521}");
        m.insert("dharl;", "\u{21C3}");
        m.insert("dharr;", "\u{21C2}");
        m.insert("diam;", "\u{22C4}");
        m.insert("diamond;", "\u{22C4}");
        m.insert("diamondsuit;", "\u{2666}");
        m.insert("diams;", "\u{2666}");
        m.insert("die;", "\u{A8}");
        m.insert("digamma;", "\u{3DD}");
        m.insert("disin;", "\u{22F2}");
        m.insert("div;", "\u{F7}");
        m.insert("divide", "\u{F7}");
        m.insert("divide;", "\u{F7}");
        m.insert("divideontimes;", "\u{22C7}");
        m.insert("divonx;", "\u{22C7}");
        m.insert("djcy;", "\u{452}");
        m.insert("dlcorn;", "\u{231E}");
        m.insert("dlcrop;", "\u{230D}");
        m.insert("dollar;", "\u{24}");
        m.insert("dopf;", "\u{1D555}");
        m.insert("dot;", "\u{2D9}");
        m.insert("doteq;", "\u{2250}");
        m.insert("doteqdot;", "\u{2251}");
        m.insert("dotminus;", "\u{2238}");
        m.insert("dotplus;", "\u{2214}");
        m.insert("dotsquare;", "\u{22A1}");
        m.insert("doublebarwedge;", "\u{2306}");
        m.insert("downarrow;", "\u{2193}");
        m.insert("downdownarrows;", "\u{21CA}");
        m.insert("downharpoonleft;", "\u{21C3}");
        m.insert("downharpoonright;", "\u{21C2}");
        m.insert("drbkarow;", "\u{2910}");
        m.insert("drcorn;", "\u{231F}");
        m.insert("drcrop;", "\u{230C}");
        m.insert("dscr;", "\u{1D4B9}");
        m.insert("dscy;", "\u{455}");
        m.insert("dsol;", "\u{29F6}");
        m.insert("dstrok;", "\u{111}");
        m.insert("dtdot;", "\u{22F1}");
        m.insert("dtri;", "\u{25BF}");
        m.insert("dtrif;", "\u{25BE}");
        m.insert("duarr;", "\u{21F5}");
        m.insert("duhar;", "\u{296F}");
        m.insert("dwangle;", "\u{29A6}");
        m.insert("dzcy;", "\u{45F}");
        m.insert("dzigrarr;", "\u{27FF}");
        m.insert("eDDot;", "\u{2A77}");
        m.insert("eDot;", "\u{2251}");
        m.insert("eacute", "\u{E9}");
        m.insert("eacute;", "\u{E9}");
        m.insert("easter;", "\u{2A6E}");
        m.insert("ecaron;", "\u{11B}");
        m.insert("ecir;", "\u{2256}");
        m.insert("ecirc", "\u{EA}");
        m.insert("ecirc;", "\u{EA}");
        m.insert("ecolon;", "\u{2255}");
        m.insert("ecy;", "\u{44D}");
        m.insert("edot;", "\u{117}");
        m.insert("ee;", "\u{2147}");
        m.insert("efDot;", "\u{2252}");
        m.insert("efr;", "\u{1D522}");
        m.insert("eg;", "\u{2A9A}");
        m.insert("egrave", "\u{E8}");
        m.insert("egrave;", "\u{E8}");
        m.insert("egs;", "\u{2A96}");
        m.insert("egsdot;", "\u{2A98}");
        m.insert("el;", "\u{2A99}");
        m.insert("elinters;", "\u{23E7}");
        m.insert("ell;", "\u{2113}");
        m.insert("els;", "\u{2A95}");
        m.insert("elsdot;", "\u{2A97}");
        m.insert("emacr;", "\u{113}");
        m.insert("empty;", "\u{2205}");
        m.insert("emptyset;", "\u{2205}");
        m.insert("emptyv;", "\u{2205}");
        m.insert("emsp13;", "\u{2004}");
        m.insert("emsp14;", "\u{2005}");
        m.insert("emsp;", "\u{2003}");
        m.insert("eng;", "\u{14B}");
        m.insert("ensp;", "\u{2002}");
        m.insert("eogon;", "\u{119}");
        m.insert("eopf;", "\u{1D556}");
        m.insert("epar;", "\u{22D5}");
        m.insert("eparsl;", "\u{29E3}");
        m.insert("eplus;", "\u{2A71}");
        m.insert("epsi;", "\u{3B5}");
        m.insert("epsilon;", "\u{3B5}");
        m.insert("epsiv;", "\u{3F5}");
        m.insert("eqcirc;", "\u{2256}");
        m.insert("eqcolon;", "\u{2255}");
        m.insert("eqsim;", "\u{2242}");
        m.insert("eqslantgtr;", "\u{2A96}");
        m.insert("eqslantless;", "\u{2A95}");
        m.insert("equals;", "\u{3D}");
        m.insert("equest;", "\u{225F}");
        m.insert("equiv;", "\u{2261}");
        m.insert("equivDD;", "\u{2A78}");
        m.insert("eqvparsl;", "\u{29E5}");
        m.insert("erDot;", "\u{2253}");
        m.insert("erarr;", "\u{2971}");
        m.insert("escr;", "\u{212F}");
        m.insert("esdot;", "\u{2250}");
        m.insert("esim;", "\u{2242}");
        m.insert("eta;", "\u{3B7}");
        m.insert("eth", "\u{F0}");
        m.insert("eth;", "\u{F0}");
        m.insert("euml", "\u{EB}");
        m.insert("euml;", "\u{EB}");
        m.insert("euro;", "\u{20AC}");
        m.insert("excl;", "\u{21}");
        m.insert("exist;", "\u{2203}");
        m.insert("expectation;", "\u{2130}");
        m.insert("exponentiale;", "\u{2147}");
        m.insert("fallingdotseq;", "\u{2252}");
        m.insert("fcy;", "\u{444}");
        m.insert("female;", "\u{2640}");
        m.insert("ffilig;", "\u{FB03}");
        m.insert("fflig;", "\u{FB00}");
        m.insert("ffllig;", "\u{FB04}");
        m.insert("ffr;", "\u{1D523}");
        m.insert("filig;", "\u{FB01}");
        m.insert("fjlig;", "\u{66}\u{6A}");
        m.insert("flat;", "\u{266D}");
        m.insert("fllig;", "\u{FB02}");
        m.insert("fltns;", "\u{25B1}");
        m.insert("fnof;", "\u{192}");
        m.insert("fopf;", "\u{1D557}");
        m.insert("forall;", "\u{2200}");
        m.insert("fork;", "\u{22D4}");
        m.insert("forkv;", "\u{2AD9}");
        m.insert("fpartint;", "\u{2A0D}");
        m.insert("frac12", "\u{BD}");
        m.insert("frac12;", "\u{BD}");
        m.insert("frac13;", "\u{2153}");
        m.insert("frac14", "\u{BC}");
        m.insert("frac14;", "\u{BC}");
        m.insert("frac15;", "\u{2155}");
        m.insert("frac16;", "\u{2159}");
        m.insert("frac18;", "\u{215B}");
        m.insert("frac23;", "\u{2154}");
        m.insert("frac25;", "\u{2156}");
        m.insert("frac34", "\u{BE}");
        m.insert("frac34;", "\u{BE}");
        m.insert("frac35;", "\u{2157}");
        m.insert("frac38;", "\u{215C}");
        m.insert("frac45;", "\u{2158}");
        m.insert("frac56;", "\u{215A}");
        m.insert("frac58;", "\u{215D}");
        m.insert("frac78;", "\u{215E}");
        m.insert("frasl;", "\u{2044}");
        m.insert("frown;", "\u{2322}");
        m.insert("fscr;", "\u{1D4BB}");
        m.insert("gE;", "\u{2267}");
        m.insert("gEl;", "\u{2A8C}");
        m.insert("gacute;", "\u{1F5}");
        m.insert("gamma;", "\u{3B3}");
        m.insert("gammad;", "\u{3DD}");
        m.insert("gap;", "\u{2A86}");
        m.insert("gbreve;", "\u{11F}");
        m.insert("gcirc;", "\u{11D}");
        m.insert("gcy;", "\u{433}");
        m.insert("gdot;", "\u{121}");
        m.insert("ge;", "\u{2265}");
        m.insert("gel;", "\u{22DB}");
        m.insert("geq;", "\u{2265}");
        m.insert("geqq;", "\u{2267}");
        m.insert("geqslant;", "\u{2A7E}");
        m.insert("ges;", "\u{2A7E}");
        m.insert("gescc;", "\u{2AA9}");
        m.insert("gesdot;", "\u{2A80}");
        m.insert("gesdoto;", "\u{2A82}");
        m.insert("gesdotol;", "\u{2A84}");
        m.insert("gesl;", "\u{22DB}\u{FE00}");
        m.insert("gesles;", "\u{2A94}");
        m.insert("gfr;", "\u{1D524}");
        m.insert("gg;", "\u{226B}");
        m.insert("ggg;", "\u{22D9}");
        m.insert("gimel;", "\u{2137}");
        m.insert("gjcy;", "\u{453}");
        m.insert("gl;", "\u{2277}");
        m.insert("glE;", "\u{2A92}");
        m.insert("gla;", "\u{2AA5}");
        m.insert("glj;", "\u{2AA4}");
        m.insert("gnE;", "\u{2269}");
        m.insert("gnap;", "\u{2A8A}");
        m.insert("gnapprox;", "\u{2A8A}");
        m.insert("gne;", "\u{2A88}");
        m.insert("gneq;", "\u{2A88}");
        m.insert("gneqq;", "\u{2269}");
        m.insert("gnsim;", "\u{22E7}");
        m.insert("gopf;", "\u{1D558}");
        m.insert("grave;", "\u{60}");
        m.insert("gscr;", "\u{210A}");
        m.insert("gsim;", "\u{2273}");
        m.insert("gsime;", "\u{2A8E}");
        m.insert("gsiml;", "\u{2A90}");
        m.insert("gt", "\u{3E}");
        m.insert("gt;", "\u{3E}");
        m.insert("gtcc;", "\u{2AA7}");
        m.insert("gtcir;", "\u{2A7A}");
        m.insert("gtdot;", "\u{22D7}");
        m.insert("gtlPar;", "\u{2995}");
        m.insert("gtquest;", "\u{2A7C}");
        m.insert("gtrapprox;", "\u{2A86}");
        m.insert("gtrarr;", "\u{2978}");
        m.insert("gtrdot;", "\u{22D7}");
        m.insert("gtreqless;", "\u{22DB}");
        m.insert("gtreqqless;", "\u{2A8C}");
        m.insert("gtrless;", "\u{2277}");
        m.insert("gtrsim;", "\u{2273}");
        m.insert("gvertneqq;", "\u{2269}\u{FE00}");
        m.insert("gvnE;", "\u{2269}\u{FE00}");
        m.insert("hArr;", "\u{21D4}");
        m.insert("hairsp;", "\u{200A}");
        m.insert("half;", "\u{BD}");
        m.insert("hamilt;", "\u{210B}");
        m.insert("hardcy;", "\u{44A}");
        m.insert("harr;", "\u{2194}");
        m.insert("harrcir;", "\u{2948}");
        m.insert("harrw;", "\u{21AD}");
        m.insert("hbar;", "\u{210F}");
        m.insert("hcirc;", "\u{125}");
        m.insert("hearts;", "\u{2665}");
        m.insert("heartsuit;", "\u{2665}");
        m.insert("hellip;", "\u{2026}");
        m.insert("hercon;", "\u{22B9}");
        m.insert("hfr;", "\u{1D525}");
        m.insert("hksearow;", "\u{2925}");
        m.insert("hkswarow;", "\u{2926}");
        m.insert("hoarr;", "\u{21FF}");
        m.insert("homtht;", "\u{223B}");
        m.insert("hookleftarrow;", "\u{21A9}");
        m.insert("hookrightarrow;", "\u{21AA}");
        m.insert("hopf;", "\u{1D559}");
        m.insert("horbar;", "\u{2015}");
        m.insert("hscr;", "\u{1D4BD}");
        m.insert("hslash;", "\u{210F}");
        m.insert("hstrok;", "\u{127}");
        m.insert("hybull;", "\u{2043}");
        m.insert("hyphen;", "\u{2010}");
        m.insert("iacute", "\u{ED}");
        m.insert("iacute;", "\u{ED}");
        m.insert("ic;", "\u{2063}");
        m.insert("icirc", "\u{EE}");
        m.insert("icirc;", "\u{EE}");
        m.insert("icy;", "\u{438}");
        m.insert("iecy;", "\u{435}");
        m.insert("iexcl", "\u{A1}");
        m.insert("iexcl;", "\u{A1}");
        m.insert("iff;", "\u{21D4}");
        m.insert("ifr;", "\u{1D526}");
        m.insert("igrave", "\u{EC}");
        m.insert("igrave;", "\u{EC}");
        m.insert("ii;", "\u{2148}");
        m.insert("iiiint;", "\u{2A0C}");
        m.insert("iiint;", "\u{222D}");
        m.insert("iinfin;", "\u{29DC}");
        m.insert("iiota;", "\u{2129}");
        m.insert("ijlig;", "\u{133}");
        m.insert("imacr;", "\u{12B}");
        m.insert("image;", "\u{2111}");
        m.insert("imagline;", "\u{2110}");
        m.insert("imagpart;", "\u{2111}");
        m.insert("imath;", "\u{131}");
        m.insert("imof;", "\u{22B7}");
        m.insert("imped;", "\u{1B5}");
        m.insert("in;", "\u{2208}");
        m.insert("incare;", "\u{2105}");
        m.insert("infin;", "\u{221E}");
        m.insert("infintie;", "\u{29DD}");
        m.insert("inodot;", "\u{131}");
        m.insert("int;", "\u{222B}");
        m.insert("intcal;", "\u{22BA}");
        m.insert("integers;", "\u{2124}");
        m.insert("intercal;", "\u{22BA}");
        m.insert("intlarhk;", "\u{2A17}");
        m.insert("intprod;", "\u{2A3C}");
        m.insert("iocy;", "\u{451}");
        m.insert("iogon;", "\u{12F}");
        m.insert("iopf;", "\u{1D55A}");
        m.insert("iota;", "\u{3B9}");
        m.insert("iprod;", "\u{2A3C}");
        m.insert("iquest", "\u{BF}");
        m.insert("iquest;", "\u{BF}");
        m.insert("iscr;", "\u{1D4BE}");
        m.insert("isin;", "\u{2208}");
        m.insert("isinE;", "\u{22F9}");
        m.insert("isindot;", "\u{22F5}");
        m.insert("isins;", "\u{22F4}");
        m.insert("isinsv;", "\u{22F3}");
        m.insert("isinv;", "\u{2208}");
        m.insert("it;", "\u{2062}");
        m.insert("itilde;", "\u{129}");
        m.insert("iukcy;", "\u{456}");
        m.insert("iuml", "\u{EF}");
        m.insert("iuml;", "\u{EF}");
        m.insert("jcirc;", "\u{135}");
        m.insert("jcy;", "\u{439}");
        m.insert("jfr;", "\u{1D527}");
        m.insert("jmath;", "\u{237}");
        m.insert("jopf;", "\u{1D55B}");
        m.insert("jscr;", "\u{1D4BF}");
        m.insert("jsercy;", "\u{458}");
        m.insert("jukcy;", "\u{454}");
        m.insert("kappa;", "\u{3BA}");
        m.insert("kappav;", "\u{3F0}");
        m.insert("kcedil;", "\u{137}");
        m.insert("kcy;", "\u{43A}");
        m.insert("kfr;", "\u{1D528}");
        m.insert("kgreen;", "\u{138}");
        m.insert("khcy;", "\u{445}");
        m.insert("kjcy;", "\u{45C}");
        m.insert("kopf;", "\u{1D55C}");
        m.insert("kscr;", "\u{1D4C0}");
        m.insert("lAarr;", "\u{21DA}");
        m.insert("lArr;", "\u{21D0}");
        m.insert("lAtail;", "\u{291B}");
        m.insert("lBarr;", "\u{290E}");
        m.insert("lE;", "\u{2266}");
        m.insert("lEg;", "\u{2A8B}");
        m.insert("lHar;", "\u{2962}");
        m.insert("lacute;", "\u{13A}");
        m.insert("laemptyv;", "\u{29B4}");
        m.insert("lagran;", "\u{2112}");
        m.insert("lambda;", "\u{3BB}");
        m.insert("lang;", "\u{27E8}");
        m.insert("langd;", "\u{2991}");
        m.insert("langle;", "\u{27E8}");
        m.insert("lap;", "\u{2A85}");
        m.insert("laquo", "\u{AB}");
        m.insert("laquo;", "\u{AB}");
        m.insert("larr;", "\u{2190}");
        m.insert("larrb;", "\u{21E4}");
        m.insert("larrbfs;", "\u{291F}");
        m.insert("larrfs;", "\u{291D}");
        m.insert("larrhk;", "\u{21A9}");
        m.insert("larrlp;", "\u{21AB}");
        m.insert("larrpl;", "\u{2939}");
        m.insert("larrsim;", "\u{2973}");
        m.insert("larrtl;", "\u{21A2}");
        m.insert("lat;", "\u{2AAB}");
        m.insert("latail;", "\u{2919}");
        m.insert("late;", "\u{2AAD}");
        m.insert("lates;", "\u{2AAD}\u{FE00}");
        m.insert("lbarr;", "\u{290C}");
        m.insert("lbbrk;", "\u{2772}");
        m.insert("lbrace;", "\u{7B}");
        m.insert("lbrack;", "\u{5B}");
        m.insert("lbrke;", "\u{298B}");
        m.insert("lbrksld;", "\u{298F}");
        m.insert("lbrkslu;", "\u{298D}");
        m.insert("lcaron;", "\u{13E}");
        m.insert("lcedil;", "\u{13C}");
        m.insert("lceil;", "\u{2308}");
        m.insert("lcub;", "\u{7B}");
        m.insert("lcy;", "\u{43B}");
        m.insert("ldca;", "\u{2936}");
        m.insert("ldquo;", "\u{201C}");
        m.insert("ldquor;", "\u{201E}");
        m.insert("ldrdhar;", "\u{2967}");
        m.insert("ldrushar;", "\u{294B}");
        m.insert("ldsh;", "\u{21B2}");
        m.insert("le;", "\u{2264}");
        m.insert("leftarrow;", "\u{2190}");
        m.insert("leftarrowtail;", "\u{21A2}");
        m.insert("leftharpoondown;", "\u{21BD}");
        m.insert("leftharpoonup;", "\u{21BC}");
        m.insert("leftleftarrows;", "\u{21C7}");
        m.insert("leftrightarrow;", "\u{2194}");
        m.insert("leftrightarrows;", "\u{21C6}");
        m.insert("leftrightharpoons;", "\u{21CB}");
        m.insert("leftrightsquigarrow;", "\u{21AD}");
        m.insert("leftthreetimes;", "\u{22CB}");
        m.insert("leg;", "\u{22DA}");
        m.insert("leq;", "\u{2264}");
        m.insert("leqq;", "\u{2266}");
        m.insert("leqslant;", "\u{2A7D}");
        m.insert("les;", "\u{2A7D}");
        m.insert("lescc;", "\u{2AA8}");
        m.insert("lesdot;", "\u{2A7F}");
        m.insert("lesdoto;", "\u{2A81}");
        m.insert("lesdotor;", "\u{2A83}");
        m.insert("lesg;", "\u{22DA}\u{FE00}");
        m.insert("lesges;", "\u{2A93}");
        m.insert("lessapprox;", "\u{2A85}");
        m.insert("lessdot;", "\u{22D6}");
        m.insert("lesseqgtr;", "\u{22DA}");
        m.insert("lesseqqgtr;", "\u{2A8B}");
        m.insert("lessgtr;", "\u{2276}");
        m.insert("lesssim;", "\u{2272}");
        m.insert("lfisht;", "\u{297C}");
        m.insert("lfloor;", "\u{230A}");
        m.insert("lfr;", "\u{1D529}");
        m.insert("lg;", "\u{2276}");
        m.insert("lgE;", "\u{2A91}");
        m.insert("lhard;", "\u{21BD}");
        m.insert("lharu;", "\u{21BC}");
        m.insert("lharul;", "\u{296A}");
        m.insert("lhblk;", "\u{2584}");
        m.insert("ljcy;", "\u{459}");
        m.insert("ll;", "\u{226A}");
        m.insert("llarr;", "\u{21C7}");
        m.insert("llcorner;", "\u{231E}");
        m.insert("llhard;", "\u{296B}");
        m.insert("lltri;", "\u{25FA}");
        m.insert("lmidot;", "\u{140}");
        m.insert("lmoust;", "\u{23B0}");
        m.insert("lmoustache;", "\u{23B0}");
        m.insert("lnE;", "\u{2268}");
        m.insert("lnap;", "\u{2A89}");
        m.insert("lnapprox;", "\u{2A89}");
        m.insert("lne;", "\u{2A87}");
        m.insert("lneq;", "\u{2A87}");
        m.insert("lneqq;", "\u{2268}");
        m.insert("lnsim;", "\u{22E6}");
        m.insert("loang;", "\u{27EC}");
        m.insert("loarr;", "\u{21FD}");
        m.insert("lobrk;", "\u{27E6}");
        m.insert("longleftarrow;", "\u{27F5}");
        m.insert("longleftrightarrow;", "\u{27F7}");
        m.insert("longmapsto;", "\u{27FC}");
        m.insert("longrightarrow;", "\u{27F6}");
        m.insert("looparrowleft;", "\u{21AB}");
        m.insert("looparrowright;", "\u{21AC}");
        m.insert("lopar;", "\u{2985}");
        m.insert("lopf;", "\u{1D55D}");
        m.insert("loplus;", "\u{2A2D}");
        m.insert("lotimes;", "\u{2A34}");
        m.insert("lowast;", "\u{2217}");
        m.insert("lowbar;", "\u{5F}");
        m.insert("loz;", "\u{25CA}");
        m.insert("lozenge;", "\u{25CA}");
        m.insert("lozf;", "\u{29EB}");
        m.insert("lpar;", "\u{28}");
        m.insert("lparlt;", "\u{2993}");
        m.insert("lrarr;", "\u{21C6}");
        m.insert("lrcorner;", "\u{231F}");
        m.insert("lrhar;", "\u{21CB}");
        m.insert("lrhard;", "\u{296D}");
        m.insert("lrm;", "\u{200E}");
        m.insert("lrtri;", "\u{22BF}");
        m.insert("lsaquo;", "\u{2039}");
        m.insert("lscr;", "\u{1D4C1}");
        m.insert("lsh;", "\u{21B0}");
        m.insert("lsim;", "\u{2272}");
        m.insert("lsime;", "\u{2A8D}");
        m.insert("lsimg;", "\u{2A8F}");
        m.insert("lsqb;", "\u{5B}");
        m.insert("lsquo;", "\u{2018}");
        m.insert("lsquor;", "\u{201A}");
        m.insert("lstrok;", "\u{142}");
        m.insert("lt", "\u{3C}");
        m.insert("lt;", "\u{3C}");
        m.insert("ltcc;", "\u{2AA6}");
        m.insert("ltcir;", "\u{2A79}");
        m.insert("ltdot;", "\u{22D6}");
        m.insert("lthree;", "\u{22CB}");
        m.insert("ltimes;", "\u{22C9}");
        m.insert("ltlarr;", "\u{2976}");
        m.insert("ltquest;", "\u{2A7B}");
        m.insert("ltrPar;", "\u{2996}");
        m.insert("ltri;", "\u{25C3}");
        m.insert("ltrie;", "\u{22B4}");
        m.insert("ltrif;", "\u{25C2}");
        m.insert("lurdshar;", "\u{294A}");
        m.insert("luruhar;", "\u{2966}");
        m.insert("lvertneqq;", "\u{2268}\u{FE00}");
        m.insert("lvnE;", "\u{2268}\u{FE00}");
        m.insert("mDDot;", "\u{223A}");
        m.insert("macr", "\u{AF}");
        m.insert("macr;", "\u{AF}");
        m.insert("male;", "\u{2642}");
        m.insert("malt;", "\u{2720}");
        m.insert("maltese;", "\u{2720}");
        m.insert("map;", "\u{21A6}");
        m.insert("mapsto;", "\u{21A6}");
        m.insert("mapstodown;", "\u{21A7}");
        m.insert("mapstoleft;", "\u{21A4}");
        m.insert("mapstoup;", "\u{21A5}");
        m.insert("marker;", "\u{25AE}");
        m.insert("mcomma;", "\u{2A29}");
        m.insert("mcy;", "\u{43C}");
        m.insert("mdash;", "\u{2014}");
        m.insert("measuredangle;", "\u{2221}");
        m.insert("mfr;", "\u{1D52A}");
        m.insert("mho;", "\u{2127}");
        m.insert("micro", "\u{B5}");
        m.insert("micro;", "\u{B5}");
        m.insert("mid;", "\u{2223}");
        m.insert("midast;", "\u{2A}");
        m.insert("midcir;", "\u{2AF0}");
        m.insert("middot", "\u{B7}");
        m.insert("middot;", "\u{B7}");
        m.insert("minus;", "\u{2212}");
        m.insert("minusb;", "\u{229F}");
        m.insert("minusd;", "\u{2238}");
        m.insert("minusdu;", "\u{2A2A}");
        m.insert("mlcp;", "\u{2ADB}");
        m.insert("mldr;", "\u{2026}");
        m.insert("mnplus;", "\u{2213}");
        m.insert("models;", "\u{22A7}");
        m.insert("mopf;", "\u{1D55E}");
        m.insert("mp;", "\u{2213}");
        m.insert("mscr;", "\u{1D4C2}");
        m.insert("mstpos;", "\u{223E}");
        m.insert("mu;", "\u{3BC}");
        m.insert("multimap;", "\u{22B8}");
        m.insert("mumap;", "\u{22B8}");
        m.insert("nGg;", "\u{22D9}\u{338}");
        m.insert("nGt;", "\u{226B}\u{20D2}");
        m.insert("nGtv;", "\u{226B}\u{338}");
        m.insert("nLeftarrow;", "\u{21CD}");
        m.insert("nLeftrightarrow;", "\u{21CE}");
        m.insert("nLl;", "\u{22D8}\u{338}");
        m.insert("nLt;", "\u{226A}\u{20D2}");
        m.insert("nLtv;", "\u{226A}\u{338}");
        m.insert("nRightarrow;", "\u{21CF}");
        m.insert("nVDash;", "\u{22AF}");
        m.insert("nVdash;", "\u{22AE}");
        m.insert("nabla;", "\u{2207}");
        m.insert("nacute;", "\u{144}");
        m.insert("nang;", "\u{2220}\u{20D2}");
        m.insert("nap;", "\u{2249}");
        m.insert("napE;", "\u{2A70}\u{338}");
        m.insert("napid;", "\u{224B}\u{338}");
        m.insert("napos;", "\u{149}");
        m.insert("napprox;", "\u{2249}");
        m.insert("natur;", "\u{266E}");
        m.insert("natural;", "\u{266E}");
        m.insert("naturals;", "\u{2115}");
        m.insert("nbsp", "\u{A0}");
        m.insert("nbsp;", "\u{A0}");
        m.insert("nbump;", "\u{224E}\u{338}");
        m.insert("nbumpe;", "\u{224F}\u{338}");
        m.insert("ncap;", "\u{2A43}");
        m.insert("ncaron;", "\u{148}");
        m.insert("ncedil;", "\u{146}");
        m.insert("ncong;", "\u{2247}");
        m.insert("ncongdot;", "\u{2A6D}\u{338}");
        m.insert("ncup;", "\u{2A42}");
        m.insert("ncy;", "\u{43D}");
        m.insert("ndash;", "\u{2013}");
        m.insert("ne;", "\u{2260}");
        m.insert("neArr;", "\u{21D7}");
        m.insert("nearhk;", "\u{2924}");
        m.insert("nearr;", "\u{2197}");
        m.insert("nearrow;", "\u{2197}");
        m.insert("nedot;", "\u{2250}\u{338}");
        m.insert("nequiv;", "\u{2262}");
        m.insert("nesear;", "\u{2928}");
        m.insert("nesim;", "\u{2242}\u{338}");
        m.insert("nexist;", "\u{2204}");
        m.insert("nexists;", "\u{2204}");
        m.insert("nfr;", "\u{1D52B}");
        m.insert("ngE;", "\u{2267}\u{338}");
        m.insert("nge;", "\u{2271}");
        m.insert("ngeq;", "\u{2271}");
        m.insert("ngeqq;", "\u{2267}\u{338}");
        m.insert("ngeqslant;", "\u{2A7E}\u{338}");
        m.insert("nges;", "\u{2A7E}\u{338}");
        m.insert("ngsim;", "\u{2275}");
        m.insert("ngt;", "\u{226F}");
        m.insert("ngtr;", "\u{226F}");
        m.insert("nhArr;", "\u{21CE}");
        m.insert("nharr;", "\u{21AE}");
        m.insert("nhpar;", "\u{2AF2}");
        m.insert("ni;", "\u{220B}");
        m.insert("nis;", "\u{22FC}");
        m.insert("nisd;", "\u{22FA}");
        m.insert("niv;", "\u{220B}");
        m.insert("njcy;", "\u{45A}");
        m.insert("nlArr;", "\u{21CD}");
        m.insert("nlE;", "\u{2266}\u{338}");
        m.insert("nlarr;", "\u{219A}");
        m.insert("nldr;", "\u{2025}");
        m.insert("nle;", "\u{2270}");
        m.insert("nleftarrow;", "\u{219A}");
        m.insert("nleftrightarrow;", "\u{21AE}");
        m.insert("nleq;", "\u{2270}");
        m.insert("nleqq;", "\u{2266}\u{338}");
        m.insert("nleqslant;", "\u{2A7D}\u{338}");
        m.insert("nles;", "\u{2A7D}\u{338}");
        m.insert("nless;", "\u{226E}");
        m.insert("nlsim;", "\u{2274}");
        m.insert("nlt;", "\u{226E}");
        m.insert("nltri;", "\u{22EA}");
        m.insert("nltrie;", "\u{22EC}");
        m.insert("nmid;", "\u{2224}");
        m.insert("nopf;", "\u{1D55F}");
        m.insert("not", "\u{AC}");
        m.insert("not;", "\u{AC}");
        m.insert("notin;", "\u{2209}");
        m.insert("notinE;", "\u{22F9}\u{338}");
        m.insert("notindot;", "\u{22F5}\u{338}");
        m.insert("notinva;", "\u{2209}");
        m.insert("notinvb;", "\u{22F7}");
        m.insert("notinvc;", "\u{22F6}");
        m.insert("notni;", "\u{220C}");
        m.insert("notniva;", "\u{220C}");
        m.insert("notnivb;", "\u{22FE}");
        m.insert("notnivc;", "\u{22FD}");
        m.insert("npar;", "\u{2226}");
        m.insert("nparallel;", "\u{2226}");
        m.insert("nparsl;", "\u{2AFD}\u{20E5}");
        m.insert("npart;", "\u{2202}\u{338}");
        m.insert("npolint;", "\u{2A14}");
        m.insert("npr;", "\u{2280}");
        m.insert("nprcue;", "\u{22E0}");
        m.insert("npre;", "\u{2AAF}\u{338}");
        m.insert("nprec;", "\u{2280}");
        m.insert("npreceq;", "\u{2AAF}\u{338}");
        m.insert("nrArr;", "\u{21CF}");
        m.insert("nrarr;", "\u{219B}");
        m.insert("nrarrc;", "\u{2933}\u{338}");
        m.insert("nrarrw;", "\u{219D}\u{338}");
        m.insert("nrightarrow;", "\u{219B}");
        m.insert("nrtri;", "\u{22EB}");
        m.insert("nrtrie;", "\u{22ED}");
        m.insert("nsc;", "\u{2281}");
        m.insert("nsccue;", "\u{22E1}");
        m.insert("nsce;", "\u{2AB0}\u{338}");
        m.insert("nscr;", "\u{1D4C3}");
        m.insert("nshortmid;", "\u{2224}");
        m.insert("nshortparallel;", "\u{2226}");
        m.insert("nsim;", "\u{2241}");
        m.insert("nsime;", "\u{2244}");
        m.insert("nsimeq;", "\u{2244}");
        m.insert("nsmid;", "\u{2224}");
        m.insert("nspar;", "\u{2226}");
        m.insert("nsqsube;", "\u{22E2}");
        m.insert("nsqsupe;", "\u{22E3}");
        m.insert("nsub;", "\u{2284}");
        m.insert("nsubE;", "\u{2AC5}\u{338}");
        m.insert("nsube;", "\u{2288}");
        m.insert("nsubset;", "\u{2282}\u{20D2}");
        m.insert("nsubseteq;", "\u{2288}");
        m.insert("nsubseteqq;", "\u{2AC5}\u{338}");
        m.insert("nsucc;", "\u{2281}");
        m.insert("nsucceq;", "\u{2AB0}\u{338}");
        m.insert("nsup;", "\u{2285}");
        m.insert("nsupE;", "\u{2AC6}\u{338}");
        m.insert("nsupe;", "\u{2289}");
        m.insert("nsupset;", "\u{2283}\u{20D2}");
        m.insert("nsupseteq;", "\u{2289}");
        m.insert("nsupseteqq;", "\u{2AC6}\u{338}");
        m.insert("ntgl;", "\u{2279}");
        m.insert("ntilde", "\u{F1}");
        m.insert("ntilde;", "\u{F1}");
        m.insert("ntlg;", "\u{2278}");
        m.insert("ntriangleleft;", "\u{22EA}");
        m.insert("ntrianglelefteq;", "\u{22EC}");
        m.insert("ntriangleright;", "\u{22EB}");
        m.insert("ntrianglerighteq;", "\u{22ED}");
        m.insert("nu;", "\u{3BD}");
        m.insert("num;", "\u{23}");
        m.insert("numero;", "\u{2116}");
        m.insert("numsp;", "\u{2007}");
        m.insert("nvDash;", "\u{22AD}");
        m.insert("nvHarr;", "\u{2904}");
        m.insert("nvap;", "\u{224D}\u{20D2}");
        m.insert("nvdash;", "\u{22AC}");
        m.insert("nvge;", "\u{2265}\u{20D2}");
        m.insert("nvgt;", "\u{3E}\u{20D2}");
        m.insert("nvinfin;", "\u{29DE}");
        m.insert("nvlArr;", "\u{2902}");
        m.insert("nvle;", "\u{2264}\u{20D2}");
        m.insert("nvlt;", "\u{3C}\u{20D2}");
        m.insert("nvltrie;", "\u{22B4}\u{20D2}");
        m.insert("nvrArr;", "\u{2903}");
        m.insert("nvrtrie;", "\u{22B5}\u{20D2}");
        m.insert("nvsim;", "\u{223C}\u{20D2}");
        m.insert("nwArr;", "\u{21D6}");
        m.insert("nwarhk;", "\u{2923}");
        m.insert("nwarr;", "\u{2196}");
        m.insert("nwarrow;", "\u{2196}");
        m.insert("nwnear;", "\u{2927}");
        m.insert("oS;", "\u{24C8}");
        m.insert("oacute", "\u{F3}");
        m.insert("oacute;", "\u{F3}");
        m.insert("oast;", "\u{229B}");
        m.insert("ocir;", "\u{229A}");
        m.insert("ocirc", "\u{F4}");
        m.insert("ocirc;", "\u{F4}");
        m.insert("ocy;", "\u{43E}");
        m.insert("odash;", "\u{229D}");
        m.insert("odblac;", "\u{151}");
        m.insert("odiv;", "\u{2A38}");
        m.insert("odot;", "\u{2299}");
        m.insert("odsold;", "\u{29BC}");
        m.insert("oelig;", "\u{153}");
        m.insert("ofcir;", "\u{29BF}");
        m.insert("ofr;", "\u{1D52C}");
        m.insert("ogon;", "\u{2DB}");
        m.insert("ograve", "\u{F2}");
        m.insert("ograve;", "\u{F2}");
        m.insert("ogt;", "\u{29C1}");
        m.insert("ohbar;", "\u{29B5}");
        m.insert("ohm;", "\u{3A9}");
        m.insert("oint;", "\u{222E}");
        m.insert("olarr;", "\u{21BA}");
        m.insert("olcir;", "\u{29BE}");
        m.insert("olcross;", "\u{29BB}");
        m.insert("oline;", "\u{203E}");
        m.insert("olt;", "\u{29C0}");
        m.insert("omacr;", "\u{14D}");
        m.insert("omega;", "\u{3C9}");
        m.insert("omicron;", "\u{3BF}");
        m.insert("omid;", "\u{29B6}");
        m.insert("ominus;", "\u{2296}");
        m.insert("oopf;", "\u{1D560}");
        m.insert("opar;", "\u{29B7}");
        m.insert("operp;", "\u{29B9}");
        m.insert("oplus;", "\u{2295}");
        m.insert("or;", "\u{2228}");
        m.insert("orarr;", "\u{21BB}");
        m.insert("ord;", "\u{2A5D}");
        m.insert("order;", "\u{2134}");
        m.insert("orderof;", "\u{2134}");
        m.insert("ordf", "\u{AA}");
        m.insert("ordf;", "\u{AA}");
        m.insert("ordm", "\u{BA}");
        m.insert("ordm;", "\u{BA}");
        m.insert("origof;", "\u{22B6}");
        m.insert("oror;", "\u{2A56}");
        m.insert("orslope;", "\u{2A57}");
        m.insert("orv;", "\u{2A5B}");
        m.insert("oscr;", "\u{2134}");
        m.insert("oslash", "\u{F8}");
        m.insert("oslash;", "\u{F8}");
        m.insert("osol;", "\u{2298}");
        m.insert("otilde", "\u{F5}");
        m.insert("otilde;", "\u{F5}");
        m.insert("otimes;", "\u{2297}");
        m.insert("otimesas;", "\u{2A36}");
        m.insert("ouml", "\u{F6}");
        m.insert("ouml;", "\u{F6}");
        m.insert("ovbar;", "\u{233D}");
        m.insert("par;", "\u{2225}");
        m.insert("para", "\u{B6}");
        m.insert("para;", "\u{B6}");
        m.insert("parallel;", "\u{2225}");
        m.insert("parsim;", "\u{2AF3}");
        m.insert("parsl;", "\u{2AFD}");
        m.insert("part;", "\u{2202}");
        m.insert("pcy;", "\u{43F}");
        m.insert("percnt;", "\u{25}");
        m.insert("period;", "\u{2E}");
        m.insert("permil;", "\u{2030}");
        m.insert("perp;", "\u{22A5}");
        m.insert("pertenk;", "\u{2031}");
        m.insert("pfr;", "\u{1D52D}");
        m.insert("phi;", "\u{3C6}");
        m.insert("phiv;", "\u{3D5}");
        m.insert("phmmat;", "\u{2133}");
        m.insert("phone;", "\u{260E}");
        m.insert("pi;", "\u{3C0}");
        m.insert("pitchfork;", "\u{22D4}");
        m.insert("piv;", "\u{3D6}");
        m.insert("planck;", "\u{210F}");
        m.insert("planckh;", "\u{210E}");
        m.insert("plankv;", "\u{210F}");
        m.insert("plus;", "\u{2B}");
        m.insert("plusacir;", "\u{2A23}");
        m.insert("plusb;", "\u{229E}");
        m.insert("pluscir;", "\u{2A22}");
        m.insert("plusdo;", "\u{2214}");
        m.insert("plusdu;", "\u{2A25}");
        m.insert("pluse;", "\u{2A72}");
        m.insert("plusmn", "\u{B1}");
        m.insert("plusmn;", "\u{B1}");
        m.insert("plussim;", "\u{2A26}");
        m.insert("plustwo;", "\u{2A27}");
        m.insert("pm;", "\u{B1}");
        m.insert("pointint;", "\u{2A15}");
        m.insert("popf;", "\u{1D561}");
        m.insert("pound", "\u{A3}");
        m.insert("pound;", "\u{A3}");
        m.insert("pr;", "\u{227A}");
        m.insert("prE;", "\u{2AB3}");
        m.insert("prap;", "\u{2AB7}");
        m.insert("prcue;", "\u{227C}");
        m.insert("pre;", "\u{2AAF}");
        m.insert("prec;", "\u{227A}");
        m.insert("precapprox;", "\u{2AB7}");
        m.insert("preccurlyeq;", "\u{227C}");
        m.insert("preceq;", "\u{2AAF}");
        m.insert("precnapprox;", "\u{2AB9}");
        m.insert("precneqq;", "\u{2AB5}");
        m.insert("precnsim;", "\u{22E8}");
        m.insert("precsim;", "\u{227E}");
        m.insert("prime;", "\u{2032}");
        m.insert("primes;", "\u{2119}");
        m.insert("prnE;", "\u{2AB5}");
        m.insert("prnap;", "\u{2AB9}");
        m.insert("prnsim;", "\u{22E8}");
        m.insert("prod;", "\u{220F}");
        m.insert("profalar;", "\u{232E}");
        m.insert("profline;", "\u{2312}");
        m.insert("profsurf;", "\u{2313}");
        m.insert("prop;", "\u{221D}");
        m.insert("propto;", "\u{221D}");
        m.insert("prsim;", "\u{227E}");
        m.insert("prurel;", "\u{22B0}");
        m.insert("pscr;", "\u{1D4C5}");
        m.insert("psi;", "\u{3C8}");
        m.insert("puncsp;", "\u{2008}");
        m.insert("qfr;", "\u{1D52E}");
        m.insert("qint;", "\u{2A0C}");
        m.insert("qopf;", "\u{1D562}");
        m.insert("qprime;", "\u{2057}");
        m.insert("qscr;", "\u{1D4C6}");
        m.insert("quaternions;", "\u{210D}");
        m.insert("quatint;", "\u{2A16}");
        m.insert("quest;", "\u{3F}");
        m.insert("questeq;", "\u{225F}");
        m.insert("quot", "\u{22}");
        m.insert("quot;", "\u{22}");
        m.insert("rAarr;", "\u{21DB}");
        m.insert("rArr;", "\u{21D2}");
        m.insert("rAtail;", "\u{291C}");
        m.insert("rBarr;", "\u{290F}");
        m.insert("rHar;", "\u{2964}");
        m.insert("race;", "\u{223D}\u{331}");
        m.insert("racute;", "\u{155}");
        m.insert("radic;", "\u{221A}");
        m.insert("raemptyv;", "\u{29B3}");
        m.insert("rang;", "\u{27E9}");
        m.insert("rangd;", "\u{2992}");
        m.insert("range;", "\u{29A5}");
        m.insert("rangle;", "\u{27E9}");
        m.insert("raquo", "\u{BB}");
        m.insert("raquo;", "\u{BB}");
        m.insert("rarr;", "\u{2192}");
        m.insert("rarrap;", "\u{2975}");
        m.insert("rarrb;", "\u{21E5}");
        m.insert("rarrbfs;", "\u{2920}");
        m.insert("rarrc;", "\u{2933}");
        m.insert("rarrfs;", "\u{291E}");
        m.insert("rarrhk;", "\u{21AA}");
        m.insert("rarrlp;", "\u{21AC}");
        m.insert("rarrpl;", "\u{2945}");
        m.insert("rarrsim;", "\u{2974}");
        m.insert("rarrtl;", "\u{21A3}");
        m.insert("rarrw;", "\u{219D}");
        m.insert("ratail;", "\u{291A}");
        m.insert("ratio;", "\u{2236}");
        m.insert("rationals;", "\u{211A}");
        m.insert("rbarr;", "\u{290D}");
        m.insert("rbbrk;", "\u{2773}");
        m.insert("rbrace;", "\u{7D}");
        m.insert("rbrack;", "\u{5D}");
        m.insert("rbrke;", "\u{298C}");
        m.insert("rbrksld;", "\u{298E}");
        m.insert("rbrkslu;", "\u{2990}");
        m.insert("rcaron;", "\u{159}");
        m.insert("rcedil;", "\u{157}");
        m.insert("rceil;", "\u{2309}");
        m.insert("rcub;", "\u{7D}");
        m.insert("rcy;", "\u{440}");
        m.insert("rdca;", "\u{2937}");
        m.insert("rdldhar;", "\u{2969}");
        m.insert("rdquo;", "\u{201D}");
        m.insert("rdquor;", "\u{201D}");
        m.insert("rdsh;", "\u{21B3}");
        m.insert("real;", "\u{211C}");
        m.insert("realine;", "\u{211B}");
        m.insert("realpart;", "\u{211C}");
        m.insert("reals;", "\u{211D}");
        m.insert("rect;", "\u{25AD}");
        m.insert("reg", "\u{AE}");
        m.insert("reg;", "\u{AE}");
        m.insert("rfisht;", "\u{297D}");
        m.insert("rfloor;", "\u{230B}");
        m.insert("rfr;", "\u{1D52F}");
        m.insert("rhard;", "\u{21C1}");
        m.insert("rharu;", "\u{21C0}");
        m.insert("rharul;", "\u{296C}");
        m.insert("rho;", "\u{3C1}");
        m.insert("rhov;", "\u{3F1}");
        m.insert("rightarrow;", "\u{2192}");
        m.insert("rightarrowtail;", "\u{21A3}");
        m.insert("rightharpoondown;", "\u{21C1}");
        m.insert("rightharpoonup;", "\u{21C0}");
        m.insert("rightleftarrows;", "\u{21C4}");
        m.insert("rightleftharpoons;", "\u{21CC}");
        m.insert("rightrightarrows;", "\u{21C9}");
        m.insert("rightsquigarrow;", "\u{219D}");
        m.insert("rightthreetimes;", "\u{22CC}");
        m.insert("ring;", "\u{2DA}");
        m.insert("risingdotseq;", "\u{2253}");
        m.insert("rlarr;", "\u{21C4}");
        m.insert("rlhar;", "\u{21CC}");
        m.insert("rlm;", "\u{200F}");
        m.insert("rmoust;", "\u{23B1}");
        m.insert("rmoustache;", "\u{23B1}");
        m.insert("rnmid;", "\u{2AEE}");
        m.insert("roang;", "\u{27ED}");
        m.insert("roarr;", "\u{21FE}");
        m.insert("robrk;", "\u{27E7}");
        m.insert("ropar;", "\u{2986}");
        m.insert("ropf;", "\u{1D563}");
        m.insert("roplus;", "\u{2A2E}");
        m.insert("rotimes;", "\u{2A35}");
        m.insert("rpar;", "\u{29}");
        m.insert("rpargt;", "\u{2994}");
        m.insert("rppolint;", "\u{2A12}");
        m.insert("rrarr;", "\u{21C9}");
        m.insert("rsaquo;", "\u{203A}");
        m.insert("rscr;", "\u{1D4C7}");
        m.insert("rsh;", "\u{21B1}");
        m.insert("rsqb;", "\u{5D}");
        m.insert("rsquo;", "\u{2019}");
        m.insert("rsquor;", "\u{2019}");
        m.insert("rthree;", "\u{22CC}");
        m.insert("rtimes;", "\u{22CA}");
        m.insert("rtri;", "\u{25B9}");
        m.insert("rtrie;", "\u{22B5}");
        m.insert("rtrif;", "\u{25B8}");
        m.insert("rtriltri;", "\u{29CE}");
        m.insert("ruluhar;", "\u{2968}");
        m.insert("rx;", "\u{211E}");
        m.insert("sacute;", "\u{15B}");
        m.insert("sbquo;", "\u{201A}");
        m.insert("sc;", "\u{227B}");
        m.insert("scE;", "\u{2AB4}");
        m.insert("scap;", "\u{2AB8}");
        m.insert("scaron;", "\u{161}");
        m.insert("sccue;", "\u{227D}");
        m.insert("sce;", "\u{2AB0}");
        m.insert("scedil;", "\u{15F}");
        m.insert("scirc;", "\u{15D}");
        m.insert("scnE;", "\u{2AB6}");
        m.insert("scnap;", "\u{2ABA}");
        m.insert("scnsim;", "\u{22E9}");
        m.insert("scpolint;", "\u{2A13}");
        m.insert("scsim;", "\u{227F}");
        m.insert("scy;", "\u{441}");
        m.insert("sdot;", "\u{22C5}");
        m.insert("sdotb;", "\u{22A1}");
        m.insert("sdote;", "\u{2A66}");
        m.insert("seArr;", "\u{21D8}");
        m.insert("searhk;", "\u{2925}");
        m.insert("searr;", "\u{2198}");
        m.insert("searrow;", "\u{2198}");
        m.insert("sect", "\u{A7}");
        m.insert("sect;", "\u{A7}");
        m.insert("semi;", "\u{3B}");
        m.insert("seswar;", "\u{2929}");
        m.insert("setminus;", "\u{2216}");
        m.insert("setmn;", "\u{2216}");
        m.insert("sext;", "\u{2736}");
        m.insert("sfr;", "\u{1D530}");
        m.insert("sfrown;", "\u{2322}");
        m.insert("sharp;", "\u{266F}");
        m.insert("shchcy;", "\u{449}");
        m.insert("shcy;", "\u{448}");
        m.insert("shortmid;", "\u{2223}");
        m.insert("shortparallel;", "\u{2225}");
        m.insert("shy", "\u{AD}");
        m.insert("shy;", "\u{AD}");
        m.insert("sigma;", "\u{3C3}");
        m.insert("sigmaf;", "\u{3C2}");
        m.insert("sigmav;", "\u{3C2}");
        m.insert("sim;", "\u{223C}");
        m.insert("simdot;", "\u{2A6A}");
        m.insert("sime;", "\u{2243}");
        m.insert("simeq;", "\u{2243}");
        m.insert("simg;", "\u{2A9E}");
        m.insert("simgE;", "\u{2AA0}");
        m.insert("siml;", "\u{2A9D}");
        m.insert("simlE;", "\u{2A9F}");
        m.insert("simne;", "\u{2246}");
        m.insert("simplus;", "\u{2A24}");
        m.insert("simrarr;", "\u{2972}");
        m.insert("slarr;", "\u{2190}");
        m.insert("smallsetminus;", "\u{2216}");
        m.insert("smashp;", "\u{2A33}");
        m.insert("smeparsl;", "\u{29E4}");
        m.insert("smid;", "\u{2223}");
        m.insert("smile;", "\u{2323}");
        m.insert("smt;", "\u{2AAA}");
        m.insert("smte;", "\u{2AAC}");
        m.insert("smtes;", "\u{2AAC}\u{FE00}");
        m.insert("softcy;", "\u{44C}");
        m.insert("sol;", "\u{2F}");
        m.insert("solb;", "\u{29C4}");
        m.insert("solbar;", "\u{233F}");
        m.insert("sopf;", "\u{1D564}");
        m.insert("spades;", "\u{2660}");
        m.insert("spadesuit;", "\u{2660}");
        m.insert("spar;", "\u{2225}");
        m.insert("sqcap;", "\u{2293}");
        m.insert("sqcaps;", "\u{2293}\u{FE00}");
        m.insert("sqcup;", "\u{2294}");
        m.insert("sqcups;", "\u{2294}\u{FE00}");
        m.insert("sqsub;", "\u{228F}");
        m.insert("sqsube;", "\u{2291}");
        m.insert("sqsubset;", "\u{228F}");
        m.insert("sqsubseteq;", "\u{2291}");
        m.insert("sqsup;", "\u{2290}");
        m.insert("sqsupe;", "\u{2292}");
        m.insert("sqsupset;", "\u{2290}");
        m.insert("sqsupseteq;", "\u{2292}");
        m.insert("squ;", "\u{25A1}");
        m.insert("square;", "\u{25A1}");
        m.insert("squarf;", "\u{25AA}");
        m.insert("squf;", "\u{25AA}");
        m.insert("srarr;", "\u{2192}");
        m.insert("sscr;", "\u{1D4C8}");
        m.insert("ssetmn;", "\u{2216}");
        m.insert("ssmile;", "\u{2323}");
        m.insert("sstarf;", "\u{22C6}");
        m.insert("star;", "\u{2606}");
        m.insert("starf;", "\u{2605}");
        m.insert("straightepsilon;", "\u{3F5}");
        m.insert("straightphi;", "\u{3D5}");
        m.insert("strns;", "\u{AF}");
        m.insert("sub;", "\u{2282}");
        m.insert("subE;", "\u{2AC5}");
        m.insert("subdot;", "\u{2ABD}");
        m.insert("sube;", "\u{2286}");
        m.insert("subedot;", "\u{2AC3}");
        m.insert("submult;", "\u{2AC1}");
        m.insert("subnE;", "\u{2ACB}");
        m.insert("subne;", "\u{228A}");
        m.insert("subplus;", "\u{2ABF}");
        m.insert("subrarr;", "\u{2979}");
        m.insert("subset;", "\u{2282}");
        m.insert("subseteq;", "\u{2286}");
        m.insert("subseteqq;", "\u{2AC5}");
        m.insert("subsetneq;", "\u{228A}");
        m.insert("subsetneqq;", "\u{2ACB}");
        m.insert("subsim;", "\u{2AC7}");
        m.insert("subsub;", "\u{2AD5}");
        m.insert("subsup;", "\u{2AD3}");
        m.insert("succ;", "\u{227B}");
        m.insert("succapprox;", "\u{2AB8}");
        m.insert("succcurlyeq;", "\u{227D}");
        m.insert("succeq;", "\u{2AB0}");
        m.insert("succnapprox;", "\u{2ABA}");
        m.insert("succneqq;", "\u{2AB6}");
        m.insert("succnsim;", "\u{22E9}");
        m.insert("succsim;", "\u{227F}");
        m.insert("sum;", "\u{2211}");
        m.insert("sung;", "\u{266A}");
        m.insert("sup1", "\u{B9}");
        m.insert("sup1;", "\u{B9}");
        m.insert("sup2", "\u{B2}");
        m.insert("sup2;", "\u{B2}");
        m.insert("sup3", "\u{B3}");
        m.insert("sup3;", "\u{B3}");
        m.insert("sup;", "\u{2283}");
        m.insert("supE;", "\u{2AC6}");
        m.insert("supdot;", "\u{2ABE}");
        m.insert("supdsub;", "\u{2AD8}");
        m.insert("supe;", "\u{2287}");
        m.insert("supedot;", "\u{2AC4}");
        m.insert("suphsol;", "\u{27C9}");
        m.insert("suphsub;", "\u{2AD7}");
        m.insert("suplarr;", "\u{297B}");
        m.insert("supmult;", "\u{2AC2}");
        m.insert("supnE;", "\u{2ACC}");
        m.insert("supne;", "\u{228B}");
        m.insert("supplus;", "\u{2AC0}");
        m.insert("supset;", "\u{2283}");
        m.insert("supseteq;", "\u{2287}");
        m.insert("supseteqq;", "\u{2AC6}");
        m.insert("supsetneq;", "\u{228B}");
        m.insert("supsetneqq;", "\u{2ACC}");
        m.insert("supsim;", "\u{2AC8}");
        m.insert("supsub;", "\u{2AD4}");
        m.insert("supsup;", "\u{2AD6}");
        m.insert("swArr;", "\u{21D9}");
        m.insert("swarhk;", "\u{2926}");
        m.insert("swarr;", "\u{2199}");
        m.insert("swarrow;", "\u{2199}");
        m.insert("swnwar;", "\u{292A}");
        m.insert("szlig", "\u{DF}");
        m.insert("szlig;", "\u{DF}");
        m.insert("target;", "\u{2316}");
        m.insert("tau;", "\u{3C4}");
        m.insert("tbrk;", "\u{23B4}");
        m.insert("tcaron;", "\u{165}");
        m.insert("tcedil;", "\u{163}");
        m.insert("tcy;", "\u{442}");
        m.insert("tdot;", "\u{20DB}");
        m.insert("telrec;", "\u{2315}");
        m.insert("tfr;", "\u{1D531}");
        m.insert("there4;", "\u{2234}");
        m.insert("therefore;", "\u{2234}");
        m.insert("theta;", "\u{3B8}");
        m.insert("thetasym;", "\u{3D1}");
        m.insert("thetav;", "\u{3D1}");
        m.insert("thickapprox;", "\u{2248}");
        m.insert("thicksim;", "\u{223C}");
        m.insert("thinsp;", "\u{2009}");
        m.insert("thkap;", "\u{2248}");
        m.insert("thksim;", "\u{223C}");
        m.insert("thorn", "\u{FE}");
        m.insert("thorn;", "\u{FE}");
        m.insert("tilde;", "\u{2DC}");
        m.insert("times", "\u{D7}");
        m.insert("times;", "\u{D7}");
        m.insert("timesb;", "\u{22A0}");
        m.insert("timesbar;", "\u{2A31}");
        m.insert("timesd;", "\u{2A30}");
        m.insert("tint;", "\u{222D}");
        m.insert("toea;", "\u{2928}");
        m.insert("top;", "\u{22A4}");
        m.insert("topbot;", "\u{2336}");
        m.insert("topcir;", "\u{2AF1}");
        m.insert("topf;", "\u{1D565}");
        m.insert("topfork;", "\u{2ADA}");
        m.insert("tosa;", "\u{2929}");
        m.insert("tprime;", "\u{2034}");
        m.insert("trade;", "\u{2122}");
        m.insert("triangle;", "\u{25B5}");
        m.insert("triangledown;", "\u{25BF}");
        m.insert("triangleleft;", "\u{25C3}");
        m.insert("trianglelefteq;", "\u{22B4}");
        m.insert("triangleq;", "\u{225C}");
        m.insert("triangleright;", "\u{25B9}");
        m.insert("trianglerighteq;", "\u{22B5}");
        m.insert("tridot;", "\u{25EC}");
        m.insert("trie;", "\u{225C}");
        m.insert("triminus;", "\u{2A3A}");
        m.insert("triplus;", "\u{2A39}");
        m.insert("trisb;", "\u{29CD}");
        m.insert("tritime;", "\u{2A3B}");
        m.insert("trpezium;", "\u{23E2}");
        m.insert("tscr;", "\u{1D4C9}");
        m.insert("tscy;", "\u{446}");
        m.insert("tshcy;", "\u{45B}");
        m.insert("tstrok;", "\u{167}");
        m.insert("twixt;", "\u{226C}");
        m.insert("twoheadleftarrow;", "\u{219E}");
        m.insert("twoheadrightarrow;", "\u{21A0}");
        m.insert("uArr;", "\u{21D1}");
        m.insert("uHar;", "\u{2963}");
        m.insert("uacute", "\u{FA}");
        m.insert("uacute;", "\u{FA}");
        m.insert("uarr;", "\u{2191}");
        m.insert("ubrcy;", "\u{45E}");
        m.insert("ubreve;", "\u{16D}");
        m.insert("ucirc", "\u{FB}");
        m.insert("ucirc;", "\u{FB}");
        m.insert("ucy;", "\u{443}");
        m.insert("udarr;", "\u{21C5}");
        m.insert("udblac;", "\u{171}");
        m.insert("udhar;", "\u{296E}");
        m.insert("ufisht;", "\u{297E}");
        m.insert("ufr;", "\u{1D532}");
        m.insert("ugrave", "\u{F9}");
        m.insert("ugrave;", "\u{F9}");
        m.insert("uharl;", "\u{21BF}");
        m.insert("uharr;", "\u{21BE}");
        m.insert("uhblk;", "\u{2580}");
        m.insert("ulcorn;", "\u{231C}");
        m.insert("ulcorner;", "\u{231C}");
        m.insert("ulcrop;", "\u{230F}");
        m.insert("ultri;", "\u{25F8}");
        m.insert("umacr;", "\u{16B}");
        m.insert("uml", "\u{A8}");
        m.insert("uml;", "\u{A8}");
        m.insert("uogon;", "\u{173}");
        m.insert("uopf;", "\u{1D566}");
        m.insert("uparrow;", "\u{2191}");
        m.insert("updownarrow;", "\u{2195}");
        m.insert("upharpoonleft;", "\u{21BF}");
        m.insert("upharpoonright;", "\u{21BE}");
        m.insert("uplus;", "\u{228E}");
        m.insert("upsi;", "\u{3C5}");
        m.insert("upsih;", "\u{3D2}");
        m.insert("upsilon;", "\u{3C5}");
        m.insert("upuparrows;", "\u{21C8}");
        m.insert("urcorn;", "\u{231D}");
        m.insert("urcorner;", "\u{231D}");
        m.insert("urcrop;", "\u{230E}");
        m.insert("uring;", "\u{16F}");
        m.insert("urtri;", "\u{25F9}");
        m.insert("uscr;", "\u{1D4CA}");
        m.insert("utdot;", "\u{22F0}");
        m.insert("utilde;", "\u{169}");
        m.insert("utri;", "\u{25B5}");
        m.insert("utrif;", "\u{25B4}");
        m.insert("uuarr;", "\u{21C8}");
        m.insert("uuml", "\u{FC}");
        m.insert("uuml;", "\u{FC}");
        m.insert("uwangle;", "\u{29A7}");
        m.insert("vArr;", "\u{21D5}");
        m.insert("vBar;", "\u{2AE8}");
        m.insert("vBarv;", "\u{2AE9}");
        m.insert("vDash;", "\u{22A8}");
        m.insert("vangrt;", "\u{299C}");
        m.insert("varepsilon;", "\u{3F5}");
        m.insert("varkappa;", "\u{3F0}");
        m.insert("varnothing;", "\u{2205}");
        m.insert("varphi;", "\u{3D5}");
        m.insert("varpi;", "\u{3D6}");
        m.insert("varpropto;", "\u{221D}");
        m.insert("varr;", "\u{2195}");
        m.insert("varrho;", "\u{3F1}");
        m.insert("varsigma;", "\u{3C2}");
        m.insert("varsubsetneq;", "\u{228A}\u{FE00}");
        m.insert("varsubsetneqq;", "\u{2ACB}\u{FE00}");
        m.insert("varsupsetneq;", "\u{228B}\u{FE00}");
        m.insert("varsupsetneqq;", "\u{2ACC}\u{FE00}");
        m.insert("vartheta;", "\u{3D1}");
        m.insert("vartriangleleft;", "\u{22B2}");
        m.insert("vartriangleright;", "\u{22B3}");
        m.insert("vcy;", "\u{432}");
        m.insert("vdash;", "\u{22A2}");
        m.insert("vee;", "\u{2228}");
        m.insert("veebar;", "\u{22BB}");
        m.insert("veeeq;", "\u{225A}");
        m.insert("vellip;", "\u{22EE}");
        m.insert("verbar;", "\u{7C}");
        m.insert("vert;", "\u{7C}");
        m.insert("vfr;", "\u{1D533}");
        m.insert("vltri;", "\u{22B2}");
        m.insert("vnsub;", "\u{2282}\u{20D2}");
        m.insert("vnsup;", "\u{2283}\u{20D2}");
        m.insert("vopf;", "\u{1D567}");
        m.insert("vprop;", "\u{221D}");
        m.insert("vrtri;", "\u{22B3}");
        m.insert("vscr;", "\u{1D4CB}");
        m.insert("vsubnE;", "\u{2ACB}\u{FE00}");
        m.insert("vsubne;", "\u{228A}\u{FE00}");
        m.insert("vsupnE;", "\u{2ACC}\u{FE00}");
        m.insert("vsupne;", "\u{228B}\u{FE00}");
        m.insert("vzigzag;", "\u{299A}");
        m.insert("wcirc;", "\u{175}");
        m.insert("wedbar;", "\u{2A5F}");
        m.insert("wedge;", "\u{2227}");
        m.insert("wedgeq;", "\u{2259}");
        m.insert("weierp;", "\u{2118}");
        m.insert("wfr;", "\u{1D534}");
        m.insert("wopf;", "\u{1D568}");
        m.insert("wp;", "\u{2118}");
        m.insert("wr;", "\u{2240}");
        m.insert("wreath;", "\u{2240}");
        m.insert("wscr;", "\u{1D4CC}");
        m.insert("xcap;", "\u{22C2}");
        m.insert("xcirc;", "\u{25EF}");
        m.insert("xcup;", "\u{22C3}");
        m.insert("xdtri;", "\u{25BD}");
        m.insert("xfr;", "\u{1D535}");
        m.insert("xhArr;", "\u{27FA}");
        m.insert("xharr;", "\u{27F7}");
        m.insert("xi;", "\u{3BE}");
        m.insert("xlArr;", "\u{27F8}");
        m.insert("xlarr;", "\u{27F5}");
        m.insert("xmap;", "\u{27FC}");
        m.insert("xnis;", "\u{22FB}");
        m.insert("xodot;", "\u{2A00}");
        m.insert("xopf;", "\u{1D569}");
        m.insert("xoplus;", "\u{2A01}");
        m.insert("xotime;", "\u{2A02}");
        m.insert("xrArr;", "\u{27F9}");
        m.insert("xrarr;", "\u{27F6}");
        m.insert("xscr;", "\u{1D4CD}");
        m.insert("xsqcup;", "\u{2A06}");
        m.insert("xuplus;", "\u{2A04}");
        m.insert("xutri;", "\u{25B3}");
        m.insert("xvee;", "\u{22C1}");
        m.insert("xwedge;", "\u{22C0}");
        m.insert("yacute", "\u{FD}");
        m.insert("yacute;", "\u{FD}");
        m.insert("yacy;", "\u{44F}");
        m.insert("ycirc;", "\u{177}");
        m.insert("ycy;", "\u{44B}");
        m.insert("yen", "\u{A5}");
        m.insert("yen;", "\u{A5}");
        m.insert("yfr;", "\u{1D536}");
        m.insert("yicy;", "\u{457}");
        m.insert("yopf;", "\u{1D56A}");
        m.insert("yscr;", "\u{1D4CE}");
        m.insert("yucy;", "\u{44E}");
        m.insert("yuml", "\u{FF}");
        m.insert("yuml;", "\u{FF}");
        m.insert("zacute;", "\u{17A}");
        m.insert("zcaron;", "\u{17E}");
        m.insert("zcy;", "\u{437}");
        m.insert("zdot;", "\u{17C}");
        m.insert("zeetrf;", "\u{2128}");
        m.insert("zeta;", "\u{3B6}");
        m.insert("zfr;", "\u{1D537}");
        m.insert("zhcy;", "\u{436}");
        m.insert("zigrarr;", "\u{21DD}");
        m.insert("zopf;", "\u{1D56B}");
        m.insert("zscr;", "\u{1D4CF}");
        m.insert("zwj;", "\u{200D}");
        m.insert("zwnj;", "\u{200C}");
        m
    };
}
