use {
    crate::{
        errors::ProgramError,
        skin::Skin,
    },
    std::io::Write,
    super::{
        TreeGitStatus,
    },
};

pub struct GitStatusDisplay<'a, 's> {
    status: &'a TreeGitStatus,
    skin: &'s Skin,
    show_branch: bool,
    show_wide: bool,
    show_stats: bool,
    pub width: usize,
}

impl<'a, 's> GitStatusDisplay<'a, 's> {
    pub fn from(
        status: &'a TreeGitStatus,
        skin: &'s Skin,
        available_width: usize,
    ) -> Self {
        let mut show_branch = false;
        let mut width = 0;
        if let Some(branch) = &status.current_branch_name {
            let branch_width = branch.chars().count();
            if branch_width < available_width {
                width += branch_width;
                show_branch = true;
            }
        }
        let mut show_stats = false;
        let unstyled_stats = format!("+{}-{}", status.insertions, status.deletions);
        let stats_width = unstyled_stats.len();
        if width + stats_width < available_width {
            width += stats_width;
            show_stats = true;
        }
        let mut show_wide = false;
        if width + 3 < available_width {
            width += 3; // difference between compact and wide format widths
            show_wide = true;
        }
        Self {
            status,
            skin,
            show_branch,
            show_stats,
            show_wide,
            width,
        }
    }

    pub fn write(
        &self,
        f: &mut impl Write,
        selected: bool,
    ) -> Result<(), ProgramError> {
        if self.show_branch {
            cond_bg!(branch_style, self, selected, self.skin.git_branch);
            if let Some(name) = &self.status.current_branch_name {
                if self.show_wide {
                    branch_style.queue(f, format!(" ᚜ {} ", name))?;
                } else {
                    branch_style.queue(f, format!(" {}", name))?;
                }
            }
        }
        if self.show_stats {
            cond_bg!(insertions_style, self, selected, self.skin.git_insertions);
            insertions_style.queue(f, format!("+{}", self.status.insertions))?;
            cond_bg!(deletions_style, self, selected, self.skin.git_deletions);
            deletions_style.queue(f, format!("-{}", self.status.deletions))?;
        }
        Ok(())
    }
}

